#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: release-qc.pl,v 1.2 2007/05/29 13:52:22 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::CDBI::Ontology;
use Gramene::CDBI::Qtl;
use Gramene::QTL::DB;
use Gramene::Marker::DB;
use Pod::Usage;
use Readonly;

Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.2 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version );
GetOptions(
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my @Traits = Gramene::CDBI::Qtl::QtlTrait->retrieve_all;
for my $Trait ( @Traits ) {
    my $to_acc = $Trait->to_accession or next;
    my ($Ont) = Gramene::CDBI::Ontology::Term->search(
        term_accession => $to_acc
    ) or printf "Bad TO accession '%s' for trait '%s' (%s)\n",
        $to_acc, $Trait->trait_name, $Trait->trait_symbol;
}

my $mdb = Gramene::Marker::DB->new;
my $qdb = Gramene::QTL::DB->new;

my $num_in_markers = $mdb->db->selectrow_array(
    q[
        select count(m.marker_id)
        from   marker m, marker_type mt
        where  m.marker_type_id=mt.marker_type_id
        and    mt.marker_type='QTL'
    ]
);
my $num_in_qtl_db = $qdb->db->selectrow_array('select count(*) from qtl');

print join("\n",
    "Num in QTL    : $num_in_qtl_db",
    "Num in Markers: $num_in_markers",
    ''
);    

__END__

# ----------------------------------------------------

=pod

=head1 NAME

release-qc.pl - run some basic QC on data

=head1 VERSION

This documentation refers to version $Revision: 1.2 $

=head1 SYNOPSIS

  release-qc.pl 

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

Describe what the script does, what input it expects, what output it
creates, etc.

=head1 SEE ALSO

perl.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
