#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: grain-genes-export.pl,v 1.1 2007/04/02 15:47:20 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::QTL::DB;
use Gramene::CDBI::Qtl;
use Pod::Usage;
use Readonly;

Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version );
GetOptions(
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $qdb = Gramene::QTL::DB->new;
my $db  = $qdb->db;

my @qtls;
for my $species ( 
    'Triticum aestivum', 'Triticum turgidum', 'Hordeum vulgare',
    'Avena sativa'
) {
    push @qtls, $qdb->search( species => $species );
}

if ( !@qtls ) {
    die "Found no QTLs.\n";
}

print join("\t", qw[
    species
    gramene_qtl_accession
    grain_genes_dbxref
    ontology_associations
    gramene_trait_name
    trait_symbol
]), "\n";

my $num_qtls = 0;
for my $qtl ( @qtls ) {
    my $Qtl  = Gramene::CDBI::Qtl::Qtl->retrieve( $qtl->{'qtl_id'} );
    my $onts = $qdb->get_qtl_ontology_associations( 
        qtl_id => $Qtl->id 
    );

    my $ontologies = @$onts 
        ? join(', ', 
            map  { "$_->{'term_accession'} ($_->{'term_name'})" } 
            grep { $_->{'term_type'} !~ /^species/i } # no species
            @$onts
        )
        : ''
    ;

    my $gg_xref = $db->selectrow_array(
        q[
            select x.xref_value
            from   xref x, xref_type xt
            where  x.table_name='qtl'
            and    x.record_id=?
            and    x.xref_type_id=xt.xref_type_id
            and    xt.xref_type='GrainGenes'
        ],
        {},
        ( $Qtl->id )
    );

    print join("\t",
        $Qtl->species->species,
        $Qtl->qtl_accession_id,
        $gg_xref || '',
        $ontologies,
        $Qtl->qtl_trait->trait_name,
        $Qtl->qtl_trait->trait_symbol,
    ), "\n";

    $num_qtls++;
}

print STDERR "Done, exported $num_qtls QTLs.\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

grain-genes-export.pl - a script

=head1 VERSION

This documentation refers to version $Revision: 1.1 $

=head1 SYNOPSIS

  grain-genes-export.pl 

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

Describe what the script does, what input it expects, what output it
creates, etc.

=head1 SEE ALSO

perl.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
