#!/usr/local/bin/perl

# $Id: get-related-markers.pl,v 1.4 2007/06/05 20:21:27 kclark Exp $

use strict;
use English qw( -no_match_vars );
use Getopt::Long;
use Gramene::CDBI::Markers;
use Gramene::Marker::DB;
use Number::Range;
use Pod::Usage;
use Readonly;

Readonly my $ANALYTICAL_CORRESPONDENCE_TYPE => 'qtl_marker';
Readonly my $COLOCALIZED   => 'colocalized_qtl_marker';
Readonly my $NEIGHBORING   => 'neighboring_qtl_marker';
Readonly my $COMMA         => q{,};
Readonly my $COMMA_SPACE   => q{, };
Readonly my $EMPTY_STR     => q{};
Readonly my $TAB           => "\t";
Readonly my $NL            => "\n";
Readonly my $VERSION 
    => sprintf '%d.%02d', qq$Revision: 1.4 $ =~ /(\d+)\.(\d+)/;

Readonly my $INCLUDED_MARKER_TYPES
    => join( $COMMA, map { qq['$_'] } qw( RFLP SSR ) );

Readonly my $SQL_MARKERS_AT_POS => qq[
    select   mapping.marker_id, s.marker_name
    from     mapping, marker m, marker_type mt, marker_synonym s
    where    mapping.map_id=?
    and      mapping.start between ? and ?
    and      mapping.marker_id=m.marker_id
    and      m.marker_type_id=mt.marker_type_id
    and      mapping.display_synonym_id=s.marker_synonym_id
    and      mt.marker_type in ($INCLUDED_MARKER_TYPES)
    order by marker_name
];

Readonly my $SQL_MARKERS_OVERLAP_POS => qq[
    select   mapping.marker_id, s.marker_name
    from     mapping, marker m, marker_type mt, marker_synonym s
    where    mapping.map_id=?
    and      (
             ( mapping.start < ? and mapping.end > ? )
        or   ( mapping.start between ? and ? )
        or   ( mapping.start < ? and mapping.end > ? )
    )
    and      mapping.display_synonym_id=s.marker_synonym_id
    and      mapping.marker_id=m.marker_id
    and      m.marker_type_id=mt.marker_type_id
    and      mt.marker_type in ($INCLUDED_MARKER_TYPES)
    order by start
];

Readonly my $SQL_MARKER_POS_BEFORE_START => qq[
    select   distinct(mapping.start)
    from     mapping, marker m, marker_type mt
    where    mapping.map_id=?
    and      mapping.start<?
    and      mapping.marker_id=m.marker_id
    and      m.marker_type_id=mt.marker_type_id
    and      mt.marker_type in ($INCLUDED_MARKER_TYPES)
    order by start desc
    limit    1
];

Readonly my $SQL_MARKER_POS_AFTER_STOP => qq[
    select   distinct(mapping.start)
    from     mapping, marker m, marker_type mt
    where    mapping.map_id=?
    and      mapping.start>?
    and      mapping.marker_id=m.marker_id
    and      m.marker_type_id=mt.marker_type_id
    and      mt.marker_type in ($INCLUDED_MARKER_TYPES)
    order by start
    limit    1
];

Readonly my $SQL_MARKERS_IN_POS => qq[
    select   m.marker_id, s.marker_name
    from     mapping, marker m, marker_type mt, marker_synonym s
    where    mapping.map_id=?
    and      mapping.start in (%s)
    and      mapping.display_synonym_id=s.marker_synonym_id
    and      mapping.marker_id=m.marker_id
    and      m.marker_type_id=mt.marker_type_id
    and      mt.marker_type in ($INCLUDED_MARKER_TYPES)
];

# ----------------------------------------------------

my $qtl_accs = $EMPTY_STR;
my ( $help, $man_page, $show_version );
GetOptions(
    'a|accessions:s' => \$qtl_accs,
    'help'           => \$help,
    'man'            => \$man_page,
    'version'        => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my %restrict_export = map { s/^\s+|\s+$//g; $_, 1 } split $COMMA, $qtl_accs;

my $mdb = Gramene::Marker::DB->new;
my $db  = $mdb->db or die $mdb->error;

my ($QtlMapType) = Gramene::CDBI::Markers::MapType->search( map_type => 'QTL' );
my @QtlMapSets   = Gramene::CDBI::Markers::MapSet->search(
    map_type_id  => $QtlMapType->id
);

print join( $TAB, 
    qw[
        from_marker_id 
        to_marker_id
        analysis_name
        analytical_correspondence_type
    ]
), $NL;

my ( $num_map_sets, $num_qtls ) = ( 0, 0 );
for my $MapSet ( @QtlMapSets ) {
    $num_map_sets++;
    for my $Map ( $MapSet->maps ) {
        my $map_id = $Map->id;
        my $qtls   = $db->selectall_arrayref(
            q[
                select mapping.marker_id, mapping.cmap_feature_accession, 
                       s.marker_name, mapping.start, mapping.end
                from   mapping, marker m, marker_type mt, marker_synonym s
                where  mapping.map_id=?
                and    mapping.display_synonym_id=s.marker_synonym_id
                and    mapping.marker_id=m.marker_id
                and    m.marker_type_id=mt.marker_type_id
                and    mt.marker_type=?
            ],
            { Columns => {} },
            ( $map_id, 'QTL' )
        );

        for my $qtl ( @$qtls ) {
            if ( %restrict_export ) {
                next if !defined 
                    $restrict_export{ $qtl->{'cmap_feature_accession'} };
            }

            $num_qtls++;
            my $qtl_start    = $qtl->{'start'};
            my $qtl_end      = $qtl->{'end'};
            my $start_bottom = sprintf( '%.0f', $qtl_start - .5 );
            my $start_top    = sprintf( '%.0f', $qtl_start + .5 );
            my $stop_bottom  = sprintf( '%.0f', $qtl_end   - .5);
            my $stop_top     = sprintf( '%.0f', $qtl_end   + .5 );
            my $colocalized  = $db->selectall_arrayref( 
                $SQL_MARKERS_OVERLAP_POS, { Columns => {} }, 
                ( $map_id, $qtl_start, $qtl_start, $qtl_start, $qtl_end, 
                  $qtl_start, $qtl_end )
            );

            for my $marker ( @$colocalized ) {
                print join( $TAB,
                    $qtl->{'marker_id'},
                    $marker->{'marker_id'},
                    $COLOCALIZED,
                    $ANALYTICAL_CORRESPONDENCE_TYPE,
                ), $NL;
            }

            my $nearest_start_pos = $db->selectcol_arrayref( 
                $SQL_MARKER_POS_BEFORE_START, {}, ( $map_id, $qtl_start )
            );

            my $nearest_end_pos = $db->selectcol_arrayref( 
                $SQL_MARKER_POS_AFTER_STOP, {}, ( $map_id, $qtl_end )
            );

            my @nearest_start_markers = 
                @$nearest_start_pos 
                ? @{ $db->selectall_arrayref( 
                        sprintf( 
                            $SQL_MARKERS_IN_POS, 
                            join($COMMA, @$nearest_start_pos)
                        ), 
                        { Columns => {} },
                        ( $map_id )
                    ) }
                : ()
            ;

            my @nearest_end_markers = 
                @$nearest_end_pos
                ? @{ $db->selectall_arrayref( 
                        sprintf( 
                            $SQL_MARKERS_IN_POS, 
                            join($COMMA, @$nearest_end_pos)
                        ), 
                        { Columns => {} },
                        ( $map_id )
                    ) }
                : ()
            ;

            for my $marker ( @nearest_start_markers, @nearest_end_markers ) {
                print join( $TAB,
                    $qtl->{'marker_id'},
                    $marker->{'marker_id'},
                    $NEIGHBORING,
                    $ANALYTICAL_CORRESPONDENCE_TYPE,
                ), $NL;
            }
        }
    }
}

print STDERR "Done, exported $num_qtls QTLs on $num_map_sets QTL map sets.\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

get-related-markers.pl - export QTL/marker relations

=head1 VERSION

This documentation refers to version $Revision: 1.4 $

=head1 SYNOPSIS

  get-related-markers.pl [-a AQG018] > related.tab

Options:

  -a|--accession=...  Comma-separated list of QTL accessions
  --help              Show brief help and exit
  --man               Show full documentation
  --version           Show version and exit

=head1 DESCRIPTION

This script exports the various associated marker IDs (left/right
flanking, linked, near to start/end) for each QTL in the markers db.
The output is a tab-delimited.

To restrict QTLs selected, define them as a comma-separated list to 
the "-a" flag.

=head1 SEE ALSO

Gramene::CDBI::Markers.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
