#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: show-map-set-summaries.pl,v 1.3 2006/10/30 16:32:33 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::CDBI::Markers;
use Pod::Usage;
use Readonly;
use Text::TabularDisplay;

Readonly my $COMMA     => q{,};
Readonly my $DASH      => q{-};
Readonly my $EMPTY_STR => q{};
Readonly my $NL        => qq{\n};
Readonly my $VERSION   => sprintf '%d.%02d', 
                        qq$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;

my $ms_accs   = '';
my $map_types = '';
my $species   = '';
my ( $help, $man_page, $show_version );
GetOptions(
    'm|map-set-acc:s' => \$ms_accs,
    't|map-type:s'    => \$map_types,
    's|species:s'     => \$species,
    'help'            => \$help,
    'man'             => \$man_page,
    'version'         => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my %map_set_restrict 
    = map { s/^\s+|\s+$//g; $_, 1 } split $COMMA, $ms_accs;
my %map_type_restrict 
    = map { s/^\s+|\s+$//g; $_, 1 } split $COMMA, $map_types;
my %species_restrict  
    = map { s/^[\s']+|[\s']+$//g; $_, 1 } split $COMMA, $species;

my @map_types = Gramene::CDBI::Markers::MapType->retrieve_all;
my @species   = Gramene::CDBI::Markers::Species->retrieve_all;

MAP_TYPE:
for my $MapType ( @map_types ) {
    if ( %map_type_restrict ) {
        next MAP_TYPE if !defined $map_type_restrict{ $MapType->map_type };
    }

    SPECIES:
    for my $Species ( @species ) {
        if ( %species_restrict ) {
            next SPECIES if !defined $species_restrict{ $Species->species };
        }

        my @map_sets = Gramene::CDBI::Markers::MapSet->search(
            map_type_id => $MapType->id,
            species_id  => $Species->id,
        );

        MAP_SET:
        for my $MapSet ( @map_sets ) {
            my $ms_acc = $MapSet->cmap_map_set_accession || $EMPTY_STR;

            if ( %map_set_restrict ) {
                next MAP_SET if !defined $map_set_restrict{ $ms_acc };
            }

            my $title = sprintf '%s-%s-%s%s',
                $MapType->map_type,
                $Species->species,
                $MapSet->map_set_name,
                $ms_acc ? " ($ms_acc)" : $EMPTY_STR
            ;

            my $tab = Text::TabularDisplay->new( 'Map', 'Num. Mappings' );

            my @maps 
                = sort { $a->display_order <=> $b->display_order } 
                $MapSet->maps;

            my $total_mappings;
            for my $Map ( @maps ) {
                my @mappings = $Map->mappings;
                my $num_mappings = scalar @mappings;
                $tab->add( $Map->map_name, $num_mappings );
                $total_mappings += $num_mappings;
            }

            $tab->add( '---', '---' );
            $tab->add( 'Total', $total_mappings );

            print join $NL, $title, $tab->render, $EMPTY_STR, $EMPTY_STR;
        }
    }
}

__END__

# ----------------------------------------------------

=pod

=head1 NAME

show-map-set-summaries.pl - show maps, number of mappings, etc.

=head1 VERSION

This documentation refers to show-map-set-summaries.pl version $Revision: 1.3 $

=head1 SYNOPSIS

  show-map-set-summaries.pl [options]

Options:

  -m|--map-set-acc=west,kl2000 Restrict just to map set accession
  -t|--map-type=QTL,Genetic    Restrict just to map types listed
  -s|--species='Oryza sativa'  Restrict just to species listed
  --help                       Show brief help and exit
  --man                        Show full documentation
  --version                    Show version and exit

=head1 DESCRIPTION

Prints a summary of map sets, number of maps, number of mappings, etc.

=head1 SEE ALSO

perl.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
