#!/usr/local/bin/perl

# $Id: set-map-display-order.pl,v 1.3 2007/02/26 21:00:52 kclark Exp $

use strict;
use IO::Prompt;
use Data::Dumper;
use Gramene::CDBI::Markers;
use Getopt::Long;
use Pod::Usage;
use File::Basename;
use Readonly;

Readonly my $DASH => q{-};
Readonly my $NL   => qq{\n};

use vars qw[ $VERSION ];
$VERSION = sprintf "%d.%02d", q$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;

my ( $help, $show_version, $ms_accs );
GetOptions(
    'h|help'         => \$help,
    'v|version'      => \$show_version,
    'ms-accs:s'      => \$ms_accs,
);
pod2usage(2) if $help;

if ( $show_version ) {
    my $prog = basename( $0 );
    print "$prog v$VERSION\n";
    exit(0);
}

my @map_sets;
if ( $ms_accs ) {
    for my $ms_acc ( split( /,/, $ms_accs ) ) {
        push @map_sets, Gramene::CDBI::Markers::MapSet->search({
            cmap_map_set_accession => $ms_acc
        });
    }
}
else {
    @map_sets = Gramene::CDBI::Markers::MapSet->retrieve_all;
}

my $ok = prompt -yn, join($NL, 
    "OK to reset display order for maps in following map sets?",
    join( $NL,
        map { join($DASH, $_->species->species, $_->map_set_name) } @map_sets
    ),
    '[yn] '
);

if ( !$ok ) {
    print "Not OK, exiting.\n";
    exit 0;
}

my %h  = (
    a  => 1,
    b  => 2,
    c  => 3,
    d  => 4,
    e  => 5,
    f  => 6,
    g  => 7,
    h  => 8,
    i  => 9,
    j  => 10,
);

for my $map_set ( @map_sets ) {
    my @maps = $map_set->maps;

    for my $map ( @maps ) {
        my $display_order = my $map_name = $map->map_name;

        if ( $display_order =~ /(\d+)/ ) {
            $display_order = $1;
        }
        elsif ( $display_order =~ /^[A-Za-z]$/ ) {
            $display_order = $h{ lc $display_order };
        }
        elsif ( $display_order eq 'UNKNOWN' ) {
            $display_order = scalar @maps;
        }

        unless ( $display_order =~ /^\d+$/ ) {
            print "+++Skipping $map_name ($display_order)\n";
            next;
        }

        print "Setting '$map_name' to '$display_order'\n";
        $map->display_order( $display_order );
        $map->update;
    }
}

print "Done.\n";

# ----------------------------------------------------

=pod

=head1 NAME

set-map-display-order.pl - Fixes map display order for CMap

=head1 SYNOPSIS

  set-map-display-order.pl [options]

Options:

  --ms-accs=msacc1[,msacc2...]  A list of map set accession IDs
  -h|--help                     Show brief help and exit
  -v|--version                  Show version and exit

=head1 DESCRIPTION

This script sets the "map.display_order" to a numerical value 
based on some numerical value in the map's name.  If you have many maps 
with the same "display_order" value and the map's have numerical names, 
they will be sorted by the ASCII value of the names, so you'll have "1, 10, 
2, 3...," and that looks silly.  Even if the map names are like "ctg1, ctg10,
ctg2..." you probably want them sorted by the numerical part of the name.

With no arguments, every map for every map set will be affected.  Limit to 
just a subset of map sets by using the "ms-accs" argument.

=head1 SEE ALSO

Gramene::CDBI::Markers.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut

