#!/usr/local/bin/perl

=head1 NAME

retrieve-datasets.pl - create gramene track sequence datasets from markers database for mapping

=head1 SYNOPSIS

  retrieve-datasets.pl [options]

Options:

  -h|--help         Show brief help and exit.
  -v|--verbose      Talk about what's happening.
  -c|--config_file  Path to DB config file, def $ENV{GrameneConfPath}.
  --dir             Directory to write output files into. Def $PWD
  --analysis        name of the analysis to retrieve, has to be the logic name for this gramene dataset, for example "Rice_est"
  --source        datasets grouped by source, the valid sources are "entrez" "ncbitrace" "unigene" "tigr_gi" "pgdb_tug" "entrez-omap"

=head1 DESCRIPTION

Create fasta sequence files for mapping, each file corresponds to a distinct
track on the gramene genome browser 

The program retrieves the requested sequences using markers database query 
formulated for each gramene track, output the sequences into a fasta file
using the track's logic name as the file name. It also produe a log file
reporting how many sequences were retieved for each dataset.

Format of config_file;

  <markers_admin>
      db_dsn   dbi:mysql:{dbname}:{myhost}:{myport}
      db_user  marker_rw_user
      db_pass  secret
  </markers_admin>


Everything printed to STDERR by the program will be logged to a
file. Lines start with '[INFO]', '[WARN]' or '[*DIE]' for ease of
grepping. The output dir can be specified as program
arguments. 


=head1 SEE ALSO

Gramene::Marker::DB, Text::RecordParser.

=head1 AUTHOR

Sharon Wei E<lt>weix@cshl.eduE<gt>.

=cut

# ----------------------------------------------------

use strict;
use Getopt::Long;

use Gramene::Ontology::OntologyDB;
use Pod::Usage;
use Text::RecordParser;

use Bio::SeqIO;
use Data::Dumper qw(Dumper);
use Date::Calc;
use Carp;

use lib "/home/weix/gramene/lib/perl";
use Gramene::Marker::DB;
use IO::Prompt;
use List::MoreUtils qw(firstval uniq);
use Readonly;

Readonly my @source => qw( entrez ncbitrace unigene tigr_gi pgdb_tug entrez-omap);

Readonly my %LOGICNAME_TO_QUERY => (
      Rice_est   => {
		     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Oryza %' ",
		     source => $source[0],
		    },

     Barley_est  => {
		     query => "select m.marker_id from marker m, species s, marker_type t  where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id  and t.marker_type = 'EST' and s.species like 'Hordeum %' ",
		     source => $source[0],
		    },


     Maize_est  => {
		     query => "select m.marker_id from marker m, species s,  marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Zea %' ",
		    source => $source[0],
		    },

    Millet_est  => {
		     query => "select m.marker_id from marker m, species s,  marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Pennisetum %' ",
		    source => $source[0],
		    },				    

    Sorghum_est => {
		    query => "select m.marker_id from marker m, species s,  marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Sorghum %' ",
		    source => $source[0],
		   },

    Sugarcane_est => {
		      query => "select m.marker_id from marker m, species s,  marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Saccharum%' ",
		      source => $source[0],
		     },

    Wheat_est => {
		  query => "select m.marker_id from marker m, species s,  marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species like 'Triticum %' ",
		  source => $source[0],
		 },

    'Other-poaceae_est' => {
		  query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'EST' and s.species not like 'Triticum %' and s.species not like 'Saccharum%' and s.species not like 'Sorghum %' and s.species not like 'Zea %' and s.species not like 'Hordeum %' and s.species not like 'Oryza %' and s.species not like 'Pennisetum %' ",	 
		    source => $source[0],
			   },

    Rice_mRNA  => {
		  query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'mRNA' and s.species like 'Oryza %' ",
		   source => $source[0],
		 },


    'Maize_mRNA' =>{
		  query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'mRNA' and s.species like 'Zea %' ",
		    source => $source[0],
		  },

    'Other-poaceae_mRNA' =>{
		  query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'mRNA' and s.species not like 'Oryza %' and s.species not like 'Zea %' ",
			    source => $source[0],
		  },

    'Rice_cDNA_KOME' =>{
		  query => "select m.marker_id from marker m, species s, marker_type t, library l where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and m.library_id=l.library_id and t.marker_type = 'mRNA' and s.species like 'Oryza %' and library_name like 'KOME%' ",
			source => $source[0],
		  },


    
    Maize_BACend  => { 
	             query => "select m.marker_id from marker m, species s , marker_type t where m.source_species_id = s.species_id  and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and s.species like 'Zea %' ",
		      source => $source[0],
		    },


    RiceAlta_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza alta' ",
			     source => $source[5],
			     #"select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OA_BBa' ",
			    },

      RiceAustraliensis_BACend_OMAP  => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza australiensis' ",
					 source => $source[5],
#		      
#query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OA_ABa' ",
					},

      RiceBrachyantha_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza brachyantha' ",	
	source => $source[5],				 

#	      #query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OB__Ba ' ",
				     },

      RiceCoarctata_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza coarctata' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OC__Ba' ",
			      },

      RiceGlaberrima_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza glaberrima' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OG_BBa' ",
			      },
      
      RiceGranulata_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza granulata' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OG_ABa' ",
			      },

      RiceJaponica_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species like 'oryza sativa%' ",	
source => $source[5],
#	      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OSJNBa' ",
			      },

      RiceMinuta_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza minuta' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OM__Ba' ",
			      },

      RiceNivara_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza nivara' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OR_BBa' ",
			      },

      RicePunctata_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza punctata' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OP__Ba' ",
		      #marker_type => '',
			      },

      RiceRufipogon_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza rufipogon' ",
source => $source[5],
#		      query => "select m.marker_id from marker m, library l where m.library_id = l.library_id  and l.library_name = 'OR_CBa' ",
		      #marker_type => '',
			      },

     RiceOfficinalis_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza officinalis' ",
				     source => $source[5],			
	    },

     RiceRidleyi_BACend_OMAP => {
			     query => "select m.marker_id from marker m, species s, marker_type t where m.source_species_id = s.species_id and m.marker_type_id = t.marker_type_id and t.marker_type = 'GSS' and species = 'oryza ridleyi' ",
				 source => $source[5],
				},
				    
    Rice_FstTransposon => { #4183 sequences
                      query => "select m.marker_id
                                from marker m, marker_details_gss md, library l
                                where m.marker_id=md.marker_id
                                and m.library_id=l.library_id
                                and l.library_name IN (
                                'UCD RdSpm Rice Insertions',
                                'UCD RGT Rice Insertions',
                                'UCD RDs Rice Insertions',
                                'UCD RGdSpm Rice Insertions' )",
			  },

    Rice_T_DNA_Insert => { #14533 sequences
		       query => "select m.marker_id
                                 from marker m, marker_details_gss md, library l
                                 where m.marker_id=md.marker_id
                                 and m.library_id=l.library_id
                                 and l.library_name IN (
                                 'AS_TRIM_TDNA_B1',
                                 'Flanking Sequence Tag of Oryza sativa T-DNA insertion lines' )",
			  },


    Rice_jap_cDNA_KOME => { #32127 sequences
		     query => "select m.marker_id
                               from marker m, marker_details_mrna md, library l
                               where m.marker_id=md.marker_id
                               and m.library_id=l.library_id
                               and l.library_name like 'KOME database of full-length cDNA%' ",
			 },
				    
    Rice_tos17_insert  => { #32127 sequences,
		      query => "select m.marker_id
                                from marker m, marker_details_gss md, library l
                                where m.marker_id=md.marker_id
                                and m.library_id=l.library_id
                                and l.library_name IN(
                                'PCR product directly amplified from rice genomic DNA' )",
			  },



      Barley_GI => {
		      query => " select m.marker_id from marker m, analysis a, species s, marker_details_est_cluster d where m.analysis_id = a.analysis_id  and m.source_species_id = s.species_id and m.marker_id = d.marker_id and a.analysis_name = 'tigr_gene_index' and s.species = 'Hordeum vulgare' and d.version = 'HVGI release 9' ",
		      #marker_type => '',
			      },

      Maize_GI => {
		      query => " select m.marker_id from marker m, analysis a, species s, marker_details_est_cluster d where m.analysis_id = a.analysis_id  and m.source_species_id = s.species_id and m.marker_id = d.marker_id and a.analysis_name = 'tigr_gene_index' and s.species = 'Zea mays' and d.version = 'ZMGI release 15' ",
		      #marker_type => '',
			      },
      Rice_GI => {
		      query => " select m.marker_id from marker m, analysis a, species s, marker_details_est_cluster d where m.analysis_id = a.analysis_id  and m.source_species_id = s.species_id and m.marker_id = d.marker_id and a.analysis_name = 'tigr_gene_index' and s.species = 'Oryza sativa' and d.version = 'OGI release 16' ",
		      #marker_type => '',
			      },

      Sorghum_GI => {
		      query => " select m.marker_id from marker m, analysis a, species s, marker_details_est_cluster d where m.analysis_id = a.analysis_id  and m.source_species_id = s.species_id and m.marker_id = d.marker_id and a.analysis_name = 'tigr_gene_index' and s.species = 'Sorghum bicolor' and d.version = 'SBGI release 1-1' ",
		      #marker_type => '',
			      },
      Wheat_GI => {
		      query => " select m.marker_id from marker m, analysis a, species s, marker_details_est_cluster d where m.analysis_id = a.analysis_id  and m.source_species_id = s.species_id and m.marker_id = d.marker_id and a.analysis_name = 'tigr_gene_index' and s.species = 'Triticum aestivum' and d.version = 'TAGI release 10' ",
		      #marker_type => '',
			      },
				    

      Barley_ESTCluster_PlantGDB => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Barley_ESTCluster_PlantGDB' ",
			      },
      Maize_ESTCluster_PlantGDB => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Maize_ESTCluster_PlantGDB' ",
			      },
      Rice_ESTCluster_PlantGDB => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Rice_ESTCluster_PlantGDB' ",
			      },
      Sorghum_ESTCluster_PlantGDB => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Sorghum_ESTCluster_PlantGDB' ",
			      },
      Wheat_ESTCluster_PlantGDB => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Wheat_ESTCluster_PlantGDB' ",
			      },

      Sorghum_ESTCluster3P_LGBPratt => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Sorghum_ESTCluster3P_LGBPratt' "
			      },
      
      Rice_ind_cluster => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Rice_ind_cluster' ",
			      },
      Rice_ind_est => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Rice_ind_est' ",
			      },


      Maize_hi_cot_TIGR => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Maize_hi_cot_TIGR' ",
			      },

     Maize_meth_filt_hi_cot_cluster => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Maize_meth_filt_hi_cot_cluster' ",
			      },

      Maize_MAGI_ISU => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Maize_MAGI_ISU' ",
			      },


      Ryegrass_Sequence => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Ryegrass_Sequence' ",
			      },
      Ryegrass_Assembly => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Ryegrass_Assembly' ",
			      },

      'Sorghum_gss-read_Klein' => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Sorghum_gss-read_Klein' ",
			      },


      Sorghum_orion => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Sorghum_orion' ",
			      },

     Sorghum_CDNA => {
		      query => "select m.marker_id from marker m, analysis a where m.analysis_id = a.analysis_id and a.analysis_name = 'Sorghum_CDNA' ",
			      },


    #Maize_MarkerRFLP => {
	#		 query => "",
		#	},
);


=stub

     Maize_hi_cot_Bennetzen => { # 446926 sequences, 186047 in markers21
		     query => "select m.marker_id
                               from marker m, marker_details_gss md, library l
                               where m.marker_id=md.marker_id
                               and m.library_id=l.library_id
                               and l.library_name IN (
                               'ZM_0.6_1.0_KB',
                               'Maize Cot library',
                               'Maize Cot100 library',
                               'Maize Cot200 library',
                               'Maize Cot466 library' )",
			       },

    Maize_meth_filt_CSHL_Mccombie => { #66390 sequences, 0 in markers21
		      query => "select m.marker_id
			       from marker m, marker_details_gss md, library l
                               where m.marker_id=md.marker_id
                               and m.library_id=l.library_id
                               and l.library_name IN (
                               'WGS-ZmaysF (DH5a methyl filtered)',
                               'WGS-ZmaysF (JM107 adapted methyl filtered)',
                               'JM107 adapted methyl filtered library' )",
			       },

    Maize_meth_filt_TIGR => {#450197 sequences, 199096 in markers21
		      query => "select m.marker_id
                              from marker m, marker_details_gss md, library l
                              where m.marker_id=md.marker_id
                              and m.library_id=l.library_id
                              and l.library_name IN (
                              'ZM_0.7_1.5_KB',
                              'ZM2_0.7_1.5_KB' )",
			    },

    Maize_Mu_Insert  => { #191715 sequences, 60851 in markers21
                      query => "select m.marker_id
                               from marker m, marker_details_gss md, library l
                               where m.marker_id=md.marker_id
                               and m.library_id=l.library_id
                               and l.library_name LIKE '% - RescueMu Grid %' ",
			},

=cut



$SIG{'INT'} = sub { exit(0) };

my ( $help, $v, $conffile, $dir, @analyses, $src);

GetOptions(
           'help'           => \$help,
           'config_file:s'  => \$conffile,
           'verbose'        => \$v,
           'dir:s'          => \$dir,
	   'analysis:s'     => \@analyses,
	   'source:s'       => \$src,
);
pod2usage(-verbose => 2) if $help;

#----
# Validate params
if( defined $conffile ){ 

  unless( -e $conffile ){
    warn( "\n[*DIE] File $conffile does not exist\n\n" );
    pod2usage;
  } unless( -r $conffile ){
    warn( "\n[*DIE] Cannot read $conffile\n\n" );
    pod2usage;
  } unless( -f $conffile ){
    warn( "\n[*DIE] File $conffile is not plain-text\n\n" );
    pod2usage;
  } unless( -s $conffile ){
    warn( "\n[*DIE] File $conffile is empty\n\n" );
    pod2usage;
  }

  $ENV{GrameneConfPath} = $conffile ;
}

#----
# output dir
$dir ||= $ENV{PWD};

#---
# logfile
my( $file ) = ( $0 =~ m/([^\/]+)$/ );
$file =~ s/\.\w+$//;
my $date = sprintf('%4.4i%2.2i%2.2i',Date::Calc::Today);
my $logfile = join( ".", $date, $$, $file, 'log' );
$logfile    = "$dir/$logfile";

=stub

my $run = prompt -yn, "Creating gramene track seq datasets using the following parameters? 
config file = $conffile (default /usr/local/gramene/conf/gramene.conf)
output dir  = $dir
log file    = $logfile
[y/n]
";

exit unless ( $run );

=cut

# Log to file
open( LOG, ">$logfile" ) or die( $! );
open( STDERR, ">&LOG" ) or die( $! );


my $MDB = Gramene::Marker::DB->new ||
    die "\n[*DIE] " . Gramene::Marker::DB->error . "\n\n";


my ( $num_of_created, $num_of_errors ) = ( 0, 0 );

if( $src ){
  $src = lc ($src);
  push @analyses, grep { $LOGICNAME_TO_QUERY{$_}->{source} eq $src } 
    keys %LOGICNAME_TO_QUERY;  

  my $track_cnt = scalar @analyses;
  print LOG "There are $track_cnt tracks needs update\n";
  print LOG join "\n", @analyses;
  print LOG "\n";
  
}else{
  @analyses = sort keys %LOGICNAME_TO_QUERY unless (scalar @analyses > 0);
}

my @uniq_analyses = uniq @analyses;
for my $track_logic_name( @uniq_analyses ){

  print "Logic_name = $track_logic_name\n" if $v;

  my $track_query = $LOGICNAME_TO_QUERY{$track_logic_name}->{query};

  unless ( $track_query ){
   print STDERR "[*ERR] $track_logic_name: No query defined for it\n";
   next ;
  }

  my @marker_ids = $MDB->search_marker_ids_by_query( $track_query ); 

  my $num_of_markers = scalar @marker_ids;
  print "found $num_of_markers markers for $track_logic_name\n" if $v;

  my $outfile = "$dir/$track_logic_name.fa";

  ( $num_of_created, $num_of_errors ) = create_fasta( $outfile, \@marker_ids);

  print LOG "\n", (join "\t", ($track_logic_name, $num_of_markers, $num_of_created, $num_of_errors));

}

close LOG;

#======================================================================
#
sub create_fasta{

 # my $MDB             = shift;
  my $outfile_name    = shift;
  my $marker_ids_ref  = shift;

  my ($track_logic_name) = ($outfile_name =~ m=([^/]+) \. fa* =xms);

  my ( $num_of_created, $num_of_errors ) = (0, 0);
  my $seqio = Bio::SeqIO->new(-file=>">$outfile_name", -format=>'fasta');


  for my $marker_id ( @{$marker_ids_ref} ){

    
    unless( $marker_id ){
      print STDERR "[*ERR] $track_logic_name: No marker_id $marker_id\n";
      $num_of_errors++;
      next;
    }

    print "$marker_id\n" if $v;

    my @marker_synonyms = $MDB->get_marker_synonyms( marker_id => $marker_id );

#    for (keys %{$marker_synonyms[0]}){
#      print "$_ => $marker_synonyms[0]->{$_}\n";
#    }

    my ($marker_display_syn) = (firstval { $_->{synonym_type} eq 'GENBANK_VERSION'} @marker_synonyms)
                         || (firstval {$_->{synonym_type} eq 'GENBANK_ACCESSION'} @marker_synonyms)
	                 || (firstval {$_->{synonym_type} eq 'GENBANK_GI'} @marker_synonyms);
  
    my $marker_display_name = $marker_display_syn ? 
      $marker_display_syn->{marker_name} :
	$MDB->get_marker_display_name( 
				      marker_id => $marker_id,
				     );

    unless( $marker_display_name ){
      print STDERR "[*ERR] $track_logic_name: No marker_display_name found for marker_id $marker_id\n";
      $num_of_errors++;
      next;
    }

    print "$marker_id => $marker_display_name\n" if $v;

    my $marker_details = $MDB->get_marker_details(
						  marker_id => $marker_id,
						 );

    my $marker_seq_obj = Gramene::CDBI::Markers::MarkerSequence->retrieve($marker_id);
    my $seq;
    eval{ $seq = $marker_seq_obj->seq()};

    if($@){
      print STDERR "[*ERR] $track_logic_name: No sequence found for marker_id $marker_id|$marker_display_name\n";
      $num_of_errors++;
      next;
    }

#    unless( $seq ){
#      print STDERR "[*ERR] $track_logic_name: No sequence found for marker_id $marker_id|$marker_display_name\n";
#      $num_of_errors++;
#      next;
#    }

    my $seq_obj = Bio::Seq->new(
			   -display_id => "mi|${marker_id}|mn|$marker_display_name|",
			   -seq       => $seq,
			  );

    $seqio->write_seq($seq_obj) && $num_of_created++;

  }
  
  return( $num_of_created, $num_of_errors );
}

__END__

#======================================================================


