#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: preprocess-cmap-export.pl,v 1.1 2006/11/10 03:20:53 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::DB;
use Gramene::CDBI::Markers;
use Gramene::Marker::DB;
use Pod::Usage;
use Readonly;

Readonly my $EMPTY_STRING => q{};
Readonly my $VERSION 
    => sprintf '%d.%02d', qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my $map_set_accs = $EMPTY_STRING;
my ( $help, $man_page, $show_version );
GetOptions(
    'm|map-set-accs=s' => \$map_set_accs,
    'help'             => \$help,
    'man'              => \$man_page,
    'version'          => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( !$map_set_accs ) {
    pod2usage('No map set accession(s) to process');
}

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $db        = Gramene::CDBI::Markers->db_Main;
my $db_select = Gramene::DB->new('markers');
$db_select->{'mysql_use_result'} = 1;

my %map_type_id = map { $_->map_type, $_->id }
    Gramene::CDBI::Markers::MapType->retrieve_all;
my $exclude_map_type_ids = join( ', ', 
    $map_type_id{'Physical'}, $map_type_id{'Sequence'} 
);
my $exlude_analysis_id = '78';

my ( $num_inspected, $num_added ) = ( 0, 0 );
for my $map_set_acc ( split /,/, $map_set_accs ) {
    my ($MapSet) = Gramene::CDBI::Markers::MapSet->search(
        cmap_map_set_accession => $map_set_acc
    ) or die "Bad map set accession '$map_set_acc'\n";

    my $sth = $db_select->prepare(
        qq[
            select m.mapping_id, m.marker_id
            from   mapping m
            where  m.map_id=?
            and    m.cmap_feature_accession is null
            and    m.analysis_id<>$exlude_analysis_id
        ]
    );

    for my $Map ( $MapSet->maps ) {
        next unless $Map->cmap_map_accession;

        printf STDERR "Processing map %s (%s)\n", $Map->map_name, $Map->id;

        $sth->execute( $Map->id );
        while ( my $mapping = $sth->fetchrow_hashref ) {
            $num_inspected++;
        
            my $num_other_mappings = $db->selectrow_array(
                qq[
                    select count(m.mapping_id)
                    from   mapping m, map, map_set ms
                    where  m.marker_id=?
                    and    m.cmap_feature_accession is not null
                    and    m.map_id=map.map_id
                    and    map.map_set_id<>?
                    and    map.map_set_id=ms.map_set_id
                    and    ms.map_type_id not in ($exclude_map_type_ids)
                ],
                {},
                ( $mapping->{'marker_id'}, $MapSet->id )
            );

            my $num_correspondences;
            if ( !$num_other_mappings ) {
                for my $dir1 ( qw[ from to ] ) {
                    my $dir2 = $dir1 eq 'from' ? 'to' : 'from';
                    $num_correspondences = $db->selectrow_array(
                        qq[
                            select count(ac.analytical_correspondence_id)
                            from   analytical_correspondence ac,
                                   mapping m,
                                   map,
                                   map_set ms
                            where  ac.${dir1}_marker_id=?
                            and    ac.${dir2}_marker_id=m.marker_id
                            and    m.cmap_feature_accession is not null
                            and    m.map_id=map.map_id
                            and    map.map_set_id<>?
                            and    map.map_set_id=ms.map_set_id
                            and    ms.map_type_id not in ($exclude_map_type_ids)
                        ],
                        {},
                        ( $mapping->{'marker_id'}, $MapSet->id )
                    );

                    last if $num_correspondences > 0;
                }
            }

            if ( $num_other_mappings > 0 || $num_correspondences > 0 ) {
#                my $Mapping = Gramene::CDBI::Markers::Mapping->retrieve(
#                    $mapping->{'mapping_id'}
#                );
#
#                $Mapping->cmap_feature_accession( $Mapping->id );
#                $Mapping->update;
                $num_added++;

#                print "$mapping->{'mapping_id'} ($mapping->{'marker_id'})\n";
                print join("\t",
                    $mapping->{'mapping_id'},
                    $mapping->{'marker_id'},
                ), "\n";
            }
        }
    }

    $sth->finish;
}

print STDERR "Done, inspected $num_inspected, added $num_added.\n";
exit 0;

#$db_select->{'mysql_use_result'} = 0;
#$db_select->disconnect;
#$db->disconnect;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

preprocess-cmap-export.pl - check map sets for mappings to export to CMap

=head1 VERSION

This documentation refers to version $Revision: 1.1 $

=head1 SYNOPSIS

  preprocess-cmap-export.pl -m|--map-set-acc=gt0506[,jrgp-rflp-2000]

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

Checks the specified map sets to see if mapping not currently tagged
for export to CMap (don't have a "cmap_feature_accession") should be 
exported because they have another mapping that will be exported or
a correpondence to a mapping that will be exported.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
