#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: merge-markers.pl,v 1.5 2006/10/30 16:32:15 kclark Exp $

package main;

use strict;
use warnings;
use Carp;
use English qw( -no_match_vars );
use IO::Prompt;
use IO::Scalar;
use Getopt::Long;
use Gramene::Marker::DB;
use File::Basename;
use Pod::Usage;
use Readonly;

Readonly my $SPACE   => q{ };
Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.5 $ =~ /(\d+)\.(\d+)/;

my $deprecated_id = 0;
my $target_id     = 0;
my $target_name   = '';
my ( $help, $show_version, $man_page ); 
GetOptions(
    'd|deprecated=i' => \$deprecated_id,
    't|target=i'     => \$target_id,
    'n|name:s'       => \$target_name,
    'help'           => \$help,
    'man'            => \$man_page,
    'version'        => \$show_version,
) or pod2usage;

my $file = shift @ARGV;

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $input;
if ( $file ) {
    open $input, '<', $file or die "Can't read '$file': $!\n";
}
else {
    if ( !$deprecated_id ) {
        pod2usage('No deprecated marker ID');
    }

    if ( !$target_id ) {
        pod2usage('No target marker ID');
    }

    if ( $deprecated_id eq $target_id ) {
        croak "Deprecated and target marker IDs are the same ($target_id)!";
    }

    my $data = join("\t", $deprecated_id, $target_id, $target_name) . "\n";
    $input = IO::Scalar->new( \$data );
}

my $question = sprintf "OK to merge %s [yn] \n",
               $file ? " markers in '$file'" : " $deprecated_id => $target_id";
my $answer   = prompt -yn, $question;

if ( $answer eq 'n' ) {
    print "Not OK, exiting.\n";
    exit;
}

my $mdb = Gramene::Marker::DB->new( admin => 1 );

my $i = 0;
while ( my $line = <$input> ) {
    chomp $line;
    my ( $dep_id, $tar_id, $display_name ) = split /\s+/, $line;

    next unless $dep_id && $tar_id;

    print "Merging $dep_id => $tar_id\n";

    $mdb->merge_markers(
        deprecated_marker_id => $dep_id,
        target_marker_id     => $tar_id,
    );

    if ( $display_name ) {
        my $Marker = Gramene::CDBI::Markers::Marker->retrieve( $tar_id );

        SYNONYM:
        for my $Syn ( $Marker->marker_synonyms ) {
            if ( $Syn->marker_name eq $display_name ) {
                print "Setting display name to '$display_name'\n";
                $Marker->display_synonym_id( $Syn->marker_synonym_id );
                $Marker->update;
                last SYNONYM;
            }
        }
    }

    $i++;
}

print "Done, processed $i merges.\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

merge-markers.pl - merge markers

=head1 SYNOPSIS

  merge-markers.pl file.tab

  Or:

  merge-markers.pl -d 42 -t 43 [options] 

Options:

  -d|--deprecated  Marker ID of deprecated marker
  -t|--target      Marker ID of targe marker

  --help           Show brief help and exit
  --man            Show full documentation
  --version        Show version and exit

=head1 DESCRIPTION

Merges "deprecated" marker into "target" marker.  Alternately, you can 
provide a space-delimited file containing the deprecated marker ids in
the first column and the target ids in the second.  An optional third
column can list the marker name you wish to be the display name for 
the merged pair.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2005 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
