#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: load-markers.pl,v 1.11 2007/05/23 01:31:24 kclark Exp $

use strict;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::Marker::Import;
use Pod::Usage;
use Readonly;

Readonly my $EMPTY_STR => q{};
Readonly my $VERSION 
    => sprintf '%d.%02d', qq$Revision: 1.11 $ =~ /(\d+)\.(\d+)/;

my $search_only_primary_name = 0;
my $analysis                 = $EMPTY_STR;
my $library                  = $EMPTY_STR;
my ( $help, $verbose, $man_page, $show_version );
GetOptions(
    'a|analysis:s' => \$analysis,
    'l|library:s'  => \$library,
    'p|primary'    => \$search_only_primary_name,
    'v|verbose'    => \$verbose,
    'help'         => \$help,
    'man'          => \$man_page,
    'version'      => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

if ( !@ARGV ) {
    pod2usage('No input files');
}

my $importer = Gramene::Marker::Import->new;

$importer->import_tab(
    files                    => \@ARGV,
    analysis                 => $analysis,
    library                  => $library,
    verbose                  => $verbose,
    search_only_primary_name => $search_only_primary_name,
);

__END__

# ----------------------------------------------------

=pod

=head1 NAME

load-markers.pl - load marker data into marker db

=head1 VERSION

This documentation refers to load-markers.pl version $Revision: 1.11 $

=head1 SYNOPSIS

  load-markers.pl [options] markers.dat [markers.dat2 ...]

Options:

  -a|--analysis=foo|183  Either an analysis name or id
  -l|--library=foo|183   Either a library name or id
  -p|--primary           Search only the primary (display) marker name

  --help                 Show brief help and exit
  --man                  Show full documentation
  --verbose              Talk about what's happening

=head1 DESCRIPTION

Loads markers into marker db.  The data file should be a tab-delimited 
with the first line containing the fields in their order.  Different
marker types have different "details" fields according to the 
"marker_details_*" table definitions, but all types support the following
fields:

=over 4

=item * marker_name

The marker's primary name [required]

=item * marker_type

The marker's type, e.g., "RFLP," "SSR."  [required]

=item * species

The marker's species, e.g., "Rice," "Oryza nivara." [required]

=item * synonyms

A comma-delimited list of synonyms, e.g., "RG234A,RG234B"

=item * germplasm

The germplasm from the "species" from which the marker was derived

=item * analysis_name

An analysis name for the marker (will be created if it doesn't exist)

=item * analysis_id

An analysis ID for the marker

=item * description

A free-text description

=item * reference

A Gramene literature ID.

=back

All fields not marked "required" are optional.

=head1 SEE ALSO

Gramene::Marker::Import.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
