#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: id-xtype-assoc.pl,v 1.1 2006/10/16 15:55:51 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::CDBI::Markers;
use Gramene::Marker::DB;
use Pod::Usage;
use Readonly;

Readonly my $ANALYSIS => 'UNKNOWN';
Readonly my $ANALYTICAL_CORRESPONDENCE_TYPE => 'UNKNOWN';
Readonly my $COMMA => q{,};
Readonly my $NL    => qq{\n};
Readonly my $TAB   => qq{\t};
Readonly my %CHECK => (
    QTL  => [ 'Gene' ],
);
Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version );
GetOptions(
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

print join( $TAB, qw[ 
    from_marker_id 
    from_marker_name
    from_marker_species
    from_marker_type
    from_chromosome
    to_marker_id 
    to_marker_name
    to_marker_species
    to_marker_type
    to_chromosome
    analysis_name
    analytical_correspondence_type
] ), $NL;

my $mdb = Gramene::Marker::DB->new;
my $db  = $mdb->db;

while ( my ( $from_type, $to_types ) = each %CHECK ) {
    my @from = $mdb->marker_search( marker_type => $from_type );

    for my $from ( @from ) {
        my $from_marker 
            = Gramene::CDBI::Markers::Marker->retrieve( $from->{'marker_id'} );

        my $names = join( $COMMA, 
            map { s/\*/\\\*/g; $_ }
            map { $_->marker_name } 
            $from_marker->marker_synonyms
        );

        my $fdetails = $mdb->get_marker_details(marker_id => $from_marker->id); 
        my $from_chr = $fdetails->{'chromosome'} || '';

#        print STDERR "Checking $from_type $from->{'marker_id'} ($names)\n";

        for my $to_type ( @$to_types ) {
            my @to = $mdb->marker_search( 
                marker_type => $to_type,
                marker_name => $names,
            );

            TO_MARKER:
            for my $to ( @to ) {
                my $exists = $db->selectrow_array(
                    q[
                        select count(*)
                        from   analytical_correspondence
                        where  from_marker_id=?
                        and    to_marker_id=?
                    ],
                    {},
                    ( $from->{'marker_id'}, $to->{'marker_id'} )
                );

                next TO_MARKER if $exists;

                my $tdetails 
                    = $mdb->get_marker_details(marker_id => $to->{'marker_id'});
                my $to_chr = $tdetails->{'chromosome'} || '';

                print join( $TAB,
                    $from->{'marker_id'},
                    $from->{'marker_name'},
                    $from->{'species'},
                    $from->{'marker_type'},
                    $from_chr,
                    $to->{'marker_id'},
                    $to->{'marker_name'},
                    $to->{'species'},
                    $to->{'marker_type'},
                    $to_chr,
                    $ANALYSIS,
                    $ANALYTICAL_CORRESPONDENCE_TYPE,
                ), $NL; 
            }
        }
    }
}

__END__

# ----------------------------------------------------

=pod

=head1 NAME

id-xtype-assoc.pl 
  - identify possible associations b/w markers by type and name

=head1 VERSION

This documentation refers to id-xtype-assoc.pl version $Revision: 1.1 $

=head1 SYNOPSIS

  id-xtype-assoc.pl > possible-associations.tab
  load-marker-correspondences.pl possible-associations.tab

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

This script runs through all the markers of defined types (e.g., Genes, 
QTLs) and looks for other markers of defined types (e.g., Gene Models and
and Genes, respectively) sharing a name with the first type.  It then
prints out data in a format suitable for hand-curation that can then be
the input to "load-marker-correspondences.pl" (after deleting the lines
that are erroneous.  

Note: this script will not print out markers where associations already 
exist.

=head1 SEE ALSO

load-marker-correspondences.pl

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
