#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: find-invalid-synonyms.pl,v 1.1 2006/06/16 15:30:34 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Gramene::CDBI::Markers;
use Getopt::Long;
use Pod::Usage;
use Readonly;

Readonly my $NL      => qq{\n};
Readonly my $TAB     => qq{\t};
Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version );
GetOptions(
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my @synonym_types = Gramene::CDBI::Markers::SynonymType->retrieve_all;
print join( $TAB, qw[ marker_id marker_synonym_id marker_name synonym_type ] ), 
    $NL;

my $db  = Gramene::CDBI::Markers->db_Main;
$db->{'mysql_use_result'} = 1;
my $sth = $db->prepare(
    q[
        select marker_synonym_id, marker_id, marker_name
        from   marker_synonym
        where  synonym_type_id=?
    ]
);

for my $st ( @synonym_types ) {
    my $validation = $st->validation or next;
    my $regex      = qr/$validation/;

    $sth->execute( $st->id );

    while ( my $syn = $sth->fetchrow_hashref ) {
        if ( $syn->{'marker_name'} !~ $regex ) {
            print join( $TAB, 
                $syn->{'marker_id'},
                $syn->{'marker_synonym_id'},
                $syn->{'marker_name'},
                $st->synonym_type,
            ), $NL;
        }
    }
}

$sth->finish;
$db->disconnect;
print STDERR "Done.\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

find-invalid-synonyms.pl - a script

=head1 VERSION

This documentation refers to find-invalid-synonyms.pl version $Revision: 1.1 $

=head1 SYNOPSIS

  find-invalid-synonyms.pl 

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

This script scans all the synonyms and checks against the marker name
"synonym_type.validation" field, kicking out all bad names.

=head1 SEE ALSO

Gramene::CDBI::Markers.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
