#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: export-mappings-to-cmap.pl,v 1.3 2006/11/01 20:19:55 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use File::Spec::Functions;
use Getopt::Long;
use Gramene::DB;
use Gramene::CDBI::Markers;
use Gramene::Marker::Export;
use IO::Prompt;
use Pod::Usage;
use Readonly;

Readonly my $COMMA        => q{,};
Readonly my $COMMA_SPACE  => q{, };
Readonly my $EMPTY_STRING => q{};
Readonly my $DASH         => q{-};
Readonly my $NEWLINE      => qq{\n};
Readonly my $SLASH        => q{/};
Readonly my $TAB          => qq{\t};
Readonly my $VERSION      => sprintf '%d.%02d', 
                             qq$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;
Readonly my @OUT_FIELDS   => qw(
    map_name map_acc map_start map_stop feature_name feature_acc
    feature_aliases feature_stop feature_start feature_type_acc
);

my $ms_accs              = $EMPTY_STRING;
my $skip_ms_accs         = $EMPTY_STRING;
my $cmap_ds              = $EMPTY_STRING;
my $out_dir              = $EMPTY_STRING;
my $update               = $EMPTY_STRING;
my $correspondences_only = $EMPTY_STRING;
my $marker_types         = $EMPTY_STRING;
my ( $help, $man_page, $show_version );
GetOptions(
    'd|data-source=s'   => \$cmap_ds,
    'o|out-dir=s'       => \$out_dir,
    'm|ms-acc:s'        => \$ms_accs,
    's|skip:s'          => \$skip_ms_accs,
    'u|update'          => \$update,
    'c|correspondences' => \$correspondences_only,
    'marker-types:s'    => \$marker_types,
    'help'              => \$help,
    'man'               => \$man_page,
    'version'           => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

if ( !$out_dir ) {
    pod2usage('No out directory specified');
}

if ( !-d $out_dir ) {
    my $make_dir = prompt -yn,
        "The directory '$out_dir' does not exist.  OK to create? ";
    
    if ( !$make_dir ) {
        die "Not OK, exiting.\n";
    }
}

if ( $ms_accs !~ /\w+/ ) {
    if ( prompt -yn, 'Export all map sets? ' ) {
        $ms_accs = [ 
            map  { $_->cmap_map_set_accession }
            Gramene::CDBI::Markers::MapSet->search_like(
                cmap_map_set_accession => '%'
            )
        ];
    }
}

Gramene::Marker::Export::mappings_to_cmap({
    map_set_accessions      => $ms_accs,
    skip_map_set_accessions => $skip_ms_accs,
    out_dir                 => $out_dir,
    cmap_data_source        => $cmap_ds,
    allow_update            => $update,
    correspondences_only    => $correspondences_only,
    marker_types            => $marker_types,
});

__END__

# ----------------------------------------------------

=pod

=head1 NAME

export-mappings-to-cmap.pl - export a map set mappings to CMap

=head1 VERSION

This documentation refers to export-mappings-to-cmap.pl version $Revision: 1.3 $

=head1 SYNOPSIS

  export-mappings-to-cmap.pl 

Required Arguments:

  -d|--data-source=X    Target CMap data source
  -o|--out-dir=X        Directory to write files

Options:

  -m|--ms-acc=A[,B]         Comma-delimited list of CMap map set accessions
  -u|--update               Update CMap features (slower to import)
  -c|--correspondences      Export correspondences instead of mappings
  --marker-types=SSR[,RFLP] Export only markers of defined types

  --help                    Show brief help and exit
  --man                     Show full documentation
  --version                 Show version and exit

=head1 DESCRIPTION

This script exports the a marker db's mappings to CMap.  You may specify
a list of CMap map set accessions, separated by commas;  if you do not,
the script will export all map sets.

The "update" option tells CMap to check for the existence of a feature
first and to update any existing feature rather than create it anew.
This is somewhat slower, but necessary if you are updating existing
data in CMap.

=head1 SEE ALSO

Gramene::Marker::DB, Bio::GMOD::CMap.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
