#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: dump-genbank-acc.pl,v 1.2 2005/10/20 16:06:39 steven Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::Marker::DB;
use Pod::Usage;
use Readonly;

Readonly my $VERSION    => sprintf '%d.%02d', 
                           qq$Revision: 1.2 $ =~ /(\d+)\.(\d+)/;
Readonly my $TAB        => "\t";
Readonly my $NL         => "\n";
Readonly my @OUT_FIELDS => qw( marker_id genbank_acc marker_name marker_type );

my $take_types = '';
my $skip_types = '';

my ( $help, $man_page, $show_version );

GetOptions(
    't|type:s' => \$take_types,
    's|skip:s' => \$skip_types,
    'help'     => \$help,
    'man'      => \$man_page,
    'version'  => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

if ( $skip_types && $take_types ) {
    pod2usage('Please use just one of "skip" or "type"');
}

my %skip  = map { $_, 1 } split /\s*,\s*/, $skip_types;
my %types = map { $_, 1 } split /\s*,\s*/, $take_types;
my $mdb   = Gramene::Marker::DB->new;
my $db    = $mdb->db or die $mdb->error;

my $sql;
if ( %skip || %types ) {
    my @types = %skip ? keys %skip : keys %types;

    $sql = sprintf
        q[
            select x.record_id as marker_id, x.xref_value as genbank_acc
            from   xref x, xref_type xt, marker m, marker_type mt
            where  x.table_name=?
            and    x.xref_type_id=xt.xref_type_id
            and    xt.xref_type=?
            and    x.record_id=m.marker_id
            and    m.marker_type_id=mt.marker_type_id
            and    mt.marker_type %sin (%s)
        ],
        ( %skip ) ? 'not ' : '',
        join(', ', map { qq['$_'] } @types )
    ;
}
else {
    $sql = q[
        select x.record_id as marker_id, x.xref_value as genbank_acc
        from   xref x, xref_type xt
        where  x.table_name=?
        and    x.xref_type_id=xt.xref_type_id  
        and    xt.xref_type=?
    ];
}

my $sth  = $db->prepare( $sql );
$sth->execute( 'marker', 'GenBank' );

print join( $TAB, @OUT_FIELDS ), $NL;

while ( my $xref = $sth->fetchrow_hashref ) {
    my $marker = $mdb->view_marker( 
        marker_id  => $xref->{'marker_id'} ,
        no_details => 1,
    ) or die $mdb->error;

    print join( $TAB, 
        $xref->{'marker_id'}, 
        $xref->{'genbank_acc'},
        $marker->{'marker_name'},
        $marker->{'marker_type'},
    ), $NL;
}

__END__

# ----------------------------------------------------
=head1 NAME

dump-genbank-acc.pl - dumps all xrefs of type "GenBank"

=head1 VERSION

This documentation refers to dump-genbank-acc.pl version $Revision: 1.2 $

=head1 SYNOPSIS

  dump-genbank-acc.pl 

Options:

  -s|--skip=x[,y...]   Skip markers of listed types (e.g., EST)

  --help               Show brief help and exit
  --man                Show full documentation
  --version            Show version and exit

=head1 DESCRIPTION

This script will create a tab-delimited file containing these fields:

=over 4

=item * marker_id

=item * genbank_acc

=item * marker_name

=item * marker_type

=back

For every xref of type "GenBank" linked to a marker.

To skip markers of a particular type, e.g., EST, use the "--skip" option.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2005 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
