#!/usr/local/bin/perl

# $Id: deprecate-map-sets.pl,v 1.4 2006/10/30 16:24:30 kclark Exp $

# ----------------------------------------------------

use strict;
use Gramene::Marker::DB;
use Getopt::Long;
use File::Basename;
use IO::Prompt;
use Pod::Usage;
use Readonly;

Readonly my $NL => qq{\n};

use vars qw[ $VERSION ];
$VERSION = sprintf "%d.%02d", q$Revision: 1.4 $ =~ /(\d+)\.(\d+)/;

my $ms_accs = '';
my ( $help, $show_version );
GetOptions(
    'm|map-set-acc=s' => \$ms_accs,
    'h|help'          => \$help,
    'v|version'       => \$show_version,
);
pod2usage(2) if $help;
pod2usage('No map set accessions') unless $ms_accs;

if ( $show_version ) {
    my $prog = basename( $0 );
    print "$prog v$VERSION\n";
    exit(0);
}

my @map_set_accs = map { s/^\s+|\s+$//g; $_ } split( /,/, $ms_accs );
die 'No usable map set accessions' unless @map_set_accs;
my $answer = prompt -yn, 
    join( $NL, 'OK deprecate the following map sets?', @map_set_accs, '[yn]' );

if ( $answer eq 'n' ) {
    print "Not OK.  Exiting.\n";
    exit(0);
}

my $mdb = Gramene::Marker::DB->new;
my $db  = $mdb->db;

my $orig_re = $db->{'RaiseError'};
my $orig_pe = $db->{'PrintError'};
my $orig_ac = $db->{'AutoCommit'};

$db->{'RaiseError'} = 1;
$db->{'PrintError'} = 0;
$db->{'AutoCommit'} = 0;

my $no_deleted = 0;
eval {
    for my $ms_acc ( @map_set_accs ) {
        my $map_set_id = $db->selectrow_array(
            q[
                select map_set_id
                from   map_set
                where  cmap_map_set_accession=?
            ],
            {},
            ( $ms_acc )
        ) or die "Bad map set acc ($ms_acc)\n";

        my $maps = $db->selectall_arrayref(
            q[
                select map.map_id, map.map_name
                from   map
                where  map.map_set_id=?
            ],
            { Columns => {} },
            ( $map_set_id )
        );

        print "'$ms_acc' ($map_set_id) has ", scalar @$maps, " maps.\n";

        for my $map ( @$maps ) {
            my $no_rows = $db->do(
                q[
                    delete 
                    from   mapping
                    where  map_id=?
                ],
                {},
                ( $map->{'map_id'} )
            );

            print "Deleted $no_rows mappings for ",
                "$map->{'map_name'} ($map->{'map_id'})\n";
            $no_deleted += $no_rows;

            $db->do(
                q[
                    delete 
                    from   map        
                    where  map_id=?
                ],
                {},
                ( $map->{'map_id'} )
            );
        }

        if ( prompt( -yn, "Delete map set '$ms_acc'?[yn] " ) ) {
            $db->do(
                q[
                    delete 
                    from   map_set
                    where  map_set_id=?
                ],
                {},
                ( $map_set_id )
            );
        }
    }
};

if ( my $err = $@ ) {
    eval { $db->rollback };
    print "Error: $err\n";
}

$db->{'RaiseError'} = $orig_re;
$db->{'PrintError'} = $orig_pe;
$db->{'AutoCommit'} = $orig_ac;

print "Done.  Deleted $no_deleted mappings for ", 
    scalar @map_set_accs, " map sets.\n";

# ----------------------------------------------------

__END__

=pod

=head1 NAME

deprecate-map-sets.pl - remove deprecated map sets

=head1 SYNOPSIS

  deprecate-map-sets.pl [options]

Required Arguments:

  -m|--map-set-acc=X[,Y,Z]  CMap map set accessions IDs

Options:

  -h|--help       Show brief help and exit
  -v|--version    Show version and exit

=head1 DESCRIPTION

Removes map sets/maps/mappings for a given list of CMap map set 
accession IDs.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
