#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: delete-synonym.pl,v 1.1 2006/04/26 15:57:54 noelyap Exp $

package main;

use strict;
use warnings;
use Carp;
use English qw( -no_match_vars );
use IO::Prompt;
use IO::Scalar;
use Getopt::Long;
use Gramene::Marker::DB;
use Pod::Usage;
use Readonly;

Readonly my $SPACE   => q{ };
Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my $synonym_type = '';
my ( $help, $show_version, $man_page, $marker_id, $synonym );
GetOptions(
    'm|marker=i'       => \$marker_id,
    's|synonym=s'      => \$synonym,
    't|synonym-type:s' => \$synonym_type,
    'help'             => \$help,
    'man'              => \$man_page,
    'version'          => \$show_version,
) or pod2usage;

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

if ( !$marker_id ) {
    pod2usage('No marker ID');
}

if ( !$synonym ) {
    pod2usage('No synonym to delete');
}

my $mdb = Gramene::Marker::DB->new( admin => 1 );

my $name = $mdb->get_marker_display_name( marker_id => $marker_id );

if ( $name eq $synonym ) {
    croak "Can't delete synonym: $synonym is the primay display name";
}

print "Deleting synonym $synonym from marker_id=$marker_id...\n";

my @synonym_ids =
    map  { $_->{'marker_synonym_id'} }
    grep { $_->{'marker_name'} eq $synonym }
    grep { $synonym_type ? lc $_->{'synonym_type'} eq lc $synonym_type : $_ }
    @{ $mdb->get_marker_synonyms( marker_id => $marker_id ) };

if ( @synonym_ids ) {
    print 'Deleting synonym_ids = ', join(', ', @synonym_ids), "\n";

    for my $synonym_id ( @synonym_ids ) {
        $mdb->delete_marker_synonym( marker_synonym_id => $synonym_id );
    }
} else {
    my $type_err = defined $synonym_type ? " $synonym_type" : '';
    carp "Failed: $synonym is not a$type_err synonym of marker_id=$marker_id";
}

__END__

# ----------------------------------------------------

=pod

=head1 NAME

delete-synonym.pl - delete a marker synonym

=head1 SYNOPSIS

  delete-synonym.pl -m 42 -s foo [options] 

Required Arguments:

  -m|--marker=X           Marker ID (integer value) of target marker
  -s|--synonym=X          Synonym to delete from target marker

Optional Arguments:

  -t|--synonym-type=X     Synonym type of synonym, e.g., 
                          GENBACK_ACCESSION (in cases of duplicate 
                          synonyms of varying types)

  --help                  Show brief help and exit
  --man                   Show full documentation
  --version               Show version and exit

=head1 DESCRIPTION

Deletes a marker synonym given a target marker ID.  Will not delete
the primary display name.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Immanuel Yap E<lt>ivy1@cornell.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2005 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
