#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: delete-marker.pl,v 1.3 2007/05/23 01:09:24 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use IO::Prompt;
use File::Basename;
use Gramene::Marker::DB;
use Getopt::Long;
use Pod::Usage;
use Readonly;

Readonly my $EMPTY_STR => q{};
Readonly my $VERSION   => sprintf '%d.%02d', 
                          qq$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;

my $analysis    = $EMPTY_STR;
my $marker_name = $EMPTY_STR;
my $marker_type = $EMPTY_STR;
my $species     = $EMPTY_STR;
my ( $help, $man_page, $show_version );
GetOptions(
    'n|name:s'     => \$marker_name,
    't|type:s'     => \$marker_type,
    's|species:s'  => \$species,
    'a|analysis:s' => \$analysis,
    'help'         => \$help,
    'man'          => \$man_page,
    'version'      => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my @marker_ids = map { m/^\d+$/ ? $_ : () } map { split /,/ } @ARGV;
@ARGV = ();

my $mdb = Gramene::Marker::DB->new;

my $analysis_id;
if ( $analysis ) {
    if ( $analysis =~ /^\d+$/ ) {
        my $Analysis = Gramene::CDBI::Markers::Analysis->retrieve($analysis)
            or die "Bad analysis id ($analysis)\n";
        $analysis_id = $Analysis->id;
    }
    else {
        my ($Analysis) = Gramene::CDBI::Markers::Analysis->search(
            analysis_name => $analysis
        ) or die "Unknown analysis ($analysis)\n";
        $analysis_id = $Analysis->id;
    }
}

if ( $marker_name || $marker_type || $species || $analysis_id ) {
    my @markers = $mdb->marker_search(
        marker_name => $marker_name,
        marker_type => $marker_type,
        species     => $species,
        analysis_id => $analysis_id,
    );
    my $num_markers = scalar @markers;

    if ( $num_markers == 1 ) {
        push @marker_ids, $markers[0]->{'marker_id'};
    }
    else {
        printf "%s markers match %s%s%s%s\n.",
            $num_markers,
            $species     ? "species '$species'"   : $EMPTY_STR,
            $marker_type ? "type '$marker_type'"  : $EMPTY_STR,
            $marker_name ? "name '$marker_name'"  : $EMPTY_STR,
            $analysis    ? "analysis '$analysis'" : $EMPTY_STR,
        ;

        @marker_ids = map { $_->{'marker_id'} } @markers;
    }
}

if ( @marker_ids ) {
    my $ok = prompt -yn, 
        sprintf(
            "OK to delete %s marker%s?[yn] ", 
            scalar @marker_ids,
            scalar @marker_ids == 1 ? $EMPTY_STR : 's',
        );

    exit unless $ok;
}
else {
    pod2usage('No usable marker IDs');
}

my $num_deleted;
for my $marker_id ( @marker_ids ) {
    next unless $marker_id =~ m/^\d+$/;

    print "Deleting marker '$marker_id'\n";

    $mdb->delete_marker( marker_id => $marker_id );
    $num_deleted++;
}

print "Done, deleted $num_deleted markers.\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

delete-marker.pl - a script

=head1 VERSION

This documentation refers to delete-marker.pl version $Revision: 1.3 $

=head1 SYNOPSIS

  delete-marker.pl 1,2,3

Or:

  delete-marker.pl 1 2 3

Or:

  delete-marker.pl -n RM3 -t SSR -s 'Oryza sativa'

Options:

  -a|--analysis An analysis name or ID
  -n|--name     Marker name
  -t|--type     Marker type
  -s|--species  Marker species

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

Deletes the markers identified by the supplied marker IDs or by any
combination of name, type, analysis and species.  Marker IDs may be
comma- or space-delimited.

=head1 SEE ALSO

Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
