#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: delete-mappings.pl,v 1.5 2007/06/05 20:19:17 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Gramene::CDBI::Markers;
use Gramene::Marker::DB;
use Getopt::Long;
use Pod::Usage;
use Readonly;

Readonly my $EMPTY_STR => q{};
Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.5 $ =~ /(\d+)\.(\d+)/;

my $delete_marker  = 0;
my $delete_map_set = 0;
my $map_set_accs  = $EMPTY_STR;
my ( $help, $man_page, $show_version );
GetOptions(
    'delete-map-set'  => \$delete_map_set,
    'delete-marker'   => \$delete_marker,
    'm|map-set-acc=s' => \$map_set_accs,
    'help'            => \$help,
    'man'             => \$man_page,
    'version'         => \$show_version,
) or pod2usage(2);

my @map_set_accs = map { s/^\s+|\s+$//g; $_ } split /,/, $map_set_accs;
if ( !@map_set_accs ) {
    pod2usage('No map set accessions');
}

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $mdb = Gramene::Marker::DB->new;
my $db  = $mdb->db or die $mdb->error;

my ( $num_map_sets, $num_maps, $num_mappings ) = ( 0, 0, 0 );
for my $map_set_acc ( @map_set_accs ) {
    my ($MapSet) = Gramene::CDBI::Markers::MapSet->search(
        cmap_map_set_accession => $map_set_acc
    ) or die "Bad map set accession ($map_set_acc).\n";
    $num_map_sets++;

    printf STDERR "%s %s '%s' (%s)\n", 
        $MapSet->map_type->map_type,
        $MapSet->species->species,
        $MapSet->map_set_name,
        $MapSet->cmap_map_set_accession || $EMPTY_STR,
    ;

    my $mapping_sth = $db->prepare(
        q[
            select mapping_id, marker_id 
            from   mapping
            where  map_id=?
        ]
    );

    for my $Map ( $MapSet->maps ) {
        $num_maps++;
        printf STDERR "Map '%s' (%s)\n",
            $Map->map_name,
            $Map->cmap_map_accession || $EMPTY_STR,
        ;

        $mapping_sth->execute( $Map->id );

        while ( 
            my ( $mapping_id, $marker_id ) = $mapping_sth->fetchrow_array 
        ) {
            $num_mappings++;
            $db->do(
                'delete from mapping where mapping_id=?', {}, $mapping_id 
            );

            if ( $delete_marker && $marker_id ) {
                $mdb->delete_marker( marker_id => $marker_id );
            }
        }

        $db->do('delete from map where map_id=?', {}, $Map->id );
    }

    if ( $delete_map_set ) {
        $db->do('delete from map_set where map_set_id=?', {}, $MapSet->id );
    }
}

printf "Done, deleted %s map set%s, %s map%s, %s mapping%s\n",
    $num_map_sets,
    $num_map_sets == 1 ? '' : 's',
    $num_maps,
    $num_maps == 1 ? '' : 's',
    $num_mappings,
    $num_mappings == 1 ? '' : 's',
;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

delete-mappings.pl - a script

=head1 VERSION

This documentation refers to version $Revision: 1.5 $

=head1 SYNOPSIS

  delete-mappings.pl [options] -m map-set-acc

Options:

  -m|--map-set-acc=X[,Y,Z]  Map set accessions (comma-separated)
  --delete-marker           Delete the underlying marker for the mappings
                              (default is to leave the marker)
  --delete-map-set          Delete the map set when done with the mappings
                              (default is to leave the map set)

  --help                    Show brief help and exit
  --man                     Show full documentation
  --version                 Show version and exit

=head1 DESCRIPTION

This script deletes mappings.  You can also use it to remove the underlying 
marker and the containing map set using the appropriate flags.

=head1 SEE ALSO

perl.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
