#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: check-cmap-xfer.pl,v 1.1 2006/11/02 15:27:56 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::CDBI::Markers;
use Bio::GMOD::CMap;
use Pod::Usage;
use Readonly;
use Text::TabularDisplay;

Readonly my $EMPTY_STRING => q{};
Readonly my $NL           => qq{\n};
Readonly my $VERSION      
    => sprintf '%d.%02d', qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my $cmap_ds = $EMPTY_STRING;
my ( $help, $man_page, $show_version );
GetOptions(
    'd=s'     => $cmap_ds,
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $cmap = Bio::GMOD::CMap->new;
$cmap->data_source( $cmap_ds );
my $cdb  = $cmap->db;
my $mdb  = Gramene::CDBI::Markers->db_Main;

my @map_sets = Gramene::CDBI::Markers::MapSet->search_like(
    cmap_map_set_accession => '%'
);

my $i = 0;
my @errors;
for my $MapSet ( @map_sets ) {
    my $map_set_acc  = $MapSet->cmap_map_set_accession or next;
    my $num_mappings = $mdb->selectrow_array(
        q[
            select count(mapping.mapping_id)
            from   mapping, map
            where  mapping.cmap_feature_accession is not null
            and    mapping.map_id=map.map_id
            and    map.map_set_id=?
        ],
        {},
        ( $MapSet->id )
    );

    my $num_features = $cdb->selectrow_array(
        q[
            select count(f.feature_id)
            from   cmap_feature f, cmap_map map, cmap_map_set ms
            where  f.map_id=map.map_id
            and    map.map_set_id=ms.map_set_id
            and    ms.map_set_acc=?
        ],
        {},
        ( $map_set_acc )
    );

    $i++;
    if ( $num_mappings == $num_features ) {
        print "$i: $map_set_acc OK ($num_mappings)\n";
    }
    else {
        push @errors, [
            $map_set_acc,
            $num_mappings, 
            $num_features,
        ];
    }
}

if ( @errors ) {
    my $tab = Text::TabularDisplay->new('Map Set', 'MarkersDB', 'CMap' );
    for my $err ( @errors ) {
        $tab->add( @$err );
    }

    print join($NL,
        scalar @errors . ' errors!',
        $tab->render,
        $EMPTY_STRING,
    );
}
else {
    print "No errors!\n";
}

__END__

# ----------------------------------------------------

=pod

=head1 NAME

check-cmap-xfer.pl - check that all mappings make it to CMap

=head1 VERSION

This documentation refers to version $Revision: 1.1 $

=head1 SYNOPSIS

  check-cmap-xfer.pl -d CMapDataSource

Options:

  -d            CMap datasource, e.g., "Build23"
  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

This script selects all map sets in the markers db that have a defined
CMap map set accession and then checks the indicated CMap data source
(-d arg) to see if the number of marker db mappings with defined CMap 
feature accessions matches the number of features in CMap.  If any 
errors are found, they are printed at the end of the run.

=head1 SEE ALSO

Bio::GMOD::CMap, Gramene::CDBI::Markers.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
