#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: check-mdb-mapping-xfer.pl,v 1.1 2006/10/17 19:55:43 kclark Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::CDBI::Genes;
use Gramene::CDBI::Markers;
use Pod::Usage;
use Readonly;

Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.1 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version );
GetOptions(
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $gene_map_pos_iter = Gramene::CDBI::Genes::GeneMapPosition->retrieve_all;

my ( $i, $good, $bad ) = ( 0, 0, 0 );
while ( my $pos = $gene_map_pos_iter->next ) {
    $i++;
    my $feature_acc = $pos->cmap_feature_accession or next;
    my $gene_acc    = $pos->gene->accession;

    my ($mapping) = Gramene::CDBI::Markers::Mapping->search(
        cmap_feature_accession => $feature_acc
    );

    if ( !$mapping ) {
        print STDERR "Missing $feature_acc ($gene_acc)\n";
        $bad++;
    }
    else {
        print "OK $feature_acc ($gene_acc) [", $mapping->id, "]\n";
        $good++;
    }
}

print "Done, checked $i, $good good, $bad bad\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

check-mdb-mapping-xfer.pl - check transfer of gene mappings

=head1 VERSION

This documentation refers to version $Revision: 1.1 $

=head1 SYNOPSIS

  check-mdb-mapping-xfer.pl 2>err

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

Iterates through all the CMap feature accessions in the map position 
table and checks that it made it to the mappings db.

=head1 SEE ALSO

Gramene::CDBI::Genes, Gramene::CDBI::Markers.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
