#!/usr/local/bin/perl

=head1 NAME

bac-extractor.pl

=head1 SYNOPSIS

  bac-extractor.pl [options] parsed-fpc.dat > bacs.dat

Options:

  -h|--help     Show help
  --hits=n      Only extract BACs with "n" hits (markers)
  --show-hits   Show the number of hits (marker) for each BAC

=head1 DESCRIPTION

Takes a B<parsed> FPC file (the output of "fpc_parser.pl") and extracts
just the BACs.

=cut

use strict;
use Pod::Usage;
use Getopt::Long;

my ( $minimum_hits, $show_hits, $help );
GetOptions(
    'h|help'    => \$help,
    'hits:i'    => \$minimum_hits,
    'show-hits' => \$show_hits,
);
pod2usage(2) if $help;

my $file = shift or pod2usage('No input file');

open my $fh, $file or die "Can't read '$file': $!\n";
my @headers = qw[map_name feature_name feature_start feature_stop feature_type];
push @headers, 'hits' if $show_hits;

print join( "\t", @headers ), "\n";

while ( <$fh> ) {
    my ( $bac, $contig, $start, $stop, $linkage_group, @markers ) = split /\t/;
    next unless $contig;
    my $no_hits = scalar @markers;
    next if $minimum_hits && $no_hits < $minimum_hits; 
    my $feature_type = $no_hits > 1 
        ? 'High-confidence BAC' : 'Low-confidence BAC';
    my @data = ( $contig, $bac, $start, $stop, $feature_type );
    push @data, $no_hits if $show_hits;
    print join( "\t", @data ), "\n";
}
