#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: mk-omap-matrix.pl,v 1.5 2007/06/06 19:33:27 kclark Exp $

use strict;
use warnings;
use Bio::GMOD::CMap;
use Data::Dumper;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Gramene::Utils qw( commify );
use Pod::Usage;
use Readonly;

Readonly my $DBL_COLON          => '::';
Readonly my $TIGR_ACC           => 'gt0506';
Readonly my $ORDERED_ACC_SUFFIX => '-stk';
Readonly my $EMPTY_STR          => q{};
Readonly my $NL                 => qq{\n};
Readonly my $VERSION 
    => sprintf '%d.%02d', qq$Revision: 1.5 $ =~ /(\d+)\.(\d+)/;
Readonly my %TABLE_TITLE        => (
    all   => 'Clone and Contig View',
    clone => 'Clone View',
    fpc   => 'Contig View',
);
Readonly my @FPC_ACCESSIONS     => qw(
    orcb0602h
    orbb0602h
    ogbb0602h
    op_b0607
    om_b0609
    oo_b0607
    oabb0607
    oaab0605
    oc_b0605
    ob_b0607
    orab0605
    ogab0604
);

my $data_source = $EMPTY_STR;
my ( $help, $man_page, $show_version );
GetOptions(
    'd=s'     => \$data_source,
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( !$data_source ) {
    pod2usage('No CMap data source');
}

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $cmap = Bio::GMOD::CMap->new;
$cmap->data_source( $data_source );
my $db   = $cmap->db;

my $tigr_maps = $db->selectall_arrayref(
    q[
        select   map.map_id, map.map_acc, map.map_name,
                 ms.map_set_acc
        from     cmap_map map, cmap_map_set ms
        where    map.map_set_id=ms.map_set_id
        and      ms.map_set_acc=?
        order by map.display_order
    ],
    { Columns => {} },
    ( $TIGR_ACC )
);

my $order = 0;
my %phylo_order = map { $_, $order++ } qw[
    rufipogon
    nivara
    glaberrima
    punctata
    minuta
    officinalis
    alta
    australiensis
    coarctata
    brachyantha
    ridleyi
    granulata
];

my @omap_map_sets = 
    sort { $a->{'phylo_order'} <=> $b->{'phylo_order'} }
    map  { 
        ($_->{'species'} = $_->{'species_full_name'}) =~ s/^Oryza\s+//;
        $_->{'phylo_order'} = $phylo_order{ $_->{'species'} }; 
        $_ 
    }
    @{ $db->selectall_arrayref(
        qq[
            select   ms.map_set_id, ms.map_set_short_name, ms.map_set_acc,
                     s.species_full_name 
            from     cmap_map_set ms, cmap_species s
            where    ms.map_set_acc like ('%${ORDERED_ACC_SUFFIX}')
            and      ms.species_id=s.species_id
        ],
        { Columns => {} },
    ) }
;
my $num_cmp_map_sets = scalar @omap_map_sets + 1;

my %all_correspondences;
for my $feature_type ( qw[ clone fpc all ] ) {
    my $title = $TABLE_TITLE{ $feature_type };
    print join($NL,
        '<TABLE class="data-table">',
        qq[<TR><TH COLSPAN="$num_cmp_map_sets">$title</TH></TR>],
        '<TR>',
        '<TH>&nbsp;</TH>', 
        ( map { "<TH>$_->{'species'}</TH>" } @omap_map_sets ), 
        '</TR>',
        $EMPTY_STR,
    );

    my $row_num = 0;
    for my $tigr_map ( @$tigr_maps ) {
        ( my $map_name = $tigr_map->{'map_name'} ) =~ s/\s+/&nbsp;/g;

        my $total_num = 0;
        my @rows;

        OMAP_MAP_SET:
        for my $omap_ms ( @omap_map_sets ) {
            my $sth = $db->prepare(
                qq[
                    select map.map_id, map.map_acc, map.map_name
                    from   cmap_map map
                    where  map.map_set_id=?
                    and    map.map_name like ('% on $tigr_map->{map_name}')
                ]
            );
            $sth->execute( $omap_ms->{'map_set_id'} );
            my $omap_map = $sth->fetchrow_hashref;

            my $number;
            if ( $omap_map ) {
                my $key = join( $DBL_COLON, 
                    $tigr_map->{'map_id'}, $omap_map->{'map_id'} 
                );

                $all_correspondences{ $key } ||= 0;

                if ( $feature_type eq 'all' ) {
                    $number = $all_correspondences{ $key };
                }
                else {
                    $number = $db->selectrow_array(
                        q[
                            select count(cl.feature_correspondence_id)
                            from   cmap_correspondence_lookup cl
                            where  map_id1=?
                            and    map_id2=?
                            and    feature_type_acc1=?
                            and    feature_type_acc2=?
                        ],
                        {},
                        ( $tigr_map->{'map_id'}, $omap_map->{'map_id'},
                          $feature_type, $feature_type )
                    ) || 0;

                    $all_correspondences{ $key } += $number;
                }
            }

            if ( $number ) {
                my $ft_arg = $EMPTY_STR;
                if ( $feature_type ne 'all' ) {
                    $ft_arg = "&ft_DEFAULT=0;ft_${feature_type}=1";
                }

                $total_num += $number;

                push @rows, join( $EMPTY_STR,
                    '<TD ALIGN="RIGHT">',
                    '<A HREF="/db/cmap/viewer?aggregate=1;ref_map_set_acc=',
                    $tigr_map->{map_set_acc},
                    ';ref_map_accs=',
                    $tigr_map->{map_acc},
                    ';comparative_maps=1%3dmap_acc%3d',
                    $omap_map->{map_acc},
                    $ft_arg,
                    '">',
                    commify($number),
                    "</A></TD>"
                );
            }
            else {
                push @rows, '<TD>&nbsp;</TD>';
            }
        }

        if ( $total_num > 0 ) {
            printf qq[<TR class="row%s">\n<TD>%s</TD>\n],
                $row_num++ % 2 == 0 ? 'even' : 'odd',
                $map_name;
            print join( $NL, @rows, '</TR>', $EMPTY_STR );
        }
    }

    print "</TABLE>\n<BR><BR>\n";
}

print join($NL,
    '<TABLE class="data-table">',
    '<TR><TH COLSPAN="3">Unordered FPC Maps</TH></TR>',
    '<TR><TH>Species</TH><TH>Map Set Name</TH><TH>&nbsp;</TH></TR>',
);

MAP_SET:
for my $ms_acc ( @FPC_ACCESSIONS ) {
    my $sth     = $db->prepare(
        q[
            select s.species_full_name as species,
                   ms.map_set_name
            from   cmap_map_set ms, cmap_species s
            where  ms.map_set_acc=?
            and    ms.species_id=s.species_id
        ]
    );
    $sth->execute( $ms_acc );
    my $ms = $sth->fetchrow_hashref or next MAP_SET;

    print "<TR><TD>$ms->{'species'}</TD><TD>$ms->{'map_set_name'}</TD>",
        qq[<TD><A HREF="/db/markers/marker_view?action=view_map_set],
        qq[&map_set_acc=$ms_acc">View</A></TD></TR>\n]
    ;
}

print "</TABLE>\n<BR><BR>\n";

__END__

# ----------------------------------------------------

=pod

=head1 NAME

mk-omap-matrix.pl - make OMAP/TIGR summary matrix HTML table

=head1 VERSION

This documentation refers to version $Revision: 1.5 $

=head1 SYNOPSIS

  mk-omap-matrix.pl -d BuildXX

Options:

  -d=BuildXX    The name of the CMap data source to query

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

This script will build an HTML table showing a matrix of each 
TIGR/OMAP map and the number of correspondences to the ordered
map from each OMAP species aligned for the reference map.

=head1 SEE ALSO

Bio::GMOD::CMap.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
