package Panzea::GMapView;

our $VERSION = '0.02';

# $Id: GMapView.pm,v 1.1.2.1 2007/06/14 19:03:22 kclark Exp $

=head1 NAME

Panzea::GMapView - Panzea-specific subclass of HTML::GMap

=head1 DESCRIPTION

Panzea-specific subclass of HTML::GMap.

=cut

use warnings;
use strict;

use base qw(HTML::GMap);

use Math::BigFloat;

=head1 METHODS

=cut

=head3 process_markers_pre_filter

 Function  :
 Arguments : \%markers
 Returns   : 1
 Notes     : None specified

=cut

sub process_markers_pre_filter {
    my ($self, $markers_ref) = @_;

    foreach my $key (keys %{$markers_ref}) {
        my @values = values %{$markers_ref->{$key}->{cluster_set}};
        $markers_ref->{$key}->{he_pre_filtered} =
          $self->calculate_he(\@values);
    }

    return 1;
}

=head3 calculate_he

 Function  : Calculates HE (Expected Heterozygosity)
 Arguments : \@data
 Returns   : $he
 Notes     : None specified

=cut

sub calculate_he {
    my ($self, $data_ref) = @_;

    # HE = 2n / (2n -1) * (1 - SUM(Xi^2))
    # HE : Expected Heterozygosity
    # n  : Number of assays
    # 2n : Number of alleles
    # Xi : Frequency of ith allele

    my $two_n;
    foreach my $value (@$data_ref) {
        my $float_value = Math::BigFloat->new($value);
        $two_n = $two_n + $float_value;
    }

    my $sum_xi_sq;
    foreach my $value (@$data_ref) {
        my $float_value = Math::BigFloat->new($value);
        my $xi          = $float_value / $two_n;
        my $xi_sq       = $xi * $xi;

        $sum_xi_sq = $sum_xi_sq + $xi_sq;
    }

    my $he = $two_n / ($two_n - 1) * (1 - $sum_xi_sq);

    my $rounded_he = $he->bround(2);

    return $rounded_he;
}

=head3 generate_piechart_details_html

 Function  :
 Arguments : $key_ref
 Returns   : $html
 Notes     :

=cut

sub generate_piechart_details_html {
    my ($self, $key_ref) = @_;

    my $data_ref = $key_ref->{cluster_set};

    my $session         = $self->session;
    my $color_table_ref = $session->param('color_table');
    my $temp_dir_eq     = $self->temp_dir_eq;

    my $he          = $key_ref->{he_pre_filtered};
    my $total_count = $key_ref->{cluster_data_count};

    my $html;

    $html .= qq[<table>\n];
    $html .= qq[<tr>\n];
    $html .=
      qq[<th align="left" width="50%">Total Count</th><th align="left">:&nbsp;$total_count</th>\n];
    $html .= qq[</tr>\n];
    $html .= qq[<tr>\n];
    $html .=
      qq[<th align="left" width="50%">Expected Heterozygosity</th><th align="left">:&nbsp;$he</th>\n];
    $html .= qq[</tr>\n];
    $html .= qq[</table>\n];

    $html .= qq[<table>\n];

    foreach my $label (
        sort {
            if (   ($a eq 'Clustered' || $a eq 'Other')
                && ($b eq 'Clustered' || $b eq 'Other')) {
                $a cmp $b;
            }
            elsif (($a eq 'Clustered' || $a eq 'Other')
                && ($b ne 'Clustered' && $b ne 'Other')) {
                1;
            }
            elsif (($a ne 'Clustered' && $a ne 'Other')
                && ($b eq 'Clustered' || $b eq 'Other')) {
                -1;
            }
            else { $data_ref->{$b} <=> $data_ref->{$a} }
        } keys %{$data_ref}
      ) {
        my $count    = $data_ref->{$label};
        my $color    = $color_table_ref->{$label};
        my $icon_url = "$temp_dir_eq/Legend-icon-$color.png";

        my $percent = Math::BigFloat->new($count / $total_count * 100);
        my $rounded_percent = $percent->ffround(-1);

        $html .= qq[<tr>\n];
        $html .=
          qq[<td align="left"><img src="$icon_url"/>&nbsp;&nbsp;$label ($count points)</td>\n];
        $html .= qq[<td align="right">&nbsp;&nbsp;$rounded_percent %</td>\n];
        $html .= qq[</tr>\n];
    }
    $html .= qq[</table>\n];

    return $html;
}

=head1 AUTHOR

Payan Canaran <canaran@cshl.edu>

=head1 BUGS

=head1 VERSION

Version 0.02

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright (c) 2006-2007 Cold Spring Harbor Laboratory

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See DISCLAIMER.txt for
disclaimers of warranty.

=cut

1;

