package LookAlign::Interface::Root;

our $VERSION = '0.01';

# $Id: Root.pm,v 1.1.2.1 2007/06/14 19:03:07 kclark Exp $

=head1 NAME

LookAlign::Interface::Root

=head1 DESCRIPTION

Root Interface module for alignment viewer.

=cut

use warnings;
use strict;

use Carp;
use CGI;
use CGI::Session;
use Config::General;
use DBI;
use File::Temp qw(tempfile tempdir);
use LWP::Simple;
use Time::Format qw(%time);

use LookAlign::Alignment::Container;
use LookAlign::Alignment::Container::Sq;
use LookAlign::Alignment::Renderer;

###############
# CONSTRUCTOR #
###############

# Function  : Constructor.
# Arguments : %params
# Returns   : $Root_object
# Notes     :
#
# Parameters that are provided in the %params are listed below.
#
# Parameter    Description                   Format       Default
# ---------    -----------                   ------       -------
# config_file Configuration file            $file_name   n/a
# session_id  Session id                    $session_id  undef (Auto-generated)
# session_dir Session storage directory     $session_dir n/a
# subtype     Interface subtype (parameter) $subtype     n/a
#
# Internal private read-only params (for development use).
#
# Parameter       Description          Format
# ---------       -----------          ------
# cgi            CGI params           Hashref
# config         Parsed configuration hashref
#                 hashref
# db_selected    Selected database    Scalar
#                 alias
# session        CGI::Session object  CGI::Session object
# printed_header Whether the header   0 | 1
#                 has been printed

sub new {
    my ($class, %params) = @_;

    my $self = bless {}, $class;

    eval {
        my $subtype = $params{subtype} || undef;
        $self->subtype($subtype);

        my $config_file = $params{config_file}
          or croak("A config file must be specified");

        my $config_obj =
          Config::General->new(
            -ConfigFile        => $config_file,
            -InterPolateVars   => 1,
            -AllowMultiOptions => 1
          );
        my %pre_config = $config_obj->getall;
        my %config; # lower-case params
        foreach my $key (keys %pre_config) {
            $config{lc($key)} = $pre_config{$key};
        }
        $self->config(\%config);

        my $session_id = $params{session_id} || undef;
        my $session_dir = $params{session_dir}
          or croak("A session_dir must be specified");
        my $session =
          CGI::Session->new("driver:File", $session_id,
            {Directory => $session_dir});

        $session_id = $session->id;
        $self->session_id($session_id);
        $self->session($session);

        # Assign db_selected
        if ($params{db_selected}) {
            $self->session_param('db_selected', $params{db_selected});
        }

        my %cgi = CGI->new->Vars;    # Returns a regular hash
        $self->_validate_cgi_params(\%cgi);
        $self->cgi(\%cgi);

        $self->set_cookies;
    };

    $self->die_with_error($@) if $@;

    return $self;
}

###########
# METHODS #
###########

# Function  : Return root_url.
# Arguments : None.
# Returns   : $root_url
# Notes     : None specified.

sub root_url {
    my ($self) = @_;

    my $session_id = $self->session_id;

    my ($root_url) = $ENV{SCRIPT_NAME} =~ /^([^\?]+)/;

    return $root_url;
}

# Function  : Return selection criteria as an hasref of arrays.
# Arguments : None.
# Returns   : \%selection_criteria
# Notes     : None specified.

sub selection_criteria {
    my ($self) = @_;

    my $cgi = $self->cgi;

    my %selection_criteria;

    foreach my $key (keys %$cgi) {
        if ($key =~ /^selection:/) {
            $selection_criteria{$key} =
              ref $cgi->{$key} ? $cgi->{$key} : [$cgi->{$key}];
        }
    }
    return \%selection_criteria;
}

# Function  : Return value of config key.
# Arguments : $key
# Returns   : $value
# Notes     : This method croaks if config key does not exist.
#             Keys are lc'ed before retrieval.

sub cfg {
    my ($self, $key) = @_;

    croak("A config key must be specified") unless $key;

    my $config = $self->config;

    $key = lc($key); # *** Keys are lc'ed before retrieval ***
    
    unless (exists $config->{$key}) {
        croak("Config key ($key) is not defined");
    }

    my $value = $config->{$key};

    if ($value =~ s/^GET://) {
        my $self_url = CGI::self_url();
        my ($current_url, $current_args) = split(/\?/, $self_url);
        $current_url  ||= '';
        $current_args ||= '';
        $current_url =~ s!^http://[^/]+!!;
        $current_url  = CGI::escape($current_url);
        $current_args = CGI::escape($current_args);
        $value =~ s/__CURRENT_URL__/$current_url/ if $value;
        $value =~ s/__CURRENT_ARGS__/$current_args/ if $value;
        if ($value) {
            my $content = get($value) or croak("Cannot get value ($value)!");
            $value      = $content;
        }
    }
    return $value;
}

# Function  : Return value of cgi_param.
# Arguments : $key, $value
# Returns   : $value
# Notes     : This method returns undef if config key does not exist.

sub cgi_param {
    my ($self, $key, $value) = @_;

    croak("A cgi_param key must be specified") unless $key;

    if (defined $value) { $self->cgi->{$key} = $value; }

    return $self->cgi->{$key};
}

# Function  : Return value of session_param.
# Arguments : $key, $value
# Returns   : $value
# Notes     : This method returns undef if session key does not exist.

sub session_param {
    my ($self, $key, $value) = @_;

    croak("A session_param key must be specified") unless $key;

    if (defined $value) { $self->session->param($key, $value); }

    return $self->session->param($key);
}

# Function  : Validates CGI params.
# Arguments : \%cgi
# Returns   : 1
# Notes     : Checks the CGI parameters passed on as a hashref and croaks
#             if an error is found.

sub _validate_cgi_params {
    my ($self, $cgi) = @_;

    foreach my $key (keys %$cgi) {
        next unless defined $cgi->{$key};

        my @value = split("\0", $cgi->{$key});

        foreach (@value) {
            $_ =~ s/^\s+//;
            $_ =~ s/\s+$//;

            # Check all values against this
            if ($_ =~ /[\&\;]/) {
                croak("Illegal character in value ($key:$_)!");
            }

            # Check individual values
            elsif ($key eq 'pic_threshold') {
                unless ($_ =~ /^[0-9\.]+$/ and $_ >= 0 and $_ < 1) {
                    croak("Invalid PIC threshold!");
                }
            }

            elsif ($key eq 'n_threshold') {
                unless ($_ =~ /^[0-9]+$/ and $_ >= 2) {
                    croak("Invalid N threshold!");
                }
            }

            elsif ($key eq 'qscore_threshold') {
                unless ($_ =~ /^[0-9]+$/ and $_ >= 0 and $_ < 100) {
                    croak("Invalid Quality Score threshold!");
                }
            }

            elsif ($key eq 'identity_threshold') {
                unless ($_ =~ /^[0-9]+$/ and $_ >= 0 and $_ < 100) {
                    croak("Invalid Identity Percentage threshold!");
                }
            }

            elsif ($key eq 'suppress_gaps') {
                unless ($_ == 0 or $_ == 1) {
                    croak("Invalid Suppress Gaps threshold!");
                }
            }

            elsif ($key eq 'filter') {
                unless ($_ == 0 or $_ == 1) {
                    croak("Invalid Filter threshold!");
                }
            }

        }

        $cgi->{$key} = @value > 1 ? \@value : $value[0];
    }
    return 1;
}

# Function  : Performs the necessary operations, determines which
#             page to display and displays a complete page. Displays
#             an error page if anything fails.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_page {
    my ($self) = @_;

    # Execute corresponding function based on type of page
    my $page = $self->cgi_param('page') || 'welcome';

    eval {
        if    ($page eq 'welcome')       { $self->make_welcome_page; }
        elsif ($page eq 'overview')      { $self->make_overview_page; }
        elsif ($page eq 'search_result') { $self->make_search_result_page; }
        elsif ($page eq 'region')        { $self->make_region_page; }
        elsif ($page eq 'centered')      { $self->make_centered_page; }
        elsif ($page eq 'text_dump')     { $self->make_text_dump_page; }
        elsif ($page eq 'list')          { $self->make_list_page; }
        elsif ($page eq 'edit_params')   { $self->make_edit_params_page; }
        else                             { $self->make_welcome_page; }
    };

    $self->die_with_error($@) if $@;

    return 1;
}

# Function  : Generates initial welcome page.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_welcome_page {
    my ($self) = @_;

    $self->display_header;
    $self->display_stored_params;
    $self->display_search;
    $self->display_footer;
}

# Function  : Generates header.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_header {
    my ($self) = @_;

    my $css         = $self->_get_file($self->cfg('CSS'));
    my $page_header = $self->cfg('PAGE_HEADER');
    my $page_title  = $self->cfg('PAGE_TITLE');
    my $root_url    = $self->root_url;
    my $session_id  = $self->session_id;

    my $cookie                 = $self->cfg('COOKIE');
    my $cookie_expires_in_min  = $self->cfg('COOKIE_EXPIRES_IN_MIN');

    my $db_selector = $self->make_db_selector;

    my $cookie_obj = CGI::cookie(
        -name    => $cookie,
        -value   => $session_id,
        -expires => "+${cookie_expires_in_min}m",
    );

    print CGI::header(-cookie => $cookie_obj);

    print <<HTML;
<head>
<title>SNP Alignment Display</title>
<meta name="robots" content="noindex,nofollow" />
<meta http-equiv="CACHE-CONTROL" content="NO-CACHE"/>
<meta http-equiv="PRAGMA" content="NO-CACHE"/>
<style type="text/css">
<!--
$css
-->
</style>
</head>
<body>
$page_header
<table width="940" border="0">
        <tr>
        <td></td>
        <td>
        <table>
                <tr>
                <td>&nbsp;</td>
                <td>
                <p>
                <h1>$page_title</h1>
                <p>
                </td>
                </tr>
        <tr>
                <td>&nbsp;</td>
                <td>
                <table>
                <tr>
                <td><a href="$root_url?session_id=$session_id"><b>[New Search]</b></a></td>
                <td><a href="$root_url?session_id=$session_id&page=list"><b>[List of Assays]</b></a></td>
                <td>$db_selector</td>
                </tr>
                </table>
                <p>
                </td>
                </tr>
        </table>
        </td>
        <td></td>
        </tr>
        <tr>
                <td>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</td>
                <td>
                <br>
HTML

    $self->printed_header(1);

    return 1;
}

# Function  : Generates footer.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_footer {
    my ($self) = @_;

    my $debug            = $self->cfg('DEBUG');
    my $page_footer_note = $self->cfg('PAGE_FOOTER_NOTE');
    my $page_footer      = $self->cfg('PAGE_FOOTER');

    my $root_url = $self->root_url;

    my $supplement;
    if ($debug > 0) {
        $supplement .=
            '<pre class="small_box2">' . "\n\n"
          . "<b>&nbsp;Version information:</b>\n\n"
          . $self->_get_id_information
          . "\n\n</pre>\n";
    }
    if ($debug > 2) {
        $supplement .=
            '<pre class="small_box2">' . "\n\n"
          . "<b>&nbsp;Environment:</b>\n\n"
          . $self->_get_env_information
          . "\n\n</pre>\n";
    }
    if ($debug > 3) {
        $supplement .=
            '<pre class="small_box2">' . "\n\n"
          . "<b>&nbsp;Loaded files:</b>\n\n"
          . $self->_get_inc_information
          . "\n\n</pre>\n";
    }

    print <<HTML;
                </td>
                <td>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</td>
        </tr>

        <tr>
                <td>&nbsp;</td>
                <td>
                <br>
                <br>
                <br>
                <font class="highlight">$page_footer_note</font>
                <br>
$supplement
                </td>
                <td>&nbsp;</td>
        </tr>
</table>
$page_footer
HTML
}

# Function  : Generates an overview page for an alignment. Uses
#             interface-specific get_overview_info subroutine to
#             display information about sequences.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_overview_page {
    my ($self) = @_;

    # Clean temp dir
    $self->clean_tmp_dir;

    my $assay_id = $self->cgi_param('assay_id');

    my $format_align_cfg       = $self->cfg('FORMAT_ALIGN_CFG');
    my $assay_cache_dir        = $self->cfg('ASSAY_CACHE_DIR');
    my $assay_cache_dir_eq     = $self->cfg('ASSAY_CACHE_DIR_EQ');
    my $cache                  = $self->cfg('CACHE');
    my $new_window_links       = $self->cfg('NEW_WINDOW_LINKS');
    my $overview_side_info     = $self->cfg('OVERVIEW_SIDE_INFO');
    my $initial_coverage_value = $self->cfg('INITIAL_COVERAGE_VALUE');

    my $mismatch = $self->cfg('MISMATCH');

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    my $target = $new_window_links ? qq[target="_blank"] : " ";

    # Filtering options
    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $filter             = $self->cgi_param('filter');

    # Selection box
    my $display_selection_box = $self->cfg('DISPLAY_SELECTION_BOX');

    # Check validity
    if ($assay_id =~ /[^\d]/) {
        croak("Assay id ($assay_id) must be a number");
    }

    # Get data - OBSERVE CACHE RULES!!!
    my ($data_file, $al);
    eval { ($data_file, $al) = $self->cache_assay_id($assay_id); };
    croak($@) if $@;

    # Get overview info
    my $overview_info =
      $self->get_overview_info($al, $al->longest_seq_len, 'overview');

    # Create overview image  - OBSERVE CACHE RULES!!!
    my $overview_file =
      "$assay_cache_dir/" . $self->_get_base_name('image', $assay_id);
    my $html = $overview_file . '.small_alignment.html';
    my $png  = $overview_file . '.small_alignment.png';

    if (   (!-e $html or !-e $png)
        or ((-e $html and -e $png) and !$cache)) {
        my $re = LookAlign::Alignment::Renderer->new(
            'alignment' => $al,
            'config'    => $format_align_cfg,
            'globals'   => ['consensus_no_gaps'],
            'ruler'     => 1,
            'out'       => $overview_file,
            'mismatch'  => $mismatch,
        ) or croak($@);

        # Render image
        ($html) = $re->render_small_alignment($n_threshold, $pic_threshold);
    }

    # Capture html
    my $content;
    {
        local $/;
        open(IN, "<$html") or croak("Cannot open $html");
        $content = <IN>;
        close IN;
    }

    # Replace png
    $content =~ s/\[PNG=([^\[\]]+)\]/$assay_cache_dir_eq\/$1/g;

    # Replace pos and unique_id
    my ($data_filename) = $data_file =~ /([^\/]+)$/;
    $content =~
      s/\[POS=([^\[\]]+)\]/ $target href=\"$root_url?session_id=$session_id&data=$data_filename&coverage=$initial_coverage_value&centered=$1&page=centered\" /g;
    $content =~
      s/\[UNIQUE_ID=([^\[\]]+)\]/ $target href=\"$root_url?text_dump=fasta&data=$data_filename&location=cache&page=text_dump&unique_id=$1&session_id=$session_id\" /g;

    # Write HTML
    eval {
        $self->display_header;
        print qq[<table width="100%">\n];

        print qq[<tr>\n];
        print qq[<th colspan="2" valign="top" align="left">\n];
        $self->display_dump_selector('cache', "$data_filename");
        print qq[$overview_info\n];
        print
          qq[Please click on a SNP candidate to retrieve a complete alignment of that region<p>\n];
        print qq[</th>\n];
        print qq[</tr>\n];

        print qq[<tr>\n];

        print qq[<td width="35%" valign="top" align="left">\n];
        $self->display_filtering_report($al);
        print qq[</td>\n];

        print qq[<td width="25%" valign="top" align="left">\n];
        print qq[<table>\n];
        print qq[<TR valign="top" align="left"><td>\n];
        $self->display_selection_box($al) if $display_selection_box;
        print qq[</td></tr>\n];
        print qq[<tr><td>\n];
        $self->display_selection_report if $display_selection_box;
        print qq[</td></tr>\n];
        print qq[</table>\n];
        print qq[</td>\n];

        print qq[<td width="40%" valign="top" align="left">\n];
        print $overview_side_info;
        print qq[</td>\n];

        print qq[</tr>\n];

        print qq[</table>\n];

        print qq[<P>\n];
        print qq[$content<p>\n];

        $self->display_footer;
    } or croak($@);

    return 1;
}

# Function  : Generates section that allows text dumps.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_dump_selector {
    my ($self, $location, $data) =
      @_;    # Special case, different types of locations can be present

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    print <<FORM;
<form method="get" action="$root_url">
     <i>Select format to dump these sequences in text</i>&nbsp;
     <select name="text_dump" value="tab_delimited" MAXLENGTH="40">
     <option>tab_delimited</option>
     <option>fasta</option>
     </select>&nbsp;
     <input type="hidden" name="data" value="$data">
     <input type="hidden" name="location" value="$location">
     <input type="hidden" name="page" value="text_dump">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="dump">
</form>
FORM

    return 1;
}

# Function  : Generates filtering report section.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_filtering_report {
    my ($self, $al) = @_;

    my $filter = $self->cgi_param('filter');

    unless ($filter) {
        print <<HTML;
<table class="small_box">
<tr><td><b>Filtering Report:</b></td><td></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
<tr><td>- no filtering was applied -</td><td></td></tr>
</table>
<br>
<br>
HTML
        return 1;
    }

    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $suppress_gaps_txt  = $suppress_gaps ? 'ON' : 'OFF';

    my @ref_sequences;
    my @disabled_sequences;

    my @sequences = $al->sequences;
    foreach my $s (@sequences) {
        if ($s->exists_attribute('ref_sequence')) {
            push @ref_sequences,
              join(' ', '<b>', $s->label, '</b>', '&nbsp;length:', $s->len,
                '&nbsp;valid_length:', $s->valid_len);
        }
        if (    $s->exists_attribute('disable')
            and $s->attribute_value('disable') eq 'filter') {
            push @disabled_sequences,
              join(' ', '<b>', $s->label, '</b>', '&nbsp;length:', $s->len,
                '&nbsp;valid_length:', $s->valid_len);
        }
    }

    my $ref_sequence_rows;
    foreach my $r (@ref_sequences) {
        $ref_sequence_rows .= qq[<tr><td>$r</td><td></td></tr>\n];
    }

    unless ($ref_sequence_rows) {
        $ref_sequence_rows = qq[<tr><td> - none - </td><td></td></tr>\n];
    }

    my $disabled_sequence_rows;
    foreach my $r (@disabled_sequences) {
        $disabled_sequence_rows .= qq[<tr><td>$r</td><td></td></tr>\n];
    }

    unless ($disabled_sequence_rows) {
        $disabled_sequence_rows = qq[<tr><td> - none - </td><td></td></tr>\n];
    }

    print <<HTML;
<table class="small_box">
<tr><td><b>Filtering Report:</b></td><td></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
<tr><td>Quality Score Threshold: <b>$qscore_threshold</b></td><td></td></tr>
<tr><td>Identity Percent Similarity to Ref Sequence Threshold: <b>$identity_threshold</b></td><td></td></tr>
<tr><td>Suppress Common Gaps: <b>$suppress_gaps_txt</b></td><td></td></tr>
<tr><td>&nbsp;</b></td><td></td></tr>
<tr><td><b>The following sequence was used as the reference sequence to eliminate sequences:</b></td><td></td></tr>
$ref_sequence_rows
<tr><td>&nbsp;</b></td><td></td></tr>
<tr><td><b>The following sequences were eliminated during filtering and not considered for calculations:</b></td><td></td></tr>
$disabled_sequence_rows
</table>
HTML

    return 1;
}

# Function  : Generates selection criteria box.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_selection_box {
    my ($self, $al) = @_;

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;
    my $assay_id   = $self->cgi_param('assay_id');

    my %selection;
    my (@sequences) = $al->sequences;
    foreach my $sequence (@sequences) {
        my @attributes = $sequence->attributes;

        foreach my $attribute (@attributes) {
            if ($attribute =~ /^selection:/) {
                my $value = $sequence->attribute_value($attribute);

                $selection{$attribute}{$value}++;
            }
        }
    }

    print <<HTML;
<table class="small_box" valign="top" align="left">
<tr><td colspan="2"><b>The Following Selection Criteria are Available:</b></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
<form method="get" action="$root_url">
HTML

    foreach my $selection (sort keys %selection) {
        my ($selection_name) = $selection =~ /^selection:(.+)/;
        $selection_name =~ s/_/ /g;
        $selection_name =~ s/^(.)/uc($1)/e;

        print qq[<tr><td><B>$selection_name</B></td><td>\n];
        my @criteria = sort keys %{$selection{$selection}};
        print qq[<select name="$selection" size="2" multiple>\n];
        foreach my $criteria (@criteria) {
            if ($selection{$selection}{$criteria} == 1) {
                print
                  qq[<option value="$criteria">$criteria (single seq)</option>\n];
            }
            else {
                print qq[<option value="$criteria">$criteria</option>\n];
            }
        }
        print qq[</select>\n];
        print qq[</td></tr>\n];
    }

    print <<HTML;
<input type="hidden" name="assay_id" value="$assay_id">
<input type="hidden" name="page" value="search_result">
<input type="hidden" name="search_type" value="assay_id">
<input type="hidden" name="session_id" value="$session_id">
<tr><td>&nbsp;</td><td>
<INPUT type="submit" value="re-display">
</td></tr>
</form>
</td></tr>
</table>
HTML

    return 1;
}

# Function  : Generates selection report section.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_selection_report {
    my ($self) = @_;

    my $sc = $self->selection_criteria;

    unless (keys %$sc) {
        print <<HTML;
<table class="small_box">
<tr><td><b>Selection Report:</b></td><td></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
<tr><td>- no selection criteria selected -</td><td></td></tr>
</table>
<br>
<br>
HTML
        return 1;
    }

    print <<HTML;
<table class="small_box">
<tr><td><b>Selection Report:</b></td><td></td></tr>
<tr><td>Currently displaying sequences based on the following selections</td><td></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
HTML

    foreach my $selection (sort keys %$sc) {
        my @criteria = sort @{$sc->{$selection}};
        my ($selection_name) = $selection =~ /^selection:(.+)/;
        $selection_name =~ s/_/ /g;
        $selection_name =~ s/^(.)/uc($1)/e;

        print qq[<tr><td><b>$selection_name: </b>]
          . join('; ', @criteria)
          . qq[</td><td></td></tr>\n];
    }

    print <<HTML;
<tr><td>&nbsp;</b></td><td></td></tr>
</table>
HTML

    return 1;
}

# Function  : Generates a search result page. If a search of type
#             "assay_id" is called - this is a generic call - uses
#             the make_overview_page sub. Otherwise, calls the
#             make_interface_result_page sub, which is the
#             interface-specific sub. Later decisions are made by the
#             interface-specific sub.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_search_result_page {
    my ($self) = @_;

    my $search_type = $self->cgi_param('search_type');

    if ($search_type eq 'assay_id') { $self->make_overview_page; }
    else { $self->make_interface_result_page() }  # *** INTERFACE-SPECIFIC ***

    return 1;
}

# Function  : Generates an alignment page centered around a point.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_centered_page {
    my ($self) = @_;

    my $coverage = $self->cgi_param('coverage');
    my $centered = $self->cgi_param('centered');

    # Check validity
    if ($centered =~ /[^\d]/) {
        croak("Centered ($centered) must be a (positive) number");
    }
    if ($coverage =~ /[^\d]/) {
        croak("Value ($coverage) must be a (positive) number");
    }

    my ($start, $end) = ($centered - $coverage, $centered + $coverage);

    if ($start < 1) { $start = 1 }

    $self->cgi_param('start', $start);
    $self->cgi_param('end',   $end);

    # Write HTML
    eval {
        $self->display_header;
        print qq[<p>\n];
        $self->display_alignment;
        $self->display_footer;
    };
    croak($@) if $@;

    return 1;
}

# Function  : Generates an alignment page for given start and end points.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_region_page {
    my ($self) = @_;

    my $coverage = $self->cgi_param('coverage');
    my $centered = $self->cgi_param('centered');
    my $start    = $self->cgi_param('start');
    my $end      = $self->cgi_param('end');

    # Check validity
    if ($centered =~ /[^\d]/) {
        croak("Centered ($centered) must be a (positive) number");
    }
    if ($coverage =~ /[^\d]/) {
        croak("Value ($coverage) must be a (positive) number");
    }
    if ($start =~ /[^\d]/) {
        croak("Start ($start) must be a (positive) number");
    }
    if ($end =~ /[^\d]/) { croak("End ($end) must be a (positive) number"); }

    # Write HTML
    eval {
        $self->display_header;
        print qq[<p>\n];
        $self->display_alignment;
        $self->display_footer;
    };
    croak($@) if $@;

    return 1;
}

# Function  : Generates a page with list of alignments stored in the system.
#             Calls the interface specific display_list subroutine.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_list_page {
    my ($self) = @_;

    eval {
        $self->display_header;
        $self->display_list;
        $self->display_footer;
    };
    croak($@) if $@;

    return 1;
}

# Function  : Generates parameter editing page.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_edit_params_page {
    my ($self) = @_;

    eval {
        $self->display_header;
        $self->display_edit_params_form;
        $self->display_footer;
    };
    croak($@) if $@;

    return 1;
}

# Function  : Generates text dumps. Uses interface-specific get_text_dump_info
#             subroutine to display information about sequences.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_text_dump_page {
    my ($self) = @_;

    my $location  = $self->cgi_param('location');
    my $data      = $self->cgi_param('data');
    my $text_dump = $self->cgi_param('text_dump');
    my $unique_id = $self->cgi_param('unique_id');

    my $tmp_dir         = $self->cfg('TMP_DIR');
    my $assay_cache_dir = $self->cfg('ASSAY_CACHE_DIR');

    my $mismatch = $self->cfg('MISMATCH');

    # Check validity
    unless ($text_dump eq 'fasta' or $text_dump eq 'tab_delimited') {
        croak("Textdump type ($text_dump) is not valid");
    }

    my $data_file;
    if ($location eq 'tmp') { $data_file = $tmp_dir . '/' . $data }
    elsif ($location eq 'cache') {
        $data_file = $assay_cache_dir . '/' . $data;
    }
    else { croak("Invalid URL parameter (location:$location)"); }

    # Create alignment object from data_file
    my $al = LookAlign::Alignment::Container->new or croak($@);
    $al->load_from_file($data_file) or croak($@);

    # Get information
    my $title = $self->get_text_dump_info($al);

    my $tab_delimited_dump;
    $tab_delimited_dump .= "Label\tPosition on Alignment\tAssay\tSequence\n";
    $tab_delimited_dump .=
      "         \t                     \t     \t"
      . $self->generate_ruler($al->offset, $al->longest_seq_len) . "\n";

    my $fasta_dump;

    my @sequences = $al->valid_sequences;
    foreach my $s (sort { $a->label cmp $b->label } @sequences) {
        if ($unique_id and $s->attribute_value('unique_id') ne $unique_id)
        {    # if a unique id is provided, we want a single sequence
            next;
        }

        my $label = $s->label;

        my $sequence = $s->sequence;

        # Get start end
        my $pos = ($al->offset + 1) . "-" . ($al->offset + length($sequence));

        $tab_delimited_dump .= "$label\t$pos\t$title\t$sequence\n";

        my $header = $label . ':' . $pos . ' ' . $title;
        $sequence =~ s/(.{60})/$1\n/g;
        chomp $sequence;
        $fasta_dump .= ">$header\n$sequence\n";
    }

    if (!$unique_id)
    {    # if a unique id is provided, we don't want a consensus sequence
            # Determine which consensus to use based on mismatch type
        my $consensus;

        if ($mismatch eq 'IUPAC') {
            my @consensus = $al->global_value_ary('consensus_no_gaps_iupac');
            foreach (@consensus) { s/[ ]/-/; }; # Replace space with dash when dumping
            $consensus = join('', @consensus);
        }
        elsif ($mismatch =~ /^.$/) {
            my @consensus = $al->global_value_ary('consensus_no_gaps');
            foreach (@consensus) { s/[ ]/-/; }; # Replace space with dash when dumping
            $consensus = join('', @consensus);
            $consensus =~ s/\*/$mismatch/g;
        }
        else {
            croak("Invalid mismatch representation ($mismatch)!");
        }

        # Get start end
        my $pos =
          ($al->offset + 1) . "-" . ($al->offset + length($consensus));

        $tab_delimited_dump .= "CONS\t$pos\t$title\t$consensus\n";

        $consensus =~ s/(.{60})/$1\n/g;
        chomp $consensus;
        $fasta_dump .=
          '>' . 'CONS' . ':' . $pos . ' ' . $title . "\n" . $consensus . "\n";
    }

    # Write HTML
    print CGI::header(-type => 'text/plain');
    if ($text_dump eq 'fasta')         { print qq[$fasta_dump]; }
    if ($text_dump eq 'tab_delimited') { print qq[$tab_delimited_dump]; }

    return 1;
}

###################
# PAGE GENERATION #
###################

# Function  : Generates an alignment given start and end points.
#             Uses interface-specific get_overview_info subroutine to display
#             information about sequences.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_alignment {
    my ($self) = @_;

    my $data     = $self->cgi_param('data');
    my $start    = $self->cgi_param('start');
    my $end      = $self->cgi_param('end');
    my $coverage = $self->cgi_param('coverage');
    my $centered = $self->cgi_param('centered');

    my $assay_cache_dir  = $self->cfg('ASSAY_CACHE_DIR');
    my $tmp_dir          = $self->cfg('TMP_DIR');
    my $tmp_dir_eq       = $self->cfg('TMP_DIR_EQ');
    my $format_align_cfg = $self->cfg('FORMAT_ALIGN_CFG');

    my $alignment_side_info = $self->cfg('ALIGNMENT_SIDE_INFO');

    my $mismatch = $self->cfg('MISMATCH');

    # Check validity
    if ($start =~ /[^\d\-]/) { croak("Start ($start) must be a number"); }
    if ($end =~ /[^\d]/) { croak("End ($end) must be a (positive) number"); }
    if ($start > $end) {
        croak("Start ($start) cannot be greater than end ($end)");
    }

    my $data_file = $assay_cache_dir . '/' . $data;

    # Create alignment object from data_file
    my $al = LookAlign::Alignment::Container->new or croak($@);
    $al->load_from_file($data_file) or croak($@);

    # Check number of sequences in alignment
    if ($al->sequences < 2) {
        my $message = <<MESSAGE;
At least two sequences are required to display an alignment and processing
information. Use your browser's back button and make a new selection
that contains at least two sequences. Please note that
selection criteria that contain only one sequence are labeled.
MESSAGE
        $self->die_with_error($message);
    }

    # Get data subset
    my ($subset_fh, $subset_file) =
      tempfile('XXXXXXXX', DIR => $tmp_dir, SUFFIX => '.xml');
    $al->dump_container($subset_file, $start, $end);

    # Generate alignment
    my $al_sub = LookAlign::Alignment::Container->new or croak($@);
    $al_sub->load_from_file($subset_file) or croak($@);
    my ($subset_filename) = $subset_file =~ /([^\/]+)$/;

    # Get frequencies, pic_values and N (number of seqs used) - stat content
    my @f = $al_sub->global_value_ary('frequencies');
    my @p = $al_sub->global_value_ary('pic_values');
    my @n = $al_sub->global_value_ary('N');

    my @c = $al_sub->global_value_ary('consensus_no_gaps');

    my $stat_content = '';
    foreach my $i (0 .. $#f) {
        if ($f[$i]) {
            my $pos = $al_sub->offset + $i + 1;
            $stat_content .=
              "<b>POS:</b> $pos <b>FREQUENCIES:</b> $f[$i] <b>N:</b> $n[$i] <b>PIC:</b> $p[$i]\n"
              ;    # .. $c[$i]\n"; # - DEBUG
        }
    }

    # Get alignment info
    my $alignment_info =
      $self->get_overview_info($al_sub, $al->longest_seq_len, 'alignment');

    # Temp file - alignment image set
    my ($alignment_fh, $alignment_file) =
      tempfile('XXXXXXXX', DIR => $tmp_dir);

    # Create Renderer object
    my $re = LookAlign::Alignment::Renderer->new(
        'alignment' => $al_sub,
        'config'    => $format_align_cfg,
        'globals'   => ['consensus_no_gaps'],
        'ruler'     => 1,
        'out'       => $alignment_file,
        'mismatch'  => $mismatch,
    ) or croak($@);

    # Render image
    my ($html) = $re->render_alignment;

    # Capture html
    my $content;
    {
        local $/;
        open(IN, "<$html") or croak("Cannot open $html");
        $content = <IN>;
        close IN;
    }

    # Replace png
    $content =~ s/\[PNG=([^\[\]]+)\]/$tmp_dir_eq\/$1/g;

    # Write HTML
    $self->display_align_adjuster;
    $self->display_specific_align;
    # Make sure this is the subset file, otherwise complete sequences will be dumped
    $self->display_dump_selector('tmp', $subset_filename);
    print qq[<br>\n];
    print qq[<table width="100%" border="0">\n];
    print qq[<tr>\n];
    print qq[<td width="40%" valign="top" align="left">\n];
    print qq[$alignment_info\n];
    print qq[<p>\n];
    print qq[<pre>\n];
    print qq[$stat_content<p>];
    print qq[</pre>\n];
    print qq[<p>\n];
    print qq[</td>\n];
    print qq[<td width="60%" valign="top" align="left">\n];
    print $alignment_side_info;
    print qq[</td>\n];
    print qq[</tr>\n];
    print qq[</table>\n];
    print qq[<P>\n];

    print qq[$content<p>\n];
    print qq[<p>\n];

    return 1;
}

# Function  : Generates a box that contains parameters in effect.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_stored_params {
    my ($self) = @_;

    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $suppress_gaps_txt  = 'ON';
    $suppress_gaps_txt = 'OFF' unless $suppress_gaps;
    my $filter     = $self->cgi_param('filter');
    my $filter_txt = 'ON';
    $filter_txt = 'OFF' unless $filter;

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    my $edit_params_link =
      "$root_url?session_id=$session_id&page=edit_params";

    print <<HTML;
<table class="small_box">
<tr><td><b>Currently the following parameters are stored and in effect where applicable:</b></td><td></td></tr>
<tr><td>&nbsp;</td><td></td></tr>
<tr><td>PIC (Polmorphism Information Content) Threshold:      </td><td><b>$pic_threshold</b></td></tr>
<tr><td>N (Number of sequences) Threshold:                    </td><td><b>$n_threshold</b></td></tr>
<tr><td>Quality Score Threshold (in effect when filtering is on): </td><td><b>$qscore_threshold</b></td></tr>
<tr><td>Identity Percent Similarity to Ref Sequence Threshold (in effect when filtering is on):&nbsp;&nbsp;</td><td><b>$identity_threshold</b></td></tr>
<tr><td>Suppress Common Gaps (in effect when filtering is on): </td><td><b>$suppress_gaps_txt</b></td></tr>
<tr><td>Filtering is currently:                               </td><td><b>$filter_txt</b></td></tr>
<tr><td>&nbsp;</b></td><td></td></tr>
<tr><td><b>Click <a href="$edit_params_link">here</a> to edit parameters ...<b></td><td></td></tr>
</table>
<br>
<br>
HTML
    return 1;
}

# Function  : Generates parameter editing section.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_edit_params_form {
    my ($self) = @_;

    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps') ? 1 : 0;
    my $filter             = $self->cgi_param('filter') ? 1 : 0;

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    my $OPTIONS_PIC_THRESHOLD      = $self->cfg('OPTIONS_PIC_THRESHOLD');
    my $OPTIONS_N_THRESHOLD        = $self->cfg('OPTIONS_N_THRESHOLD');
    my $OPTIONS_QSCORE_THRESHOLD   = $self->cfg('OPTIONS_QSCORE_THRESHOLD');
    my $OPTIONS_IDENTITY_THRESHOLD = $self->cfg('OPTIONS_IDENTITY_THRESHOLD');
    my $OPTIONS_SUPPRESS_GAPS      = $self->cfg('OPTIONS_SUPPRESS_GAPS');
    my $OPTIONS_FILTER             = $self->cfg('OPTIONS_FILTER');

    my $select_pic_threshold =
      $self->_select_box('pic_threshold',
        [split(',', $OPTIONS_PIC_THRESHOLD)], $pic_threshold);
    my $select_n_threshold =
      $self->_select_box('n_threshold', [split(',', $OPTIONS_N_THRESHOLD)],
        $n_threshold);
    my $select_qscore_threshold =
      $self->_select_box('qscore_threshold',
        [split(',', $OPTIONS_QSCORE_THRESHOLD)], $qscore_threshold);
    my $select_identity_threshold =
      $self->_select_box('identity_threshold',
        [split(',', $OPTIONS_IDENTITY_THRESHOLD)], $identity_threshold);
    my $select_suppress_gaps =
      $self->_select_box('suppress_gaps',
        [split(',', $OPTIONS_SUPPRESS_GAPS)], $suppress_gaps);
    my $select_filter =
      $self->_select_box('filter', [split(',', $OPTIONS_FILTER)], $filter);

    print <<HTML;
<form method=get action="$root_url">
     <h3>Change parameters and click save:</h3>
        <table class="box">
        <tr><td>&nbsp;</td><td></td></tr>
        <tr><td>PIC (Polmorphism Information Content):                            </td><td>$select_pic_threshold</td></tr>
        <tr><td>N (Number of sequences) Threshold:                                </td><td>$select_n_threshold</td></tr>
        <tr><td>Quality Score Threshold:                                          </td><td>$select_qscore_threshold</td></tr>
        <tr><td>Identity Percent Similarity to Ref Sequence Threshold:&nbsp;&nbsp;</td><td>$select_identity_threshold</tr>
        <tr><td>Suppress Common Gaps:                                             </td><td>$select_suppress_gaps</td></tr>
        <tr><td>Filtering:                                                        </td><td>$select_filter</td></tr>
        <tr><td>&nbsp;</td><td></td></tr>
        <tr><td>&nbsp;</b></td><td><input type="hidden" name="session_id" value="$session_id"></td></tr>
        <tr><td>&nbsp;</b></td><td><input type="submit" value="Save"></td></tr>
        </table>
     <p>
     <p>
<input type="hidden" name="modify_params" value="1">
</form>
HTML
    return 1;
}

# Function  : Generates section in the welcome page that displays
#             available search boxes. Calls the interface-specific
#             get_search_forms subroutine.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_search {
    my ($self) = @_;

    my $search_forms = $self->get_search_forms;   # *** INTERFACE-SPECIFIC ***

    print <<FORM;
<table>

<tr>
<td>
<h2>This viewer can retrieve alignments by the following methods:</h2>
<br>
</td>
</tr>

$search_forms

</table>
FORM

    return 1;
}

# Function  : Generates section that allows extending alignment
#             view around a SNP candidate.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_align_adjuster {
    my ($self) = @_;

    my $data     = $self->cgi_param('data');
    my $coverage = $self->cgi_param('coverage');
    my $centered = $self->cgi_param('centered');

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    print <<FORM;
<form method="get" action="$root_url">
     <i>Enter number of bases to view around the SNP candidate to expand/shorten display:</i>&nbsp;
     <input type="text" name="coverage" value="$coverage" size="10" MAXLENGTH="40">&nbsp;
     <input type="hidden" name="data" value="$data">
     <input type="hidden" name="centered" value="$centered">
     <input type="hidden" name="page" value="centered">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="re-display">
</form>
FORM

    return 1;
}

# Function  : Generates section that allows specifying arbitrary
#             start and end positions in an alignment.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_specific_align {
    my ($self) = @_;

    my $data     = $self->cgi_param('data');
    my $coverage = $self->cgi_param('coverage');
    my $centered = $self->cgi_param('centered');
    my $start    = $self->cgi_param('start');
    my $end      = $self->cgi_param('end');

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    print <<FORM;
<form method="get" action="$root_url">
     <i>Enter a start position</i>&nbsp;<input type="text" name="start" value="$start" size="10" MAXLENGTH="40">&nbsp;
     <i>and an end position</i>&nbsp;<input type="text" value="$end" name="end" size="10" MAXLENGTH="40">&nbsp;
     <i>to display a specific region</i>&nbsp;
     <input type="hidden" name="data" value="$data">
     <input type="hidden" name="coverage" value="$coverage">
     <input type="hidden" name="centered" value="$centered">
     <input type="hidden" name="page" value="region">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="re-display">
</form>
FORM
}

###################
# UTILITY METHODS #
###################

# Function  : Generates a select box given options and the selected option.
# Arguments : $label, \@options, $selected_option
# Returns   : $html_code
# Notes     : None specified.

sub _select_box {
    my ($self, $label, $ref_options, $selected_option) = @_;
    my @options = @$ref_options;

    my $select;

    if ($options[0] =~ /text/i) {
        $select =
          qq[<input type="text" name="$label" value="$selected_option" size="5" MAXLENGTH="3">];
        return $select;
    }

    $select = qq[<select name="$label" class="fixed_width">];
    foreach my $i (0 .. $#options) {
        my ($option, $option_label) =
          $options[$i] =~ /\[([^=]+)=([^=]+)\]/
          ? ($1, $2)
          : ($options[$i], $options[$i]);
        $option_label = sprintf("%3s", $option_label);
        $option_label =~ s/ /&nbsp;/g;

        my $selected = $option == $selected_option
            ? 'selected'
            : '';
        $select .=
          qq[<option value="$option" $selected>$option_label</option>];
    }
    $select .= qq[</select>];

    return $select;
}

# Function  : Returns content of a file.
# Arguments : $file
# Returns   : $file_content
# Notes     : None specified.

sub _get_file {
    my ($self, $file) = @_;
    open(IN, "<$file") or croak("Cannot read file ($file)");
    my $content;
    { local $/; $content = <IN>; };
    close IN;
    return $content;
}

# Function  : Generates environment variable information for debug section.
# Arguments : None
# Returns   : $html_code
# Notes     : None specified.

sub _get_env_information {
    my ($self) = @_;

    my $env;
    foreach my $env_variable (sort keys %ENV) {
        my $value = $ENV{$env_variable};

        #       $value =~ s/(.{100})/$1<br>/g if $value =~ /\S{100,}/;
        $env .= "&nbsp;<b>$env_variable</b>: " . $value . "\n";
    }
    return $env;
}

# Function  : Generates @INC information for debug section.
# Arguments : None
# Returns   : $html_code
# Notes     : None specified.

sub _get_inc_information {
    my ($self) = @_;

    my $inc;
    foreach my $inc_variable (sort keys %INC) {
        $inc .= "&nbsp;<b>$inc_variable</b>: " . $INC{$inc_variable} . "\n";
    }
    return $inc;
}

# Function  : Generates CVS id information of files for debug section.
# Arguments : None
# Returns   : $html_code
# Notes     : None specified.

sub _get_id_information {
    my ($self) = @_;

    my $time_stamp = $time{'dd-Mon-yyyy hh:mm:ss tz'};

    my $url = $ENV{HTTP_HOST} . $ENV{REQUEST_URI};
    $url =~ s/(.{100})/$1<br>/g if $url =~ /\S{100,}/;

    my $software;
    my @int_specific_modules =
      $self->get_critical_modules;    # *** INTERFACE-SPECIFIC ***
    foreach my $file ($0, $INC{'LookAlign/Alignment/Container.pm'},
        $INC{'LookAlign/Alignment/Container/Sq.pm'}, $INC{'LookAlign/Alignment/Renderer.pm'},
        @int_specific_modules) {
        my ($id) = $self->_get_version_information($file);
        my ($file_name) = $file =~ /([^\/]+)$/;
        $software .= "&nbsp;<b>$file_name</b>: $id\n";
    }
    chomp $software;

    my $int_specific_content =
      $self->get_critical_content;    # *** INTERFACE-SPECIFIC ***

    return
      join("\n", "&nbsp;<b>URL</b>: $url", "&nbsp;<b>Time</b>: $time_stamp",
        $software, "&nbsp;$int_specific_content");
}

# Function  : Captures CVS version information of files for _get_id_information.
# Arguments : $file
# Returns   : $version_id
# Notes     : None specified.

sub _get_version_information {
    my ($self, $file) = @_;
    open(IN, "<$file") or croak("Cannot read file ($file)");
    my $content;
    { local $/; $content = <IN>; }
    close IN;
    my ($id) = $content =~ /(\$Id[^\$]*\$)/;
    return ($id);
}

# Function  : Generates an error page.
# Arguments : $error
# Returns   : - exits with 0 -
# Notes     : None specified.

sub die_with_error {
    my ($self, @message) = @_;

    $self->display_header unless $self->printed_header;
    print qq[<font class="attention">ERROR:</font><br>\n];
    print qq[<p><p>\n];

    foreach my $message (@message) {

        # Replace newline with HTML breaks
        $message =~ s/\n+/<br>/g;

        # Format locations
        $message =~ s/(at \S+ line \d+)/<br>[Trace: $1]/g;

        # More cleaning/formatting
        $message =~ s/(<br>\s*<br>)/<br>/g;
        $message =~ s/<br>\[Trace: at/<br><br>[Trace: at/;

        # Now ready to print out
        print qq[<b>$message</b><br>\n];
    }

    print qq[<p><p>\n];
    $self->display_footer;

    exit 0;
}

# Function  : Generates a text-based ruler for text dumps.
# Arguments : $offset, $len
# Returns   : $text_content
# Notes     : None specified.

sub generate_ruler {
    my ($self, $offset, $len) = @_;
    my $ruler;

    for (my $i = $offset + 1 ; $i <= $offset + $len ; $i++) {
        if ($i == $offset + 1) { $ruler .= $i; $i += length($i) - 1; next; }
        elsif ($i % 10 == 0) { $ruler .= $i; $i += length($i) - 1; next; }
        else { $ruler .= ' '; }
    }

    return $ruler;
}

# Function  : Sets params.
# Arguments : None
# Returns   : 1
# Notes     : This method does *not* set cookies anymore,
#             uses session object to store params in bulk.

sub set_cookies {
    my ($self) = @_;

    my @params = qw[pic_threshold
      n_threshold
      qscore_threshold
      identity_threshold
      suppress_gaps
      filter];

    foreach my $param (@params) {
        if (defined $self->cgi_param($param)) {
            $self->session_param($param, $self->cgi_param($param));
        }

        elsif (defined $self->session_param($param)) {
            $self->cgi_param($param, $self->session_param($param));
        }

        else {
            my $default_param = 'DEFAULT_' . uc($param);
            $self->session_param($param, $self->cfg($default_param));
            $self->cgi_param($param, $self->cfg($default_param));
        }
    }

    return 1;
}

# Function  : Cleans temp directory.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub clean_tmp_dir {
    my ($self) = @_;

    my $tmp_dir                = $self->cfg('TMP_DIR');
    my $tmp_dir_expires_in_min = $self->cfg('TMP_DIR_EXPIRES_IN_MIN');

    foreach my $file_wild_card ('*.*.png', '*.xml', '*.alignment.html',
        '*.snp_search', '????????', 'cgisess_*') {
        my $clean_cmd =
          "find $tmp_dir -name \'$file_wild_card\' -cmin +$tmp_dir_expires_in_min -exec rm -f {} \\;";
        system $clean_cmd;
    }

    return 1;
}

#########
# OTHER #
#########

# Function  : Retrieves an assay and dumps in a file. uses the
#             interface-specific get_by_assay_id subroutine.
# Arguments : $assay_id
# Returns   : $data_file, $al
# Notes     : None specified.

sub cache_assay_id {
    my ($self, $assay_id) = @_;

    my $assay_cache_dir = $self->cfg('ASSAY_CACHE_DIR');
    my $tmp_dir         = $self->cfg('TMP_DIR');
    my $cache           = $self->cfg('CACHE');

    # Filtering options
    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $filter             = $self->cgi_param('filter');

    my $data_file =
      "$assay_cache_dir/" . $self->_get_base_name('data', $assay_id) . '.xml';

    my $al;

    if (!-e $data_file or !$cache) {

        # Dump raw data into temp dir
        my ($tmp_fh, $tmp_file) =
          tempfile('XXXXXXXX', DIR => $tmp_dir, SUFFIX => '.xml');
        eval { $self->get_by_assay_id($assay_id, $tmp_file) }; # *** INTERFACE-SPECIFIC ***
        croak($@) if $@;

        # Create alignment object from data_file
        $al = LookAlign::Alignment::Container->new or croak($@);
        $al->load_from_file($tmp_file) or croak($@);

        # Determine whether there are selection criteria
        my $selection_criteria = $self->selection_criteria;

        # Do filtering, calculate consensus
        eval {
            if (%{$selection_criteria}) {
                $al->filter_by_selection($selection_criteria);
            }
            if ($filter and $qscore_threshold) {
                $al->mask_low_quality_bases($qscore_threshold);
            }
            if ($filter and $identity_threshold) {
                $al->filter_alignment_cluster($identity_threshold);
            }
            if ($filter and $suppress_gaps) { $al->remove_common_gaps }
            $al->calculate_consensus;
        } or croak($@);

        # Dump into cache file
        $al->dump_container($data_file);
    }

    else {
        $al = LookAlign::Alignment::Container->new() or croak($@);
        $al->load_from_file($data_file) or croak($@);
    }

    return ($data_file, $al);
}

# Function  : Generates a standard index file name and index key for
#             assays to be dumped. Accounts for filtering and
#             selection options.
# Arguments : $type, $assay_id
# Returns   : $index_file, $index_key
# Notes     : None specified.

sub _get_base_name {
    my ($self, $type, $assay_id) = @_;

    my $assay_cache_dir = $self->cfg('ASSAY_CACHE_DIR');

    # Index file
    my $index_file = "$assay_cache_dir/$assay_id.idx";

    # Indexing information
    my $datasource = $self->get_datasource;

    # Filtering options
    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $filter             = $self->cgi_param('filter');

    #Image tag
    my @image_tag = ($n_threshold, $pic_threshold);

    # Filter tag
    my @filter_tag       = ($qscore_threshold, $identity_threshold, $suppress_gaps);
    my @empty_filter_tag = ('', '', '');

    # Selection options and tag
    my $sc = $self->selection_criteria;
    my @sc_tag;
    foreach my $selection (sort keys %$sc) {
        push @sc_tag, $selection;
        push @sc_tag, (sort @{$sc->{$selection}});
    }

    my @index_key;

    if ($type eq 'data') {
        if   ($filter) { @index_key = (@filter_tag,       'filtered',   @sc_tag); }
        else           { @index_key = (@empty_filter_tag, 'unfiltered', @sc_tag); }
    }

    elsif ($type eq 'image') {
        if ($filter) {
            @index_key = (@image_tag, @filter_tag,       'filtered',   @sc_tag);
        }
        else {
            @index_key = (@image_tag, @empty_filter_tag, 'unfiltered', @sc_tag);
        }
    }

    else {
        croak("Internal error: Base name type ($type) must be data or image");
    }

    my $index_key = join('-', $datasource, $type, @index_key);

    # Read index file contents (if one available)
    my %index;
    my $max_id = 0;
    if (-e $index_file) {
        open(IN, "<$index_file")
          or croak("Cannot read file ($index_file): $!");
        while (<IN>) {
            chomp;
            my ($cache_base, $index_key) = split("\t", $_);
            croak("Index file ($index_file) is corrupt!")
              if $index{$index_key};
            $index{$index_key} = $cache_base;
            my ($id) = $cache_base =~ /^\d+\-0*(\d+)/
              or croak("Cannot parse index file ($index_file)!");
            $max_id = $id if $id > $max_id;
        }
        close IN;
    }

    # Check if we already have the file name indexed
    if ($index{$index_key}) { return $index{$index_key}; }
    else {
        open(OUT, ">>$index_file")
          or croak("Cannot append file ($index_file): $!");
        flock(OUT, 2) or croak("Cannot lock file: $!");
        my $id = $max_id + 1;
        my $cache_base = "$assay_id-" . sprintf("%03s", $id);
        print OUT join("\t", $cache_base, $index_key) . "\n";
        close OUT;
        return $cache_base;
    }

    return 1;
}

# Function  : Retrieves the content for the directive specified;
#             supports GET (retrieval by LWP), EXEC (executes a command-line
#             and captures output), FILE (retrieves a file content).
# Arguments : $directive
# Returns   : $content
# Notes     : None specified.

sub content {
    my ($self, $container) = @_;

    return '&nbsp;' unless $container;

    my $content = $container;

    if ($container =~ s/^(FILE|EXEC|GET)://) {
        my $type = $1;

        if ($type eq 'GET') {
            $content = get($container)
              or croak("Cannor get container ($container)!");
        }

        elsif ($type eq 'EXEC') {
            open(EXEC, "$container|")
              or croak("Cannot exec container ($container)! - $!");
            { local $/; $content = <EXEC>; }
            close EXEC;
        }

        elsif ($type eq 'FILE') {
            open(FILE, "<$container")
              or croak("Cannot open container ($container)! - $!");
            { local $/; $content = <FILE>; }
            close FILE;
        }
    }

    return $content;
}

# Function  : Determine what call if any search type other than assay_id is used.
# Arguments : None.
# Returns   : 1
# Notes     : Interface-specific sub overridden by subclass.

sub make_interface_result_page {
    return 1;
}

# Function  : Retrieves assays by assay id.
# Arguments : $assay_id, $out
# Returns   : 1
# Notes     : Interface-specific sub overridden by subclass.

sub get_by_assay_id {
    return 1;
}

# Function  : Generates available search boxes.
# Arguments : None
# Returns   : $html_code
# Notes     : Interface-specific sub overridden by subclass.

sub get_search_forms {
    return 1;
}

# Function  : Returns interface specific modules
#             to be displayed in debug message.
# Arguments : None
# Returns   : @critical_modules
# Notes     : Interface-specific sub overridden by subclass.

sub get_critical_modules {
    return 1;
}

# Function  : Returns interface specific infomation
#             to be displayed in debug message.
# Arguments : None
# Returns   : $html_code
# Notes     : Interface-specific sub overridden by subclass.

sub get_critical_content {
    return 1;
}

# Function  : Returns interface specific datasource infomation
#             to be included in cache index.
# Arguments : None
# Returns   : $string
# Notes     : Interface-specific sub overridden by subclass.

sub get_datasource {
    return 1;
}

# Function  : Returns a db selection box.
# Arguments :
# Returns   : $html
# Notes     : Interface-specific sub overridden by subclass.

sub make_db_selector {
    return '';
}

##########################
# SIMPLE GET/SET METHODS #
##########################

sub access {
    my ($self, $value) = @_;
    $self->{access} = $value if @_ > 1;
    return $self->{access};
}

sub cgi {
    my ($self, $value) = @_;
    $self->{cgi} = $value if @_ > 1;
    return $self->{cgi};
}

sub config {
    my ($self, $value) = @_;
    $self->{config} = $value if @_ > 1;
    return $self->{config};
}

sub config_file {
    my ($self, $value) = @_;
    $self->{config_file} = $value if @_ > 1;
    return $self->{config_file};
}

sub default {
    my ($self, $value) = @_;
    $self->{default} = $value if @_ > 1;
    return $self->{default};
}

sub printed_header {
    my ($self, $value) = @_;
    $self->{printed_header} = $value if @_ > 1;
    return $self->{printed_header};
}

sub session {
    my ($self, $value) = @_;
    $self->{session} = $value if @_ > 1;
    return $self->{session};
}

sub session_dir {
    my ($self, $value) = @_;
    $self->{session_dir} = $value if @_ > 1;
    return $self->{session_dir};
}

sub session_id {
    my ($self, $value) = @_;
    $self->{session_id} = $value if @_ > 1;
    return $self->{session_id};
}

sub subtype {
    my ($self, $value) = @_;
    $self->{subtype} = $value if @_ > 1;
    return $self->{subtype};
}

=head1 AUTHOR

Payan Canaran <canaran@cshl.edu>

=head1 BUGS

=head1 VERSION

Version 0.01

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright (c) 2004-2007 Cold Spring Harbor Laboratory

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See DISCLAIMER.txt for
disclaimers of warranty.

=cut

1;
