package LookAlign::Interface::GDPDM;

our $VERSION = '0.01';

# $Id: GDPDM.pm,v 1.1.2.1 2007/06/14 19:03:07 kclark Exp $

=head1 NAME

LookAlign::Interface::GDPDM

=head1 DESCRIPTION

GDPDM Interface module for alignemnt viewer. Documentation is added to describe each section/subroutine.

=cut

use warnings;
use strict;

use base qw(LookAlign::Interface::Root);

use strict;

use Carp;
use CGI;
use File::Temp qw(tempfile tempdir);
use List::Util qw(first);
use Time::Format qw(%time);

use LookAlign::Alignment::Container;
use LookAlign::Alignment::Container::Sq;
use LookAlign::Alignment::Renderer;

use LookAlign::Panzea::WebFormUtils;

###########
# METHODS #
###########

# Function  : Generates and returns a database handle.
# Arguments : None
# Returns   : $dbh
# Notes     : None specified.

sub dbh {
    my ($self) = @_;

    my $datasource = $self->cfg('DATASOURCE');
    my $username   = $self->cfg('USERNAME');
    my $password   = $self->cfg('PASSWORD');

    my $db_access_params = $self->cfg('DB_ACCESS_PARAMS');

    my @available_databases;
    my $selected_db;

    if ($datasource && $username) {
        @available_databases = (
            { alias      => 'default',
              datasource => $datasource,
              username   => $username,
              password   => $password,
              }
        );

        $selected_db = $available_databases[0];
    }

    else {
        my $database = $self->session_param('db_selected');

        @available_databases =
            ref $db_access_params->{database} eq 'ARRAY'
            ? @{$db_access_params->{database}}
            : $db_access_params->{database};

        unless (@available_databases) {
            croak("No database specified!");
        }

        if (!$database) {
            $database = $available_databases[0]->{alias};
        }

        $selected_db =
          first { $_->{alias} eq $database } @available_databases;

        if (!defined($selected_db)) {
            $self->session_param('db_selected', '');
            croak("Cannot determine database ($database)!");
        }
    }

    my $dbh = DBI->connect(
       $selected_db->{datasource},
       $selected_db->{username}, $selected_db->{password},
       {PrintError => 1, RaiseError => 1}
    ) || croak("Cannot connect to database!");

    $self->{dbh} = $dbh;

    $self->session_param('db_selected', $selected_db->{alias});

    $self->available_databases(\@available_databases);

    return $dbh;
}

# Function  : Directs which result page to display based on search type.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_interface_result_page {
    my ($self) = @_;

    my $search_type = $self->cgi_param('search_type');

    if    ($search_type eq 'locus')     { $self->make_detailed_search_page; }
    elsif ($search_type eq 'pz_number') { $self->make_detailed_search_page; }
    else { croak("Internal error: Unknown search type ($search_type)"); }

    return 1;
}

# Function  : Generates result page for detailed searches.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub make_detailed_search_page {
    my ($self) = @_;

    my $search_type = $self->cgi_param('search_type');
    my $locus       = $self->cgi_param('locus');
    my $pz_number   = $self->cgi_param('pz_number');

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    # Filtering options
    my $pic_threshold      = $self->cgi_param('pic_threshold');
    my $n_threshold        = $self->cgi_param('n_threshold');
    my $qscore_threshold   = $self->cgi_param('qscore_threshold');
    my $identity_threshold = $self->cgi_param('identity_threshold');
    my $suppress_gaps      = $self->cgi_param('suppress_gaps');
    my $filter             = $self->cgi_param('filter');

    # Check validity
    if ($search_type eq 'locus' && $locus =~ /[^A-Za-z0-9]/) {
        croak("Locus ($locus) is not valid!");
    }
    if ($search_type eq 'pz_number' && $pz_number =~ /\s/) {
        croak("Marker name ($pz_number) is not valid!");
    }

    # Get assay ids
    my @assay_ids;
    if ($search_type eq 'locus') {
        eval { @assay_ids = $self->get_by_locus($locus) };
        croak($@) if $@;
    }
    if ($search_type eq 'pz_number') {
        eval { @assay_ids = $self->get_by_pz_number($pz_number) };
        croak($@) if $@;
    }

    # Display something based on availablity of assay ids
    if (@assay_ids == 0 && $search_type eq 'locus') {
        croak(
            "Cannot find any assay_id for this locus ($locus)<br>If you know that this is a valid locus and it contains valid assays please contact the maintainer of this tool"
        );
    }

    elsif (@assay_ids == 0 && $search_type eq 'pz_number') {
        croak(
            "Cannot find any assay_id for this valid marker name ($pz_number)<br>If you know that this is a valid marker name please contact the maintainer of this tool"
        );
    }

    elsif (@assay_ids >= 1) {

        # Get assay id related information - OBSERVE CACHE RULES!!!
        my %assay_info;
        foreach my $assay_id (@assay_ids) {
            my ($data_file, $al);
            eval { ($data_file, $al) = $self->cache_assay_id($assay_id) };
            croak($@) if $@;
            unless ($data_file) {
                croak(
                    "Cannot retrieve any *sequence* assays<br>If you know that this is a valid query that has one or more valid sequence assays performed please contact the maintainer of this tool"
                );
            }

            my $aln_length          = $al->longest_seq_len;
            my $number_of_sequences = $al->valid_sequences;
            my %sequences;
            foreach my $s ($al->valid_sequences) {
                my ($i) = $s->label =~ /^([^\(\)]+)/;
                $sequences{$i}++;
            }
            my $number_of_accessions = scalar(keys %sequences);

            my $filtering = '(no filtering)';
            $filtering = '(filtering applied)' if $filter;

            $assay_info{$assay_id} =
              "$filtering $number_of_sequences sequences / $number_of_accessions accessions in this alignment (total alignment length $aln_length)";
        }

        # Write HTML
        eval {
            $self->display_header;
            if ($search_type eq 'locus') {
                print qq[<h2>Assays available for locus $locus</h2><p>\n];
            }
            elsif ($search_type eq 'pz_number') {
                print
                  qq[<h2>Assays available for marker $pz_number</h2><p>\n];
            }

            foreach (sort { $a <=> $b } @assay_ids) {
                print qq[<form method=get action="$root_url">\n];
                print
                  qq[<b>Assay id:&nbsp;$_&nbsp;-&nbsp;$assay_info{$_}</b>\n];
                print qq[<input type="hidden" name="assay_id" value="$_">\n];
                print
                  qq[<input type="hidden" name="page" value="overview">\n];
                print
                  qq[<input type="hidden" name="session_id" value="$session_id">\n];
                print qq[<input type="submit" value="select"><p>\n];
                print qq[</form>\n];
            }
            print qq[<p>\n];
            $self->display_footer;
        };
        croak($@) if $@;
    }
    return 1;
}

# Function  : Placeholder for a db selection box.
#             Currently, returns selected database as a string
# Arguments :
# Returns   : $html
# Notes     :

sub make_db_selector {
    my ($self) = @_;

    my $dbh = $self->dbh; # This call is necessary to parse db_access_params
                          # and populate available databases

    my @available_databases = @{$self->available_databases};

    my $database   = $self->session_param('db_selected');

    my $selected_db =
        first { $_->{alias} eq $database } @available_databases;

    my $selected_display = $selected_db->{display};

    my $html = $selected_display && $selected_display ne 'default'
        ? qq[<b><i>Database:</i></b> $selected_display]
        : '';

#    my $session_id = $self->session_id;
#
#    my $select_url = $self->root_url . "?session_id=$session_id";
#
#    my $html;
#
#    if (@available_databases > 1) {
#        my $select;
#
#        my $current_db = '';
#
#        $select .=
#          qq[<select id="db_selector" name="database" onchange="select_db()">\n];
#        foreach my $available_database (@available_databases) {
#            my $alias           = $available_database->{alias};
#            my $display         = $available_database->{display};
#            my $selected_switch = qq[];
#            if ($alias eq $database) {
#                $selected_switch = 'selected="1"';
#                $current_db      = $display;
#            }
#
#            $select .=
#              qq[<option $selected_switch value="$alias">$display</option>\n];
#        }
#        $select .= qq[</select>\n];
#
#        $html = <<HTML;
#<table>
#<tr>
#<td>
#<form method="get" action="$select_url">
#<b><i>Current Database:</i></b> $current_db
#$select
#<input type="submit" value="Change">
#</form>
#<script type="text/javascript">
#<!--
#function select_db()
#{
#var database=document.getElementById("db_selector").value;
#var target_url= location.pathname
#    + "?database="   + database
#    + "&session_id=$session_id";
#window.location=target_url;
#}
#-->
#</script>
#</td>
#<!--
#<td id="javascript_warning">
#<a style="color:red; font-style:italic">
#(Javascript must be enabled for automatic refresh!)</a>
#</td>
#-->
#</tr>
#
#</table>
#
#<script type="text/javascript">
#<!--
#document.getElementById("javascript_warning").innerHTML = '';
#-->
#</script>
#
#HTML
#
#    }
#
#    else {
#        $html = '';
#    }

    return $html;
}

# Function  : Generates a list of available assays.
# Arguments : None
# Returns   : 1
# Notes     : None specified.

sub display_list {
    my ($self) = @_;

    my @list = @{$self->get_list};

    my $root_url         = $self->root_url;
    my $session_id       = $self->session_id;
    my $new_window_links = $self->cfg('NEW_WINDOW_LINKS');

    my $target = $new_window_links ? qq[target="_blank"] : " ";

    # Write HTML
    print qq[<p>\n];
    print qq[<h2>List of Available Assays</h2><p>\n];
    print qq[<table border="0">\n];

    print
      qq[<tr class="highlight"><td>Producer</td><td>Locus Name</td><td>Assay Name</td><td>Assay Id(s)</td></tr>\n];

    my $current_group;
    foreach my $ref_row (@list) {
        my ($producer, $locus, $name, $assay_id) = @{$ref_row};
        print qq[<tr>\n];
        my @assay_id_urls;
        foreach (@{$assay_id}) {
            push @assay_id_urls,
              qq[<a href="$root_url?session_id=$session_id&assay_id=$_&page=overview" $target>$_</a>];
        }
        print qq[<td>$producer</td><td>$locus</td><td>$name</td><td>]
          . join(' ', @assay_id_urls)
          . qq[</td>\n];
        print qq[</tr>\n];
    }

    print qq[</table>\n];

    return 1;
}

# Function  : Generates content for list of available assays.
# Arguments : None
# Returns   : \@list
# Notes     : None specified.

sub get_list {
    my ($self) = @_;

    my $dbh = $self->dbh;

    my $statement = <<SQL;
SELECT distinct daa.producer, cmf.name, daa.name, daa.div_allele_assay_id
FROM div_allele_assay daa
LEFT JOIN div_scoring_tech_type dstt USING(div_scoring_tech_type_id)
LEFT JOIN cdv_marker cm ON (daa.cdv_marker_id = cm.cdv_marker_id)
LEFT JOIN cdv_map_feature cmf USING(cdv_map_feature_id)
WHERE dstt.scoring_tech_group = "Sequencing"
ORDER BY daa.producer, cmf.name
SQL

    my $sth = $dbh->prepare($statement)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth->execute or croak("Statement: $statement<br>", $dbh->errstr);

    my %list;
    while (my ($producer, $locus, $name, $assay_id) = $sth->fetchrow_array) {
        push @{$list{"$producer:$locus:$name"}}, $assay_id;
    }

    $sth->finish;
    $dbh->disconnect;

    my @list;
    foreach my $group (sort keys %list) {
        my ($producer, $locus, $name) = split(':', $group);
        my $assay_id = $list{$group};
        push @list, [$producer, $locus, $name, $assay_id];
    }
    return (\@list);
}

# Function  : Retrieves assays by assay id.
# Arguments : $assay_id, $out
# Returns   : 1
# Notes     : None specified.

sub get_by_assay_id {
    my ($self, $assay_id, $out) = @_;

    my $dbh = $self->dbh;

    # Get sequences
    my $statement = <<SQL;
SELECT dp.accename, dp.sampstat, dp.source, da.quality, da.value
FROM div_passport dp
RIGHT JOIN div_stock ds USING (div_passport_id)
RIGHT JOIN div_obs_unit dou USING (div_stock_id)
RIGHT JOIN div_obs_unit_sample dous USING (div_obs_unit_id)
RIGHT JOIN div_allele da USING (div_obs_unit_sample_id)
LEFT JOIN div_allele_assay daa USING (div_allele_assay_id)
LEFT JOIN cdv_marker cm USING (cdv_marker_id)
LEFT JOIN cdv_map_feature cmf USING (cdv_map_feature_id)
LEFT JOIN div_scoring_tech_type dstt ON(dstt.div_scoring_tech_type_id = daa.div_scoring_tech_type_id)
WHERE da.div_allele_assay_id = ? AND dstt.scoring_tech_group = 'Sequencing'
SQL

    my $sth = $dbh->prepare($statement)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth->bind_param(1, $assay_id)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth->execute or croak("Statement: $statement<br>", $dbh->errstr);

    my @sequences;
    while (my @row = $sth->fetchrow_array) { push(@sequences, \@row) }
    unless (@sequences > 0) {
        croak(
            "This assay id ($assay_id) does not retrieve any sequences<br>If you know that this is a valid assay id and it contains valid aligned sequences please contact the maintainer of this tool"
        );
    }

    # Get other info
    my $statement2 = <<SQL;
SELECT distinct cm.name, cmf.name
FROM div_allele_assay daa
LEFT JOIN cdv_marker cm USING (cdv_marker_id)
LEFT JOIN cdv_map_feature cmf USING (cdv_map_feature_id)
WHERE daa.div_allele_assay_id = ?
SQL

    my $sth2 = $dbh->prepare($statement2)
      or croak("Statement: $statement2<br>", $dbh->errstr);
    $sth2->bind_param(1, $assay_id)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth2->execute or croak("Statement: $statement2<br>", $dbh->errstr);

    my ($pz_number, $locus) = $sth2->fetchrow_array;

    my $info = "locus=${locus},pz_number=${pz_number},assay_id=${assay_id}";

    open(OUT, ">$out") or croak("Cannot open file ($out): $!");

    print OUT "<data>\n";
    print OUT "\n";

    my $unique_id = 0;

    my $code2sampstat = LookAlign::Panzea::WebFormUtils->code2sampstat();

    foreach my $sequence_ref (@sequences) {

        my ($accename, $sampstat, $source, $quality, $value) =
          @{$sequence_ref};

        $unique_id++;

        # -- Cleaning --

        # Remove preceding and leading spaces
        $quality =~ s/^\s+// if $quality;
        $quality =~ s/\s+$// if $quality;
        $value   =~ s/^\s+// if $value;
        $value   =~ s/\s+$// if $value;

        # Replace space (delimiter) with comma in quality score
        $quality =~ tr/ /,/ if $quality;

        # Discard quality score if sizes do not match
        my @quality = split(',', $quality) if $quality;
        $quality = '' unless (length($value) == scalar @quality);

        # *** TEMPORARY ***
        $accename =~ s/\&/&amp;/g if $accename;
        $source   =~ s/\&/&amp;/g if $source;
        $info     =~ s/\&/&amp;/g if $info;

        # -- End Cleaning --

        # Checking
        if ($value =~ /\d/) {
            croak(
                "This assay id ($assay_id) contains a sequence ($accename) containing numbers<br>If you know that all sequences it contains are valid please contact the maintainer of this tool"
            );
        }

        my $formatted_sampstat =
            $code2sampstat->{$sampstat}
          ? $code2sampstat->{$sampstat}
          : $sampstat;

        print OUT qq[<_sequence>\n];
        print OUT
          qq[\t<sequence label="$accename($source - $formatted_sampstat)">$value</sequence>\n];
        print OUT qq[\t<attribute label="quality">$quality</attribute>\n]
          if $quality;
        print OUT qq[\t<attribute label="unique_id">$unique_id</attribute>\n];
        print OUT
          qq[\t<attribute label="selection:germplasm_type">$formatted_sampstat</attribute>\n]
          if $formatted_sampstat;
        print OUT qq[</_sequence>\n];

    }

    print OUT qq[<_offset>0</_offset>\n];
    print OUT qq[<_info>$info</_info>\n];
    print OUT qq[</data>\n];

    close OUT;

    $sth->finish;
    $sth2->finish;
    $dbh->disconnect;

    return 1;
}

# Function  : Retrieves assay_ids by locus.
# Arguments : $locus
# Returns   : @assay_ids
# Notes     : None specified.

sub get_by_locus {
    my ($self, $locus) = @_;

    my $dbh = $self->dbh;

    #    my $locus_table = $self->_get_locus_table;

    my $statement = <<SQL;
SELECT distinct daa.div_allele_assay_id
FROM div_allele da
LEFT JOIN div_allele_assay daa USING (div_allele_assay_id)
LEFT JOIN cdv_marker cm USING (cdv_marker_id)
LEFT JOIN cdv_map_feature cmf USING (cdv_map_feature_id)
LEFT JOIN div_scoring_tech_type dstt ON (dstt.div_scoring_tech_type_id = daa.div_scoring_tech_type_id)
WHERE cmf.name = ? AND dstt.scoring_tech_group = "Sequencing"
SQL

    my $sth = $dbh->prepare($statement)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth->execute($locus) or croak("Statement: $statement<br>", $dbh->errstr);

    my @assay_ids;
    while (my @row = $sth->fetchrow_array) { push @assay_ids, shift @row }

    $sth->finish;
    $dbh->disconnect;

    return (@assay_ids);
}

# Function  : Retrieves assay_ids by pz_number.
# Arguments : $pz_number
# Returns   : @assay_ids
# Notes     : None specified.

sub get_by_pz_number {
    my ($self, $pz_number) = @_;

    my $dbh = $self->dbh;

    my $statement = <<SQL;
SELECT distinct daa.div_allele_assay_id
FROM div_allele da
LEFT JOIN div_allele_assay daa USING (div_allele_assay_id)
LEFT JOIN cdv_marker cm USING (cdv_marker_id)
LEFT JOIN div_scoring_tech_type dstt ON (dstt.div_scoring_tech_type_id = daa.div_scoring_tech_type_id)
WHERE cm.name =  ? AND dstt.scoring_tech_group = "Sequencing"
SQL

    my $sth = $dbh->prepare($statement)
      or croak("Statement: $statement<br>", $dbh->errstr);
    $sth->execute($pz_number)
      or croak("Statement: $statement<br>", $dbh->errstr);

    my @assay_ids;
    while (my @row = $sth->fetchrow_array) { push @assay_ids, shift @row }

    $sth->finish;
    $dbh->disconnect;

    return (@assay_ids);
}

# Function  : Generates available search boxes.
# Arguments : None
# Returns   : $html_code
# Notes     : None specified.

sub get_search_forms {
    my ($self) = @_;

    my $root_url   = $self->root_url;
    my $session_id = $self->session_id;

    my $search_forms = <<FORMS;
<tr>
<td class="box">
<form method=get action="$root_url">
     <h3>Enter an assay id to retrieve sequences of all individuals sequenced for the corresponding amplicon:</h3>
     Assay id (552, etc.):&nbsp;<input type="text" name="assay_id" size="10" maxlength="40">
     &nbsp;&nbsp;&nbsp;
     <input type="hidden" name="page" value="search_result">
     <input type="hidden" name="search_type" value="assay_id">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="submit">
     <p>
     <p>
</form>
</td>
</tr>

<td class="box">
<form method=get action="$root_url">
     <h3>Enter a marker name to retrieve sequences of all individuals sequenced for the corresponding amplicon:</h3>
     Marker Name (PZA00320, PZB00188, etc.):&nbsp;<input type="text" name="pz_number" size="10" maxlength="40">
     &nbsp;&nbsp;&nbsp;
     <input type="hidden" name="page" value="search_result">
     <input type="hidden" name="search_type" value="pz_number">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="submit">
     <p>
     <p>
</form>
</td>
</tr>

<tr>
<td class="box">
<form method=get action="$root_url">
     <h3>Enter a locus name to retrieve assays of all amplicons originating from that locus:</h3>
     Locus name (AY104138, asg11, etc.):&nbsp;<input type="text" name="locus" size="10" maxlength="40">
     &nbsp;&nbsp;&nbsp;
     <input type="hidden" name="page" value="search_result">
     <input type="hidden" name="search_type" value="locus">
     <input type="hidden" name="session_id" value="$session_id">
     <input type="submit" value="submit">
     <p>
     <p>
</form>
</td>
</tr>
FORMS

    return $search_forms;
}

# Function  : Returns overview info based on the alignment container object.
# Arguments : al, $aln_length, $type
# Returns   : $html_code
# Notes     : None specified.

sub get_overview_info {
    my ($self, $al, $aln_length, $type) = @_;

    my $info = $al->info;

    # Get locus/assay_id info
    my ($locus)     = $info =~ /locus=([^,]+)/;
    my ($assay_id)  = $info =~ /assay_id=([^,]+)/;
    my ($pz_number) = $info =~ /pz_number=([^,]+)/;

    # Get sequence number related information
    my @sequences           = $al->valid_sequences;
    my $number_of_sequences = @sequences;

    # my $aln_length = $al->longest_seq_len;

    # Get germplasm info
    my %sequences;
    foreach my $s (@sequences) {
        my ($i) = $s->label =~ /^([^\(\)]+)/;
        $i ||= '';
        $sequences{$i}++;
    }
    my $number_of_accessions = scalar(keys %sequences);

    # Get start and end positions
    my $start_pos = $al->offset + 1;
    my $end_pos   = $start_pos + $al->longest_seq_len - 1;

    my $overview_info;

    $overview_info .=
      qq[<h2><u>Locus</u>: $locus; <u>Assay id</u>: $assay_id; <u>Marker Name</u>: $pz_number</h2><p>\n];
    if ($type eq 'overview') {
        $overview_info .=
          qq[<h3>Alignment Overview (total alignment length $aln_length)</h3>\n];
    }
    elsif ($type eq 'alignment') {
        $overview_info .=
          qq[<h3>Alignment between position $start_pos and $end_pos (total alignment length $aln_length)</h3>\n];
    }
    else { croak("Internal error: Invalid get_overview_info type ($type)"); }
    $overview_info .=
      qq[<h4>$number_of_sequences sequences / $number_of_accessions accessions in this alignment</h4>\n];

    return $overview_info;
}

# Function  : Returns title info for text dumps.
# Arguments : al
# Returns   : $text
# Notes     : None specified.

sub get_text_dump_info {
    my ($self, $al) = @_;

    my $info = $al->info;

    my ($assay_id) = $info =~ /assay_id=([^,]+)/;
    my ($locus)    = $info =~ /locus=([^,]+)/;
    my $title = "Assay id: $assay_id Locus: $locus";

    return $title;
}

# Function  : Returns interface specific modules
#             to be displayed in debug message.
# Arguments : None
# Returns   : @critical_modules
# Notes     : None specified.

sub get_critical_modules {
    my ($self) = @_;

    return ($INC{'LookAlign/Interface/GDPDM.pm'});
}

# Function  : Returns interface specific infomation
#             to be displayed in debug message.
# Arguments : None
# Returns   : $html_code
# Notes     : None specified.

sub get_critical_content {
    my ($self) = @_;

    my $datasource = $self->get_datasource;

    return qq[<b>Datasource</b>: $datasource &nbsp;];
}

# Function  : Returns interface specific datasource infomation
#             to be included in cache index.
# Arguments : None
# Returns   : $string
# Notes     : None specified.

sub get_datasource {
    my ($self) = @_;

    my $dbh = $self->dbh;

    my $datasource =
          $dbh->{Driver}->{Name}
        . ':'
        . $dbh->{Name};

    return $datasource;
}

##########################
# SIMPLE GET/SET METHODS #
##########################

sub available_databases {
    my ($self, $value) = @_;
    $self->{available_databases} = $value if @_ > 1;
    return $self->{available_databases};
}

=head1 AUTHOR

Payan Canaran <canaran@cshl.edu>

=head1 BUGS

=head1 VERSION

Version 0.01

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright (c) 2004-2007 Cold Spring Harbor Laboratory

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See DISCLAIMER.txt for
disclaimers of warranty.

=cut

1;
