package LookAlign::Alignment::Renderer;

our $VERSION = '0.01';

# $Id: Renderer.pm,v 1.1.2.1 2007/06/14 19:03:07 kclark Exp $

=head1 NAME

LookAlign::Alignment::Renderer

=head1 DESCRIPTION

Object for rendering images to represent alignments
stored by Alignment::Container.

=cut

use warnings;
use strict;

use Bio::Graphics;
use Bio::SeqFeature::Generic;
use Carp;
use Config::General;
use GD;

###############
# CONSTRUCTOR #
###############

sub new {
    my ($class, %args) = @_;

    my %obj;

    eval {
        defined $args{alignment} or croak("An 'alignment' is required");
        defined $args{globals}   or croak("A 'globals' is required");
        defined $args{config}    or croak("A 'config' is required");
        defined $args{out}       or croak("An 'out' is required");
        defined $args{mismatch}  or croak("A 'mismatch' is required");
    } or return;

    my $al = $args{alignment};

    $obj{_ruler} = 1;
    defined $args{ruler} and $obj{_ruler} = $args{ruler};
    $obj{_ruler_offset} = $al->offset;

    # Find longest sequence length
    my $longest_seq_len = $al->longest_seq_len;

    # Determine number of tracks (this depends on: 1) number of sequences 2) number of listed globals exist within the alignment object 3) whether a ruler exists
    my $number_of_tracks = scalar $al->valid_sequences;
    foreach (@{$args{globals}}) {
        $number_of_tracks++ if $al->exists_global($_);
    }
    $number_of_tracks++ if $obj{_ruler};

    # Parse config file
    my $config =
      _process_config_file($args{config}, $number_of_tracks,
        $longest_seq_len);

    $obj{_alignment} = $al;
    $obj{_globals}   = $args{globals};
    $obj{_config}    = $config;
    $obj{_out}       = $args{out};
    $obj{_mismatch}  = $args{mismatch};

    bless \%obj, $class;
}

###########
# METHODS #
###########

sub alignment {
    my ($self) = @_;
    return $self->{_alignment};
}

sub globals {
    my ($self) = @_;
    return (@{$self->{_globals}});
}

sub ruler {
    my ($self) = @_;
    return $self->{_ruler};
}

sub ruler_offset {
    my ($self) = @_;
    return $self->{_ruler_offset};
}

sub config {
    my ($self) = @_;
    return $self->{_config};
}

sub out {
    my ($self) = @_;
    return $self->{_out};
}

sub mismatch {
    my ($self) = @_;
    return $self->{_mismatch};
}

sub _process_config_file {
    my ($file, $number_of_tracks, $longest_seq_length) = @_;

    my %cfg = Config::General::ParseConfig($file);

    $cfg{NUMBER_OF_TRACKS}   = $number_of_tracks;
    $cfg{LONGEST_SEQ_LENGTH} = $longest_seq_length;

    $cfg{CHARS_PER_LABEL} =
      int($cfg{LABEL_SEGMENT_WIDTH} / $cfg{MEDIUMBOLD_FONT_WIDTH});
    $cfg{CHARS_PER_TRACK} =
      int($cfg{TRACK_SEGMENT_WIDTH} / $cfg{LARGE_FONT_WIDTH});

    $cfg{IMAGE_WIDTH} =
      $cfg{LEFT_PAD} + $cfg{LABEL_SEGMENT_WIDTH} + $cfg{LABEL_SEQ_SPACING} +
      $cfg{TRACK_SEGMENT_WIDTH} + $cfg{RIGHT_PAD};

    $cfg{TRACK_HEIGHT} =
      $cfg{SCORE_SEGMENT_HEIGHT} + $cfg{SEQ_SEGMENT_HEIGHT} +
      $cfg{BUFFER_SEGMENT_HEIGHT};
    $cfg{PANEL_HEIGHT} =
      ($cfg{NUMBER_OF_TRACKS} * $cfg{TRACK_HEIGHT}) + $cfg{PANEL_SPACING};

    $cfg{NUMBER_OF_PANELS} =
      $cfg{LONGEST_SEQ_LENGTH} % $cfg{CHARS_PER_TRACK}
      ? int($cfg{LONGEST_SEQ_LENGTH} / $cfg{CHARS_PER_TRACK}) + 1
      : $cfg{LONGEST_SEQ_LENGTH} / $cfg{CHARS_PER_TRACK};

    $cfg{IMAGE_HEIGHT} =
      $cfg{TOP_PAD} + ($cfg{NUMBER_OF_PANELS} * $cfg{PANEL_HEIGHT});

    return \%cfg;
}

sub render_alignment {
    my ($self) = @_;

    my %cfg = %{$self->config};
    my $al  = $self->alignment;

    my $longest_seq_len = $al->longest_seq_len;

    my @consensus_no_gaps_iupac =
        $al->exists_global('consensus_no_gaps_iupac')
      ? $al->global_value_ary('consensus_no_gaps_iupac')
      : '';

    my $mismatch = $self->mismatch;

    my %fonts = (
        'gdMediumBoldFont' => gdMediumBoldFont,
        'gdTinyFont'       => gdTinyFont,
        'gdLargeFont'      => gdLargeFont
    );

    my $counter = 0;

    my %tracks;

    if ($self->ruler) {
        $counter++;
        $tracks{$counter}{type} = 'ruler';
    }

    # For future reference: This is the location to implement any sorting algorithm for the alignment

    foreach (sort { $a->label cmp $b->label } $al->valid_sequences) {
        $counter++;
        $tracks{$counter}{type} = 'sequence';
        $tracks{$counter}{ref}  = $_;
    }

    foreach ($self->globals) {
        if ($al->exists_global($_)) {
            $counter++;
            $tracks{$counter}{type} = 'global';
            $tracks{$counter}{ref}  = $_;
        }
    }

    while (my ($panel_start, $panel_end) = $self->_new_panel_limits) {

        # Generate image
        my ($im, $ref_colors) = $self->_new_panel;

        # Track ruler marks
        my $placed_ruler_mark;

        foreach my $track (
            sort {
                if (    $tracks{$a}{type} eq 'global'
                    and $tracks{$b}{type} ne 'global') {
                    return -1;
                }
                elsif ( $tracks{$b}{type} eq 'global'
                    and $tracks{$a}{type} ne 'global') {
                    return 1;
                }
                else { return $a <=> $b }
            } keys %tracks
          ) {

            # Get information about track
            my $type = $tracks{$track}{type};
            my $ref  = $tracks{$track}{ref};

            # Get label & data
            my $label;
            my @d;
            if ($type eq 'ruler') { $label = ''; }
            elsif ($type eq 'sequence') {
                $label = $ref->label;
                @d     = $ref->sequence_ary;
            }
            elsif ($type eq 'global') {
                $label = '';
                @d     = $al->global_value_ary($ref);
            }

            # Truncate label if necessary
            my ($truncated_label) = $label =~ /^(.{0,$cfg{CHARS_PER_LABEL}})/;

            # Insert label for this panel
            my ($x, $y) = $self->_get_label_coordinates($track);
            $im->string($fonts{$cfg{MEDIUMBOLD_FONT}}, $x, $y,
                $truncated_label, $ref_colors->{orange});

            # Slide through the base pairs and insert all sequences and globals
            foreach my $i ($panel_start .. $panel_end - 1) {

                # Calculate positional information
                my ($pos) = $self->_get_char_pos($i);
                my ($x, $y_sequence, $y_score) =
                  $self->_get_coordinates($track, $pos);

                if ($type eq 'ruler') {
                    my $ruler_mark = $self->ruler_offset + $i + 1;
                    if ($ruler_mark % $cfg{RULER_MARK_RESOLUTION} == 0) {
                        $im->string($fonts{$cfg{SMALL_FONT}}, $x, $y_sequence,
                            $ruler_mark, $ref_colors->{black});
                        $placed_ruler_mark = 1;
                    }

                    if ($i + 1 == $longest_seq_len and !$placed_ruler_mark)
                    {   # Place one final mark if no mark exists in this panel
                        my $final_i    = $i;
                        my $final_mark = $self->ruler_offset + $final_i + 1;
                        while ($final_mark % $cfg{RULER_MARK_RESOLUTION} != 0)
                        {
                            $final_i++;
                            $final_mark = $self->ruler_offset + $final_i + 1;
                        }
                        my ($final_pos) = $self->_get_char_pos($final_i);
                        my ($final_x, $final_y_sequence, $final_y_score) =
                          $self->_get_coordinates($track, $final_pos);
                        $im->string($fonts{$cfg{SMALL_FONT}}, $final_x,
                            $final_y_sequence, $final_mark,
                            $ref_colors->{black});
                    }

                }

                elsif ($type eq 'global' and ($ref eq 'consensus_no_gaps'))
                {    # or $ref eq 'consensus_with_gaps')) {
                        # Get sequence
                    my @data = @d;

                    my $mismatch_char = $data[$i];

                    # Place the frame of non-matches so that it can be overwritten
                    if ($data[$i] eq '*') {
                        my $top_track = 1;
                        $top_track = 2
                          if $self->ruler
                          ; # if there is a ruler, mismatch mark starts from track 2, else from track 1
                        my ($x_top, $null1, $y_top) =
                          $self->_get_coordinates($top_track, $pos);
                        my ($x_bottom, $null2, $y_bottom) =
                          $self->_get_coordinates($track + 1, $pos);
                        my ($x, $y, $z, $t) =
                          ($x_top - 2, $y_top - 2,
                            $x_bottom + $cfg{LARGE_FONT_WIDTH} - 3,
                            $y_bottom - 1);
                        $im->filledRectangle($x, $y, $z, $t,
                            $ref_colors->{yellow});

                        if ($mismatch eq 'IUPAC') {
                            $mismatch_char = $consensus_no_gaps_iupac[$i];
                        }
                        elsif ($mismatch =~ /^.$/) {
                            $mismatch_char = $mismatch;
                        }
                        else {
                            croak(
                                "Invalid mismatch representation ($mismatch)!"
                            );
                        }
                    }

                    # Place the matching consensus sequence
                    $im->string($fonts{$cfg{MEDIUMBOLD_FONT}}, $x,
                        $y_sequence, $mismatch_char, $ref_colors->{black});
                }

                elsif ($type eq 'sequence') {

                    # Get sequence
                    my @s = @d;

                    # Get quality scores if available - word: 'quality' is reserved
                    my $q;
                    $q = $ref->attribute_value('quality')
                      if $ref->exists_attribute('quality');
                    my @q = $q ? split(/,/, $q) : '';

                    my %c = (
                        'A' => $ref_colors->{blue},
                        'T' => $ref_colors->{red},
                        'C' => $ref_colors->{green},
                        'G' => $ref_colors->{orange}
                    );

                    my $color = $ref_colors->{black};

                    $color = $c{$s[$i]} if $c{$s[$i]};

                    $im->string($fonts{$cfg{LARGE_FONT}}, $x, $y_sequence,
                        $s[$i], $color);

                    if (defined $q[$i] and $q[$i] =~ /^\d+$/) {
                        $im->string($fonts{$cfg{SMALL_FONT}}, $x, $y_score,
                            $q[$i], $ref_colors->{red});
                    }

                }
            }
        }
    }
    $self->_dump_panel;

    my $out_html = $self->out . '.alignment.html';
    open(HTML, ">$out_html") or croak($!);
    my $html;
    foreach ($self->_alignment_files) {
        my ($f) = $_ =~ /([^\/]+)$/;
        $html .= qq[<img src="[PNG=$f]" border="1"></img><br>\n];
    }
    print HTML $html, "\n";
    close HTML;

    return ($out_html, $self->_alignment_files);
}

sub _new_panel_limits {
    my ($self) = @_;
    my $cfg    = $self->config;
    my $al     = $self->alignment;

    # Find longest sequence length
    my $longest_seq_len = $al->longest_seq_len;

    my $current_start;

    if (!defined $self->_current_start) {
        $self->_current_start(0);
        return (0, $cfg->{CHARS_PER_TRACK});
    }

    $current_start = $self->_current_start;

    my $start = $current_start + $cfg->{CHARS_PER_TRACK};
    my $end   = $start + $cfg->{CHARS_PER_TRACK};

    if ($start >= $longest_seq_len) { return; }
    if ($end >= $longest_seq_len) { $end = $longest_seq_len; }

    $self->_current_start($start);

    return ($start, $end);
}

sub _new_panel {
    my ($self) = @_;
    my $cfg = $self->config;

    if ($self->_current_panel) { $self->_dump_panel }

    my $im = new GD::Image($cfg->{IMAGE_WIDTH}, $cfg->{PANEL_HEIGHT});

    # Allocate colors
    my %colors = (
        'whitish' => $im->colorAllocate(245, 245, 255),
        'white'   => $im->colorAllocate(255, 255, 255),
        'black'   => $im->colorAllocate(0,   0,   0),
        'red'     => $im->colorAllocate(255, 0,   0),
        'blue'    => $im->colorAllocate(0,   0,   255),
        'gray'    => $im->colorAllocate(128, 128, 128),
        'orange'  => $im->colorAllocate(139, 37,  0),
        'yellow'  => $im->colorAllocate(255, 255, 0),
        'wheat'   => $im->colorAllocate(245, 222, 179),
        'green'   => $im->colorAllocate(0,   100, 0),
        'violet'  => $im->colorAllocate(199, 21,  133),
    );

    $self->_current_panel($im);
    my $j = $self->_current_panel_order;
    $j++;
    $self->_current_panel_order($j);
    $self->_current_panel_file($self->out . '.'
          . sprintf("%03s", $self->{_current_panel_order}) . 'of'
          . $cfg->{NUMBER_OF_PANELS}
          . ".png");
    $self->_current_panel_colors(\%colors);

    return ($self->_current_panel, $self->_current_panel_colors);
}

sub _dump_panel {
    my ($self) = @_;
    my $cfg = $self->config;

    my $out = $self->_current_panel_file;
    my $im  = $self->_current_panel;

    # Write out image
    open(OUT, ">$out") or croak($!);

    # Make sure we are writing to a binary stream
    binmode OUT;

    # Convert the image to PNG and print it on standard output
    print OUT $im->png;

    close OUT;

    $self->_alignment_files($out);

    return 1;
}

sub _get_char_pos {
    my ($self, $i) = @_;
    my $cfg = $self->config;

    $i++;

    my $pos =
        $i % $cfg->{CHARS_PER_TRACK}
      ? $i % $cfg->{CHARS_PER_TRACK}
      : $cfg->{CHARS_PER_TRACK};

    return ($pos);
}

sub _get_coordinates {
    my ($self, $track, $pos) = @_;
    my $cfg = $self->config;

    my $x =
      $cfg->{LEFT_PAD} + $cfg->{LABEL_SEGMENT_WIDTH} +
      $cfg->{LABEL_SEQ_SPACING} + $cfg->{LARGE_FONT_WIDTH} * $pos;

    my $y_sequence =
      ($track - 1) * $cfg->{TRACK_HEIGHT} + $cfg->{SCORE_SEGMENT_HEIGHT};

    my $y_score = $y_sequence - $cfg->{SCORE_SEGMENT_HEIGHT};

    return ($x, $y_sequence, $y_score);
}

sub _get_label_coordinates {
    my ($self, $track) = @_;
    my $cfg = $self->config;

    my $x = $cfg->{LEFT_PAD};
    my $y =
      ($track - 1) * $cfg->{TRACK_HEIGHT} + $cfg->{SCORE_SEGMENT_HEIGHT};

    return ($x, $y);
}

sub render_small_alignment {
    my ($self, $n_limit, $pic_limit) = @_;

    $n_limit   or $n_limit   = 0;
    $pic_limit or $pic_limit = 0;

    my %cfg = %{$self->config};
    my $al  = $self->alignment;

    # Min sequence length
    my $seq_len = 500;
    if ($seq_len < $al->longest_seq_len) { $seq_len = $al->longest_seq_len; }

    my %tracks;
    my $counter = 0;

    # For future reference: This is the location to implement any sorting algorithm for the small alignment

    foreach (sort { $a->label cmp $b->label } $al->valid_sequences) {
        $counter++;
        $tracks{$counter}{type} = 'sequence';
        $tracks{$counter}{ref}  = $_;
    }

    # Generate Panel
    my $panel = Bio::Graphics::Panel->new(
        -length     => $seq_len,
        -width      => $cfg{SMALL_ALIGNMENT_WIDTH},
        -pad_left   => 50,
        -pad_top    => 50,
        -pad_right  => 50,
        -pad_bottom => 50,
        -spacing    => 15,
        -grid       => 1,
        -grid_color => 'red',
        -key_style  => 'between',
        -bgcolor    => 'aliceblue'
    );

    my $full_length = Bio::SeqFeature::Generic->new(
        -start => 1,
        -end   => $seq_len,
        -tag   => {ruler => 1},
    );
    my $ruler_track = $panel->add_track(
        $full_length,
        -glyph   => 'arrow',
        -tick    => 2,
        -fgcolor => 'black',
        -double  => 1,
    );

    # Write SNP candidates
    # Currently only knows 'consensus_no_gaps' and 'pic_values' and 'N'
    if ($al->exists_global('consensus_no_gaps')) {

        my @p =
            $al->exists_global('pic_values')
          ? $al->global_value_ary('pic_values')
          : '';
        my @n = $al->exists_global('N') ? $al->global_value_ary('N') : '';

        my $high_snp_track = $panel->add_track(
            -glyph      => 'diamond',
            -label      => 1,
            -bgcolor    => 'blue',
            -font2color => 'blue',
            -key        => "SNP Candidates (N>=$n_limit and PIC>=$pic_limit)"
        );

        my $low_snp_track = $panel->add_track(
            -glyph      => 'diamond',
            -label      => 1,
            -bgcolor    => 'blue',
            -font2color => 'blue',
            -key        => "SNP Candidates (N<$n_limit or PIC<$pic_limit)"
        );

        my $sequence =
          join('', $al->global_value_ary('consensus_no_gaps'))
          ;    # Corrected as now strict usage of comma as the separator

        while ($sequence =~ /(\*)/g) {
            my $fragment = $1;

            my $start = pos($sequence) - 1;

            my $pic = $p[pos($sequence) - 1];
            my $n   = $n[pos($sequence) - 1];

            my $feature = Bio::SeqFeature::Generic->new(
                -start => $start,
                -end   => $start,
                -tag   => {snp => 1}
            );
            if ($n >= $n_limit and $pic >= $pic_limit) {
                $high_snp_track->add_feature($feature);
            }
            elsif ($n < $n_limit or $pic < $pic_limit) {
                $low_snp_track->add_feature($feature);
            }

        }

    }

    my $seq_track = $panel->add_track(
        -glyph      => 'graded_segments',
        -label      => 1,
        -connector  => 'dashed',
        -bgcolor    => 'orange',
        -font2color => 'orange',
        -min_score  => 0,
        -max_score  => 100,
        -sort_order => 'high_score',
        -key        => 'Amplicons from individuals'
    );

    foreach my $track (sort { $a <=> $b } keys %tracks) {

        # Get information about track
        my $type = $tracks{$track}{type};
        my $ref  = $tracks{$track}{ref};

        # Get label
        my $label;
        if    ($type eq 'ruler')    { $label = 'ruler' }
        elsif ($type eq 'sequence') { $label = $ref->label; }
        elsif ($type eq 'global')   { $label = $ref }

        # Truncate label if necessary
        my ($truncated_label) = $label =~ /^(.{0,$cfg{CHARS_PER_LABEL}})/;

        # Get sequence
        my $sequence;
        my $sequence_info;
        my $unique_id;    # Sequence unique id
        if ($type eq 'sequence') {
            $sequence      = $ref->sequence;
            $sequence_info = ' len:' . $ref->len . '/' . $ref->valid_len;
            $unique_id     = $ref->attribute_value('unique_id');
        }
        elsif ($type eq 'global') {
            $sequence = join('', $al->global_value_ary($ref));
        }    # Corrected as now strict usage of comma as the separator

        my $feature_blank =
          Bio::SeqFeature::Generic->new(
            -display_name => $label . $sequence_info, -start => 1, -end => 1);

        my $feature =
          Bio::SeqFeature::Generic->new(
            -display_name => $label . $sequence_info,
            -tag => {sequence => 1, unique_id => $unique_id});

        my $feature_check = 0;

        # Previously all sub-segments were individually represented, this creates a burden on Bio::Graphics phase
        # This has been modified to represent only 2 sub-segments:
        # (i) [ATCGatcg]+: basically anything that can be considered meaningful (ii) [\-]+ (gaps) (iii) the rest

        while ($sequence =~ /([ATCGatcg]+|[^ATCGatcg\-]+)/g) {

            $feature_check = 1;    # Now we know that features has been added

            my $fragment     = $1;
            my $len_fragment = length($fragment);

            my $start = pos($sequence) - $len_fragment + 1;
            my $end   = $start + $len_fragment - 1;

            my $score = 20;

            #            if ($fragment =~ /\+/) { $score = 80 }
            if ($fragment =~ /[ATCGatcg]/) { $score = 100 }

            my $sub_feature = Bio::SeqFeature::Generic->new(
                -score => $score,
                -start => $start,
                -end   => $end,
                -tag   => {sequence => 1}
            );

            $feature->add_sub_SeqFeature($sub_feature, 'EXPAND');
        }

        $feature_check
          ? $seq_track->add_feature($feature)
          : $seq_track->add_feature($feature_blank);
    }

    # Generate image map - knows "ruler", "snp", "sequence"
    my $image_map;
    my @boxes = $panel->boxes;

    foreach my $map_component (@boxes) {
        my ($feature, $x1, $y1, $x2, $y2, $track) = @{$map_component};
        my $resolution = $cfg{SMALL_ALIGNMENT_RULER_RESOLUTION};

        if ($feature->has_tag('ruler')) {
            my $pixels_per_residue =
              ($x2 - $x1) / ($feature->end - $feature->start);
            foreach my $start ($feature->start .. $al->longest_seq_len) {
                if ($start % $resolution == 1) {
                    my $info = "[POS=$start]";
                    my ($x_start, $y_start, $x_end, $y_end) =
                      ($x1, $y1, $x2, $y2, $track);

                    $x_start =
                      $x1 + ($start - $resolution / 2) * $pixels_per_residue;
                    $x_start = $x1 if $x_start < $x1;

                    $x_end =
                      $x1 + ($start + $resolution / 2) * $pixels_per_residue;
                    $x_end = $x2 if $x_end > $x2;

                    $image_map .=
                      qq[<area shape="rect" $info coords="$x_start, $y_start, $x_end, $y_end">\n];
                }
            }
        }

        elsif ($feature->has_tag('snp')) {
            my $start = $feature->start;
            my $info  = "[POS=$start]";
            $image_map .=
              qq[<area shape="rect" $info coords="$x1, $y1, $x2, $y2">\n];
        }

        elsif ($feature->has_tag('sequence')) {
            my ($unique_id) = $feature->get_tag_values('unique_id');
            my $info = "[UNIQUE_ID=$unique_id]";
            $image_map .=
              qq[<area shape="rect" $info coords="$x1, $y1, $x2, $y2">\n];
        }
    }

    # Print out image
    my $out_png = $self->out . '.small_alignment.png';
    open(PNG, ">$out_png") or croak($!);
    flock(PNG, 2) or croak($!);
    binmode PNG;
    print PNG $panel->png;
    close PNG;

    # finish panel
    $panel->finished();

    # Print out image map
    my $out_html = $self->out . '.small_alignment.html';
    my ($out_png_filename) = $out_png =~ /([^\/]+)$/;
    open(HTML, ">$out_html") or croak($!);
    flock(HTML, 2) or croak($!);
    print HTML <<MAP_CONTENT;
<img src="[PNG=$out_png_filename]" usemap="#SNP-MAP" border="1">

<map name="SNP-MAP">

$image_map

</map>

MAP_CONTENT
    close HTML;

    return ($out_html, $out_png);
}

sub _current_start {
    my ($self, $value) = @_;
    defined $value and $self->{_current_start} = $value;
    return $self->{_current_start};
}

sub _current_panel {
    my ($self, $value) = @_;
    defined $value and $self->{_current_panel} = $value;
    return $self->{_current_panel};
}

sub _current_panel_order {
    my ($self, $value) = @_;
    defined $value and $self->{_current_panel_order} = $value;
    return $self->{_current_panel_order};
}

sub _current_panel_file {
    my ($self, $value) = @_;
    defined $value and $self->{_current_panel_file} = $value;
    return $self->{_current_panel_file};
}

sub _current_panel_colors {
    my ($self, $value) = @_;
    defined $value and $self->{_current_panel_colors} = $value;
    return $self->{_current_panel_colors};
}

sub _alignment_files {
    my ($self, $value) = @_;
    defined $value and push @{$self->{_alignment_files}}, $value;
    return @{$self->{_alignment_files}};
}

=head1 AUTHOR

Payan Canaran <canaran@cshl.edu>

=head1 BUGS

=head1 VERSION

Version 0.01

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright (c) 2004-2007 Cold Spring Harbor Laboratory

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See DISCLAIMER.txt for
disclaimers of warranty.

=cut

1;
