package JIM::Session;

#JIM::Session, copyright and (c) 2005 Jim Thomason

$VERSION = '2.00';

use Basset::Object;
our @ISA = Basset::Object->pkg_for_type('persistentobject');

use Date::Calc qw(Add_Delta_DHMS);
use Basset::Encryption::WeakKey;

use strict;
use warnings;

=pod

=head1 JIM::Session

=head1 ATTRIBUTES

=over

=cut

=item sessionkey
=item permanent
=item user_id
=item packedip
=item expires
=item data
=item created_date
=item last_modified

=cut

__PACKAGE__->add_class_attr('rolling_expiration');
__PACKAGE__->add_class_attr('cookiename');

__PACKAGE__->add_attr(
	qw(
		sessionkey
		permanent
		user_id
		packedip
		expires
		created_date
		last_modified
		hours
	),
	[
		'data', '_isa_committing_accessor',
		sub {
			my $self = shift;
			my $prop = shift;
			
			return Basset::Encryption::WeakKey->encrypt($self->dump($self->$prop()), $ENV{'HTTP_USER_AGENT'});
		}
	],
);

__PACKAGE__->add_tables(
	__PACKAGE__->factory(
		'type' 					=> 'table',
		'name'					=> 'session',
		'primary_column'		=> 'sessionkey',
		'autogenerated'			=> 0,
		'non_primary_columns'	=> [qw(permanent user_id packedip expires data created_date last_modified)],
	)
);

sub setup {
	my $self = shift;

	my $datastring = Basset::Encryption::WeakKey->decrypt($self->data, $ENV{'HTTP_USER_AGENT'});

	$datastring =~ /^(\$\w+)/;
	local $@ = undef;
	if ($datastring) {
		my $data = eval qq{
			my $1;
			eval \$datastring;
		};

		$self->data($data);
	}
	
	return $self;
};

sub init {
	my $self = shift;
	
	$self = $self->SUPER::init(
		'permanent' => 0,
		'data' => {},
		@_
	);
	
	$self->generate_sessionkey unless defined $self->sessionkey;
	$self->generate_expiration unless defined $self->expires;
	
	return $self;
};

sub generate_expiration {
	my $self	= shift;
	my $never	= @_ ? shift : $self->permanent();

	my $delta_days	= $never ? 5000 : 0;

	my ($sec,$min,$hour,$day,$mon,$year) = (localtime(time))[0..5];
	$mon++;
	$year+= 1900;
	($year,$mon,$day,$hour,$min,$sec) =
	Add_Delta_DHMS($year,$mon,$day, $hour,$min,$sec, $delta_days,$self->hours,0,0);
	foreach ($sec, $min, $hour, $day, $mon){
		$_ = sprintf("%02d", $_);
	};

	return $self->expires("$year-$mon-$day $hour:$min:$sec");
};

=pod

=begin btest generate_expiration

=end btest

=cut


sub expired {
	my $self = shift;
	
	return 0 if $self->permanent;

	my ($sec,$min,$hour,$day,$mon,$year) = (localtime(time))[0..5];

	$year+= 1900;
	$mon++;
	foreach ($sec, $min, $hour, $day, $mon){
		$_ = sprintf("%02d", $_);
	};
	my $now = "$year-$mon-$day $hour:$min:$sec";
	if ($now gt $self->expires) {	#if we're expired
		return 1;
	}
	else {							#otherwise, it's valid
		return 0;
	};
	
};

=pod

=begin btest expired

=end btest

=cut


sub ip {
	my $self = shift;
	
	if (@_){
		$self->packedip($self->pack_ip(shift));
	};
	
	return $self->unpack_ip($self->packedip);
	
};

=pod

=begin btest ip

=end btest

=cut


sub pack_ip {
	my $self	= shift;
	my $ip		= shift || return $self->error("Cannot pack nothing", 'BS-01');
	
	return hex(join('', map {sprintf("%02lx", $_)} split(/\./, $ip)));
};

=pod

=begin btest pack_ip

=end btest

=cut


sub unpack_ip {
	my $self	= shift;
	my $packed	= shift || return $self->error("Cannot pack nothing", 'BS-01');
	
	return join('.', map {hex($_)} unpack("A2A2A2A2", sprintf("%08lx", $packed)));
	
};

=pod

=begin btest unpack_ip

=end btest

=cut


sub generate_sessionkey {
	my $self = shift;
	
	my $key = undef;
	foreach (0..10){
		$key .= ('.',',','!',0..9,'A'..'Z','a'..'z')[rand 64];
	}

	return $self->sessionkey($key);
};

=pod

=begin btest generate_sessionkey

=end btest

=cut


sub DESTROY {
	my $self = shift;
	
	unless ($self->deleted) {
		
		if ($self->rolling_expiration && ! $self->permanent) {
			$self->generate_expiration();
		};
		
		$self->commit();
	};
	
	#return $self->SUPER::DESTROY(@_);
};

=pod

=begin btest DESTROY

=end btest

=cut


1;


1;


