package Gramene::XRef;

# $Id: XRef.pm,v 1.2 2005/05/27 19:38:03 kclark Exp $

=head1 NAME

Gramene::XRef - a Gramene module

=head1 SYNOPSIS

  use Gramene::XRef;

  my $xdb = Gramene::XRef->new;

=head1 DESCRIPTION

Access to cross-reference module.

=head1 METHODS

=cut

# ----------------------------------------------------

use strict;
use Class::Base;
use Template;

use base 'Class::Base';

use vars qw[ $VERSION ];
$VERSION = sprintf "%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/;

# ----------------------------------------------------
sub init {
    my ( $self, $config ) = @_;
    $self->params( $config, qw[ admin db_name ] );
    return $self;
}

# ----------------------------------------------------
sub db {

=pod

=head2 db

  my $db = $xdb->db or die $xdb->error;

Returns a database handle to the xref database.

=cut

    my $self = shift;

    unless ( $self->{'db'} ) {
        my $db_name = 
            $self->{'admin'}   ? 'xrefs_rw'         :
            $self->{'db_name'} ? $self->{'db_name'} :
            'xrefs';
        $self->{'db'} = Gramene::DB->new( $db_name );
    }

    return $self->{'db'};
}

# ----------------------------------------------------
sub get_xrefs {

=pod

=head2 get_xrefs

  my @xrefs        = $xdb->get_xrefs(
      module_name  => 'diversity',
      table_name   => 'div_taxonomy',
      record_id    => 1984,
      # optional
      xref_type    => 'GenBank',
      # or
      xref_type_id => 12,
      process_url  => 0, # default is "1"
  ) or die $xdb->error;

Returns the cross-references for a given record in a table in a module.
Can be restricted to just the xrefs of a particular type with the 
"xref_type" or "xref_type_id" argument.

By default, the "url_template" will be process by Template Toolkit
unless the "process_url" argument is a false value.  The results of the 
processed template will be placed into a new "url" field.

=cut

    my ( $self, %args ) = @_;
    my $module_name     = $args{'module_name'}  || $args{'module'} || '';
    my $table_name      = $args{'table_name'}   || $args{'table'}  || '';
    my $record_id       = $args{'record_id'}    || 0;
    my $xref_type       = $args{'xref_type'}    || '';
    my $xref_type_id    = $args{'xref_type_id'} || 0;
    my $process_url     = defined $args{'process_url'} 
                          ? $args{'process_url'} : 1;

    return $self->error('Not enough arguments') 
        unless $module_name && $table_name && $record_id;

    if ( $xref_type && ! $xref_type_id ) {
        $xref_type_id = $self->get_xref_type_id( $xref_type );
    }

    my $db = $self->db;

    my $sql = sprintf(
        q[
            select x.xref_id, 
                   x.xref_value, 
                   xt.xref_type_id, 
                   xt.xref_type, 
                   xt.url_template 
            from   xref x, xref_type xt
            where  x.module_name=?
            and    x.table_name=?
            and    x.record_id=?
            %s
            and    x.xref_type_id=xt.xref_type_id
        ],
        $xref_type_id ? 'and xref_type=?' : ''
    );
    my @args = ( $module_name, $table_name, $record_id );
    push @args, $xref_type_id if $xref_type_id;

    my $xrefs = $db->selectall_arrayref( $sql, { Columns => {} }, @args );

    if ( $process_url ) {
        for my $xref ( @$xrefs ) {
            my $url_template = $xref->{'url_template'} or next;
            my $template     = Template->new;
            my $url;
            $template->process(
                \$url_template, 
                { xref_value => $xref->{'xref_value'} }, 
                \$url
            ) or next;
            $xref->{'url'} = $url;
        }
    }

    return wantarray ? @$xrefs : $xrefs;
}

# ----------------------------------------------------
sub create_xref {

=pod

=head2 create_xref

  my $xref_id = $xdb->create_xref(
      module_name  => 'diversity',
      table_name   => 'div_taxonomy',
      record_id    => 1984,
      xref_value   => 'AB99032',
      xref_type_id => 13, # OR xref_type => 'GenBank',
  ) or die $xdb->error;

Creates a new cross-reference.

=cut

    my ( $self, %args ) = @_;
    my $module_name     = $args{'module_name'}  || $args{'module'} || '';
    my $table_name      = $args{'table_name'}   || $args{'table'}  || '';
    my $record_id       = $args{'record_id'}    || 0;
    my $xref_type       = $args{'xref_type'}    || '';
    my $xref_type_id    = $args{'xref_type_id'} || 0;
    my $xref_value      = $args{'xref_value'}   || '';

    if ( $xref_type && ! $xref_type_id ) {
        $xref_type_id = $self->get_xref_type_id( $xref_type );
    }

    return $self->error('Not enough arguments') 
        unless $module_name && $table_name && $record_id && $xref_type_id;

    my $db = $self->db;

    my $xref_id = $db->selectrow_array(
        q[
            select  xref_id
            from    xref
            where   module_name=?
            and     table_name=?
            and     record_id=?
            and     xref_type_id=?
            and     xref_value=?
        ],
        {},
        ( $module_name, $table_name, $record_id, $xref_type_id, $xref_value )
    );

    unless ( $xref_id ) {
        $db->do(
            q[
                insert
                into    xref (module_name, table_name, record_id, 
                        xref_type_id, xref_value)
                values  ( ?, ?, ?, ?, ? )
            ],
            {},
            ( $module_name, $table_name, $record_id, 
              $xref_type_id, $xref_value )
        );

        $xref_id = $db->selectrow_array('select last_insert_id()');
    }

    return $xref_id;
}

# ----------------------------------------------------
sub create_xref_type {

=pod

=head2 create_xref_type

  my $xref_type_id = $xdb->create_xref_type(
      xref_type    => 'GenBank',
      url_template => 'http://www.ncbi.nlm.nih.gov/entrez/query.fcgi?'.
                      'db=Nucleotide&cmd=search&term=[% xref_value %]'
  ) or die $xdb->error;

Creates a new cross-reference type.

=cut

    my ( $self, %args ) = @_;
    my $xref_type       = $args{'xref_type'}    || '';
    my $url_template    = $args{'url_template'} || '';

    return $self->error('Missing arguments') unless $xref_type;

    my $db = $self->db;

    my $xref_type_id = $db->selectrow_array(
        q[
            select xref_type_id
            from   xref_type
            where  xref_type=?
            and    url_template=?
        ],
        {},
        ( $xref_type, $url_template )
    );

    unless ( $xref_type_id ) {
        $db->do(
            q[
                insert
                into    xref_type (xref_type, url_template)
                values  ( ?, ? )
            ],
            {},
            ( $xref_type, $url_template )
        );

        $xref_type_id = $db->selectrow_array('select last_insert_id()');
    }

    return $xref_type_id;
}

# ----------------------------------------------------
sub get_xref_type_id {

=pod

=head2 get_xref_type_id

  my $xref_type_id = $xdb->get_xref_type_id('GenBank') or die $xdb->error;

Returns the cross-references for a given record in a table in a module.

=cut

    my $self         = shift;
    my $xref_type    = shift or return;
    my $db           = $self->db or return;
    my $xref_type_id = $db->selectrow_array(
        q[
            select xref_type_id
            from   xref_type
            where  xref_type=?
        ],
        {},
        ( $xref_type )
    );

    return $xref_type_id;
}

# ----------------------------------------------------
sub find_or_create_xref_type {

=pod

=head2 find_or_create_xref_type

  my $xref_type_id = $xdb->find_or_create_xref_type(
      xref_type    => 'GenBank',
      url_template => 'http://www.ncbi.nlm.nih.gov/entrez/query.fcgi?'.
                      'db=Nucleotide&cmd=search&term=[% xref_value %]'
  ) or die $xdb->error;

Creates a new cross-reference type.

=cut

    my ( $self, %args ) = @_;
    my $xref_type       = $args{'xref_type'}    || '';
    my $url_template    = $args{'url_template'} || '';

    my $xref_type_id = $self->get_xref_type_id( $xref_type );

    unless ( $xref_type_id ) {
        $xref_type_id = $self->create_xref_type( %args ) or return;
    }

    return $xref_type_id;
}
1;

# ----------------------------------------------------

=pod

=head1 SEE ALSO

Gramene::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2005 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
