package Gramene::Search::DB;

# $Id: DB.pm,v 1.3 2007/06/05 19:27:04 kclark Exp $

=head1 NAME

Gramene::Search::DB - a Gramene module

=head1 SYNOPSIS

  use Gramene::Search::DB;

=head1 DESCRIPTION

Description of module goes here.

=head1 SEE ALSO

perl.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut

# ----------------------------------------------------

use strict;
use Gramene::Config;
use Readonly;

Readonly my $MIN_SEARCH_LEN => 3;
Readonly my $PIPE           => q{\|};
Readonly my $SPACE          => q{ };
Readonly my $EMPTY_STR      => q{};
Readonly my $VERSION   
    => sprintf "%d.%02d", q$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;

# ----------------------------------------------------
sub module_display_list {
    my $cfile        = Gramene::Config->new;
    my $modules_conf = $cfile->get('modules') or die 'No "modules" config';
    my $search_conf  
        = $cfile->get('gramene_search') or die 'No gramene search config';

    my @module_names
        = grep { !/gramene_search/ } (
            defined $modules_conf->{'module'} 
            && ref $modules_conf->{'module'} eq 'ARRAY'
            ? @{ $modules_conf->{'module'} || [] }
            : ( $modules_conf->{'module'} )
        );

    push @module_names, 'documents';

    my $display_order = 1;
    my %module_display_order
        = map { $_, $display_order++ }
        ( split /,/, $search_conf->{'module_display_order'} );

    for my $module ( @module_names ) {
        $module_display_order{ $module } ||= $display_order++;
    }

    @module_names = sort {
        $module_display_order{ $a } <=> $module_display_order{ $b }
    } @module_names;

    my %display_name = %{ $search_conf->{'module_display_name'} || {} };
    my @module_list;
    for my $module ( @module_names ) {
        my $display = $display_name{ $module } 
                   || join( $SPACE, map { ucfirst } split /_/, $module );
        push @module_list, {
            module_name  => $module,
            display_name => $display,
        };
    }

    return @module_list; 
}

# ----------------------------------------------------
sub search {
    my $args           = shift;
    my $db             = $args->{'db'}          or die 'No search db';
    my $search_conf    = $args->{'search_conf'} or die 'No search config';
    my $config         = $args->{'config'}      or die 'No config';
    my $search_value   = $args->{'search_for'}  or die 'Missing search value';
    my $search_module  = $args->{'module'}     || $EMPTY_STR;
    my $related_to     = $args->{'related_to'} || $EMPTY_STR;
    my $where          = $args->{'where'}      || $EMPTY_STR;
    my $order_by       = $args->{'order_by'}   || $EMPTY_STR;
    my $page_num       = $args->{'page'}       || $EMPTY_STR;
    my $search_url     = $args->{'search_url'} || $EMPTY_STR;
    my %object_display_name = %{ $search_conf->{'object_display_name'} || {} };

    if ( length $search_value < $MIN_SEARCH_LEN ) {
        die "Sorry, the search value &quot;$search_value&quot; is too short.",
            "<br>Please try again with a string at least ",
            "$MIN_SEARCH_LEN characters long.\n"
    }

    $search_value =~ s/%/*/g; # change SQL wildcard to asterisk for FULLTEXT
    $search_value =~ s/^\s+|\s+$//g; # trim

    my $sql = sprintf(
        qq[ 
            select module_name, table_name, record_id, record_text,
                   match(record_text) against ('%s') as score
            from   module_search
            where  match(record_text) against ('%s' in boolean mode)
            %s
            order by score desc
        ],
        $search_value,
        $search_value,
        $search_module ? "and module_name='$search_module'"   : $EMPTY_STR
    );

    my $sth = $db->prepare( $sql );
    $sth->execute;

    my $entries_per_page = 10;
    my %basket;
    while ( my $r = $sth->fetchrow_hashref ) {
        push @{ $basket{ $r->{'module_name'} }{ $r->{'table_name'} }{'data'} },
            $r;
    }

    if ( !$search_module || $search_module eq 'markers' ) {
        my $mdb = Gramene::Marker::DB->new;
        my @markers;
        for my $v ( 
            iterative_search_values($search_value, {no_leading_wildcard => 1 })
        ) { 
            @markers = $mdb->marker_search( marker_name => $v );
            last if @markers;
        }
        
        for my $marker ( @markers ) {
            push @{ $basket{'markers'}{'marker'}{'data'} }, {
                record_id   => $marker->{'marker_id'},
                record_text => join($SPACE,
                    $marker->{'species'},
                    $marker->{'marker_type'},
                ),
            };
        }
    }
    
    my @modules = module_display_list();
    my @ensembl_modules 
        = !$search_module 
        ? grep { /^ensembl_/ } @modules
        : $search_module =~ /^ensembl_/
            ? $search_module
            : ()
    ;

    ENS_MODULE:
    for my $ens_module ( @ensembl_modules ) {
        my $ens_conf = $config->get( $ens_module ) or next ENS_MODULE;
        my ($scheme, $driver, $attr_string, $attr_hash, $driver_dsn)
            = DBI->parse_dsn($ens_conf->{'db_dsn'});
        
        my $host    = 'localhost';
        my $db_name = $driver_dsn;
        if ( $driver_dsn =~ /database=([^;]+)/ ) {
            $db_name = $1;
        }
        if ( $driver_dsn =~ /host=([^;]+)/ ) {
            $host = $1;
        }
        
        my $dba = Bio::EnsEMBL::DBSQL::DBAdaptor->new(
            -user   => $ens_conf->{'db_user'},
            -pass   => $ens_conf->{'db_pass'},
            -dbname => $db_name,
            -host   => $host,
            -driver => $driver,
        );

        if ( my $feat_adaptor = $dba->get_adaptor('DnaAlignFeature') ) {
            my @features
                = @{ $feat_adaptor->fetch_all_by_hit_name( $search_value ) };

            my %seen;
            for my $f ( @features ) {
                next if $seen{ $f->hseqname }++;

                push @{ $basket{ $ens_module }{'dna_align_feature'}{'data'} },
                {   
                    record_id   => $f->dbID,
                    record_text => join( $SPACE,
                        $f->hseqname,
                        $f->analysis->display_label || $f->analysis->logic_name,
                        $f->feature_Slice->name
                    ),
                };
            }
        }

        if ( my $marker_adaptor = $dba->get_adaptor('Marker') ) {
            my @markers
                = @{ $marker_adaptor->fetch_all_by_synonym( $search_value ) };
            
            for my $marker ( @markers ) {
                push @{ $basket{ $ens_module }{'marker'}{'data'} }, {
                    record_id   => $marker->dbID,
                    record_text => join( $SPACE,
                        map { $_->name() } @{ $marker->get_all_MarkerSynonyms }
                    ),
                };
            }
        }
    }

    for my $module ( keys %basket ) {
        my %list_column_conf = %{ $search_conf->{'list_columns'} || {} };
        my %view_link_conf   = %{ $search_conf->{'view_link'}    || {} };
        
        while ( my ( $table, $basket ) = each %{ $basket{ $module } } ) {
            my @list_columns     
                = split(/,/, $list_column_conf{"$module.$table"} || $EMPTY_STR);
            my $items = $basket->{'data'};
            my $pager = Data::Page->new( scalar @$items, 10, $page_num );
            $items    = [ $pager->splice( $items ) ];
            my $class = table_name_to_gramene_cdbi_class( $module, $table );
            my $view_link = $view_link_conf{"$module.$table"};
            
            ITEM:
            for my $record ( @$items ) {
                my $object = $class->retrieve( $record->{'record_id'} )
                or do {
                    print STDERR "Can't retrieve $class $record->{record_id}\n";
                    next ITEM;
                };
                
                if ( $module eq 'markers' && $table eq 'marker' ) {
                    $record->{'record_text'} .= join( $SPACE,
                        $EMPTY_STR,
                        map { $_->marker_name } $object->marker_synonyms
                    );
                }
                
                my @view_urls;
                for my $tmpl ( split( /$PIPE/, $view_link ) ) {
                    my $link_text = 'View';
                    my $url       = $EMPTY_STR;

                    if ( $tmpl =~ /([^@]+)@(.+)/ ) {
                        $link_text = $1;
                        $tmpl      = $2;
                    }

                    my @sub_values;
                    while ( $tmpl =~ /\[(.*?)\]/g ) {
                        my ( $cur_class, @methods ) = split /\./, $1;
                        my $cur_obj     = $object;
                        my $last_method = pop @methods;

                        for (;;) {
                            if ( my $method = shift @methods ) {
                                ( $cur_obj ) = $cur_obj->$method();
                            }

                            if ( scalar @methods == 0 ) {
                                push @sub_values, $cur_obj->$last_method();
                                last;
                            }
                        }
                    }
                    
                    if ( !@sub_values ) {
                        @sub_values = $object->id;
                        $tmpl      .= '[id]';
                    }
                     
                    $tmpl =~ s/\[.*?\]/%s/g;
                    $url  = sprintf( $tmpl, @sub_values );
                    
                    push @view_urls, {
                        link_text => $link_text,
                        url       => $url,
                    } if $url;
                }
                
                push @{ $basket->{'items'} }, {
                    object    => $object,
                    score     => $record->{'score'},
                    view_urls => \@view_urls,
                    context   => match_context(
                        $search_value, $record->{'record_text'}
                    ),
                };
            }
            
            $basket->{'pager'}       = $pager;
            $basket->{'columns'}     = 
                @list_columns ? \@list_columns : [$class->columns('Essential')];
            $basket->{'has_a'}       = meta_parse( $class->meta_info('has_a') );
            $basket->{'object_name'} 
                =  $object_display_name{ "$module.$table" }
                || $class->object_type;
            $basket->{'pager_url'}
                = $search_url . "?action=list&module=$module&object=$table"
                . "&search_for=$search_value";
        }
    }

    if (
        !$search_module || ($search_module && $search_module eq 'documents')
    ) { 
        my $doc_sql = sprintf(
            qq[ 
                select   path, title, contents,
                         match(path, title, contents) against ('%s') as score
                from     doc_search
                where    match(path, title, contents)
                against  ('%s' in boolean mode)
                order by score desc
            ],
            $search_value,
            $search_value,
        );

        my $items = $db->selectall_arrayref( $doc_sql, { Columns => {} } );

        if ( @$items > 0 ) {
            my $pager = Data::Page->new( scalar @$items, 10, $page_num );
            $items    = [ $pager->splice( $items ) ];

            for my $item ( @$items ) {
                $item->{'context'}
                    = match_context( $search_value, $item->{'contents'} );
            }

            $basket{'documents'}{'files'} =  {
                items       => $items,
                pager       => $pager,
                object_name => 'Document',
                pager_url   => $search_url
                    . "?action=list&module=documents&search_for=$search_value",
            };
        }
    }

    return \%basket;
}

1;
