package Gramene::QTL::OpenCuration;

# $Id: OpenCuration.pm,v 1.2 2004/11/11 22:36:16 kclark Exp $

=head1 NAME

Gramene::QTL::OpenCuration - community curation module

=head1 SYNOPSIS

  use Gramene::QTL::OpenCuration;

  my $qdb = Gramene::QTL::OpenCuration->new;

=head1 DESCRIPTION

This is a work in progess, just ideas on how to create a community
curation interface.

=head1 METHODS

=cut

# --------------------------------------------------

use strict;
use Class::Base;
use Gramene::Config;
use Date::Format;
use Digest::MD5 qw[ md5_hex ];
use Email::Valid;
use File::Spec::Functions;
use Time::ParseDate;
use File::Temp 'tempfile';

use base 'Class::Base';

# --------------------------------------------------
sub init {
    my ( $self, $config ) = @_;
    $self->params( $config, qw[ cgi user_name session_id ] ) 
        or return $self->error;
    return $self;
}

# --------------------------------------------------
sub cgi {
    my $self = shift;
    $self->{'cgi'} = shift if @_;
    return $self->{'cgi'};
}

# --------------------------------------------------
sub config {
    my $self = shift;

    unless ( $self->{'config'} ) {
        my $conf = Gramene::Config->new;
        $self->{'config'} = $conf->get('qtl');
    }

    return $self->{'config'};
}

# --------------------------------------------------
sub curator_id {
    my $self = shift;

    unless ( $self->{'curator_id'} ) {
        my $user_name  = $self->user_name or 
            return $self->error('Object has no user name');
        my $db         = $self->db;
        my $curator_id = $db->selectrow_array(
            'select curator_id from qtl_curator where user_name=?',
            {}, ( $user_name )
        ) or return $self->error("Can't find curator_id for '$user_name'");
        $self->{'curator_id'} = $curator_id;
    }

    return $self->{'curator_id'};
}

# --------------------------------------------------
sub db {
    my $self = shift;

    unless ( $self->{'db'} ) {
        eval { $self->{'db'} = Gramene::DB->new('qtl') };
        return $self->error( $@ ) if $@;
    }

    return $self->{'db'};
}

# --------------------------------------------------
sub freshen_session {
    my ( $self, %args ) = @_;
    my $curator_id = $args{'curator_id'} || $self->curator_id;
    my $session_id = $args{'session_id'} || $self->session_id;
    my $now        = $self->now;
    my $db         = $self->db or return;

    return $self->error('Insufficient args to logout') 
        unless $curator_id && $session_id;

    $db->do(
        q[
            update qtl_curator_session 
            set    last_activity_at=?
            where  curator_id=?
            and    session_id=?
        ],
        {},
        ( $self->now, $curator_id, $session_id )
    );

    return 1;
}

# --------------------------------------------------
sub login {
    my ( $self, %args ) = @_;
    my $user_name  = $args{'user_name'} or return $self->error('No user name');
    my $password   = $args{'password'}  or return $self->error('No password');
    my $db         = $self->db or return;
    my $curator_id;

    if ( Email::Valid->address( $user_name ) ) {
        my ( $id, $name ) = $db->selectrow_array(
            q[
                select curator_id, user_name 
                from   qtl_curator 
                where  email_address=?
            ],
            {}, ( $user_name )
        ) or return $self->error("Unknown e-mail address '$user_name'");
        $user_name  = $name;
        $curator_id = $id;
    }
    else {
        $curator_id = $db->selectrow_array(
            'select curator_id from qtl_curator where user_name=?',
            {}, ( $user_name )
        ) or return $self->error("Unknown user '$user_name'");
    }

    $self->user_name( $user_name );

    my $db_pass    = $db->selectrow_array(
        'select password from qtl_curator where curator_id=?',
        {}, ( $curator_id )
    ); 
    
    return $self->error("Bad password for user '$user_name'")
        unless $password eq $db_pass;

    my $now        = $self->now;
    my $session_id = md5_hex(
        sprintf( "%x-%x-%x", time(), $$, int rand 0x1000 )
    );

    $db->do(
        q[
            insert 
            into   qtl_curator_session (session_id, curator_id, started_at) 
            values (?, ?, ?)
        ],
        {},
        ( $session_id, $curator_id, $now )
    );

    return $session_id;
}

# --------------------------------------------------
sub logout {
    my ( $self, %args ) = @_;
    my $curator_id = $args{'curator_id'} || $self->curator_id;
    my $session_id = $args{'session_id'} || $self->session_id;
    my $now        = $self->now;
    my $db         = $self->db or return;

    return $self->error('Insufficient args to logout') 
        unless $curator_id && $session_id;

    $db->do(
        q[
            update qtl_curator_session 
            set    logged_out_at=?
            where  curator_id=?
            and    session_id=?
        ],
        {},
        ( $now, $curator_id, $session_id )
    );

    return 1;
}

# --------------------------------------------------
sub make_login_cookie {
    my ( $self, %args ) = @_;
    my $user_name  = $args{'user_name'}  || $self->user_name;
    return $self->error('No user name') unless $user_name;
    my $session_id = $args{'session_id'} || $self->session_id;
    return $self->error('No session_id') unless $session_id;
    my $expires    = $args{'expires'}    || '+30m';
    my $cgi        = $self->cgi       or return $self->error('No CGI object');
    my $config     = $self->config    or return $self->error('No config');
    my $secret     = $config->{'user_secret'};
    my $hash       = md5_hex( $user_name, $session_id, $secret );
    my $cookie     = $cgi->cookie(
        -name      => 'user_name', 
        -value     => "$user_name:$session_id:$hash",
        -expires   => $expires,
    );
    return $cookie;
}

# --------------------------------------------------
sub now {
    my $self = shift;
    my @lt   = localtime;
    return strftime( '%Y-%m-%d %X', @lt );
}

# --------------------------------------------------
sub process_template {
    my ( $self, $template, $args ) = @_;

    $self->freshen_session;
    
    $args->{'cgi'} = $self->cgi;
    $args->{'user_name'} ||= $self->user_name;
    my $t = $self->template or return;
    my $html;
    $t->process( $template, $args, \$html ) or return $self->error( $t->error );
    my $cookie = $self->make_login_cookie or return;
    my $cgi    = $self->cgi or return $self->error('No CGI object');
    return $cgi->header( -type => 'text/html', -cookie => $cookie ), $html;
}

# --------------------------------------------------
sub process_upload {
    my $self = shift;
    my $upload_id = shift or return $self->error('No upload_id');
    my $db        = $self->db;
    my $sth       = $db->prepare(
        q[
            select upload_id, system_file_name
            from   qtl_curator_upload
            where  upload_id=?
        ],
        { Columns => {} }
    );
    $sth->execute( $upload_id ) or return $self->error('Bad upload_id');;

    $db->do(
        q[
            update qtl_curator_upload
            set    processed_at=now()
            where  upload_id=?
        ],
        {},
        ( $upload_id )
    );

    return $self->error("foo bar!");

    return 1;
}

# --------------------------------------------------
sub save_upload {
    my ( $self, %args )   = @_;
    my $in_file           = $args{'file'} or return $self->error('No file');
    my $session_id        = $args{'session_id'} || $self->session_id; 
    return $self->error('No session_id') unless $session_id;
    my $db                = $self->db;
    my $config            = $self->config;
    my $dir               = $config->{'upload_file_dir'};
    my ( $fh, $filename ) = tempfile( 'qtlocXXXXXX', DIR => $dir );

    while ( <$in_file> ) {
        print $fh $_;
    }
    close $fh;
    my $size = -s $filename;

    my $upload_id = $db->selectrow_array(
        'select max(upload_id) from qtl_curator_upload'
    );
    $upload_id++;

    $db->do(
        q[
            insert
            into   qtl_curator_upload
                   (upload_id, session_id, incoming_file_name,
                    system_file_name, size, uploaded_at)
            values (?, ?, ?, ?, ?, ?)
        ],
        {},
        ( $upload_id, $session_id, $in_file, $filename, $size, $self->now )
    );

    return {
        filename => $in_file,
        size     => $size,
    };
}

# --------------------------------------------------
sub session_id {
    my $self = shift;
    $self->{'session_id'} = shift if @_;
    return $self->{'session_id'};
}

# --------------------------------------------------
sub template {
    my $self = shift;

    unless ( $self->{'template'} ) {
        my $config = $self->config;
        my $template_dir = $config->{'template_dir'} or
            return $self->error('No template dir');
        $self->{'template'} = Template->new(
            INCLUDE_PATH => $template_dir,
            WRAPPER      => 'oc_wrapper'
        );
    }

    return $self->{'template'};
}

# --------------------------------------------------
sub set_user_info {
    my ( $self, %args )  = @_;
    my @missing;
    my $user_name        = $args{'user_name'}        
        or push @missing, 'user name';
    my $real_name        = $args{'real_name'}        
        or push @missing, 'real name';
    my $organization     = $args{'organization'}     
        or push @missing, 'organization';
    my $email_address    = $args{'email_address'}    
        or push @missing, 'email address';
    my $curator_id       = $args{'curator_id'}       || $self->curator_id;
    my $password         = $args{'password'}         || '';
    my $password_confirm = $args{'password_confirm'} || '';

    push @missing, 'password' if $args{'action'} eq 'insert' && !$password;
    push @missing, 'password_confirm' if $password && !$password_confirm;

    return $self->error( 
        "Missing required fields: ", join(', ', @missing), "\n" 
    ) if @missing;

    return $self->error(
        "Bad user name (only 6-10 letters and numbers, no spaces)"
    ) unless $user_name =~ /^[a-zA-Z0-9]{6,10}$/;

    return $self->error("Bad e-mail address")
        unless Email::Valid->address( $email_address );

    return $self->error("Passwords don't match") if
        $password && $password_confirm && ( $password ne $password_confirm );

    return $self->error("Password too short (min. length = 5)") if
        $password && length $password < 5;

    return $self->error("No spaces in password") if 
        $password && $password =~ /\s/;

    my $db = $self->db;

    if ( $args{'action'} eq 'insert' ) {
        my $name = $db->selectrow_array(
            'select curator_id from qtl_curator where user_name=?',
            {}, ( $user_name )
        );

        return $self->error("The user name '$user_name' is already taken.")
            if $name;

        $curator_id = $db->selectrow_array(
            'select max(curator_id) from qtl_curator',
        );
        $curator_id++;

        $db->do(
            q[
                insert
                into   qtl_curator
                       (curator_id, user_name, password, real_name,
                        organization, email_address)
                values (?, ?, ?, ?, ?, ?)
            ],
            {},
            ( $curator_id, $user_name, $password, $real_name,
              $organization, $email_address )
        );
    }
    else { 
        return $self->error('No curator_id') unless $curator_id;

        $db->do(
            q[
                update qtl_curator
                set    real_name=?, organization=?, email_address=?
                where  curator_id=?
            ],
            {},
            ( $real_name, $organization, $email_address, $curator_id )
        );
    }

    if ( $password ) {
        $db->do(
            q[
                update qtl_curator
                set    password=?
                where  curator_id=?
            ],
            {},
            ( $password, $curator_id )
        );
    }

    return $curator_id;
}

# --------------------------------------------------
sub user_name {
    my $self = shift;

    if ( my $arg = @_ ) {
        if ( my ( $user_name, $hash ) = split( /:/, $arg ) ) {
            my $config = $self->config or return $self->error('No config');
            my $secret = $config->{'user_secret'};
            my $digest = md5_hex( $user_name, $secret );
            return $self->error('Bad digest') unless $hash eq $digest;
            $self->{'user_name'} = $user_name;
        }
        else {
            $self->{'user_name'} = $arg;
        }
    }

    return $self->{'user_name'};
}

# --------------------------------------------------
sub verify_login {
    my ( $self, %args ) = @_;
    my $cgi             = $self->cgi or return;
    my $user_name       = $args{'user_name'}  || $self->user_name;
    my $session_id      = $args{'session_id'} || $self->session_id;
    my $hash            = $args{'hash'};
    my @missing;
    push @missing, 'user_name'  unless $user_name;
    push @missing, 'session_id' unless $session_id;
    push @missing, 'hash'       unless $hash;
    return $self->error("Missing required arguments: ", join(', ', @missing))
        if @missing;
    my $conf      = Gramene::Config->new;
    my $config    = $conf->get('qtl');
    my $secret    = $config->{'user_secret'};
    my $digest    = md5_hex( $user_name, $session_id, $secret );

    return $self->error("Bad login cookie") unless $digest eq $hash;

    return $session_id;
}

# --------------------------------------------------
sub view_session {
    my ( $self, %args ) = @_;
    my $session_id = $args{'session_id'} || $self->session_id;
use Data::Dumper;
print STDERR "args = ", Dumper(\%args), "\nsession id = '", $self->session_id, "'\n";
    return $self->error('No session_id') unless $session_id;
    my $curator_id = $self->curator_id or return $self->error('No curator_id');
    my $db         = $self->db;
    my $sth        = $db->prepare(
        q[
            select s.session_id, s.curator_id, s.started_at,
                   s.last_activity_at, s.logged_out_at,
                   c.user_name
            from   qtl_curator_session s,
                   qtl_curator c
            where  s.session_id=?
            and    s.curator_id=?
            and    s.curator_id=c.curator_id
        ]
    );
    $sth->execute( $session_id, $curator_id );
    my $session = $sth->fetchrow_hashref or return 
        $self->error("Invalid session ID ($session_id : $curator_id)");

    my $start    = parsedate( $session->{'started_at'} );
    my $end      = parsedate( 
        $session->{'logged_out_at'} || $session->{'last_activity_at'} 
    ); 
    my $duration = $end - $start;
    if ( $duration > 0 ) {
        my $s = Time::Seconds->new( $duration );
        $session->{'duration'} = sprintf( "%.2f", $s->minutes );
    }

    $session->{'uploads'} = $db->selectall_arrayref(
        q[
            select   incoming_file_name, system_file_name, size, 
                     uploaded_at, processed_at
            from     qtl_curator_upload
            where    session_id=?
            order by uploaded_at
        ],
        { Columns => {} },
        ( $session_id )
    );

    my ( @sizes, $total_size );
    for my $upload ( @{ $session->{'uploads'} } ) {
        my $size = $upload->{'size'} or next;
        push @sizes, $size;
        $total_size += $size;
    }
    if ( @sizes ) {
        $session->{'average_upload_size'} = $total_size / scalar @sizes;
    }
    $session->{'total_upload_size'} = $total_size;

    return $session;
}

# --------------------------------------------------
sub view_sessions {
    my $self       = shift;
    my $curator_id = $self->curator_id or return $self->error('No curator_id');
    my $db         = $self->db;
    my $sessions   = $db->selectall_arrayref(
        q[
            select   s.session_id, s.curator_id, s.started_at,
                     s.last_activity_at, s.logged_out_at,
                     c.user_name
            from     qtl_curator_session s,
                     qtl_curator c
            where    s.curator_id=?
            and      s.curator_id=c.curator_id
            order by started_at desc
        ],
        { Columns => {} },
        ( $curator_id )
    );

    return $sessions;
}

1;

# ----------------------------------------------------

=pod

=head1 SEE ALSO

Class::Base, Gramene::Config, Gramene::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
