package Gramene::QTL::DB;

# $Id: DB.pm,v 1.22 2007/06/05 19:26:11 kclark Exp $

=head1 NAME

Gramene::QTL::DB - API to QTL database

=head1 SYNOPSIS

  use Gramene::QTL::DB;

=head1 DESCRIPTION

This is an interface to the QTL database.  

Note: All methods throw exceptions via Carp::croak, so be prepared to 
catch them!

=head1 METHODS

=cut

# ----------------------------------------------------

use strict;
use Bio::GMOD::CMap::Admin;
use Bio::GMOD::CMap;
use Carp qw( croak );
use Class::Base;
use Data::Dumper;
use Gramene::CDBI::Literature;
use Gramene::CDBI::Ontology;
use Gramene::CDBI::Qtl;
use Gramene::Config;
use Gramene::DB;
use Gramene::Utils qw( get_logger similarity_search );

use vars qw[ $VERSION ];
$VERSION = sprintf "%d.%02d", q$Revision: 1.22 $ =~ /(\d+)\.(\d+)/;

use base 'Class::Base';

# ----------------------------------------------------
sub init {

=pod

=head2 new

  my $qdb = Gramene::QTL::DB->new( 
      admin   => 1 ,
      db_name => '...',
  );

Creates a new QTL-db object.  The optional "admin" argument indicates
you need read/write access to the database.  The optional "db_name" can
be used to indicate the database name (as defined in "gramene.conf")
to connect to.

=cut

    my ( $self, $config ) = @_;
    $self->params( $config, qw[ admin db_name ] );
    return $self;
}

# ----------------------------------------------------
sub add_trait_ontology_association {

=pod

=head2 add_trait_ontology_association

  my $assoc_id = $qdb->add_trait_ontology_association(
      qtl_trait_id   => 42,
      term_accession => 'PO:0007038',
      species_id     => 13,
  ) or die $qdb->error;

Creates an association between a trait and an ontology accession.

=cut

    my ( $self, %args )   = @_;
    my $to_accession      = $args{'to_accession'} or croak('No to_accession');
    my $related_accession = $args{'related_accession'} 
                            or croak('No related accession');
    my $species_id        = $args{'species_id'} || 0;

    my $Assoc = Gramene::CDBI::Qtl::TraitOntologyAssociation->find_or_create({
        to_accession      => $to_accession,
        related_accession => $related_accession,
        species_id        => $species_id,
    });

    return $Assoc->id;
}

# ----------------------------------------------------
sub add_qtl_ontology_association {

=pod

=head2 add_qtl_ontology_association

  my $assoc_id = $qdb->add_qtl_ontology_association(
      qtl_id               => 42,
      term_accession       => 'PO:0007038',
      evidence_code        => 'SM', # optional
      gramene_reference_id => 1882, # optional
  ) or die $qdb->error;

Creates an association between a QTL and an ontology accession.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id          = $args{'qtl_id'}         or croak('No QTL ID');
    my $term_accession  = $args{'term_accession'} or croak('No term accession');
    my $reference_id    = $args{'gramene_reference_id'} || '';
    my $evidence_code   = $args{'evidence_code'}        || '';

    if ( !$reference_id && !$evidence_code ) {
        croak(
              'Need either a lit. reference ID or an evidence code '
            . 'to create association'
        );
    }

    my ($Ont) = Gramene::CDBI::Ontology::Term->search(
        term_accession => $term_accession
    ) or croak("Bad term accession '$term_accession'");

    my $Assoc = Gramene::CDBI::Qtl::QtlOntologyAssociation->find_or_create({
        qtl_id         => $qtl_id,
        term_type      => $Ont->term_type->term_type,
        term_accession => $term_accession,
    });

    if ( $reference_id ) {
        my ($Lit) = Gramene::CDBI::Literature::Reference->retrieve(
            $reference_id
        ) or croak("Non-existent reference ID '$reference_id'");
    }
    else {
        my ($LitXrefType) = Gramene::CDBI::Qtl::XrefType->search(
            xref_type     => 'Gramene Literature'
        );

        my ($LitXref)    = Gramene::CDBI::Qtl::Xref->search(
            table_name   => 'qtl',
            record_id    => $qtl_id,
            xref_type_id => $LitXrefType->id,
        );

        if ( $LitXref ) {
            $reference_id = $LitXref->id;
        }
    }

    if ( $reference_id && !$evidence_code ) {
        $evidence_code = 'SM';
    }

    my $Evidence
        = Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->find_or_create({
            qtl_ontology_association_id => $Assoc->id,
            evidence_code               => $evidence_code,
            gramene_reference_id        => $reference_id,
        });

    return $Assoc->id;
}

# ----------------------------------------------------
sub add_xref {

=pod

=head2 add_xref

  my $xref_id    = $qdb->add_xref(
      table_name   => 'qtl';
      record_id    => 182,
      xref_type_id => 13,
      xref_value   => 2888,
  ) or die $qdb->error;

Creates an xref;

=cut

    my ( $self, %args )  = @_;
    my $table_name       = $args{'table_name'}   or croak('No table name');
    my $record_id        = $args{'record_id'}    or croak('No record id');
    my $xref_type_id     = $args{'xref_type_id'} or croak('No xref type id');
    my $xref_value       = $args{'xref_value'}   or croak('No xref value');

    my $Xref = Gramene::CDBI::Qtl::Xref->insert({
        table_name   => $table_name,
        record_id    => $record_id,
        xref_type_id => $xref_type_id,
        xref_value   => $xref_value,
    });

    return $Xref->id;
}

# ----------------------------------------------------
sub add_trait_synonym {

=pod

=head2 add_trait_synonym

  my $trait_syn_id = $qdb->add_trait_synonym(
      trait_id     => 7782,
      synonym      => 'foo',
  ) or die $qdb->error;

Creates a synonym for a trait;  returns the trait_synonym_id.

=cut

    my ( $self, %args )  = @_;
    my $trait_id         = $args{'trait_id'} or croak('No trait id');
    my $synonym          = $args{'synonym'}  or croak('No synonym');
    my $db               = $self->db or return;
    my $trait_synonym_id = $db->selectrow_array(
        q[
            select qtl_trait_synonym_id
            from   qtl_trait_synonym
            where  qtl_trait_id=?
            and    upper(trait_synonym)=?
        ],
        {},
        ( $trait_id, uc $synonym )
    );

    unless ( $trait_synonym_id ) {
        $trait_synonym_id = $db->selectrow_array(
            'select max(qtl_trait_synonym_id) from qtl_trait_synonym'
        );
        $trait_synonym_id++;
        
        $db->do(
            q[
                insert
                into   qtl_trait_synonym
                       (qtl_trait_synonym_id, qtl_trait_id, trait_synonym)
                values (?, ?, ?)
            ],
            {},
            ( $trait_synonym_id, $trait_id, $synonym )
        );
    }

    return $trait_synonym_id;
}

# ----------------------------------------------------
sub cmap_admin {

=pod

=head2 cmap_admin

  my $admin = $qdb->cmap_admin;

Returns a CMap Admin object.

=cut

    my $self    = shift;
    my $config  = $self->config;
    my $cmap_ds = $config->{'cmap_datasource'} || '';
    my $admin   = Bio::GMOD::CMap::Admin->new( data_source => $cmap_ds ) or 
                  croak( Bio::GMOD::CMap::Admin->error );

    return $admin;
}

# ----------------------------------------------------
sub cmap_db {

=pod

=head2 cmap_db

  my $dbh = $qdb->cmap_db;

Returns a handle to the CMap database.  If the "<qtl>" section of the
Gramene config file defines a "cmap_datasource," then that database is
opened;  otherwise, it will be whatever is the default CMap database.

=cut

    my $self   = shift;

    unless ( defined $self->{'cmap'} ) {
        my $cmap = Bio::GMOD::CMap->new or croak( Bio::GMOD::CMap->error );

        my $config = $self->config;
        if ( my $cmap_ds = $config->{'cmap_datasource'} ) {
            $cmap->data_source( $cmap_ds ) or croak( $cmap->error ); 
        }

        $self->{'cmap'} = $cmap;
    }

    my $cmap    = $self->{'cmap'};
    my $cmap_db = $cmap->db or croak( $cmap->error );

    return $cmap_db;
}

# ----------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $qdb->config;

Returns a hashref representing the "<qtl>" section of the Gramene config.

=cut

    my $self  = shift;

    unless ( $self->{'config'} ) {
        my $cfile = Gramene::Config->new;
        $self->{'config'} = $cfile->get('qtl');
    }

    return $self->{'config'};
}

# ----------------------------------------------------
sub create_qtl {

=pod

=head2 create_qtl

  $qdb->create_qtl(
      cmap_map_aid     => 'foo-1',
      qtl_accession_id => 'foo',
      published_symbol => 'foo-1'
      qtl_trait_id     => 8,
      linkage_group    => '3',
      chromosome       => '3',
      start_position   => 190.1,
      stop_position    => 198.2,
      comments         => '',
      species          => 'Rice',
  ) or die $qdb->error;

Creates the QTL and CMap position info.

=cut

    my ( $self, %args )  = @_;
    my @missing;
    my $cmap_map_aid     = $args{'cmap_map_aid'}          
                           or push @missing, 'CMap map aid';
    my $qtl_accession_id = $args{'qtl_accession_id'} 
                           or push @missing, 'QTL accession ID';
    my $published_symbol = $args{'published_symbol'} 
                           or push @missing, 'Published Symbol';
    my $qtl_trait_id     = $args{'qtl_trait_id'}     
                           or push @missing, 'Trait';
    my $linkage_group    = $args{'linkage_group'}    
                           or push @missing, 'Linkage Group';
    my $chromosome       = $args{'chromosome'} || '';
    my $species          = $args{'species'}          
                           or push @missing, 'Species';
    my $start_position   = $args{'start_position'};
    my $stop_position    = $args{'stop_position'};
    my $comments         = $args{'comments'}         || '';
    push @missing, 'Start Position' unless defined $start_position;
    push @missing, 'Stop Position'  unless defined $stop_position;

    if ( @missing ) {
        croak('Missing required fields: '.join(', ', @missing));
    }

    if ( !$chromosome ) {
        ( $chromosome = $linkage_group ) =~ s/[a-zA-Z]$//;
    }

    my $species_id = $args{'species_id'} || '';
    if ( !$species_id && $species ) {
        my $species = Gramene::CDBI::Qtl::Species->find_or_create({
            species => $species
        });

        $species_id = $species->id;
    }

    my $db     = $self->db or return;
    my $qtl_id = $db->selectrow_array( 'select max(qtl_id) from qtl' );
    $qtl_id++;
    $db->do(
        q[
            insert
            into   qtl (qtl_id, qtl_accession_id, published_symbol,
                   qtl_trait_id, linkage_group, chromosome, start_position,
                   stop_position, comments, species_id)
            values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ],
        {},
        ($qtl_id, $qtl_accession_id, $published_symbol, $qtl_trait_id, 
         $linkage_group, $chromosome, $start_position, $stop_position, 
         $comments, $species_id)
    );

    my $cmap_db = $self->cmap_db or return;
    if ( my $position = $self->get_cmap_position( qtl_id => $qtl_id ) ) {
        $cmap_db->do(
            q[
                update cmap_feature
                set    feature_start=?, feature_stop=?
                where  feature_id=?
            ],
            {},
            ( $start_position, $stop_position, $position->{'feature_id'} )
        );
    }
    else {
        my $map_id = $cmap_db->selectrow_array(
            'select map_id from cmap_map where map_acc=?', 
            {}, ( $cmap_map_aid )
        ) or croak("No map with acc. ID '$cmap_map_aid' exists in CMap");

        my $trait_symbol = $db->selectrow_array(
            'select trait_name from qtl_trait where qtl_trait_id=?',
            {}, ( $qtl_trait_id )
        );

        my $cmap_admin = $self->cmap_admin;
        $cmap_admin->feature_create(
            map_id         => $map_id,            
            feature_name   => $trait_symbol,
            accession_id   => $qtl_accession_id,
            start_position => $start_position,
            stop_position  => $stop_position,
        ) or croak( $cmap_admin->error );
    }

    return $qtl_id;
}

# ----------------------------------------------------
sub delete_qtl_ontology_association {

=pod

=head2 delete_qtl_ontology_association

  $qdb->delete_qtl_ontology_association(
      qtl_ontology_association_id => 42,
  ) or die $qdb->error;

Deletes an association between a QTL and an ontology accession.

=cut

    my ( $self, %args ) = @_;
    my $qtl_ontology_assoc_id = $args{'qtl_ontology_association_id'} 
                                or croak('No QTL ontology association ID');

    my $Assoc = Gramene::CDBI::Qtl::QtlOntologyAssociation->retrieve(
        $qtl_ontology_assoc_id 
    );

    $Assoc->delete;

    return 1;
}

# ----------------------------------------------------
sub delete_xref {

=pod

=head2 delete_xref

  $qdb->delete_xref( xref_id => 9987 ) or die $qdb->error;

Deletes an xref.

=cut

    my ( $self, %args ) = @_;

    my $xref_id = $args{'xref_id'} || $args{'dbxref_id'};

    if ( !$xref_id ) {
        croak('No dbxref_id');
    }

    my $Xref = Gramene::CDBI::Qtl::Xref->retrieve( $xref_id )
        or croak("Bad xref id ($xref_id)");

    $Xref->delete;

    return 1;
}

# ----------------------------------------------------
sub delete_trait {

=pod

=head2 delete_trait

  $qdb->delete_trait( trait_id => 884 ) or die $qdb->error;

Deletes a trait if no QTL are related to it.

=cut

    my ( $self, %args ) = @_;

    my $trait_id = $args{'trait_id'} or croak('No trait id');
    my $db       = $self->db or return;
    my $no_qtl   = $db->selectrow_array(
        q[
            select count(*)
            from   qtl
            where  qtl_trait_id=?
        ],
        {},
        ( $trait_id )
    );

    if ( $no_qtl == 0 ) {
        my $trait_category_id = $db->selectrow_array(
            q[
                select qtl_trait_category_id
                from   qtl_trait
                where  qtl_trait_id=?
            ],
            {},
            ( $trait_id )
        );

        $db->do(
            q[
                delete 
                from   qtl_trait_synonym
                where  qtl_trait_id=?
            ],
            {},
            ( $trait_id )
        );

        $db->do(
            q[
                delete 
                from   qtl_trait
                where  qtl_trait_id=?
            ],
            {},
            ( $trait_id )
        );

        return $trait_category_id;
    }
    else {
        my $trait_name = $db->selectrow_array(
            q[
                select trait_name
                from   qtl_trait
                where  qtl_trait_id=?
            ],
            {},
            ( $trait_id )
        );

        croak(
            "Cannot delete the trait '$trait_name' ($trait_id) " .
            "because it has $no_qtl QTL associated with it.\n"
        );
    }
}

# ----------------------------------------------------
sub delete_trait_synonym {

=pod

=head2 delete_trait_synonym

  $qdb->delete_trait_synonym( trait_synonym_id => 8892 ) or die $qdb->error;

Deletes a trait synonym.

=cut

    my ( $self, %args )  = @_;
    my $trait_synonym_id = $args{'trait_synonym_id'} or 
                           croak('No trait synonym id');

    my $db = $self->db or return;
    $db->do(
        q[
            delete 
            from   qtl_trait_synonym
            where  qtl_trait_synonym_id=?
        ],
        {},
        ( $trait_synonym_id )
    );

    return 1;
}

# ----------------------------------------------------
sub delete_qtl {

=pod

=head2 delete_qtl

  $qdb->delete_qtl( qtl_id => 1789 ) or die $qdb->error;

Deletes a QTL record and all associated information.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id          = $self->get_qtl_id( %args );
    my $db              = $self->db or return;

    #
    # Delete dbxrefs.
    #
    $db->do(
        'delete from qtl_dbxref_to_object where table_name=? and record_id=?',
        {}, ( 'qtl', $qtl_id )
    );

    #
    # Delete CMap positions.
    #
    if ( my $position = $self->get_cmap_position( qtl_id => $qtl_id ) ) {
        my $cmap_admin = $self->cmap_admin;
        $cmap_admin->feature_delete(
            feature_id => $position->{'feature_id'}
        ) or die $cmap_admin->error;
    }

    #
    # Delete QTL.
    #
    $db->do( 'delete from qtl where qtl_id=?', {}, ( $qtl_id ) );

    return 1;
}

# ----------------------------------------------------
sub db {

=pod

=head2 db

  my $dbh = $qdb->db or die $qdb->error;

Returns a handle to the QTL database as defined in the config file.

=cut

    my $self = shift;

    if ( !defined $self->{'db'} ) {
        $self->{'db'} = Gramene::DB->new('qtl');
    }

    return $self->{'db'};
}

# ----------------------------------------------------
sub get_qtl_lit_id {

=pod

=head2 get_qtl_lit_id

  my $lit_id = $qdb->get_qtl_lit_id( 
      qtl_id => 42,
  );

Returns the lit id xref value for a QTL.

=cut

    my ( $self, %args ) = @_;

    my $qtl_id    = $self->get_qtl_id( %args );
    my $db        = $self->db or return;
    my @lit_ids   = map { $_->{'xref_value'} } $self->get_xrefs(
        qtl_id    => $qtl_id,
        xref_type => 'Gramene Literature',
    );

    my $lit_id;
    if ( @lit_ids == 0 ) {
        $lit_id = 0;
    }
    elsif ( @lit_ids == 1 ) {
        $lit_id = shift @lit_ids;
    }
    else {
        my @lits =
            sort {
                $a->year <=> $b->year
                ||
                $a->volume <=> $b->volume
                ||
                $a->start_page <=> $b->start_page
            }
            map { Gramene::CDBI::Literature::Reference->retrieve($_) }
            @lit_ids;

        $lit_id = $lits[0]->id;
    }

    return $lit_id;
}

# ----------------------------------------------------
sub get_xrefs {

=pod

=head2 get_xrefs

  my $xrefs = $qdb->get_xrefs( 
      qtl_id    => 42,
      xref_type => 'Gramene Literature' # optional
  );

Returns all xrefs for a QTL.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id = $self->get_qtl_id( %args );
    my $db     = $self->db or return;
    my $sql    = sprintf(
        q[
            select xt.xref_type,
                   xt.url_template,
                   x.xref_value,
                   x.comments
            from   xref x, xref_type xt
            where  x.table_name=?
            and    x.record_id=?
            and    x.xref_type_id=xt.xref_type_id
            %s
        ],  
        $args{'xref_type'} ? "and xt.xref_type='$args{xref_type}'" : ''
    );

    my @xrefs = @{ $db->selectall_arrayref(
        $sql, { Columns => {} }, ( 'qtl', $qtl_id )
    ) };  

    return wantarray ? @xrefs : \@xrefs;
}

# ----------------------------------------------------
sub get_xref_types {

=pod

=head2 get_xref_types

  my $xref_types = $qdb->get_xref_types;

Returns all xref_types.

=cut

    my $self = shift;
    my $db   = $self->db or return;

    return $db->selectall_arrayref(
        q[
            select   xref_type_id, xref_type, url_template
            from     xref_type
            order by xref_type
        ],
        { Columns => {} }
    );
}

# ----------------------------------------------------
sub get_cmap_position {

=pod

=head2 get_cmap_position

  my $position         = $qdb->get_cmap_position(
      qtl_id           => 1998,      # either
      qtl_accession_id => 'ACQ9813', # or
  ) or die $qdb->error;

Returns a hashref of a QTL's CMap position from the CMap db.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id        = $self->get_qtl_id(%args);
    my $db            = $self->db or return;
    my $qtl_accession = $db->selectrow_array(
        'select qtl_accession_id from qtl where qtl_id=?',
        {}, ($qtl_id) );

    my $cmap_admin = $self->cmap_admin;
    my $cmap_sql   = $cmap_admin->sql;
    my $features   = $cmap_sql->get_features(
        cmap_object    => $cmap_admin,
        feature_acc    => $qtl_accession,
        ignore_aliases => 1,
    );
    my $feature;

    if ( $features and @$features ) {
        $feature = $features->[0];

        # Make some key name changes to blend with this code base
        $feature->{'feature_aid'}        = $feature->{'feature_acc'};
        $feature->{'map_set_aid'}        = $feature->{'map_set_acc'};
        $feature->{'map_set_short_name'} = $feature->{'map_set_name'};
        $feature->{'species'}            = $feature->{'species_common_name'};
    }

    return $feature;
}

# ----------------------------------------------------
sub get_trait_by_to {

=pod

=head2 get_trait_by_to

  my $trait = $qdb->get_trait_by_to("TO:0000529") or die $qdb->error;

Returns a hashref representing the trait associated with a TO accession.  
Fields include:

=over 4

=item * trait_symbol

=item * trait_name

=item * trait_category

=item * no_qtl

=back

=cut

    my $self         = shift;
    my $to_accession = shift or croak('No TO accession');

    my $db = $self->db;
    my $sth = $db->prepare(
        q[
            select   t.trait_symbol, t.trait_name, tc.trait_category,
                     count(q.qtl_id) as no_qtl
            from     qtl_trait t, qtl_trait_category tc, qtl q
            where    t.to_accession=?
            and      t.qtl_trait_category_id=tc.qtl_trait_category_id
            and      t.qtl_trait_id=q.qtl_trait_id
            group by t.trait_symbol, t.trait_name, tc.trait_category
        ]
    );
    $sth->execute( $to_accession );
    my $trait = $sth->fetchrow_hashref;

    return $trait;
}

# ----------------------------------------------------
sub get_qtl_id {

=pod

=head2 get_qtl_id

  my $qtl_id           = $qdb->get_qtl_id( 
      qtl_accession_id => 'ACQ9908', # either
      qtl_id           => 8872       # or
  ) or die $qdb->error;

Returns the "qtl_id" given either a "qtl_id" or "qtl_accession_id."  The 
idea is to use this method when you're not sure which field you're being
provided.

=cut

    my ( $self, %args )  = @_;
    my $qtl_id           = $args{'qtl_id'}           ||  0;
    my $qtl_accession_id = $args{'qtl_accession_id'} || '';

    return $qtl_id if $qtl_id;

    if ( !$qtl_accession_id ) {
        croak('No QTL accession');
    }

    my $db  = $self->db or return;
    $qtl_id = $db->selectrow_array(
        'select qtl_id from qtl where qtl_accession_id=?', 
        {}, ( $qtl_accession_id )
    );

    return $qtl_id;
}

# ----------------------------------------------------
sub get_qtl {

=pod

=head2 get_qtl

  my $qtl              = $qdb->get_qtl(
      qtl_accession_id => 'ACQ9908', # either
      qtl_id           => 8872       # or
  ) or die $qtl->error;

Returns all the known info on a QTL.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id          = $self->get_qtl_id( %args ) or return;
    my $db              = $self->db or return;
    my $sth             = $db->prepare( 
        q[
            select q.qtl_id,
                   q.qtl_accession_id,
                   q.published_symbol,
                   q.linkage_group,
                   q.start_position,
                   q.stop_position,
                   q.species_id,
                   s.species,
                   q.comments,
                   qt.qtl_trait_id,
                   qt.trait_symbol,
                   qt.trait_name,
                   qtc.qtl_trait_category_id,
                   qtc.trait_category
            from   qtl q, 
                   species s,
                   qtl_trait qt, 
                   qtl_trait_category qtc
            where  q.qtl_id=?
            and    q.qtl_trait_id=qt.qtl_trait_id
            and    q.species_id=s.species_id
            and    qt.qtl_trait_category_id=qtc.qtl_trait_category_id 
        ] 
    );
    $sth->execute( $qtl_id ) ;
    my $qtl = $sth->fetchrow_hashref or die "Can't find QTL";

    #
    # Get trait synonyms
    #
    $qtl->{'trait_synonyms'} = $db->selectcol_arrayref(
        q[
            select   trait_synonym
            from     qtl_trait_synonym
            where    qtl_trait_id=?
            order by trait_synonym
        ],
        {},
        ( $qtl->{'qtl_trait_id'} )
    );

    #
    # Get the QTL's xrefs
    #
    my $xrefs = $db->selectall_arrayref(
        q[
            select xt.xref_type, 
                   xt.url_template, 
                   x.xref_id,
                   x.xref_value
            from   xref x, xref_type xt
            where  x.table_name=?
            and    x.record_id=?
            and    x.xref_type_id=xt.xref_type_id
        ],
        { Columns => {} },
        ( 'qtl', $qtl->{'qtl_id'} )
    );

    #
    # Process any existing xref templates
    #
    for my $xref ( @$xrefs ) {
        my $name_string = $xref->{'xref_type'};
        $name_string =~ s/\./ /g;
        $name_string .= " " . $xref->{'xref_value'};

        $xref->{'name'} = $name_string;

        my $url_template = $xref->{'url_template'} or next;

        if ( $url_template ) {
            $xref->{'url'}  = sprintf( $url_template, $xref->{'xref_value'} );
        }
    }

    $qtl->{'xrefs'} = $xrefs;

    # CMap position
    $qtl->{'cmap_position'} = $self->get_cmap_position( qtl_id => $qtl_id );

    return $qtl;
}

# ----------------------------------------------------
sub get_qtl_ontology_associations {

=pod

=head2 get_qtl_ontology_associations

  my $assocs = $qdb->get_qtl_ontology_associations(
      qtl_id => 42,
  );

Returns the ontology terms associated with a QTL.

=cut

    my ( $self, %args ) = @_;
    my $qtl_id          = $self->get_qtl_id( %args );
    my $db              = $self->db or return;

    my $ont_assoc = $db->selectall_arrayref(
        q[  
            select   oa.qtl_ontology_association_id,
                     oa.term_accession,
                     oa.term_type,
                     e.evidence_code,
                     e.gramene_reference_id
            from     qtl_ontology_association oa,
                     qtl_ontology_association_evidence e
            where    oa.qtl_id=?
            and      oa.qtl_ontology_association_id=
                     e.qtl_ontology_association_id
            order by 1,2
        ],
        { Columns => {} },
        ( $qtl_id )
    );

    my $Qtl = Gramene::CDBI::Qtl::Qtl->retrieve( $qtl_id );
    my @lit_xrefs = $self->get_xrefs( 
        qtl_id    => $qtl_id,
        xref_type => 'Gramene Literature',
    );

    for my $lit_xref ( @lit_xrefs ) {
        push @$ont_assoc,
            {   
                term_accession       => $Qtl->qtl_trait->to_accession,
                term_type            => 'Trait Ontology',
                gramene_reference_id => $lit_xref->{'xref_value'},
                evidence_code        => 'IAGP',
            },
            {   
                term_accession       => $Qtl->species->gramene_taxonomy_id,
                term_type            => 'Species Ontology',
                gramene_reference_id => $lit_xref->{'xref_value'},
                evidence_code        => 'SM',
            },
        ;
    }

    for my $o ( @$ont_assoc ) {
        my ($Ont) = Gramene::CDBI::Ontology::Term->search(
            term_accession => $o->{'term_accession'},
        );

        if ( $Ont ) {
            $o->{'term_name'} = $Ont->term_name;
        }
        else {
            print STDERR "Unknown ontology accession '$o->{term_accession}'\n";
        }
    }

    return wantarray ? @$ont_assoc : $ont_assoc;
}

# ----------------------------------------------------
sub get_traits {

=pod

=head2 get_traits

  my $traits            = $qdb->get_traits(
      trait_category_id => 189,          # optional
      with_qtl_counts   => 1,            # optional
      no_synonyms       => 1,            # optional
      order_by          => 'trait_name', # optional
  );

Returns all available QTL traits.  If "trait_category_id" is present, only 
those traits associated with the given trait category will be returned.
If "with_qtl_counts" is a true value, then each trait will include the number
of QTL associated with it.  Unless "no_synonyms" is true, each trait will
include an arrayref of its synonyms.

=cut

    my ( $self, %args ) = @_;
    my $trait_cat_id    = $args{'trait_category_id'} || 0;
    my $db              = $self->db or return;
    my $order_by        = $args{'order_by'} || 'trait_symbol';
    my $sql             = sprintf(
        q[
            select   t.qtl_trait_id, 
                     t.trait_symbol, 
                     t.trait_name,
                     t.to_accession,
                     c.qtl_trait_category_id, 
                     c.trait_category
            from     qtl_trait t, 
                     qtl_trait_category c
            where    t.qtl_trait_category_id=c.qtl_trait_category_id
            %s
            %s
        ],
        $trait_cat_id ? 'and t.qtl_trait_category_id=?' : '',
        $order_by ne 'no_qtl' ? 'order by trait_symbol' : '',
    );

    my $traits = $db->selectall_arrayref(
        $sql, { Columns => {} }, ( $trait_cat_id || () )
    );

    if ( $args{'with_qtl_counts'} ) {
        my ( $count_sql, @bind );
        if ( $trait_cat_id ) {
            $count_sql = q[
                select    count(q.qtl_id) as no_qtl, q.qtl_trait_id
                from      qtl_trait t
                left join qtl q
                on        q.qtl_trait_id=t.qtl_trait_id
                where     t.qtl_trait_category_id=?
                group by  q.qtl_trait_id
            ];
            @bind = $trait_cat_id;
        }
        else {
            $count_sql = q[
                select   count(qtl_trait_id) as no_qtl, qtl_trait_id
                from     qtl
                group by qtl_trait_id
            ];
        }

        my $counts = $db->selectall_hashref(
            $count_sql, 'qtl_trait_id', {}, @bind
        );

        for my $t ( @$traits ) {
            $t->{'no_qtl'} = $counts->{ $t->{'qtl_trait_id'} }{'no_qtl'} || 0;
        }
    }

    unless ( $args{'no_synonyms'} ) {
        my $syn_sql = sprintf(
            q[
                select   s.qtl_trait_synonym_id,
                         s.qtl_trait_id,
                         s.trait_synonym
                from     qtl_trait_synonym s,
                         qtl_trait t
                where    s.qtl_trait_id=t.qtl_trait_id
                %s
                order by trait_synonym
            ],
            $trait_cat_id ? 'and t.qtl_trait_category_id=?' : ''
        );

        my $synonyms = $db->selectall_arrayref(
            $syn_sql, { Columns => {} }, ( $trait_cat_id || () )
        );

        my %syn_lookup;
        for my $s ( @$synonyms  ) {
            push @{ $syn_lookup{ $s->{'qtl_trait_id'} } }, 
                $s->{'trait_synonym'};
        }

        for my $t ( @$traits ) {
            $t->{'synonyms'} = $syn_lookup{ $t->{'qtl_trait_id'} };
        }
    }

    if ( $order_by eq 'no_qtl' ) {
        $traits = [
            map  { $_->[0] }
            sort { $b->[1] <=> $a->[1] }
            map  { [ $_, $_->{'no_qtl'} ] }
            @$traits
        ];
    }

    return $traits;
}

# ----------------------------------------------------
sub get_trait_categories {

=pod

=head2 get_trait_categories

  my $trait_categories = $qdb->get_trait_categories or die $qdb->error;

Returns all available QTL trait categories.

=cut

    my ( $self, %args ) = @_;
    my $db              = $self->db or return;

    return $db->selectall_arrayref(
        q[
            select   qtl_trait_category_id, trait_category
            from     qtl_trait_category
            order by trait_category
        ],
        { Columns => {} },
    );
}

# ----------------------------------------------------
sub search {

=pod

=head2 search

  my $qtl = $qdb->search(
      query             => 'ACQ*',
      search_field      => 'trait_symbol'
      species           => 'Rice',
      # or
      trait_category_id => 4,
      # or
      trait_id          => 887,
      # optional
      order_by          => 'trait_symbol',
  ) or die $qdb->error;

Searches for all QTL matching whatever criteria provided (if any).
The search parameters work in independent groups, so if
"trait_category_id" or "trait_id" are provided, then all QTL for those
are returned;  otherwise, a search is done for the "query" string in
the "search_field" for records matching the "species," if provided.
Valid search fields include "trait_name," "trait_symbol,"
"trait_synonym," "trait_category," "linkage_group,"
"published_symbol," and "qtl_accession_id." If no "search_field" is
indicated, then all fields will be searched.

=cut

    my ( $self, %args )   = @_;
    my $query             = $args{'query'}                || '';
    my $search_field      = $args{'search_field'}         || '';
    my $species           = $args{'species'}              || '';
    my $species_id        = $args{'species_id'}           || '';
    my $trait_id          = $args{'trait_id'}             ||  0;
    my $trait_category_id = $args{'trait_category_id'}    ||  0;
    my $sim_threshold     = $args{'similarity_threshold'} || '';
    my @literature_ids    = ref $args{'lit_id'} eq 'ARRAY'
                            ? @{ $args{'lit_id'} }
                            : split(/,/, $args{'lit_id'} || '');
    my $order_by          = $args{'order_by'} || 'trait_name';

    my ( @wheres, @args );
    if ( $trait_category_id ) {
        push @args, $trait_category_id;
        push @wheres, ' qt.qtl_trait_category_id=? ';
    }

    if ( $trait_id ) {
        push @args, $trait_id;
        push @wheres, 'q.qtl_trait_id=?';
    }

    if ( $species ) {
        push @wheres, 's.species=?';
        push @args, $species;
    }

    if ( $species_id ) {
        push @wheres, 's.species_id=?';
        push @args, $species_id;
    }

    $query =~ s/%/*/g;
    $query =~ s/\*{2,}/*/g;

    if ( $query && $query ne '*' ) {
        my $cmp = ( $query =~ s/\*/%/g ) ? 'like' : '=';
        if ( $search_field ) {
            if ( $search_field eq 'to_accession' ) {
                $search_field = 'qt.to_accession';
            }

            push @wheres, "$search_field $cmp ?";
            push @args, $query;
        } 
        else {
            push @wheres, qq[
                q.qtl_accession_id $cmp ? 
                or q.published_symbol $cmp ?
                or q.linkage_group $cmp ?
                or q.chromosome $cmp ?
                or qt.trait_symbol $cmp ?
                or qt.trait_name $cmp ?
                or qt.to_accession $cmp ?
                or qtc.trait_category $cmp ?
                or qts.trait_synonym $cmp ?
            ];

            push @args, $query for 1..9;
        }
    }

    if ( my @ps_traits = map { s/^\s+|\s+$//g; $_ || () }
        split( /[,\s]+/, $args{'ps_trait_inc'} )
    ) {
        push @wheres, join( ' OR ',
            ( map { "qt.trait_symbol = '$_'" } @ps_traits )
        );
    }

    if ( my @ps_traits = map { s/^\s+|\s+$//g; $_ || () }
        split( /[,\s]+/, $args{'ps_trait_exc'} )
    ) {
        push @wheres, join( ' AND ',
            ( map { "qt.trait_symbol != '$_'" } @ps_traits )
        );
    }

    if ( my @ps_species = map { s/^\s+|\s+$//g; $_ || () }
        split( /,/, $args{'ps_species_inc'} )
    ) {
        push @wheres, join( ' OR ',
            ( map { "species = '$_'" } @ps_species )
        );
    }

    if ( my @ps_species = map { s/^\s+|\s+$//g; $_ || () }
        split( /,/, $args{'ps_species_exc'} )
    ) {
        push @wheres, join( ' AND ', 
            ( map { "species != '$_'" } @ps_species )
        );
    }

    if ( my @ps_lg = map { s/^\s+|\s+$//g; $_ || () }
        split( /[,\s]+/, $args{'ps_linkage_group_inc'} )
    )
    {
        my @cond;
        for my $lg ( @ps_lg ) {
            my $cmp = $lg =~ /\*/ ? 'like' : '=';
            $lg     =~ s/\*/%/g;
            push @cond, "linkage_group $cmp '$lg'"
        }

        push @wheres, join( ' OR ', @cond );
    }

    if ( my @ps_lg = map { s/^\s+|\s+$//g; $_ || () }
        split( /[,\s]+/, $args{'ps_linkage_group_exc'} )
    )
    {
        my @cond;
        for my $lg ( @ps_lg ) {
            my $cmp = $lg =~ /\*/ ? 'not like' : '!=';
            $lg     =~ s/\*/%/g;
            push @cond, "linkage_group $cmp '$lg'"
        }
        
        push @wheres, join( ' AND ', @cond );
    }

    my $sql = qq[
        select     q.qtl_id,
                   q.qtl_accession_id,
                   q.published_symbol,
                   q.linkage_group,
                   q.chromosome,
                   s.species,
                   q.cmap_map_accession,
                   q.start_position,
                   q.stop_position,
                   qt.qtl_trait_id,
                   qt.trait_symbol,
                   qt.trait_name,
                   qtc.qtl_trait_category_id,
                   qtc.trait_category,
                   qts.trait_synonym
        from       qtl q
        inner join species s
        on         q.species_id=s.species_id
        inner join qtl_trait qt
        on         q.qtl_trait_id=qt.qtl_trait_id
        inner join qtl_trait_category qtc
        on         qt.qtl_trait_category_id=qtc.qtl_trait_category_id 
        left join  qtl_trait_synonym qts
        on         qt.qtl_trait_id=qts.qtl_trait_id
    ];

    if ( @literature_ids ) {
        $sql .= q[
            left join  xref x
            on         q.qtl_id=x.record_id
            inner join xref_type xt
            on         x.xref_type_id=xt.xref_type_id
        ];

        push @wheres,
            q[x.table_name='qtl'],
            @literature_ids == 1 
                ? qq[x.xref_value='$literature_ids[0]']
                : qq[x.xref_value in (] 
                . join(', ', map {qq['$_']} @literature_ids) 
                . ')',
            q[xt.xref_type='Gramene Literature'],
        ;
    }

    if ( @wheres ) {
        $sql .= ' where ' . shift @wheres;

        if ( @wheres ) {
            $sql .= ' and '. join( ' and ', map { "($_)" } @wheres );
        }
    }

    $sql .= " order by $order_by";

    my $db  = $self->db or return;
    my $tmp = $db->selectall_arrayref( $sql, { Columns => {} }, @args );

    #
    # Because there is a one-to-many relationship of traits
    # to synonyms, I had to do the following aggregation of
    # the records (too many QTL are returned by the query if
    # a trait has multiple synonyms).  I create a hash keyed
    # on the QTL accession ID, mark the order as returned from
    # the database and then push on the trait synonym, then
    # turn the values of the hash into an array sorted on the
    # original order.  It's a bit much, I know, but it seemed
    # the easiest way. - ky [12/23/03]
    #
    my ( %data, $i, %trait_synonyms );
    for my $rec ( @$tmp ) {
        $data{ $rec->{'qtl_accession_id'} } = $rec;
        $data{ $rec->{'qtl_accession_id'} }{'order'} = ++$i;
        if ( $rec->{'trait_synonym'} ) {
            push @{ $trait_synonyms{ $rec->{'qtl_accession_id'} } },
                $rec->{'trait_synonym'};
        }
    }
        
    for my $qtl_accession_id ( keys %trait_synonyms ) {
        $data{ $qtl_accession_id }{'trait_synonyms'} = [
            sort @{ $trait_synonyms{ $qtl_accession_id } }
        ];
    }              
                   
    my $qtl = [       
        sort { $a->{'order'} <=> $b->{'order'} }
        values %data
    ];      

    return wantarray ? @$qtl : $qtl;
}

# ----------------------------------------------------
sub trait_search {

=pod

=head2 trait_search

  my $traits = $qdb->trait_search(
      query      => 'FOO',          # wildcards acceptable, e.g., 'foo*'
      field_name => 'trait_symbol', # or 'trait_name'
      order_by   => 'trait_symbol', # a field name in the result set
  ) or die $qdb->error;

Searches for a trait.  Returns an arrayref of hashrefs.

=cut

    my ( $self, %args ) = @_;
    my $query           = $args{'query'}      || '';
    my $field_name      = $args{'field_name'} || '';
    my $order_by        = $args{'order_by'}   || 'trait_symbol';

    if ( !$field_name && !$query ) {
        croak('Nothing to search for');
    }

    if ( $field_name !~ /^(trait_name|trait_symbol|to_accession)$/ ) {
        croak("Bad field name ($field_name)");
    }

    my %search_values = map { s/^\s+|\s+$//g; $_, 1 } split(/,/, $query);
    my ( @wheres, @args );
    for my $val ( keys %search_values ) {
        my $cmp = ( $val =~ s/\*/%/ ) ? 'like' : '=';
        push @wheres, "t.$field_name $cmp ?";
        push @args, $val;
    }

    my $sql = q[
        select     t.qtl_trait_id,
                   t.trait_name,
                   t.trait_symbol,
                   t.to_accession,
                   tc.qtl_trait_category_id,
                   tc.trait_category,
                   count(q.qtl_id) as no_qtl
        from       qtl_trait_category tc,
                   qtl_trait t
        left join  qtl q
        on         t.qtl_trait_id=q.qtl_trait_id
        where      t.qtl_trait_category_id=tc.qtl_trait_category_id
    ];

    if ( @wheres ) {
        $sql .= ' AND ( ' . join( ' OR ', map { " ($_) " } @wheres ) . ' ) ';
    }

    $sql .= qq[
        group by   t.qtl_trait_id, t.trait_name, t.trait_symbol, 
                   tc.qtl_trait_category_id, tc.trait_category
        order by   $order_by
    ];

    my $db     = $self->db or return;
    my $traits = $db->selectall_arrayref( $sql, { Columns => {} }, @args );

    return $traits;
}

# ----------------------------------------------------
sub update_trait {

=pod

=head2 update_trait

  $qdb->update_trait(
      trait_id          => 8892,
      trait_symbol      => 'FOO',
      trait_name        => 'foobar',
      to_accession      => 'TO:0000673',
      trait_category_id => 8,
  ) or die $qdb->error;

Updates a trait.

=cut

    my ( $self, %args )   = @_;
    my $trait_id          = $args{'trait_id'}     or croak('No trait id');
    my $trait_symbol      = $args{'trait_symbol'} or croak('No trait symbol');
    my $trait_name        = $args{'trait_name'}   or croak('No trait name');
    my $to_accession      = $args{'to_accession'} || '';
    my $trait_category_id = $args{'trait_category_id'} or
                            croak('No trait category id');

    my $db = $self->db or return;
    $db->do(
        q[
            update qtl_trait
            set    trait_symbol=?, trait_name=?, 
                   qtl_trait_category_id=?, to_accession=?
            where  qtl_trait_id=?
        ],
        {},
        ( $trait_symbol, $trait_name, $trait_category_id, $to_accession,
          $trait_id )
    );

    return 1;
}

# ----------------------------------------------------
sub update_trait_synonym {

=pod

=head2 update_trait_synonym

  $qdb->update_trait_synonym(
      trait_synonym_id => 8892,
      synonym          => 'foo',
  ) or die $qdb->error;

Updates a trait synonym.

=cut

    my ( $self, %args )  = @_;
    my $trait_synonym_id = $args{'trait_synonym_id'} or 
                           croak('No trait synonym id');
    my $synonym          = $args{'synonym'} or croak('No synonym');

    my $db = $self->db or return;
    $db->do(
        q[
            update qtl_trait_synonym
            set    trait_synonym=?
            where  qtl_trait_synonym_id=?
        ],
        {},
        ( $synonym, $trait_synonym_id )
    );

    return 1;
}

# ----------------------------------------------------
sub update_qtl {

=pod

=head2 update_qtl

  $qdb->update_qtl(
      qtl_id           => 4328,
      qtl_accession_id => 'foo',
      published_symbol => 'foo-1'
      qtl_trait_id     => 8,
      linkage_group    => '3',
      chromosome       => '3',
      start_position   => 190.1,
      stop_position    => 198.2,
      comments         => '',
      species          => 'Rice',
  ) or die $qdb->error;

Updates the QTL and CMap position info.

=cut

    my ( $self, %args )  = @_;
    my $qtl_id           = $args{'qtl_id'}           ||  0;
    my $qtl_accession_id = $args{'qtl_accession_id'} || '';
    my $published_symbol = $args{'published_symbol'} || '';
    my $qtl_trait_id     = $args{'qtl_trait_id'}     ||  0;
    my $linkage_group    = $args{'linkage_group'}    || '';
    my $chromosome       = $args{'chromosome'}       || '';
    my $start_position   = $args{'start_position'};
    my $stop_position    = $args{'stop_position'};
    my $comments         = $args{'comments'}         || '';
    my $species          = $args{'species'}          || '';

    if ( !$chromosome ) {
        ( $chromosome = $linkage_group ) =~ s/[a-zA-Z]$//;
    }

    my $species_id = $args{'species_id'} || '';
    if ( !$species_id && $species ) {
        my $species = Gramene::CDBI::Qtl::Species->find_or_create({
            species => $species
        });

        $species_id = $species->id;
    }

    my $db = $self->db or return;
    $db->do(
        q[
            update qtl
            set    qtl_accession_id=?, published_symbol=?,
                   qtl_trait_id=?, linkage_group=?, chromosome=?,
                   start_position=?, stop_position=?, comments=?, 
                   species_id=?
            where  qtl_id=?
        ],
        {},
        ( $qtl_accession_id, $published_symbol, $qtl_trait_id, $linkage_group, 
          $chromosome, $start_position, $stop_position, $comments, $species_id, 
          $qtl_id )
    );

    my $cmap_db  = $self->cmap_db or return;
    if ( my $position = $self->get_cmap_position( qtl_id => $qtl_id ) ) {
        $cmap_db->do(
            q[
                update cmap_feature
                set    feature_start=?, feature_stop=?
                where  feature_id=?
            ],
            {},
           ( $start_position, $stop_position, $position->{'feature_id'} )
        );
    }

    return 1;
}

1;

# ----------------------------------------------------

=pod

=head1 SEE ALSO

Bio::GMOD::CMap, Class::Base, Gramene::Config, Gramene::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
