package Gramene::Protein::ProteinDB;

=head1 NAME

Gramene::Protein::ProteinDB

=head1 SYNOPSIS

use Gramene::Protein::ProteinDB;

=head1 DESCRIPTION

A module for querying gene database  

=cut

# ----------------------------------------------------------------

use strict;
use DBI;

use Gramene::DB;

sub new {

	my $class = shift;
	my $self  = {};
	bless( $self, $class );
	return $self;

}

# destructor
sub DESTROY {
	my $self = shift;

	# $self->terminate_database;
}

sub terminate_database {
	my $self = shift;
	$self->{'db'}->disconnect()            if $self->{'db'};
	$self->{'ontology_db'}->disconnect()   if $self->{'ontology_db'};
	$self->{'literature_db'}->disconnect() if $self->{'literature_db'};
    $self->{'gene_db'}->disconnect()            if $self->{'gene_db'};
}

sub connect_to_db {
	my $self = shift;

	eval {

		# get  db connection from configuration file
		$self->{'db'} = Gramene::DB->new('protein');

		# get literature db connection from configuration file
		$self->{'literature_db'} = Gramene::DB->new('literature');

        $self->{'ontology_db'}    = Gramene::DB->new('ontology');

		
		$self->{'gene_db'} = Gramene::DB->new('genes');
	};
	if ($@) {
		die "DB connection failed:$@\n";
	}

}

sub get_protein_id {

	my ( $self, $acc ) = @_;

	my ($gp_id) = $self->{'db'}->selectrow_array(
		q[
             SELECT gene_product_id FROM gene_product_helper
             WHERE  swissprot_acc = ?
         ],
		{},
		($acc)
	);

	return $gp_id;

}

# --------------------------------------------------------------------------------

=pod

=head2 get_gene_references

Get the gene gene's or allele's  literature references.

Return an array of reference object.

=cut

sub get_references {

	my ( $self, $gp_id) = @_;


    my $xref_dbname = 'gramene.literature';
    my $xref_keytype = 'id';

    my @ref_ids = get_xrefs($self, $gp_id, $xref_dbname, $xref_keytype);

	my %unsorted_references;
	my @references;


	foreach my $ref_id (@ref_ids){
		my $ref_obj = _get_reference_detail( $self, $ref_id );
		my $pub_year = $ref_obj->{'year'};
		push @{ $unsorted_references{$pub_year} }, $ref_obj;
	}



	# sort references by publish year
	foreach my $pub_year ( sort { $a <=> $b } keys %unsorted_references ) {
		my @ref_objs = @{ $unsorted_references{$pub_year} };
		push @references, @ref_objs;

	}

	return \@references;

}

#
# get the reference's title,source_name,volume,start_page,end_page and author
#
sub _get_reference_detail {

	my ( $self, $ref_id ) = @_;

	my $ref_obj;

	my $sth = $self->{'literature_db'}->prepare(
		q[
                           SELECT R.reference_id, R.title, R.year,volume,
                                  start_page,end_page, source, author
                           FROM   reference R, reference_extra RE
                           WHERE  R.reference_id = RE.reference_id	
                           AND    R.reference_id = ?
                         ]
	);
	$sth->execute( ($ref_id) ) or die $self->{'literature_db'}->errstr;

	$ref_obj = $sth->fetchrow_hashref;

	

	$sth->finish;



	return $ref_obj;

}




sub get_proteins_by_name {
	my ( $self, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';

	my $sql = qq[
                   SELECT gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM   gene_product_helper
		   WHERE  ( gene_product_name like ? 
		           OR gene_product_gene_name like ?
			  ) 
    ];

	my $condition = '';
	my @params;

	push @params, ($query, $query);

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		push @candidates, $gp_obj;
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_helper_field

Get the candidate protein objects by searching  fields in gene_product_helper table 

=cut

sub get_proteins_by_helper_field {
	my ( $self, $search_field, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';

	my $sql = qq[
                   SELECT gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM   gene_product_helper
    ];

	my $condition = '';
	my @params;

	$condition = qq[ WHERE UPPER( $search_field ) like ? ];
	push @params, $query;

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		push @candidates, $gp_obj;
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_pid

Get the candidate protein objects by searching  pid 

=cut

sub get_proteins_by_pid {

	my ( $self, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';
	my %uniq_ids;

	my $sql = qq[
                   SELECT     GH.gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM       gene_product_helper GH
                   INNER JOIN  gene_product_to_pid GP
		                 on   GH.gene_product_id = GP.gene_product_id
    ];

	my $condition = '';
	my @params;

	$condition = qq[ WHERE UPPER( pid ) like ? ];
	push @params, $query;

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		my $gp_id = $gp_obj->{'gene_product_id'};
		unless ( $uniq_ids{$gp_id} ) {
			push @candidates, $gp_obj;
			$uniq_ids{$gp_id} = 1;
		}
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_gi

Get the candidate protein objects by searching  gi 

=cut

sub get_proteins_by_gi {

	my ( $self, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';
	my %uniq_ids;

	my $sql = qq[
                   SELECT     GH.gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM       gene_product_helper GH
                   INNER JOIN  gene_product_to_gi_number GI
		                 on   GH.gene_product_id = GI.gene_product_id
    ];

	my $condition = '';
	my @params;

	$condition = qq[ WHERE UPPER( gi_number ) like ? ];
	push @params, $query;

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		my $gp_id = $gp_obj->{'gene_product_id'};
		unless ( $uniq_ids{$gp_id} ) {
			push @candidates, $gp_obj;
			$uniq_ids{$gp_id} = 1;
		}
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_ec

Get the candidate protein objects by searching  ec

=cut

sub get_proteins_by_ec {

	my ( $self, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';
	my %uniq_ids;

	my $sql = qq[
                   SELECT     GH.gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM       gene_product_helper GH
                   INNER JOIN objectxref OX
                   ON         GH.gene_product_id = OX.row_id
                   INNER JOIN dbxref DX
                   ON         OX.dbxref_id    = DX.dbxref_id
                   WHERE      OX.table_name   = 'gramene.protein'
                   AND        DX.xref_dbname  = 'ENZYME'
                   AND        DX.xref_keytype = 'EC'
            
    ];

	my $condition = '';
	my @params;

	$condition = qq[ AND UPPER( xref_key ) like ? ];
	push @params, $query;

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}
	print STDERR "sql=$sql $condition\n";
	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		my $gp_id = $gp_obj->{'gene_product_id'};
		unless ( $uniq_ids{$gp_id} ) {
			push @candidates, $gp_obj;
			$uniq_ids{$gp_id} = 1;
		}
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_cultivar

Get the candidate protein objects by searching  cultivar 

=cut

sub get_proteins_by_cultivar {
	my ( $self, $query, $organism, $order_by ) = @_;
	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';

	my $sql = qq[
                   SELECT     GH.gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM       gene_product_helper GH
                   INNER JOIN gene_product_to_cultivar GC
                   ON         GH.gene_product_id = GC.gene_product_id
                   INNER JOIN cultivar C 
                   ON         GC.cultivar_id = C.cultivar_id
    ];

	my $condition = '';
	my @params;

	$condition = qq[ WHERE UPPER( cultivar_name ) like ? ];
	push @params, $query;

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		push @candidates, $gp_obj;
	}

	return @candidates;
}

# -----------------------------------------------------------------------------------

=pod

=head2 get_proteins_by_all_search_fields

Get the candidate protein objects by all search fields

=cut

sub get_proteins_by_all_search_fields {

	my ( $self, $query, $organism, $order_by ) = @_;

	$query =~ s/^\s+|\s+$//g;
	$query = uc($query);
	$query =~ s/^\*+|\*+$//g;    # remvoe the user defined wildcard
	$query = "*$query*";         # set wildard as default
	$query =~ s/\*/%/g;
	$organism = uc($organism) . '%';
	my %uniq_ids;

	my $sql = qq[
                   SELECT     GH.gene_product_id, swissprot_acc, swissprot_id, gene_product_name, organism, evidence_codes
                   FROM       gene_product_helper GH
                      LEFT JOIN  gene_product_to_pid GP
		                 on   GH.gene_product_id = GP.gene_product_id
		          LEFT JOIN  gene_product_to_gi_number GI
		                 on   GH.gene_product_id = GI.gene_product_id
		          WHERE       (
		                      UPPER(swissprot_acc) like ?
		          OR          UPPER(swissprot_id)  like ?
		          OR          UPPER(gene_product_name) like ?
		          OR          UPPER(gene_product_gene_name) like ?
		          OR          UPPER(pid) like ?
		          OR          UPPER(gi_number) like ?
			      OR          UPPER(organism) like ?
		                     )     
    ];

	my $condition = '';
	my @params;

	push @params, $query for ( 1 .. 7 );

	if ($organism) {
		$condition .= ' AND UPPER( organism ) like ? ';
		push @params, $organism;
	}

	if ($order_by) {

		# use if ( is null) to set null or empty field displayed at last
		$condition .=
		  " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by";
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );
	$sth->execute(@params) or die $self->{'db'}->errstr;

	my @candidates;
	while ( my $gp_obj = $sth->fetchrow_hashref ) {
		my $gp_id = $gp_obj->{'gene_product_id'};
		unless ( $uniq_ids{$gp_id} ) {
			push @candidates, $gp_obj;
			$uniq_ids{$gp_id} = 1;
		}
	}

	return @candidates;
}

sub validate_id {
	my ( $self, $id ) = @_;
	my $sth1 =
	  $self->{'db'}->prepare(
"select gene_product_id from gene_product_helper where gene_product_id = ?"
	  );
	$sth1->bind_param( 1, $id );
	$sth1->execute();
	my ($p_id) = $sth1->fetchrow_array();
	$sth1->finish;
	return $p_id;
}

sub get_id_by_acc {
	my ( $self, $acc ) = @_;
	my $sth1 =
	  $self->{'db'}->prepare(
"select gene_product_id from gene_product_helper where UPPER(swissprot_acc) = ?"
	  );
	$sth1->bind_param( 1, $acc );
	$sth1->execute();
	my ($p_id) = $sth1->fetchrow_array();
	$sth1->finish;
	return $p_id;

}

sub get_id_by_swall {
	my ( $self, $swall_id ) = @_;
	my $sth1 =
	  $self->{'db'}->prepare(
"select gene_product_id from gene_product_helper where UPPER(swissprot_id) = ?"
	  );
	$sth1->bind_param( 1, $swall_id );
	$sth1->execute();
	my $id = $sth1->fetchrow_array();
	$sth1->finish;
	return $id;

}

sub get_id_by_pid {
	my ( $self, $pid ) = @_;
	my $sth =
	  $self->{'db'}
	  ->prepare("select gene_product_id from gene_product_to_pid where pid=?");
	$sth->bind_param( 1, $pid );
	$sth->execute;
	my ($gp_id) = $sth->fetchrow_array();
	$sth->finish;
	return $gp_id;

}

sub get_id_by_gi {
	my ( $self, $gi ) = @_;
	my $sth =
	  $self->{'db'}->prepare(
"select gene_product_id from gene_product_to_gi_number where gi_number=?"
	  );
	$sth->bind_param( 1, $gi );
	$sth->execute;
	my ($gp_id) = $sth->fetchrow_array();
	$sth->finish;
	return $gp_id;

}

sub get_protein_object {
	my ( $self, $protein_id ) = @_;
	my $sth = $self->{'db'}->prepare(
		q[
	   SELECT 	gene_product_id,
	             gene_product_full_name, 
	             gene_product_symbol
	   FROM      gene_product 
	   WHERE     gene_product_id = ?
	]
	);

	$sth->execute( ($protein_id) ) or die $self->{'db'}->errstr;

	my $protein_obj = $sth->fetchrow_hashref;
	$sth->finish;

	return $protein_obj;
}

sub get_swall {
    my ($self, $p_id) = @_; 
    my $sth1 = $self->{'db'}->prepare(
        "select swissprot_id from gene_product_helper
                               where gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my $acc = $sth1->fetchrow_array();
    $sth1->finish;
    return $acc;

}

sub get_protein_acc_by_id {
	my ( $self, $id ) = @_;
	my $sth1 =
	  $self->{'db'}->prepare(
"select swissprot_acc from gene_product_helper where gene_product_id = ?"
	  );
	$sth1->bind_param( 1, $id );
	$sth1->execute();
	my ($acc) = $sth1->fetchrow_array();
	$sth1->finish;
	return $acc;

}

sub get_synonyms {
	my ( $self, $p_id ) = @_;

	my $sth2 =
	  $self->{'db'}->prepare(
"SELECT gene_product_synonym_symbol FROM gene_product_synonym WHERE gene_product_id = ?"
	  );
	$sth2->bind_param( 1, $p_id );
	$sth2->execute();
	my ( $syn, $syns );
	while ( $syn = $sth2->fetchrow_array() ) {

		if ( ( $syn =~ /\)/ ) && ( !( $syn =~ /\(/ ) ) ) {
			$syn =~ s/\)//;
		}
		unless ( $syn eq "FRAGMENT" ) {
			if ($syns) {
				$syns = $syns . ", " . $syn;
			}
			else {
				$syns = $syn;
			}
		}
	}

	$syns = 'Not available' unless $syns;

	$sth2->finish;

	return $syns;

}

sub get_keywords {
    my $self = $_[0];
    my $p_id = $_[1];

    my $sth1 = $self->{'db'}->prepare(
        "SELECT KW.keyword
                               FROM   keyword KW, gene_product_to_keyword GPKW
                               WHERE  KW.keyword_id =GPKW.keyword_id
                                AND   gene_product_id = ?
                              "
    );

    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ( $gns, $gn );
    while ( $gn = $sth1->fetchrow_array() ) {

        #$gns = $gns.", ".$gn;
        if ($gns) {
            $gns = $gns . ", " . $gn;
        }
        else {
            $gns = $gn;
        }
    }
    $sth1->finish;
    unless ($gns) {
        $gns = "Not available";
    }
    return $gns;

}

sub get_gene_names {
	my ($self, $p_id) = @_; 
	
	my $sth1 = $self->{'db'}->prepare(
		"SELECT gene_name FROM gene_product_gene_name  WHERE gene_product_id = ?"
	);
	$sth1->bind_param( 1, $p_id );
	$sth1->execute();
	my ( $gns, $gn );
	while ( $gn = $sth1->fetchrow_array() ) {
		if ($gns) {
			$gns = $gns . ", " . $gn;
		}
		else {
			$gns = $gn;
		}
	}
	$sth1->finish;
	unless ($gns) {
		$gns = "Not available";
	}
	return $gns;

}

sub get_ecs {
    my ($self, $p_id) = @_; 
    
    my $xref_dbname = 'ENZYME';
    my $xref_keytype = 'EC';
    my @ecs = get_xrefs($self,$p_id, $xref_dbname, $xref_keytype);
 
    return \@ecs;

}

sub get_pids {
    my ($self, $id) = @_; 
    my $sth1 = $self->{'db'}->prepare(
        "SELECT pid
                               FROM gene_product_to_pid
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $id );
    $sth1->execute();
    my ( @pids, $pid );
    while ( $pid = $sth1->fetchrow_array ) {
        @pids = ( @pids, $pid );
    }
    $sth1->finish;
    return \@pids;

}

sub get_organelle {
    my ($self, $p_id) =@_; 
    my $sth1 = $self->{'db'}->prepare(
        "SELECT organelle
                               FROM gene_product_organelle
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my $tissue = $sth1->fetchrow_array();
    unless ($tissue) {
        $tissue = "Not available";
    }
    $sth1->finish;
    return $tissue;

}

sub get_organisms {
    my ($self, $p_id ) = @_;
 
    my $sth2 = $self->{'db'}->prepare(
        "SELECT genus, species, ncbi_taxa_id, cultivar_name  
         FROM   gene_product_to_cultivar GC, species S, cultivar C
         WHERE  GC.cultivar_id = C.cultivar_id
         AND    S.species_id = C.species_id
         AND    gene_product_id = ?"
                             
    );
    $sth2->bind_param( 1, $p_id );
    $sth2->execute();
    my ( $genus, $spe, $taxa_id, $cul ) = $sth2->fetchrow_array;
    $cul =~s/Unknown/Not available/i;
 
    $sth2->finish;

    return ( $genus, $spe, $taxa_id, $cul );

}

sub get_phenotype_genes {
    my ($self, $p_id) = @_;

    my $xref_dbname = 'gramene.gene';
    my $xref_keytype = 'acc';

    my @gp_mus = get_xrefs($self,$p_id,$xref_dbname, $xref_keytype);
    my @genes;
    my $sth_mu = $self->{'gene_db'}->prepare(
        " SELECT accession, symbol,name  FROM  gene_gene WHERE accession = ? "
    );

    foreach  my $mu_acc (@gp_mus) {
      
        $sth_mu->execute( ($mu_acc) );
        my $gene = $sth_mu->fetchrow_hashref;

        push( @genes, $gene );

    }

    $sth_mu->finish;
    return \@genes;
}


sub get_protein_xref_db {
    my $self = $_[0];
    my $gpid = $_[1];
    my $sth  = $self->{'db'}->prepare(
        q[
                         SELECT xref_dbname
                         FROM   gene_product,dbxref
                         WHERE  organism_dbxref_id = dbxref_id 
                         AND    gene_product_id = ?
                       ]
    );
    $sth->execute( ($gpid) );
    my ($xref_dbname) = $sth->fetchrow_array();
    $sth->finish;
    return $xref_dbname;

}

# only show public comment, hide curator comment
sub get_comment {
    my ($self, $p_id) = @_; 
    my $sth1 = $self->{'db'}->prepare(
        "SELECT public_comment
                               FROM gene_product_comment
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ($comment) = $sth1->fetchrow_array();
    $sth1->finish;
    return $comment;

}


sub get_gis {
    my ($self, $p_id) = @_;
    my $sth1 = $self->{'db'}->prepare(
        "SELECT gi_number
                               FROM gene_product_to_gi_number
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ( @gis, $gi );
    while ( $gi = $sth1->fetchrow_array ) {
        @gis = ( @gis, $gi );
    }
    $sth1->finish;
    return \@gis;

}

sub get_pfams {
    my ($self, $p_id) = @_; 

    my $sth1 = $self->{'db'}->prepare(
        "SELECT xref_key as pfam_acc, xref_desc as pfam_desc
                               FROM   dbxref DBX,objectxref OBX
                               WHERE  DBX.dbxref_id = OBX.dbxref_id 
                                 AND  xref_keytype = 'ACC'
                                 AND  xref_dbname = 'Pfam'
                                 AND  row_id = ?
                                 AND  table_name = 'gramene.protein'
                              "
    );

    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ( @pfams );
    while ( my $pfam  = $sth1->fetchrow_hashref() ) {
        @pfams  = ( @pfams,  $pfam );
      
    }

    $sth1->finish;
    return \@pfams;

}

sub get_prosites {
    my ($self, $p_id ) = @_; 

# You don't want to display those entries with something like "(355 residues" in the PROSITE_SEQUENCE field.
    my $sth1 = $self->{'db'}->prepare("
        SELECT prosite_id, prosite_desc, prosite_sequence
        FROM   gene_product_prosite
        WHERE gene_product_id = ?
        order by prosite_id 
     " );
                        
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my @prosites;
 
    while ( my $prosite = $sth1->fetchrow_hashref ) {
        my $sequence = $prosite->{'prosite_sequence'};
        if($sequence && $sequence =~/residues/i){
           next;
        }
        push @prosites, $prosite;
    }

    $sth1->finish;
    return \@prosites;

}

sub get_features {
    my ($self, $pid) =@_; 
    my $sth1 = $self->{'db'}->prepare( "
                               SELECT feature_type, from_position,to_position,xref_dbname,xref_key 
                               FROM gene_product_feature GF 
                               INNER JOIN dbxref   
                               ON GF.dbxref_id = dbxref.dbxref_id                                  
                               INNER JOIN gene_product_feature_type GFT 
                               ON GF.feature_type_id = GFT.feature_type_id
                               WHERE gene_product_id = ?
                               ORDER BY feature_type,from_position,to_position,xref_dbname
                            " );

    $sth1->bind_param( 1, $pid );
    $sth1->execute();
    my @features;
    while ( my $feature = $sth1->fetchrow_hashref ){
        push @features, $feature;
    }
    $sth1->finish;
    return \@features;

}


sub get_related_refs {
    my $self  = $_[0];
    my $gp_id = $_[1];
    my $sth1  = $self->{'db'}->prepare(
        "select gene_product_name
                               from gene_product_helper
                               where gene_product_id = ?"
    );
    $sth1->bind_param( 1, $gp_id );
    $sth1->execute();
    my $gp_name = $sth1->fetchrow_array();
    $gp_name =~s/^\s+|\s+$//g;
    my @names = split( /   /, $gp_name );

    my $query = "select reference_id 
                 from reference_extra
                 where UPPER(title) like ?
                 or UPPER(abstract_part_a) like ?
                 or UPPER(abstract_part_b) like ?";
    my $i = 2;
    while ( $i <= scalar(@names) ) {
        $query = $query . " or UPPER(title) like ?
                          or UPPER(abstract_part_a) like ?
                          or UPPER(abstract_part_b) like ?";
        $i++;
    }
    my $sth2 = $self->{'literature_db'}->prepare($query);
    my $j    = 1;
    while ( $j <= scalar(@names) ) {
        my $l = $j * 3 - 2;
        my $m = $j * 3 - 1;
        my $n = $j * 3;
        $names[ $j - 1 ] =~ tr/a-z/A-Z/;
        $names[ $j - 1 ] = "% " . $names[ $j - 1 ] . " %";
        $sth2->bind_param( $l, $names[ $j - 1 ] );
        $sth2->bind_param( $m, $names[ $j - 1 ] );
        $sth2->bind_param( $n, $names[ $j - 1 ] );
        $j++;
    }
    $sth2->execute();
    my ( $ref_id, @ref_ids );
    while ( $ref_id = $sth2->fetchrow_array ) {
        @ref_ids = ( @ref_ids, $ref_id );
    }
    $sth1->finish;
    $sth2->finish;
    return \@ref_ids;

}

sub get_associations{
	my ($self, $gp_id) = @_;
	my $sth = $self->{'db'}->prepare(q[
	    SELECT term_type, term_accession, evidence_id, xref_dbname, xref_key, evidence_code, seq_acc
	    FROM   association A, evidence E, dbxref X
	    WHERE  A.association_id = E.association_id
	    AND    E.dbxref_id      = X.dbxref_id
	    AND    gene_product_id = ?
	]);
	
	$sth->execute(($gp_id));
	
	my $terms = {};
	my $associations = {};
	while(my ($term_type,$term_acc, $evn_id, $xdb, $xkey,$evn_code, $seq_acc) = $sth->fetchrow_array){
	    my $term_name = $terms->{$term_acc};
	    unless($term_name){
	    	    $term_name = _get_term_name($self,  $term_acc );
	    	    $term_name ||= $term_acc;
	    	    $terms->{$term_acc} = $term_name;
	    }
	    
	    $seq_acc ||= 'NA';
	    $associations->{$term_type}->{$term_acc}->{"$xdb:$xkey"}->{$evn_code}=$seq_acc;
	    
	    
	}
	
	return ($associations,$terms);
}


sub _get_term_name{
	my ($self, $term_acc) = @_;
	my $sth = $self->{'ontology_db'}->prepare(
            "SELECT term_name  FROM term
	        WHERE term_accession = ?"
        );
    $sth->bind_param( 1, $term_acc );
    $sth->execute();
    my ( $term_name ) = $sth->fetchrow_array;
    $sth->finish;
    return $term_name;
}


sub get_ensembl_xrefs {
    my ($self, $p_id) = @_; 
    
    my $xref_dbname = 'gramene.ensembl';
    my $xref_keytype = 'id';
    my @xrefs = get_xrefs($self,$p_id, $xref_dbname, $xref_keytype);
 
    return \@xrefs;

}


sub get_sequence {
    my ( $self, $gp_id ) = @_;
    my $sth = $self->{'db'}->prepare(
        q[
		SELECT seq
		FROM   gene_product_seq GPS, seq
		WHERE  GPS.seq_id = seq.id
		AND    gene_product_id = ?
	    ]
    );
    $sth->execute( ($gp_id) );
    my ($seq) = $sth->fetchrow_array();
    $seq =~ s/^\s+|\s+$//g if $seq;

    $sth->finish;
    return $seq;
}


sub get_xrefs{
    my ($self, $obj_id, $xref_dbname, $xref_keytype) = @_;


    my $table_name = 'gramene.protein';
    my $sql = (
        " SELECT distinct xref_key
	  FROM  objectxref OX, dbxref DX
	  WHERE OX.dbxref_id = DX.dbxref_id
	  AND   row_id = ? 
	  AND   table_name = ?
	  AND   xref_dbname = ?
	"
    );


    if($xref_keytype){
	$sql.= ' AND xref_keytype = ? ';
    }

    my $sth = $self->{'db'}->prepare($sql);
    my @params = ($obj_id, $table_name,$xref_dbname);
    push(@params, $xref_keytype) if $xref_keytype;

    $sth->execute( @params );
    my @xref_keys;
    while (my ($xref_key) = $sth->fetchrow_array){
	push @xref_keys,$xref_key;
    }

    return @xref_keys;

}

sub get_all_xref_db_urls{
    my ($self ) = @_;
    my ($url_templates) = $self->{'db'}->selectall_hashref(
	q[
	    SELECT upper(name) as db_name, url_syntax FROM data_base
	],('db_name') );

    return $url_templates;

}

1;
