package Gramene::Protein::EditProteinDB;

=head1 NAME

Gramene::Protein::EditProteinDB

=head1 SYNOPSIS

  

=head1 DESCRIPTION

A module for operating protein database when editing protein 

=cut

# ----------------------------------------------------------------

use strict;
use POSIX qw(strftime);
use DBI;

use Gramene::DB;
use Gramene::Ontology::OntologyDB;

sub new {

    #my $invocant = shift;
    #my $class = ref($invocant) || $invocant;
    my $class = shift;
    my $self  = {};
    bless( $self, $class );
    return $self;

}

sub terminate_database {
    my $self = shift;
    $self->{'db'}->disconnect() if $self->{'db'};
    $self->{'ontology_db'}->disconnect() if $self->{'ontology_db'};
}

sub connect_to_db {

    my $self = shift;
    eval { 
	$self->{'db'} = Gramene::DB->new('protein_editing'); 

	# get ontology db connection from ontology DB module for using its methods
	$self->{'OntologyDB'} = Gramene::Ontology::OntologyDB->new();


    };

    if ($@) {
        die "DB connection to protein DB failed\n";

    }

}

sub get_protein_id {

    my ($self, $acc) = @_;
    $acc =~ tr/a-z/A-Z/;

    my $sth =
      $self->{'db'}->prepare(q[
	SELECT gene_product_id
	FROM   gene_product, dbxref
	WHERE  organism_dbxref_id = dbxref.dbxref_id
	AND    xref_keytype = 'ACC'
	AND    UPPER(xref_key) = ?
    ]);
	

    $sth->bind_param( 1, $acc );
    $sth->execute() or die $self->{'db'}->errstr;
    my ($gp_id) = $sth->fetchrow_array();
    $sth->finish;

    return $gp_id;

}

sub get_protein_acc {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 =
      $self->{'db'}->prepare(q[
	SELECT xref_key 
	FROM   gene_product, dbxref
	WHERE  organism_dbxref_id = dbxref.dbxref_id
	AND    xref_keytype = 'ACC'
	AND    gene_product_id = ?
    ]);
    $sth1->bind_param( 1, $p_id );
    $sth1->execute() or die $self->{'db'}->errstr;
    my ($acc) = $sth1->fetchrow_array();
    $sth1->finish;


    return $acc;

}
# -------------

sub get_symbol {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT gene_product_id, gene_product_symbol
                               FROM gene_product
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ( $id, $symbol ) = $sth1->fetchrow_array();

    $sth1->finish;
    return $symbol;

}

sub update_symbol {
    my ( $self, $id, $new_symbol ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update gene_product 
		set gene_product_symbol = ?
		where gene_product_id = ?
	    ]);

	$sth->execute(($new_symbol, $id) ) or die $self->{'db'}->errstr;
	$sth->finish;

	_update_name_in_helper_table($self,$id);

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

# -----------

sub get_name {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT gene_product_id, gene_product_full_name
                               FROM gene_product
                               WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ( $id, $name ) = $sth1->fetchrow_array();

    $sth1->finish;
    return $name;

}

sub update_name {
    my ( $self, $id, $new_name ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update gene_product 
		set gene_product_full_name = ?
		where gene_product_id = ?
	    ]);

	$sth->execute(($new_name, $id) ) or die $self->{'db'}->errstr;
	$sth->finish;

	_update_name_in_helper_table($self,$id);

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_synonyms {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT * FROM gene_product_synonym
         WHERE gene_product_id = ? "
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();

    my @synonyms;
    while ( my $synonym_hash = $sth1->fetchrow_hashref() ) {
        push @synonyms, $synonym_hash;
    }

    $sth1->finish;
    return \@synonyms;

}

sub add_synonym {

    my ( $self, $gpid, $synonym ) = @_;

    $synonym =~ s/^\s+|\s+$//g;
    $synonym =~ s/\s+/ /g;

    return unless $synonym;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {

	my $sth0 = $self->{'db'}->prepare(
	    "SELECT gene_product_synonym_id 
	    FROM gene_product_synonym 
	    WHERE gene_product_id=? 
	    and gene_product_synonym_symbol=?"
	  );

	my $sth = $self->{'db'}->prepare(
	    "INSERT INTO gene_product_synonym VALUES (?,?,?) ");

	my $sth1 = $self->{'db'}->prepare(
	    "UPDATE gene_product_synonym 
	    SET gene_product_synonym_symbol=? 
	    WHERE gene_product_synonym_id=?"
	  );



	$sth0->execute( ( $gpid, $synonym ) );
	my ($db_gp_syn_id) = $sth0->fetchrow_array;
	if ($db_gp_syn_id) {
	    $sth1->execute( ( $synonym, $db_gp_syn_id ) );
	}else {
	    $sth->execute( (
			    &_next_id($self->{'db'},
				      'gene_product_synonym',
				      'gene_product_synonym_id'
				      ),
			    $gpid, 
			    $synonym )
			    );

	}

	_update_name_in_helper_table($self,$gpid);

	$self->{'db'}->commit();

	$sth0->finish;
	$sth->finish;
	$sth1->finish;
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub delete_synonyms {
    my ( $self, $id, $synref ) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{
	my $sth = $self->{'db'}->prepare(
	    q[
		DELETE FROM gene_product_synonym
		WHERE  gene_product_id = ?
		AND    gene_product_synonym_symbol = ?
	    ]);
	foreach my $syn (@$synref) {
	    $sth->bind_param( 1, $id );
	    $sth->bind_param( 2, $syn );
	    $sth->execute;

	}

	_update_name_in_helper_table($self, $id);

	$self->{'db'}->commit();
	$sth->finish;
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);


}

sub get_ecs {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        q[
	    SELECT objectxref_id as gene_product_ec_id,
		   xref_key as ec
	    FROM   objectxref OX, dbxref DX 
	    WHERE  OX.dbxref_id = DX.dbxref_id
            AND    OX.table_name = ?
	    AND    OX.row_id = ?
	    AND    DX.xref_dbname = ?
	    AND    DX.xref_keytype = ?
	]);
    
    my $table_name = 'gramene.protein';
    my $xref_dbname = 'ENZYME';
    my $xref_keytype = 'EC';
    $sth1->execute(($table_name, $p_id, $xref_dbname, $xref_keytype));

    my @ecs;
    while ( my $ec_hash = $sth1->fetchrow_hashref() ) {
        push @ecs, $ec_hash;
    }

    $sth1->finish;
    return \@ecs;

}

sub add_ec {

    my ( $self, $gpid, $ec ) = @_;

    $ec =~ s/^\s+|\s+$//g;
    $ec =~ s/\s+/ /g;

    return unless $ec;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $xref_dbname = 'ENZYME';
	my $xref_keytype = 'EC';
	my $table_name = 'gramene.protein';

	my ($dbxref_id) = _get_dbxref_id($self,$xref_dbname, $ec, $xref_keytype);

	my $objxref_id = 0;

	if($dbxref_id){

	    ($objxref_id) = $self->{'db'}->selectrow_array(
		q[
		    SELECT objectxref_id FROM objectxref
		    WHERE  table_name = ?
		    AND    row_id = ?
		    AND    dbxref_id = ?
		],{},($table_name,$gpid,$dbxref_id));

	}else{
	    $dbxref_id = _add_dbxref($self, $xref_dbname, $ec, $xref_keytype);
	}

	unless($objxref_id){

	    ($objxref_id) = $self->{'db'}->selectrow_array(
		q[
		    SELECT max(objectxref_id) FROM objectxref
		]) || 0;

	    $objxref_id++;
	

	    my $sth_objxref = $self->{'db'}->prepare(
		q[
		    INSERT INTO objectxref
		    (objectxref_id, table_name, row_id, dbxref_id)
		    VALUES (?,?,?,?)
		]);

	    $sth_objxref->execute(($objxref_id,$table_name,$gpid,$dbxref_id));

	    $sth_objxref->finish;
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub delete_ec {
    my ($self, $gpid, $ec) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{
	my $xref_dbname = 'ENZYME';
	my $xref_keytype = 'EC';
	my $table_name = 'gramene.protein';

	my ($dbxref_id) = _get_dbxref_id($self,$xref_dbname, $ec, $xref_keytype);

	if($dbxref_id){
	    $self->{'db'}->do(
		q[
		    DELETE FROM  objectxref
		    WHERE  table_name = ?
		    AND    row_id = ?
		    AND    dbxref_id = ?
		],{},($table_name,$gpid,$dbxref_id));
	    }

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}

sub delete_ecs {
    my ( $self, $ecref ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{
	foreach my $ecid (@$ecref) {
	    $self->{'db'}->do(
		q[
		    DELETE FROM objectxref
		    WHERE objectxref_id = ?
		],{},($ecid) );
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_gene_names {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT * FROM gene_product_gene_name
         WHERE gene_product_id = ?
        ");
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();

    my @gene_names;
    while ( my $gene_name_hash = $sth1->fetchrow_hashref() ) {
        push @gene_names, $gene_name_hash;
    }

    $sth1->finish;
    return \@gene_names;

}

sub add_gene_name {

    my ( $self, $gpid, $gene_name ) = @_;

    $gene_name =~ s/^\s+|\s+$//g;
    $gene_name =~ s/\s+/ /g;

    return unless $gene_name;


    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {

	my $sth0 = $self->{'db'}->prepare(
	    "SELECT * 
	    FROM gene_product_gene_name 
	    WHERE gene_product_id=? 
	    and gene_name=?"
	  );

	my $sth = $self->{'db'}->prepare(
	    "INSERT INTO gene_product_gene_name VALUES (?,?,?) ");

	my $sth1 = $self->{'db'}->prepare(
	    "UPDATE gene_product_gene_name 
	    SET gene_name=? 
	    WHERE gene_product_id=?"
	  );



	$sth0->execute( ( $gpid, $gene_name ) );
	my ($db_gp_id, $db_synoym) = $sth0->fetchrow_array;
	if ($db_gp_id && $db_synoym) {
	    $sth1->execute( ( $gene_name, $gpid ) );
	}else {
	    $sth->execute(
		    (&_next_id (
				$self->{'db'},
				'gene_product_gene_name',
				'gene_product_gene_name_id'
				),
		    $gpid,
		    $gene_name )
		    );

	}

	_update_gene_name_in_helper_table($self, $gpid);

	$self->{'db'}->commit();

	$sth0->finish;
	$sth->finish;
	$sth1->finish;
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}

sub delete_gene_names {
    my ( $self, $gpid, $synref ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{
	my $sth = $self->{'db'}->prepare(
	    q[
		DELETE FROM gene_product_gene_name
		WHERE  gene_product_id = ?
		AND    gene_name = ?
	    ]);
	foreach my $syn (@$synref) {
	    $sth->bind_param( 1, $gpid );
	    $sth->bind_param( 2, $syn );
	    $sth->execute;

	}

	_update_gene_name_in_helper_table($self, $gpid);

	$self->{'db'}->commit();
	$sth->finish;
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_organism {
    my ($self, $gp_id) = @_;
    my $organism = $self->{'db'}->selectrow_hashref("
	SELECT species_id, C.cultivar_id as cultivar_id
	FROM   gene_product_to_cultivar GC, cultivar C
	WHERE  GC.cultivar_id = C.cultivar_id
	AND    gene_product_id = ?
    ", {}, ($gp_id));

    return $organism;
}

sub update_organism {
    my ($self, $gp_id, $species_id, $cul_id) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{
	unless($cul_id){ # no cultivar for the species, set the cultivar as unkown
	    my $cul_name = 'Unknown';
	    my ($db_cul_id) = $self->{'db'}->selectrow_array("
		SELECT cultivar_id FROM cultivar
		WHERE  species_id = ? AND cultivar_name = ? 
	    ", {}, ($species_id, $cul_name));

	    if($db_cul_id){
		$cul_id = $db_cul_id;
	    }else{
		($cul_id ) = $self->{'db'}->selectrow_array("SELECT MAX(cultivar_id) FROM cultivar");
		$cul_id++;
		my $sth = $self->{'db'}->prepare("INSERT INTO cultivar VALUES(?,?,?)");
		$sth->execute( ( $cul_id, $species_id, $cul_name ) );
		$sth->finish;
	    }
	}

	my ($db_gp_cul_id) = $self->{'db'}->selectrow_array("
	    SELECT cultivar_id FROM gene_product_to_cultivar
	    WHERE  gene_product_id = ?", {}, ($gp_id));

	    
	if($db_gp_cul_id){
	    $self->{'db'}->do("
		UPDATE gene_product_to_cultivar
		SET    cultivar_id = ?  WHERE  gene_product_id = ?
	    ", {}, ($cul_id, $gp_id));
	}else{
	    my $gp_cul_id = _next_id($self->{'db'},'gene_product_to_cultivar','gene_product_to_cultivar_id');
	    my $sth2 = $self->{'db'}->prepare("INSERT INTO gene_product_to_cultivar VALUES(?,?,?)");
	    $sth2->execute(($gp_cul_id,$gp_id, $cul_id));
	    $sth2->finish;
	}

        _update_organism_in_helper_table($self,$gp_id);

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}
    
sub get_associations {
    my $self = $_[0];
    my $gp_id = $_[1];
    my @associations;

    my $sth = $self->{'db'}->prepare(
        " SELECT * from association 
	  WHERE gene_product_id = ? 
	  AND   term_type <> 'Cereal Plant Growth Stage'
	  order by term_type
	"
    );

    my $sth1 = $self->{'db'}->prepare(
        "SELECT evidence_id,
		association_id,
		evidence_code,
		xref_key,
		xref_dbname
	FROM evidence, dbxref
	WHERE evidence.dbxref_id = dbxref.dbxref_id
	AND   association_id = ?
	order by xref_dbname,xref_key,evidence_code
      ");

    my $sth2 = $self->{'db'}->prepare(q[
	SELECT evidence_dbxref_id,xref_dbname,xref_key 
	FROM   evidence_dbxref EDX, dbxref DX
	WHERE  EDX.dbxref_id = DX.dbxref_id
	AND    evidence_id = ?
    ]);
	
    $sth->bind_param( 1, $gp_id );
    $sth->execute;
    my $i = 0;
    while ( my $assoc_ref = $sth->fetchrow_hashref ) {
	my $term_acc = $assoc_ref->{'term_accession'};
	my $term_name = _get_term_name($self, $term_acc);
	$term_name ||= $term_acc;    # for term not in ontology DB yet
	$assoc_ref->{'term_name'} = $term_name ;

        $sth1->bind_param( 1, $assoc_ref->{'association_id'} );
        $sth1->execute();
        my @evns;
        while ( my $evn_ref = $sth1->fetchrow_hashref ) {
	    my $evn_id = $evn_ref->{'evidence_id'};
	    $sth2->execute(($evn_id));
	    my @evn_dbxrefs;
	    while(my $evn_dbxref = $sth2->fetchrow_hashref){
		push @evn_dbxrefs, $evn_dbxref;
	    }
	    $evn_ref->{'evidence_dbxref'}=\@evn_dbxrefs;
            push @evns, $evn_ref;
        }
        $assoc_ref->{'evidences'} = \@evns;
        push @associations, $assoc_ref;
    }

    $sth->finish;
    $sth1->finish;
    $sth2->finish;
    return \@associations;

}


sub delete_associations {

    my ( $self, $association_id ) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{

	my $evn_ids = $self->{'db'}->selectall_arrayref(q[
	    SELECT evidence_id FROM evidence
	    WHERE  association_id = ?
	], {},($association_id));

	my $sth2 = $self->{'db'}->prepare(
	    "DELETE FROM evidence  
	     WHERE evidence_id =?"
	);	


	my $sth3 = $self->{'db'}->prepare(
	    "DELETE FROM evidence_dbxref  
	     WHERE evidence_id =?"
	);	

       foreach my $evn_idRef (@$evn_ids){
	   my $evn_id = $evn_idRef->[0];
	   $sth2->execute(($evn_id));
	   $sth3->execute(($evn_id));
	}


	my $sth1 = $self->{'db'}->prepare(
	    "DELETE FROM association 
	     WHERE association_id =?"
	);
	$sth1->execute(($association_id));


	$sth1->finish;
	$sth2->finish;
	$sth3->finish;
	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub delete_evidence {

    my ( $self, $evn_id ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $sth_dbx = $self->{'db'}->prepare(
	    "DELETE FROM evidence_dbxref
	     WHERE  evidence_id = ?"
	);
	$sth_dbx->execute(($evn_id));


	my $sth = $self->{'db'}->prepare(
	    "DELETE FROM evidence
	     WHERE evidence_id =? "
	);

	$sth->execute(($evn_id));

	$sth_dbx->finish;
	$sth->finish;

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}


sub update_evidence_code {

    my ( $self, $evn_id,$evn_code ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $sth = $self->{'db'}->prepare(
	    "UPDATE evidence
	     SET evidence_code = ?
	     WHERE evidence_id =? "
	);

	$sth->execute(($evn_code,$evn_id));

	$sth->finish;

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub delete_evidence_dbxref {

    my ( $self, $evn_dbx_ids,$evn_id ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $sth_dbx = $self->{'db'}->prepare(
	    "DELETE FROM evidence_dbxref
	     WHERE  evidence_dbxref_id = ?"
	);
	foreach my $evn_dbx_id (@$evn_dbx_ids){
	    $sth_dbx->execute(($evn_dbx_id));
	}

	&_update_evidence_seq_acc($self,$evn_id);

	$sth_dbx->finish;

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_association_id {
    my ( $self, $gpid, $term_acc ) = @_;
    my ($assoc_id) = $self->{'db'}->selectrow_array(
	    q[
		SELECT association_id
		FROM   association
		WHERE  term_accession = ?
		AND    gene_product_id = ?
	    ],{}, ($term_acc, $gpid));

    return $assoc_id;
}

sub add_association {

    my ( $self, $gpid, $term_acc, $term_type ) = @_;

    my $date=strftime "%Y%m%d", localtime;

    $term_acc =~ s/^\s+|\s+$//g;

    return unless $term_acc;

    my $assoc_id;
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{
	($assoc_id) = get_association_id($self, $gpid, $term_acc );

	if($assoc_id){
	    my $sth = $self->{'db'}->do(
		q[
		    UPDATE association
		    SET term_accession = ?,
			gene_product_id = ?,
			term_type = ?,
			assocdate = ?
		    WHERE association_id = ?
		], {},($term_acc, $gpid, $term_type,$date, $assoc_id) );

	}else{
	    ($assoc_id) = $self->{'db'}->selectrow_array(
		q[
		    SELECT MAX(association_id) FROM association
		]) || 0;
	    $assoc_id ++;

	    my $sth = $self->{'db'}->prepare(
		q[
		    INSERT INTO association
		    (association_id,term_accession,gene_product_id,term_type,assocdate)
		    VALUES(?,?,?,?,?)
		]);

	    $sth->execute(($assoc_id,$term_acc,$gpid, $term_type,$date));

	}

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

    return $assoc_id;
}

sub get_evidence_id{
    my ($self, $evn_code, $assoc_id, $dbxref_id) = @_;
    my ($evn_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT evidence_id FROM evidence
	    WHERE evidence_code = ?
	    AND   association_id = ?
	    AND   dbxref_id = ?
	], {}, ($evn_code, $assoc_id, $dbxref_id));

    return $evn_id;

}

sub update_evidence{
    my ($self, $evn_code, $assoc_id, $dbxref_id,$evn_id ) = @_;

    my $sth = $self->{'db'}->prepare(
	q[
	    UPDATE evidence
	    SET evidence_code = ? ,
		association_id = ?,
		dbxref_id = ?
	    WHERE evidence_id= ?
	]);
    $sth->execute(($evn_code, $assoc_id, $dbxref_id,$evn_id));
    $sth->finish;
}

sub add_evidence {

    my ( $self, $gp_id, $assoc_id, $db_name, $xref_key, $evn_code ) = @_;

    return unless ( $db_name && $xref_key && $evn_code );
    
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	$xref_key =~ s/^\s+|\s+$//g;
	$xref_key =~ s/\s+/ /g;
	my $dbxref_id = _get_dbxref_id($self, $db_name, $xref_key);

	my $evn_id;
	if($dbxref_id) {
	    ($evn_id) = &get_evidence_id($self, $evn_code, $assoc_id, $dbxref_id);
	}else{
	    $dbxref_id = _add_dbxref($self, $db_name, $xref_key);
	}

	unless($evn_id){
	    $evn_id = $self->{'db'}->selectrow_array(
		q[
		    SELECT max(evidence_id) FROM evidence
		]) || 0;

	    $evn_id ++;

	    my $sth = $self->{'db'}->prepare(
		q[
		    INSERT INTO evidence
		    (evidence_id, evidence_code, association_id, dbxref_id)
		    VALUES (?, ?, ?, ?)
		]);

	    $sth->execute(($evn_id,$evn_code, $assoc_id, $dbxref_id));
	    $sth->finish;
	}

	# update protein references
	if($db_name eq 'gramene.literature'){
	    my %db_refs = _get_objectxref_literature_refs($self,$gp_id);
	    unless($db_refs{$dbxref_id}){
		_add_objectxref($self, $gp_id,$dbxref_id);
	    }
	}


	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}




sub add_evidence_dbxref {

    my ( $self, $evn_id, $db_name, $xref_key ) = @_;

    return unless ( $db_name && $xref_key  );
    
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	$xref_key =~ s/^\s+|\s+$//g;
	$xref_key =~ s/\s+/ /g;
	my $dbxref_id = _get_dbxref_id($self, $db_name, $xref_key);

	my $evn_dbx_id;

	if($dbxref_id) {

	    ($evn_dbx_id ) = $self->{'db'}->selectrow_array(q[
		SELECT evidence_dbxref_id 
		FROM   evidence_dbxref
		WHERE  evidence_id = ?
		AND    dbxref_id = ?
	    ],{},($evn_id,$dbxref_id ));

	}else {
	    $dbxref_id = _add_dbxref($self, $db_name, $xref_key);
	}

	unless($evn_dbx_id){
	    $evn_dbx_id = $self->{'db'}->selectrow_array(
		q[
		    SELECT max(evidence_dbxref_id) FROM evidence_dbxref
		]) || 0;

	    $evn_dbx_id ++;
	}

	my $sth = $self->{'db'}->prepare(
	    q[
		INSERT INTO evidence_dbxref
		(evidence_dbxref_id, evidence_id, dbxref_id)
		VALUES (?, ?, ?)
	]);

	$sth->execute(($evn_dbx_id,$evn_id, $dbxref_id));
	$sth->finish;

	&_update_evidence_seq_acc($self,$evn_id);


	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_gene_associations {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        q[
	    SELECT objectxref_id as gene_product_to_gene_id,
		   xref_key as gene_accession 
	    FROM   objectxref OX, dbxref DX 
	    WHERE  OX.dbxref_id = DX.dbxref_id
            AND    OX.table_name = ?
	    AND    OX.row_id = ?
	    AND    DX.xref_dbname = ?
	    AND    DX.xref_keytype = ?
	]);
    
    my $table_name = 'gramene.protein';
    my $xref_dbname = 'gramene.gene';
    my $xref_keytype = 'acc';
    $sth1->execute(($table_name, $p_id, $xref_dbname, $xref_keytype));

    my @associations;
    while ( my $assoc_hash = $sth1->fetchrow_hashref() ) {
        push @associations, $assoc_hash;
    }

    $sth1->finish;
    return \@associations;

}

sub delete_gene_association_by_acc{
    my ($self, $gp_id, $gene_acc) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{

	my $xref_dbname = 'gramene.gene';
	my $xref_keytype = 'acc';
	my $table_name = 'gramene.protein';

	my $dbxref_id = _get_dbxref_id($self, $xref_dbname, $gene_acc, $xref_keytype);

	if($dbxref_id){
	    &_delete_objectxref($self,$gp_id,$dbxref_id);
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}
	    




sub delete_gene_association {

    my ( $self, $gene_product_to_gene_idRefs ) = @_;


    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{
	foreach my $mu_id (@$gene_product_to_gene_idRefs) {
	    $self->{'db'}->do(
		q[
		    DELETE FROM objectxref
		    WHERE objectxref_id = ?
		],{},($mu_id) );
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub add_gene_association {

    my ( $self, $gpid, $gene_acc ) = @_;

    $gene_acc =~ s/^\s+|\s+$//g;

    return unless ( $gene_acc && $gpid );


    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $xref_dbname = 'gramene.gene';
	my $xref_keytype = 'acc';
	my $table_name = 'gramene.protein';

	my $dbxref_id = 
	    _get_dbxref_id($self, $xref_dbname, $gene_acc, $xref_keytype);

	my $objxref_id = 0;

	if($dbxref_id){

	    ($objxref_id) = $self->{'db'}->selectrow_array(
		q[
		    SELECT objectxref_id FROM objectxref
		    WHERE  table_name = ?
		    AND    row_id = ?
		    AND    dbxref_id = ?
		],{},($table_name,$gpid,$dbxref_id));

	}else{
	    $dbxref_id =
		_add_dbxref($self, $xref_dbname, $gene_acc, $xref_keytype);

	}

	unless($objxref_id){

	    ($objxref_id) = $self->{'db'}->selectrow_array(
		q[
		    SELECT max(objectxref_id) FROM objectxref
		]) || 0;

	    $objxref_id++;


	    my $sth_objxref = $self->{'db'}->prepare(
		q[
		    INSERT INTO objectxref
		    (objectxref_id, table_name, row_id, dbxref_id)
		    VALUES (?,?,?,?)
		]);

	    $sth_objxref->execute(($objxref_id,$table_name,$gpid,$dbxref_id));

	    $sth_objxref->finish;
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_features {

    my $self = $_[0];
    my $p_id = $_[1];

    my $sth1 = $self->{'db'}->prepare(
        q[
	    SELECT gene_product_feature_id,
		   gene_product_id,
		   feature_type,
		   from_position,
		   to_position,
		   xref_dbname,
		   xref_key
	    FROM gene_product_feature GF,
		 gene_product_feature_type GFT,
		 dbxref DX
	    WHERE GF.feature_type_id = GFT.feature_type_id
	    AND   GF.dbxref_id = DX.dbxref_id
	    AND   gene_product_id = ?
	]);
		                                        

    $sth1->bind_param( 1, $p_id );
    $sth1->execute();

    my @features;
    while ( my $feature_hash = $sth1->fetchrow_hashref() ) {
        push @features, $feature_hash;
    }

    $sth1->finish;
    return \@features;

}

sub delete_feature {
    my ( $self, $feature_id ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $sth = $self->{'db'}->prepare(
	    q[
		DELETE FROM gene_product_feature
		WHERE gene_product_feature_id = ?
	    ]);


	$sth->bind_param( 1, $feature_id );
	$sth->execute;
	$sth->finish;

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub update_feature {
    my ( $self, $feature_id,$gp_id, $feature_type, $from_pos, $to_pos, $xref_dbname, $xref_key ) = @_;

    return unless ( $feature_type
		    && $from_pos
		    && $to_pos
		    && $xref_dbname
		    && $xref_key
		);

    my $db_attr_ref = _transact_init( $self->{'db'} );

    my $feature_type_id = _get_feature_type_id($self, $feature_type);
    my $dbxref_id = _get_dbxref_id($self,$xref_dbname, $xref_key);

    eval{

	my $db_gene_product_feature_id;
	if($feature_type_id && $dbxref_id){
	    $db_gene_product_feature_id = _get_gene_product_feature_id
		($self,$gp_id,$feature_type_id,$from_pos,$to_pos,$dbxref_id);
	}

	unless($feature_type_id){
	    $feature_type_id = add_feature_type($self, $feature_type);
	}

	unless($dbxref_id){
	    $dbxref_id = _add_dbxref($self,$xref_dbname, $xref_key);
	}

	unless($db_gene_product_feature_id){
	    my $sth = $self->{'db'}->prepare(
		q[
		    UPDATE gene_product_feature
		    SET feature_type_id = ?,
			from_position = ?,
			to_position = ? ,
			dbxref_id = ?
		    WHERE gene_product_feature_id = ?
		]);

	    $sth->execute(($feature_type_id,$from_pos,$to_pos,$dbxref_id,$feature_id));

	    $sth->finish;

	}


	# update protein references
	if($xref_dbname eq 'gramene.literature'){
	    my %db_refs = _get_objectxref_literature_refs($self,$gp_id);
	    unless($db_refs{$dbxref_id}){
		_add_objectxref($self, $gp_id,$dbxref_id);
	    }
	}

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub add_feature {
    my ( $self, $gp_id, $feature_type, $from_pos, $to_pos, $xref_dbname,
        $xref_key ) = @_;

    return unless ( $feature_type
		    && $from_pos
		    && $to_pos
		    && $xref_dbname
		    && $xref_key );

    my $db_attr_ref = _transact_init( $self->{'db'} );

    my $feature_type_id = _get_feature_type_id($self, $feature_type);
    my $dbxref_id = _get_dbxref_id($self,$xref_dbname, $xref_key);


    eval{
	my $gene_product_feature_id;
	if($feature_type_id && $dbxref_id){
	    $gene_product_feature_id = _get_gene_product_feature_id
		($self,$gp_id,$feature_type_id,$from_pos,$to_pos,$dbxref_id);
	}

	unless($feature_type_id){
	    $feature_type_id = add_feature_type($self, $feature_type);
	}

	unless($dbxref_id){
	    $dbxref_id = _add_dbxref($self,$xref_dbname, $xref_key);
	}

	unless($gene_product_feature_id){
	    $gene_product_feature_id = $self->{'db'}->selectrow_array(
		q[
		    SELECT MAX(gene_product_feature_id)
		    FROM gene_product_feature
		]) || 0;

	    $gene_product_feature_id++;

	    my $sth = $self->{'db'}->prepare(
		q[
		    INSERT INTO gene_product_feature
		    (gene_product_feature_id,
		     gene_product_id,
		     feature_type_id,
		     from_position,
		     to_position,
		     dbxref_id
		    ) 
		    VALUES (?,?,?,?,?,?)
		]);
	    $sth->execute(($gene_product_feature_id,
		$gp_id,$feature_type_id,$from_pos,$to_pos,$dbxref_id));

	    $sth->finish;

	}



	# update protein references
	if($xref_dbname eq 'gramene.literature'){
	    my %db_refs = _get_objectxref_literature_refs($self,$gp_id);
	    unless($db_refs{$dbxref_id}){
		_add_objectxref($self, $gp_id,$dbxref_id);
	    }
	}

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}

sub delete_all_features {

    my ( $self, $gp_id ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{

	my $sth = $self->{'db'}->prepare(
	    q[
		DELETE FROM gene_product_feature
		WHERE gene_product_id = ?
	    ]);

	$sth->bind_param( 1, $gp_id );
	$sth->execute;

	$sth->finish;

	$self->{'db'}->commit();
    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_public_comment {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT public_comment
	   FROM gene_product_comment
	   WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ($comment) = $sth1->fetchrow_array();

    $sth1->finish;
    return $comment;

}

sub get_curator_comment {

    my $self = $_[0];
    my $p_id = $_[1];
    my $sth1 = $self->{'db'}->prepare(
        "SELECT curator_comment
	   FROM gene_product_comment
	   WHERE gene_product_id = ?"
    );
    $sth1->bind_param( 1, $p_id );
    $sth1->execute();
    my ($comment) = $sth1->fetchrow_array();

    $sth1->finish;
    return $comment;

}

sub update_public_comment {
    my ( $self, $id, $comment ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );
    my $time=localtime;

    eval{

	my $sth_check = $self->{'db'}->prepare(
	    "SELECT gene_product_id FROM gene_product_comment
	     WHERE gene_product_id=?"
	);
	$sth_check->execute( ($id) );
	my ($db_id) = $sth_check->fetchrow_array();
	if ($db_id) {
	    my $sth = $self->{'db'}->prepare(
		"UPDATE gene_product_comment 
		SET  public_comment = ?,
		p_comment_curation_date = ?
		WHERE gene_product_id =?"
	    );

	    $sth->bind_param( 1, $comment );
	    $sth->bind_param( 2, $time );
	    $sth->bind_param( 3, $id );
	    $sth->execute;
	    $sth->finish;
	}else {

	    my $sth = $self->{'db'}->prepare(
		"INSERT INTO gene_product_comment
		( gene_product_id,public_comment,p_comment_curation_date)
		VALUES(?,?,?)");
	    $sth->execute( ( $id, $comment, $time));
	    $sth->finish;
	}
	$sth_check->finish;
	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}

sub update_curator_comment {
    my ( $self, $id, $comment ) = @_;


    my $db_attr_ref = _transact_init( $self->{'db'} );
    my $time=localtime;

    eval{

	my $sth_check = $self->{'db'}->prepare(
	    "SELECT gene_product_id FROM gene_product_comment
	     WHERE gene_product_id=?"
	);
	$sth_check->execute( ($id) );
	my ($db_id) = $sth_check->fetchrow_array();
	if ($db_id) {
	    my $sth = $self->{'db'}->prepare(
		"UPDATE gene_product_comment 
		SET  curator_comment = ?,
		c_comment_curation_date = ?
		WHERE gene_product_id =?"
	    );

	    $sth->bind_param( 1, $comment );
	    $sth->bind_param( 2, $time );
	    $sth->bind_param( 3, $id );
	    $sth->execute;
	    $sth->finish;
	}else {

	    my $sth = $self->{'db'}->prepare(
		"INSERT INTO gene_product_comment
		( gene_product_id,curator_comment,c_comment_curation_date)
		VALUES(?,?,?)");
	    $sth->execute( ( $id, $comment, $time));
	    $sth->finish;
	}
	$sth_check->finish;
	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_all_feature_type_names {

    my $self = shift;
    my %types;

    my $sth =
      $self->{'db'}->prepare(
        "SELECT DISTINCT feature_type FROM gene_product_feature_type");
    $sth->execute;
    while ( my ($type) = $sth->fetchrow_array ) {
        $types{$type} = 1;
    }

    my @types_sorted = sort { lc($a) cmp lc($b) } keys(%types);

    $sth->finish;

    return \@types_sorted;

}

sub get_all_feature_types {
    my $self = shift;
    my $sth = $self->{'db'}->prepare("SELECT * from gene_product_feature_type order by feature_type_id");
    $sth->execute();
    my @feature_types;
    while(my $ft = $sth->fetchrow_hashref){
	push @feature_types, $ft;
    }

    $sth->finish;
    return @feature_types;
}


sub get_feature_type {
    my ($self,$type) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from gene_product_feature_type WHERE feature_type_id = ?");
    $sth->execute(($type));
    my $ft = $sth->fetchrow_hashref;

    $sth->finish;
    return $ft;
}


sub add_feature_type {

    my ( $self, $feature_type ) = @_;

    return unless $feature_type;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	my %types;
	my $sth = $self->{'db'}->prepare("SELECT feature_type_id, feature_type FROM gene_product_feature_type");
	$sth->execute;
	while ( my ($type_id,$type) = $sth->fetchrow_array ) {
	    $types{$type} = $type_id;
	}

	my $feature_type_id = $types{$feature_type};

	unless ( $feature_type_id ) {
	    my $sth1 = $self->{'db'}->prepare("SELECT MAX(feature_type_id) FROM gene_product_feature_type");
	    my $sth2 = $self->{'db'}->prepare("INSERT INTO gene_product_feature_type VALUES(?,?)");
	    $sth1->execute;
	    $feature_type_id = $sth1->fetchrow_array;
	    $feature_type_id++;
	    $sth2->execute( ( $feature_type_id, $feature_type ) );
	    $sth1->finish;
	    $sth2->finish;

	}
	$sth->finish;


	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}


sub update_feature_type {
    my ( $self, $ft_id, $ft ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update gene_product_feature_type 
		set feature_type = ?
		where feature_type_id = ?
	    ]);

	$sth->execute(($ft,$ft_id) ) or die $self->{'db'}->errstr;
	$sth->finish;

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub get_all_species{
    my $self = shift;
    my $sth = $self->{'db'}->prepare("SELECT * from species order by genus, species");
    $sth->execute();
    my @all_spes;
    while(my $spe = $sth->fetchrow_hashref){
	push @all_spes, $spe;
    }

    $sth->finish;
    return @all_spes;
}


sub get_species {
    my ($self,$spe_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from species WHERE species_id = ?");
    $sth->execute(($spe_id));
    my $spe = $sth->fetchrow_hashref;

    $sth->finish;
    return $spe;
}


sub update_species {
    my ( $self, $spe_id, $genus, $spe, $name, $taxid ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    my @spe_vals = map { defined $_ ? $_ : '' } ($genus, $spe, $name, $taxid, $spe_id);

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update species 
		set genus = ?,
		species = ?,
		common_name = ?, 
		ncbi_taxa_id = ?
		where species_id = ?
	    ]);

	$sth->execute(@spe_vals ) or die $self->{'db'}->errstr;
	$sth->finish;

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}


sub add_species {
    my ( $self, $genus, $spe, $name, $taxid ) = @_;

    my @spe_vals = map { defined $_ ? $_ : '' } ($genus, $spe, $name, $taxid);
    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval{
	
	my $sth2 = $self->{'db'}->prepare("SELECT MAX(species_id) FROM species");
	$sth2->execute;
	my ($sp_id) = $sth2->fetchrow_array();
	$sp_id++;
	@spe_vals = ($sp_id, @spe_vals);
	my $sth3 = $self->{'db'}->prepare(
	      "INSERT INTO species 
	      (species_id, genus, species, common_name, ncbi_taxa_id)
	      VALUES(?, ?, ?, ?, ?)");
	$sth3->execute( @spe_vals);

	$sth2->finish;
	$sth3->finish;

	$self->{'db'}->commit();

    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}



sub get_all_species_cultivar{
    my ($self, $species_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from cultivar where species_id= ? order by cultivar_name");
    $sth->execute(($species_id));
    my @all_culs;
    while(my $cul = $sth->fetchrow_hashref){
	push @all_culs, $cul;
    }

    $sth->finish;
    return @all_culs;
}

sub get_cultivar {
    my ($self,$spe_id, $cul_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from cultivar WHERE species_id = ? and cultivar_id = ?");
    $sth->execute(($spe_id, $cul_id));
    my $cul = $sth->fetchrow_hashref;

    $sth->finish;
    return $cul;
}


sub update_cultivar {
    my ( $self, $spe_id, $cul_id, $cul_name ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );


    eval {
	my $gp_idsRef = $self->{'db'}->selectall_arrayref(
	    qq[
		SELECT gene_product_id FROM gene_product_to_cultivar
		WHERE  cultivar_id = ?
	    ], {}, ($cul_id) );

	my ($new_cul_id) = $self->{'db'}->selectrow_array(
	    qq[
		SELECT cultivar_id
		FROM   cultivar
		WHERE  UPPER(cultivar_name) = ?
		AND    species_id = ?
	    ], {}, (uc($cul_name),$spe_id));

	if($new_cul_id && $new_cul_id != $cul_id){ 
	    # different cul under different species
	    _merge_cutlivars($self, $cul_id, $new_cul_id);
	    $cul_id = $new_cul_id;
	}else{
	    my $sth = $self->{'db'}->prepare(
		qq[
		    Update cultivar 
		    set cultivar_name = ? ,
		    species_id = ?
		    where  cultivar_id = ?
		]);

	    $sth->execute(($cul_name, $spe_id, $cul_id) ) or die $self->{'db'}->errstr;
	    $sth->finish;
        }
	

	foreach my $gp_idRef (@$gp_idsRef){
	    my $gp_id = $gp_idRef->[0];
	    _update_organism_in_helper_table($self,$gp_id);
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

    return ($spe_id, $cul_id);

}



sub merge_cultivars {
    my ( $self, $cul_id, $new_cul_id ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    my ($db_new_cul_id, $species_id);


    eval {
	my $gp_idsRef = $self->{'db'}->selectall_arrayref(
	    qq[
		SELECT gene_product_id FROM gene_product_to_cultivar
		WHERE  cultivar_id = ?
	    ], {}, ($cul_id) );

	($db_new_cul_id, $species_id) = $self->{'db'}->selectrow_array(
	    qq[
		SELECT cultivar_id,species_id
		FROM   cultivar
		WHERE  cultivar_id = ?
	    ], {}, ($new_cul_id));

	if($db_new_cul_id && $db_new_cul_id != $cul_id){ 
	    _merge_cutlivars($self, $cul_id, $db_new_cul_id);
	}

	foreach my $gp_idRef (@$gp_idsRef){
	    my $gp_id = $gp_idRef->[0];
	    _update_organism_in_helper_table($self,$gp_id);
	}

	$self->{'db'}->commit();

    };

    _transact_finish ($self->{'db'}, $db_attr_ref,$@);


    return ($db_new_cul_id, $species_id);

}

sub delete_cultivar {

    my ( $self, $cul_id) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{
	my $gp_idsRef = $self->{'db'}->selectall_arrayref(
	    qq[
		SELECT gene_product_id FROM gene_product_to_cultivar
		WHERE  cultivar_id = ?
	    ], {}, ($cul_id) );

	unless(scalar(@$gp_idsRef) >0 ){
	    $self->{'db'}->do(
		q[
		    DELETE FROM cultivar_synonym
		    WHERE  cultivar_id = ?
		], {}, ($cul_id));

	    $self->{'db'}->do(
		q[
		    DELETE FROM cultivar
		    WHERE  cultivar_id = ?
		], {}, ($cul_id));

	    $cul_id = '';
	}


	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
    return $cul_id;
}


sub add_cultivar {

    my ( $self, $spe_id, $cul_name ) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    my $cul_id;
    eval{
	my ($species_id) = $self->{'db'}->selectrow_array("
	    SELECT species_id FROM species where species_id = ?
	", {}, $spe_id);


	if( $species_id ) {
	    my $sth1 = $self->{'db'}->prepare("SELECT MAX(cultivar_id) FROM cultivar");
	    my $sth2 = $self->{'db'}->prepare("INSERT INTO cultivar VALUES(?,?,?)");
	    $sth1->execute;
	    ($cul_id ) = $sth1->fetchrow_array;
	    $cul_id++;
	    $sth2->execute( ( $cul_id, $species_id, $cul_name ) );
	    $sth1->finish;
	    $sth2->finish;

	}

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
    return $cul_id;
}





sub get_all_xref_dbnames {

    my $self = shift;
    my $sth  = $self->{'db'}->prepare("SELECT DISTINCT name FROM data_base");
    $sth->execute;
    my %dbnames;
    while ( my ($dbname) = $sth->fetchrow_array ) {
        $dbnames{$dbname} = 1;
    }

    my @dbnames_sorted = sort { lc($a) cmp lc($b) } keys(%dbnames);

    $sth->finish;
    return \@dbnames_sorted;
}


sub get_all_xref_dbs{
    my $self = shift;
    my $sth = $self->{'db'}->prepare("SELECT * from data_base");
    $sth->execute();
    my @xref_dbs;
    while(my $xref_db = $sth->fetchrow_hashref){
	push @xref_dbs, $xref_db;
    }

    $sth->finish;
    return @xref_dbs;
}


sub get_xref_db{
    my ($self, $xdb_id)  = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from data_base WHERE data_base_id = ?");
    $sth->execute(($xdb_id));
    my ($xref_db) = $sth->fetchrow_hashref();

    $sth->finish;
    return $xref_db;
}
sub update_dbxref_source {
    my ( $self, $dbx_id, $dbx_name, $dbx_url ) = @_;

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update data_base 
		set name = ?,
		url_syntax = ?
		where data_base_id = ?
	    ]);

	$sth->execute(($dbx_name, $dbx_url, $dbx_id) ) or die $self->{'db'}->errstr;
	$sth->finish;

	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);

}

sub add_dbxref_source {
    my ( $self, $xref_dbname, $url ) = @_;

    return unless ( $xref_dbname );

    my $db_attr_ref = _transact_init( $self->{'db'} );

    eval {
	
	my $sth0 = $self->{'db'}->prepare(
	    "SELECT data_base_id FROM data_base 
	    WHERE name=? and url_syntax = ?"
	  );
	$sth0->execute( ($xref_dbname,$url ) );
	my ($db_id) = $sth0->fetchrow_array;

	unless ($db_id) {
	    my $sth2 = $self->{'db'}->prepare("SELECT MAX(data_base_id) FROM data_base");
	    $sth2->execute;
	    ($db_id) = $sth2->fetchrow_array();
	    $db_id++;
	    my $sth3 = $self->{'db'}->prepare(
	      "INSERT INTO data_base (data_base_id, name, url_syntax) VALUES(?, ?, ?)");
	    $sth3->execute( ( $db_id, $xref_dbname, $url));
	    $sth2->finish;
	    $sth3->finish;
	}

	$sth0->finish;


	$self->{'db'}->commit();
    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}



sub get_dbxref_by_objectxref_id {

    my ($self, $objx_id) = @_;


    my $dbx = $self->{'db'}->selectrow_hashref(
	q[
	    SELECT DX.dbxref_id,xref_dbname,xref_keytype,xref_key
	    FROM dbxref DX, objectxref OX
	    WHERE DX.dbxref_id = OX.dbxref_id 
	    AND   objectxref_id = ?
	],{},($objx_id));

    return $dbx;
}

sub get_dbxref_id {
    my ($self, $xref_dbname, $xref_key, $xref_keytype) = @_;
    return &_get_dbxref_id($self, $xref_dbname, $xref_key, $xref_keytype);
}

sub _get_dbxref_id {

    my ($self, $xref_dbname, $xref_key, $xref_keytype) = @_;

    unless($xref_keytype){
	if($xref_key =~/\D+/){
	    $xref_keytype = 'acc';
	}else{
	    $xref_keytype = 'id';
	}

    }

    my ($dbxref_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT dbxref_id FROM dbxref
	    WHERE  xref_dbname = ?
	    AND   xref_keytype = ?
	    AND   xref_key = ?
	],{},($xref_dbname,$xref_keytype,$xref_key));

    return $dbxref_id;
}

sub _add_dbxref{
    my ($self, $xref_dbname, $xref_key, $xref_keytype, $xref_desc) = @_;


    unless($xref_keytype){
	if($xref_key =~/\D+/){
	    $xref_keytype = 'acc';
	}else{
	    $xref_keytype = 'id';
	}

    }

    $xref_desc = '' unless defined $xref_desc;

    my ($dbxref_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT max(dbxref_id) FROM dbxref
	]) || 0;

    $dbxref_id ++;
    my $sth_dbxref = $self->{'db'}->prepare(
	q[
	    INSERT INTO dbxref
	    (dbxref_id,xref_key,xref_keytype,xref_dbname,xref_desc)
	    VALUES (?,?,?,?,?)
	]);


    $sth_dbxref->execute(($dbxref_id,$xref_key,$xref_keytype,$xref_dbname,$xref_desc));

    $sth_dbxref->finish;

    return $dbxref_id;


}

sub _get_objectxref_literature_refs{
    my ($self, $gp_id) = @_;


    my $table_name = 'gramene.protein';
    my $xref_dbname = 'gramene.literature';
    my $xref_keytype = 'id';
    my %db_refs;
    my $sth = $self->{'db'}->prepare(
	q[
	    SELECT DX.dbxref_id
	    FROM objectxref OX, dbxref DX
	    WHERE OX.dbxref_id = DX.dbxref_id
	    AND   row_id = ?
	    AND   table_name = ?
	    AND   xref_dbname = ?
	    AND   xref_keytype = ?
	]);

    $sth->execute(($gp_id,$table_name,$xref_dbname,$xref_keytype));
    while(my ($dbx_id) = $sth->fetchrow_array){
	$db_refs{$dbx_id} = 1;
    }
    
    return %db_refs;
}

sub _get_gene_product_all_literatures{
    my ($self, $gp_id) = @_;

    my $table_name = 'gramene.protein';
    my $xref_dbname = 'gramene.literature';
    my $xref_keytype = 'id';
    
    my $db_assoc_refs = $self->{'db'}->selectall_hashref(
	q[
	    SELECT DX.dbxref_id
	    FROM association A, evidence E, dbxref DX
	    WHERE A.association_id = E.association_id
	    AND E.dbxref_id = DX.dbxref_id
	    AND gene_product_id = ?
	    AND xref_dbname = ?
	    AND   xref_keytype = ?
	], 'dbxref_id', ($gp_id, $xref_dbname,$xref_keytype));

    my $db_feature_refs = $self->{'db'}->selectall_hashref(
	q[
	    SELECT DX.dbxref_id
	    FROM gene_product_feature GF, dbxref DX
	    WHERE GF.dbxref_id = DX.dbxref_id
	    AND gene_product_id = ?
	    AND xref_dbname = ?
	    AND   xref_keytype = ?
	], 'dbxref_id', ($gp_id, $xref_dbname,$xref_keytype));

    return (keys(%$db_assoc_refs),keys(%$db_feature_refs));

}

sub _merge_cutlivars{
    my ($self, $old_cul_id, $new_cul_id) = @_;
    my $dbh = $self->{'db'};
    $dbh->do(
	q[
	    UPDATE gene_product_to_cultivar
	    SET    cultivar_id = ?
	    WHERE  cultivar_id = ?
	], {}, ($new_cul_id, $old_cul_id));

    my $new_synonyms = $dbh->selectall_hashref(
	q[
	    SELECT cultivar_synonym
	    FROM   cultivar_synonym
	    WHERE  cultivar_id = ?
	], 'cultivar_synonym',{}, ($new_cul_id));

    my $sth_syn = $dbh->prepare(
	q[
	    UPDATE cultivar_synonym
	    SET    cultivar_id = ?
	    WHERE  cultivar_synonym_id = ?
	]);
    
    
    my ($max_cul_syn_id ) = $dbh->selectrow_array(
	q[
	    SELECT MAX(cultivar_synonym_id) FROM cultivar_synonym
	]);

    my $sth_syn_add = $dbh->prepare(
	q[
	    INSERT INTO cultivar_synonym
	    (cultivar_synonym_id, cultivar_id, cultivar_synonym)
	    VALUES (?, ?, ?)
	]);

    my $sth_old_syn = $dbh->prepare(
	q[
	    SELECT cultivar_synonym_id, cultivar_synonym
	    FROM  cultivar_synonym
	    WHERE  cultivar_id = ?
	]);
    
    $sth_old_syn->execute(($old_cul_id));


    my ($old_cul_name) = $dbh->selectrow_array(
	q[
	    SELECT cultivar_name FROM cultivar
	    WHERE  cultivar_id = ?
	], {}, ($old_cul_id) );

    unless(exists $new_synonyms->{$old_cul_name}){
	$max_cul_syn_id ++ ;
	$sth_syn_add->execute(($max_cul_syn_id,$new_cul_id,$old_cul_name));
    }

    while(my ($syn, $id) = $sth_old_syn->fetchrow_array){
	if(exists $new_synonyms->{$syn}){
	    $sth_syn->execute(($new_cul_id,$id));
	}else{
	    $max_cul_syn_id ++ ;
	    $sth_syn_add->execute(($max_cul_syn_id,$new_cul_id,$syn));
	}
    }



    $sth_syn->finish;
    $sth_syn_add->finish;
    $sth_old_syn->finish;

    $dbh->do(
	q[
	    DELETE FROM cultivar
	    WHERE  cultivar_id = ?
	], {}, ($old_cul_id));


    $dbh->do(
	q[
	    DELETE FROM cultivar_synonym
	    WHERE  cultivar_id = ?
	], {}, ($old_cul_id));
}
    


sub _add_objectxref{
    my ($self,$row_id,$dbxref_id) = @_;
    my ($objxref_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT objectxref_id
	    FROM   objectxref
	    WHERE  table_name = 'gramene.protein'
	    AND    row_id = ?
	    AND    dbxref_id = ?
	],{},($row_id,$dbxref_id)
    );

    return if $objxref_id;
    ($objxref_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT max(objectxref_id) FROM objectxref
	]);
    $objxref_id ++;
    $self->{'db'}->do(
	q[
	    INSERT INTO objectxref
	    (objectxref_id,table_name,row_id,dbxref_id)
	    VALUES(?,?,?,?)
	],{}, ($objxref_id, 'gramene.protein', $row_id,$dbxref_id));

}

sub _delete_objectxref{
    my ($self,$row_id,$dbxref_id) = @_;
    $self->{'db'}->do(
	q[
	    DELETE FROM objectxref
	    WHERE  table_name = ?
	    AND    row_id = ?
	    AND    dbxref_id = ?
	], {}, ('gramene.protein', $row_id,$dbxref_id));
}


sub _get_feature_type_id {

    my ( $self, $feature_type ) = @_;
    return unless $feature_type;

    $feature_type =~s/^\s+|\s+$//g;

    my ($feature_type_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT feature_type_id
	    FROM  gene_product_feature_type
	    WHERE  feature_type = ?
	], {}, ($feature_type) );

    return $feature_type_id;

}


sub _get_gene_product_feature_id {
    my ($self, $gpid, $feature_type_id,$f_pos, $t_pos,$dbxref_id) = @_;

    my ($gene_product_feature_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT gene_product_feature_id
	    FROM gene_product_feature
	    WHERE gene_product_id = ?
	    AND feature_type_id = ?
	    AND from_position = ?
	    AND to_position = ?
	    AND dbxref_id = ?
	],{},($gpid, $feature_type_id,$f_pos, $t_pos,$dbxref_id));

    return $gene_product_feature_id;

}

sub _update_evidence_seq_acc{
    my ($self, $evn_id ) = @_;
    my $sth = $self->{'db'}->prepare(q[
	SELECT xref_dbname, xref_key
	FROM   evidence_dbxref EX, dbxref DX
	WHERE  EX.dbxref_id = DX.dbxref_id
	AND    evidence_id = ?
    ]);
    my @evn_dbxs;
    $sth->execute(($evn_id));
    while(my ($xdb,$xkey) = $sth->fetchrow_array){
	push @evn_dbxs, ("$xdb:$xkey");
    }

    $sth->finish;
    my $evn_dbx =  '';
    if(@evn_dbxs){
	$evn_dbx = join("|",@evn_dbxs);
    }

    my $sth2 = $self->{'db'}->prepare(q[
	UPDATE evidence
	SET    seq_acc = ?
	WHERE  evidence_id = ?
    ]);
    $sth2->execute(($evn_dbx,$evn_id));
    $sth2->finish;

}

sub _update_gene_name_in_helper_table{
    my ($self, $gp_id) = @_;
    my $sth = $self->{'db'}->prepare(
	q[
	    SELECT  gene_name
	    FROM    gene_product_gene_name
	    WHERE   gene_product_id = ?
	]);

    $sth->execute(($gp_id));
    my $gene_name = '';
    while ( my ($gn) = $sth->fetchrow_array ) {
	$gene_name = $gene_name."   ".$gn;
    }

    unless($gene_name){
	$gene_name = 'Not available';
    }

    my $sth2 = $self->{'db'}->prepare(
	q[
	    UPDATE gene_product_helper
	    SET gene_product_gene_name =?
	    WHERE gene_product_id = ?
	]);
    $sth2->execute(($gene_name,$gp_id));
    $sth->finish;
    $sth2->finish;
}


sub _update_name_in_helper_table{
    my ($self, $gp_id) = @_;
    my $gp_name_symbolref = $self->{'db'}->selectall_hashref(
	q[
	    SELECT gene_product_id,
		   gene_product_full_name,
		   gene_product_symbol
	    FROM   gene_product
	    WHERE  gene_product_id = ?
	],('gene_product_id'),{},($gp_id));

    my $name = $gp_name_symbolref->{$gp_id}->{'gene_product_full_name'};
    my $symbol = $gp_name_symbolref->{$gp_id}->{'gene_product_symbol'};
    unless( ( $symbol eq "Not available" ) || ( $symbol eq "FRAGMENT" ) ) {
	$name = $name."   ".$symbol;
    }

    my $gp_synsRef = $self->{'db'}->selectall_arrayref(
	q[
	    SELECT gene_product_synonym_symbol
	    FROM gene_product_synonym
	    WHERE  gene_product_id = ?
	],{},($gp_id));

    foreach my $ref (@$gp_synsRef){
	my ($syn) = @$ref;
	$name = $name."   ".$syn;
    }


    my $sth = $self->{'db'}->prepare(
	q[
	    UPDATE gene_product_helper 
	    SET    gene_product_name =?
	    WHERE gene_product_id = ?
	]);

    $sth->execute(($name,$gp_id));

    $sth->finish;

}



sub _update_organism_in_helper_table{
    my ($self, $gp_id) = @_;
    my $gp_org_ref = $self->{'db'}->selectall_hashref(
	q[
	    SELECT gene_product_id, genus, species, cultivar_name
	    FROM   gene_product_to_cultivar GC, species, cultivar
	    WHERE  GC.cultivar_id = cultivar.cultivar_id
	    AND    species.species_id = cultivar.species_id
	    AND    gene_product_id = ?
	],('gene_product_id'),{},($gp_id));

    my $genus = $gp_org_ref->{$gp_id}->{'genus'};
    my $species = $gp_org_ref->{$gp_id}->{'species'};
    my $cul = $gp_org_ref->{$gp_id}->{'cultivar_name'};
    my $organism = $genus." ".$species."   ".$cul;

    my $organism_rank;
    if($organism =~/sativa/i){
	if( $organism =~ /japonica/i ) {
	    if($organism =~ /Nipponbare/i ) {
		$organism_rank = 1;
	    }else{
		$organism_rank = 2;
	    }
	}elsif($organism =~ /indica/i ) {
	    $organism_rank = 3;
	}else{
	    $organism_rank = 4;
	}
    }else{
	$organism_rank = 5;
    }



    my $sth = $self->{'db'}->prepare(
	q[
	    UPDATE gene_product_helper 
	    SET    organism = ?,
	    organism_rank = ?
	    WHERE gene_product_id = ?
	]);

    $sth->execute(($organism,$organism_rank,$gp_id));

    $sth->finish;

}


sub _transact_init {
    my $dbh = shift;
    my $attr_ref = {};
    $attr_ref->{AutoCommit} = $dbh->{AutoCommit};

    $dbh->{AutoCommit}= 0;

    return ($attr_ref);
}

sub _transact_finish {
    my ($dbh, $attr_ref, $error) = @_;
    if ( $error ){
	print "Database update failed, rolling back. Error
	was:\n$error\n";
	eval { $dbh->rollback; };
    }
    $dbh->{AutoCommit} = $attr_ref->{AutoCommit};
}



sub _next_id {
    my ( $db, $table_name, $field_name ) = @_;
    my $id = $db->selectrow_array("select max($field_name) from $table_name");
    return ++$id;
}


sub _get_term_name {

    my ( $self, $acc ) = @_;

    return $self->{'OntologyDB'}->get_term_name_by_term_accession($acc);

}

1;
