package Gramene::Page;

# $Id: Page.pm,v 1.45.2.2 2007/06/14 19:03:22 kclark Exp $
# vim: set ts=4:

=head1 NAME

Gramene::Page - Gramene page wrapper

=head1 DESCRIPTION

Gramene::Page creates the headers and footers and site
navigation bars around both dynamic and static content for all Gramene
pages.

=head1 USAGE

Especially by Gramene::Apache::WrapHTML

Note that PerlSetEnv WRAP_ONLY implies that WrapHTML will ignore
all urls that don't include this value in the filename part of 
the path.

=head2 tutorials and frames

Tutorial = powerpoint that was converted to html by Open Office.
This html uses frames and javascript.
Each tutorial has its own directory.
Only the main .html that defines the frames is called *tutorial*
The WRAP_ONLY PerlSetEnv  is set to 'tutorial' so that WrapHTML will
alter only this file and the ___tutorial__EMPTY.html url (see below)
And not add logo & navbar to all frame components.

On pages that contain <frameset>, WrapHTML does this:
		Add an 'empty' frame at the top to contain the logo & navbar
				For /myloc/myframe.html this is /myloc/myframe.htmlEMPTY.html
					 which is actually read from /empty.html
							(its <base> makes it's link break out of frames)
		    Com
		Treat the <noframes> <body> as a normal page



=cut

use strict;
use Apache;
use Apache::File;
use CGI;
use Carp;
use Gramene::Config;
use Gramene::Search::DB;
use constant CELL_LINEPADDING   => 0;
use constant CELL_LINESPACING   => 0;
use constant DEFAULT_PAGEWIDTH  => 500;
use constant DEFAULT_PANELWIDTH => 100;
my %PANELS;
my $bottom_menu_bar;

sub new {
    my $class     = shift;
    my $r         = shift || Apache->request;
    my $opts_ref  = shift || {};
    my $conf      = $r->dir_config('NavConf') 
        || croak( "Apache Request has no dir_config('NavConf')");
    my $conf_file = $r->server_root_relative($conf);
    -e $conf_file 
        or print STDERR "no $conf_file\n" and return;

    my $nav_panel = $class->read_configuration($conf_file) or return;
    my $hostname = $r->get_server_name
        . ( $r->get_server_port == 80 ? '' : ":" . $r->get_server_port );

    ( my $current_url = $r->uri ) =~ s,^//*,/,;
    $current_url =~ s!/perl/$ENV{ENSEMBL_SPECIES}/!/$ENV{ENSEMBL_SPECIES}/!
        if defined $ENV{ENSEMBL_SPECIES};

    #warn(join(" # ",$current_url,$hostname));
    $current_url =~ s!EMPTY.html$!!;

    $r->err_header_out( 'ensembl_headers_out' => 1 );

    return bless {
        r            => $r,
        panel        => $nav_panel,
        hostname     => $hostname,
        current_url  => $current_url,
        current_args => scalar( $r->args ),
        stylesheet   => $r->dir_config('Stylesheet'),
        stylesheets  => [ split( ':', $r->dir_config('Stylesheet') ) ],
        javascripts  => [ split( ':', $r->dir_config('JavaScript') ) ],
        background   => $r->dir_config('Background'),
        bgcolor      => $r->dir_config('Bgcolor'),
        logo         => $r->dir_config('Logo'),
        enslogo      => $r->dir_config('EnsLogo'),
        banner       => $r->dir_config('Banner'),
        pagewidth    => $r->dir_config('PageWidth') || DEFAULT_PAGEWIDTH,
        panelwidth   => $r->dir_config('PanelWidth') || DEFAULT_PANELWIDTH,
        footer       => $r->dir_config('Footer'),
        %{ $opts_ref }
    }, $class;
} ## end sub new

#
# Read the navigation bar configuration file and return it as a hash.
#
sub read_configuration {
    my $class    = shift;
    my $conf     = shift;
    my $mod_time = ( stat $conf )[9];
    return $PANELS{$conf}
        if $PANELS{$conf}
        && $PANELS{$conf}->modified >= $mod_time;

    return $PANELS{$conf} = Gramene::NavBar->new($conf);
} ## end sub read_configuration

sub panel        { shift->{panel} }
sub banner       { shift->{banner} }
sub logo         { shift->{logo} }
sub enslogo      { shift->{enslogo} }
sub background   { shift->{background} }
sub bgcolor      { shift->{bgcolor} }
sub stylesheet   { warn( "DEPRECATED; use stylesheets instead" );
                   warn "From file: ", join( " line ", (caller(0))[1..2] );
                   shift->{stylesheet} }
sub stylesheets  { @{shift->{stylesheets}||[]} }
sub javascripts  { @{shift->{javascripts}||[]} }
sub pagewidth    { shift->{pagewidth} }
sub panelwidth   { shift->{panelwidth} }
sub footer       { shift->{footer} }
sub hostname     { shift->{hostname} }
sub current_url  { shift->{current_url} }
sub current_args { shift->{current_args} }
sub modified     { shift->panel->modified }
sub r            { shift->{r} }

sub url_for_form {
    my $self         = shift;
    my $current_args = $self->current_args;
    return $self->current_url . ( $current_args ? '?' . $current_args : '' );
} ## end sub url_for_form

sub stylesheet_link {
    my $self    = shift;
    my @sheets = $self->stylesheets;
    my @scripts = $self->javascripts;
    my $css_htmlt = qq(<link rel="stylesheet" href="%s" type="text/css">\n);
    my $js_htmlt  = qq(<script type="text/javascript" src="%s"></script>\n);
    my $google    = q[
        <script src="http://www.google-analytics.com/urchin.js" type="text/javascript">
        </script>
        <script type="text/javascript">
        _uacct = "UA-1624628-5";
        urchinTracker();
        </script>
    ];
    return join( "", map{ sprintf( $css_htmlt, $_ ) } @sheets ) .
           join( "", map{ sprintf( $js_htmlt, $_ )  } @scripts ).
           qq(<link rel="Shortcut Icon" type="image/ico" href="/favicon.ico">).
           $google;
}


#
# Starts the HTML for the page, including stylesheets etc needed
# for the header. Is a wrapper for CGI::start_html, and you can pass
# appropriate args if wanted.
#
sub start_html {
  my $self = shift;
  my %args = @_;

	# Fix IE 7 which set XHTML 1.0 and HTML 4.01 Trans with URL to Standards
	# mode instead of Quirks mode. Another solution is to insert a comment 
  # before DTD (but don't know how to accomplish by CGI) 
  my $passed_style = $args{'-style'};
  $args{-style}  = [ map{{src=>$_}} 
                     $self->stylesheets() ];
  if (ref $passed_style eq 'ARRAY') {
    push @{ $args{'-style'} }, @$passed_style;
  }
  elsif ($passed_style) {
    push @{ $args{'-style'} }, $passed_style;
  };

  my $passed_script = $args{'-script'};
  $args{-script} = [ map{{src=>$_,language=>'javascript'}} 
                     $self->javascripts() ];

  if (ref $passed_script eq 'ARRAY') {
    push @{ $args{-script} }, @$passed_script;
  }
  elsif ($passed_script) {
    push @{ $args{-script} }, $passed_script;
  }

  return CGI::start_html(%args);
}

#
# This starts the table that includes the navigation panel and
# background.
#
sub start_body {
    my $self       = shift;
    # {-ensembl=> .., -bodyattr=>'..', -bodyfirst=>'...'};
    my %attr       = ref $_[0] eq 'HASH' ? %{ $_[0] } : @_; 
    my $background = $self->background;
    my $bgcolor    = $self->bgcolor;
    my $pagewidth  = $self->pagewidth;
    my $panelwidth = $self->panelwidth;
    my $ensembl    = $attr{-ensembl}   || $ENV{'ENSEMBL_SPECIES'} || 0;
    my $bodyattr   = $attr{-bodyattr}  || '';
    my $bodyfirst  = $attr{-bodyfirst} || '';
    my $onload     = $attr{-onload}    || '';
    $bodyattr     .= qq( background="$background") if $background;
    $bodyattr     .= qq( bgcolor="$bgcolor")       if $bgcolor;
    $bodyattr     .= qq( onload="$onload")         if $onload;
    my $text       = sprintf(
      "<body%s>\n%s", $bodyattr ? " $bodyattr" : '', $bodyfirst 
    );

    my $current_url  = $self->current_url;
    my $live_link = "http://www.gramene.org/" . $current_url . 
        ( $self->current_args ? '?' . $self->current_args : '' );

    $text .=
        $ENV{'DevelopmentSite'}
    ? qq( <h2>This is the Gramene Development Site. )
        . qq( For the Live Site go <a href="$live_link">here</a></h2> )
        : '';

    $text .= qq(<a name="top"></a>);
    $text .= $self->make_panel($ensembl);
    return $text;
} ## end sub start_body

sub end_body {
    my $self = shift;
    my $additional_stuff = shift || '';
    my $part = shift || 0;    # so we can put something after the navbar
    my $text = '';
    return $text if $part == 1;

    $text .= "<br>&nbsp;&nbsp;$additional_stuff";

    my $divt = qq(<div id="%s">%s\n</div>);
    my $navbar = $self->panel;
    $text .= sprintf( $divt, 'grm_footer', 
                      join( "", 
                            $navbar->render('species'),
                            $navbar->render('affiliation'),
                            sprintf( $divt, 'grm_lastmod', 
                                     $navbar->render('bottom') ) ) ,
                      );

#    my $lmod = "Last Modified: ".localtime( $self->modified );
#    my $grm_lastmod = '';
#    #$grm_lastmod .= qq(<ul id="gbot"><li id="first">$lmod</li></ul>);
#    $grm_lastmod .= $navbar->render('bottom');
#
#    $text .= sprintf( $divt, 'grm_lastmod', $navbar->render('bottom') );
    $text .= "</body></html>\n";
    return $text;
}


sub make_panel {
    my $self        = shift;
    my $ensembl     = shift;
    my $current_url = $self->current_url;
    my $panel       = $self->panel;
    my $cp          = CELL_LINEPADDING;
    my $cs          = CELL_LINESPACING;
    my $banner      = $self->banner;
    my $logo        = $self->logo;
    my $path_info = $self->r->path_info;
#    $current_url =~ s/${path_info}$//;

    my $banner_img = $banner
        ? qq(<img src="$banner" alt="[Home Page]" border="0">)
        : '<h1>Gramene</h1>';
    my $logo_img
        = $logo ? qq(<a href="/"><img src="$logo" alt="[Home Page]" title="[Home Page]" border="0"></a>) : '&nbsp';


    my $modules_select = join('', 
          '<option value="">Find anything</option>',
          map { 
              qq[<option value="$_->{module_name}">$_->{display_name}</option>]
          }
          Gramene::Search::DB->module_display_list()
    );

    # ---Search---
    my $apr                = Apache::Request->new( $self->r );
    my $current_search_val = $apr->param('search_for') || '';
    my $guide_search_html  = qq(
<form method="GET" action="/db/searches/quick_search" style="display:inline">
<select name="module">
$modules_select
</select>
<input type="text" name="search_for" size="15" value="$current_search_val" maxlength="100">
<input type="submit" value="Search">
</form>);
    # ---EnsSearch---

    #------------------------------
    # New stuff
    my $grm_header = '';
    my @url_bits 
        = grep { !/\.html?$/ } 
        map { $_ || () } 
        ( split '/', $current_url );
    my $banner_text = $url_bits[-1];
    my %tweak_banner_text = (
        'cmap'           => 'CMap',
        'diversity'      => 'Genetic Diversity',
        'diversity_view' => 'Genetic Diversity',
        'faq'            => 'FAQ',
        'mutant'         => 'Genes',
        'ontology'       => 'Ontologies',
        'plant_ontology' => 'Ontologies',
        'protein'        => 'Proteins',
        'qtl'            => 'QTL',
        'rice_mutant'    => 'Genes',
    );

    if ( $tweak_banner_text{ $banner_text } ) { 
        $banner_text = $tweak_banner_text{ $banner_text };
    }
    elsif ( $apr->param('banner_text') ) {
        $banner_text = $apr->param('banner_text');
    }
    elsif ( @url_bits == 0 ) { # home page
        $banner_text = 'Home';
    }
    else {
        $banner_text = join( q{ }, map { ucfirst } split /_/, $banner_text );
    }

    my $divt=qq(
<div id="%s">%s</div>);
    my $nbar = $self->panel;

    my $banner_code = sprintf( $divt, 'grm_logo', $logo_img);
#    $banner .= qq(<ul id="grm_banner"><li>$banner_text</li><li style="float:right">$version</li></ul>);
#    $banner .=  sprintf( $divt, 'grm_banner', $banner_text);
    my @banner_lis = sprintf( "<li>%s</li>", $banner_text );
    if( my $script = $ENV{ENSEMBL_SCRIPT} ){
      if( $script =~ /(.+)(view)/ ){
        $script = ucfirst($1).ucfirst($2);
      }
      $script = '&nbsp;'.$script;
      push @banner_lis, sprintf( "<li>%s</li>", $script );
    }

    push @banner_lis, 
        qq(<li style="float:right;margin-top:6px">$guide_search_html</li>);

    $banner_code .= sprintf( qq(<ul id="grm_banner">%s</ul>),
                        join( '', @banner_lis ) );

    $grm_header .= sprintf( $divt, 'grm_header', $banner_code );

    #$grm_header .= sprintf( $divt, 'grm_logo', $logo_img);

		$current_url =~s!/perl/$ENV{ENSEMBL_SPECIES}/!/$ENV{ENSEMBL_SPECIES}/! if defined $ENV{ENSEMBL_SPECIES};
		my $url_for_form = $current_url
				. ( $self->current_args ? '?' .  $self->current_args : '' );
		
		my $feedback_url = '/db/feedback/send_feedback';
		
		my $nbar_menu = $nbar->render;
		$nbar_menu =~s#$feedback_url#$feedback_url?refer_from=$url_for_form#g;

    #$grm_header .= sprintf( $divt, 'grm_tabbar',  $nbar->render);
    $grm_header .= sprintf( $divt, 'grm_tabbar',  $nbar_menu);
    #------------------------------

    my $guide
        = qq(<table border="0" cellpadding="$cp" cellspacing="$cs" width="100%"> <tr> <td>);

    my $guide_htmlt = qq(
<table border="0" cellpadding="0" cellspacing="0" width="100%%">
 <tr>
  <!-- Gramene banner + logo -->
  <td rowspan="2">%s<br>%s%s</td>
  <!-- Ensembl logo + species -->
  <td valign="bottom" colspan="3" width="100%%">&nbsp;%s</td>
 </tr>
 <tr>
  <!-- Search -->
  <td align="right">%s</td>
  <!-- Feedback -->
  <td>%s</td>
 </tr>
 
</table>);

    my $ensembl_link_html = '&nbsp;';
    my $species_link_html = '&nbsp;';
    if ($ensembl) {

        my $imglink_htmlt
            = qq(<a href="%s"><img src="%s" alt="%s" title="%s" border="0"></a>);

        $ensembl_link_html = sprintf( $imglink_htmlt,
            'http://www.ensembl.org', $self->enslogo || '/gfx/blank.gif',
            'Ensembl', 'Ensembl' );

        my $species = $ENV{'ENSEMBL_SPECIES'};
        $species_link_html = sprintf( $imglink_htmlt,
            "/$species",
            $self->r->dir_config("${species}_logo") || '/gfx/blank.gif',
            $species, $species );
    } ## end if ($ensembl)


    my $guide_feedback_html = qq(
               <form method=POST action="/db/feedback/send_feedback">
               <input type="hidden" name="refer_to" value=")
        . $self->url_for_form . qq(">
               <input type="submit" value="Feedback">
               </form>);

    $guide .= sprintf( $guide_htmlt,
        $banner_img, qq(<a href="/">$logo_img</a>),
        $ensembl_link_html, $species_link_html, $guide_search_html,
        $guide_feedback_html );

    $guide .= qq(</td></tr>);

    $bottom_menu_bar
        = qq(<hr><table border="0" align="center" width="100%"><tr><td align="left" >)
        . qq(<img src="/images/icons/grain_icon.jpg" alt="grain_icon" height=16 width=16 align="top">&nbsp;&nbsp;</td><td>);

    # my @labels;
    my @up_labels;
    my @bottom_labels;
    my $is_bottom_menu = 0;
    foreach my $label ( $panel->labels(), '0', $panel->bottom_labels() ) {

        my $url = $panel->url($label);

        #$label = 
        #    = $label =~ /^\"/ ? substr( $label, 1 ) : CGI->escapeHTML($label);
        if ( $url =~ /^\d$/ ) {    # all digits == not a url
                #$guide .= qq(<tr><td align="left" colspan="2">)
                #  . join ( "&nbsp;| ", @labels );

            $is_bottom_menu = 1;
            $guide
                .= qq(<tr><td><table border="0" cellpadding="5" cellspacing="2" width="100%"><tr>)
                . join( "", @up_labels )
                . qq(</tr></table></td>);

   #$bottom_menu_bar .= join ( "&nbsp;| ", @labels ); ## qq(</td></tr></table>);
   #@labels = ();

        } else {

            if ($is_bottom_menu) {

                my $cell = $current_url eq $url
                    ? qq( <span class="currentpage_bottom">$label</span>)
                    : qq( <a class="notcurrent_bottom" href="$url">$label</a>);
                push( @bottom_labels, $cell );

            } else {
                my $up_cell = $current_url eq $url
                    ? qq(<td class="currentpage" align="center" nowrap>$label</td>)
                    : qq(<td class="notcurrent" align="center" nowrap><a class="notcurrent" href="$url">$label</a></td>);
                push( @up_labels, $up_cell );
            } ## end else [ if ($is_bottom_menu)

        } ## end else [ if ( $url =~ /^\d$/ )
    } ## end foreach my $url ( $panel->urls)
    $bottom_menu_bar
        .= join( "&nbsp;| ", @bottom_labels );    ## qq(</td></tr></table>);

    $guide .= qq(</tr>);

    #$guide .= qq(<td align="right" valign="top">)
    #  . join ( "&nbsp;| ", @labels )
    #  . "</form></td></tr></table><hr></td>";

    $guide .= qq(</table><br><hr width="100%">\n);
    return $grm_header ;#. $guide;
} ## end sub make_panel

###################### package Gramene::NavBar ##############################

package Gramene::NavBar;

# create a new Gramene::NavBar object
sub new {
    my ( $class, $conf_file ) = @_;
    my ( @labels, @bottom_labels, @species_labels, @affiliation_labels,
         %urls, %sub_labels, %sub_urls );
    local $/ = "\n";    #just in case somebody's doing something tricky up there
    my $fh = Apache::File->new($conf_file)
        or print STDERR "Gramene::NavBar: open $conf_file:$!\n"
        and return;
    my $type = 'TOP'; # Bottom or top menu items?
    while (<$fh>) {
        chomp;
        s/^\s+|\s+$//g;    # trim leading and trailing whitespace
        print STDERR map {"inc\t$_\t$INC{$_}\n"} sort keys %INC and next
            if /^!inc\b/;
        next if /^#/ || /^$/;    # skip comments and empty lines

        my @bits = split( /\s+/, $_ );
        if( $bits[0] eq '0' ){ $type = $bits[1]; next; } # Switch to bottom menu
        my $label = $bits[0];
        unless( exists $urls{$label} ){ # Not seen this one b4
          $urls{$label} = undef;
          if( $type eq 'BOTTOM' ){ push @bottom_labels, $label }
          elsif( $type eq 'SPECIES' ){  push @species_labels, $label }
          elsif( $type eq 'AFFILIATION' ){ push @affiliation_labels, $label }
          else{ push @labels, $label }
        }

        if( @bits == 2 ){ # Top-level option
          my $url   = $bits[1];
          $urls{$label} = $url;       # keep its url in a hash
        }
        if( @bits == 3 ){ # 2-nd level option
          $sub_labels{$label} ||= [];
          $sub_urls{$label}   ||= {};
          my $sub_label = $bits[1];
          my $sub_url   = $bits[2];
          push @{$sub_labels{$label}}, $bits[1];
          $sub_urls{$label}->{$sub_label} = $sub_url;
        }

    } ## end while (<$fh>)
    close $fh or print STDERR "Gramene::NavBar: close $conf_file:$!\n";
    return bless {
        'urls'          => \%urls,
        'labels'        => \@labels,
        'bottom_labels' => \@bottom_labels,
        'species_labels'=> \@species_labels,
        'affiliation_labels' => \@affiliation_labels,
        'sub_urls'      => \%sub_urls,
        'sub_labels'    => \%sub_labels,
        'modified'      => ( stat $conf_file )[9]
    }, $class;
} ## end sub new

# return ordered list of all the URLs in the navigation bar
sub labels { return @{ shift->{'labels'} }; }
sub bottom_labels { return @{ shift->{'bottom_labels'}||[] }; }
sub species_labels { return @{ shift->{'species_labels'}||[] }; }
sub affiliation_labels { return @{ shift->{'affiliation_labels'}||[] }; }

# return the label for a particular URL in the navigation bar
sub url { return $_[0]->{'urls'}->{ $_[1] } || $_[1]; }

# return the sub-labels for a given label
sub sub_labels{ return @{ $_[0]->{'sub_labels'}->{$_[1]} || [] }; }

# return the sub-url for a given parent and sub label
sub sub_url{ return $_[0]->{'sub_urls'}->{$_[1]} ? 
                 $_[0]->{'sub_urls'}->{$_[1]}->{$_[2]} : '' }

# return the modification date of the configuration file
sub modified { return $_[0]->{'modified'}; }

# returns the HTML corresponding to he navbar.
# Relys on the page having loaded gramene-page.css
sub render{
  my $self = shift;
  my $type = shift || '';
  if( $type and $type ne 'bottom' ){
    return $self->render_by_type( $type );
  }

  my $id = $type eq 'bottom' ? 'gbot' : 'gnav';

  # Define HTML templates
  my $ul_t = qq(
<ul id="$id"> %s
</ul> );

  my $li_t = qq(
  <li>
   <a href="%s">%s</a> %s
  </li> );

  my $lilast_t = qq(
  <li id="last">
   <a href="%s">%s</a> %s
  </li> );

  my $ulul_t = qq(
   <ul> %s
   </ul> );

  my $ulullast_t = qq(
   <ul style="right: 0"> %s
   </ul> );

  my $lili_t = qq(
    <li><a href="%s">%s</a></li> );
  # End templates

  # Populate templates
  my @labels = $self->labels;

  if( $type eq 'bottom' ){ @labels = $self->bottom_labels };
  my $count = 0;
  my $length = @labels;
  my @lis;
  foreach my $label( @labels ){
    $count++;
    my @lilis;
    foreach my $sublabel( $self->sub_labels($label) ){
      my $suburl = $self->sub_url( $label, $sublabel ); 

      $sublabel =~ s/_/ /g;
      push @lilis, sprintf( $lili_t, $suburl, $sublabel );
    }
    my $ululthis_t = ( $length > $count ) ? $ulul_t : $ulullast_t;
    my $lithis_t   = ( $length > $count ) ? $li_t : $lilast_t; 
    my $ulul = sprintf( $ululthis_t, join( "", @lilis ) );
    my $url = $self->url($label);
    $label =~ s/_/ /g;
    push @lis, sprintf( $lithis_t, $url, $label, $ulul );
  }
  my $ul = sprintf( $ul_t, join( "", @lis ) );
  return $ul;       
}

sub render_by_type{
  my $self = shift;
  my $type = shift;

  my $htmlt = qq(
<a href="%s"><img class="%s" src="%s" alt="%s" title="%s"></a>);
  
  my @imgs;
  my @labels;
  if   ( $type eq "species"     ){ @labels = $self->species_labels }
  elsif( $type eq "affiliation" ){ @labels = $self->affiliation_labels }
  foreach my $label( @labels ){
    foreach my $src( $self->sub_labels($label) ){
      my $url = $self->sub_url( $label, $src );
      $label =~ s/_/ /g;
      push @imgs, sprintf( $htmlt, $url, $type, $src, $label, $label );
    }
  }
  return join( "", @imgs );
}

1;

=pod

=head1 AUTHOR

Lincoln Stein.  Modified by a cast of thousands.

=cut

