##########################################################################
# Date: 05/22/2002
# Version: 1.3
# Description: Ontology Database Perl Module 
##########################################################################
package Gramene::Ontology::OntologyDB;

use strict;
use DBI;
use Gramene::DB;

##########################################################################
##########################################################################
# constants
use constant ISA => 1;
use constant PARTOF => 2;
use constant DEVELOPSFROM => 4;

# Somehow Kuan has five relationship types, but there are actually only three. Constants added to deal with the duplications.
use constant ISA_TWO => 5;
use constant PARTOF_TWO => 3;


##########################################################################
# constructor
##########################################################################
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = { };
  bless($self, $class);
  return $self;
}

# destructor
sub DESTROY {
 # my $self = shift;
 # $self->terminate_database if $self->{'db'};  #obsolete, the Gramene::DB will call disconnect  
}

# database subroutines
sub db {
  my $self = shift;
  unless ( $self->{'db'} ) {
    my $dbh; 
    eval{  
      $dbh = Gramene::DB->new('ontology');  
    };  
  
    if($@){   
        die "DB connection failed\n";   
    }   

    $self->{'db'}=$dbh;

  }
  return $self->{'db'};
}

sub terminate_database{
  my $self = shift;
  $self->{'db'}->disconnect() if $self->{'db'};
}

##############################################################################
# functions for accessing ontology database
##############################################################################
sub get_termID {
  my ($self, $acc) = @_;
  my $dbh  = $self->db;
  die "Invalid input in get_termID" unless defined $acc;

  my $sth = $dbh->prepare("
        SELECT term_id
        FROM   term 
        WHERE  term_accession=?");
 
  $sth->bind_param(1,$acc);
  
  $sth->execute || die "Execute: ", $sth->errstr;
  my ($term_id) = $sth->fetchrow_array;
  $sth->finish;
 
  # some term accessions become secondary term(term synonym)
  unless($term_id){
    $sth = $dbh->prepare("
      SELECT term_id
      FROM   term_synonym
      WHERE  synonym_name=?");

    $sth->bind_param(1,$acc);
    $sth->execute || die "Execute: ", $sth->errstr;
    ($term_id) = $sth->fetchrow_array;
    $sth->finish;
  }
  return $term_id;
}


sub get_term {
  my ($self,$term_id) = @_;
  my $dbh = $self->db;
  die "Invalid input in get_term" unless defined $term_id;

  my $sth = $dbh->prepare("
                SELECT term_name, T.term_type_id,term_type, term_accession,is_obsolete
                FROM   term T, term_type TT
		WHERE  T.term_type_id = TT.term_type_id
                AND    term_id = ? 
              ");

  $sth->execute(($term_id));

  my @term = $sth->fetchrow_array;
  $sth->finish;
  return @term;

}

sub get_name {
  my ($self, $id) = @_;
  die "Invalid input in get_name" unless defined $id;
  my $dbh  = $self->db;
  my $sth = $dbh->prepare("SELECT term_name FROM term WHERE term_id= ?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my ($term_name) = $sth->fetchrow_array;
  $sth->finish;  

  return $term_name;
}

sub get_definition_and_comment {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die "Invalid input in get_definition" unless defined $id;
  my $sth = $dbh->prepare("SELECT definition, term_def_comment FROM term_definition WHERE term_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my ($term_definition,$comment) = $sth->fetchrow_array;
  $sth->finish;
  return ($term_definition,$comment);
}

sub get_synonym {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die "Invalid input in get_synonym" unless defined $id;
  my $sth = $dbh->prepare("SELECT synonym_name FROM term_synonym WHERE term_id=?")|| die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $synonym;	
  while(my $row = $sth->fetchrow_array){
    $synonym .= $row .", ";
  }
  
  if(defined $synonym){
     $synonym =~ s/, $/./;
  }

  $sth->finish;
  return $synonym;
}

sub get_term_xrefs {
    my ($self, $id ) = @_;
    die "Invalid input in get_synonym" unless defined $id;
    my $dbh  = $self->db;
    my $sth = $dbh->prepare(
	q[
	    SELECT xref_dbname, xref_key
	    FROM   term_dbxref, dbxref
	    WHERE  term_dbxref.dbxref_id = dbxref.dbxref_id
	    AND    term_id = ?
	]);

    $sth->execute(($id));

    my @term_xrefs;
    while(my ($xref_dbname, $xref_key) = $sth->fetchrow_array){
	push @term_xrefs, "$xref_dbname:$xref_key";
    }
    $sth->finish;

    return \@term_xrefs;
}


sub get_association_id {
  my ($self, $term_id, $gene_product_id) = @_;
  die "Invalid input in get_association_id" unless (defined $term_id and defined $gene_product_id);
  my $dbh  = $self->db;
  my $sth = $dbh->prepare("SELECT association_id FROM association WHERE term_id=? and gene_product_id= ?") || die $dbh->errstr;
  $sth->bind_param(1,$term_id);
  $sth->bind_param(2,$gene_product_id); 

 $sth->execute || die "Execute: ", $sth->errstr;
  my $association_id = $sth->fetchrow_array;
  $sth->finish;
  return $association_id;
}

# from association id
sub get_evidence_codes {
  my ($self, $association_id) = @_;
  my $dbh  = $self->db;
  die  "Invalid input in get_evidence_codes" unless defined $association_id;
  my $sth = $dbh->prepare("SELECT evidence_code FROM evidence WHERE association_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$association_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my @evidence_codes = ();
  while(my $row = $sth->fetchrow_array){
    push @evidence_codes, $row;
  }
  $sth->finish;
  return @evidence_codes;
}

# from association id
sub get_evidence_codes_in_once {
  my ($self, $association_id) = @_;
  my $dbh  = $self->db;
  die  "Invalid input in get_evidence_codes_in_once" unless defined $association_id;
  my $sth = $dbh->prepare("SELECT evidence_code FROM evidence WHERE association_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$association_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $evidence_codes;
  while(my $row = $sth->fetchrow_array){
    $evidence_codes .= $row . " ";
  }
  $sth->finish;
  return $evidence_codes;
}

# post: return list of IDs of parent terms;
sub get_parents {
  my $self = shift;  
  my $id = shift;
  my $dbh  = $self->db;
  die  "Invalid input in get_parents" unless defined $id;
  my $sth = $dbh->prepare("SELECT term1_id FROM term_to_term WHERE term2_id=? order by term1_id") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my @term_parents = ();
  while(my $row = $sth->fetchrow_array){
    push @term_parents, $row;
  }
  $sth->finish;
  return @term_parents;
}

# post: return list of IDs of child terms
sub get_children {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die  "Invalid input in get_children" unless defined $id;
  my $sth = $dbh->prepare("SELECT term2_id FROM term_to_term WHERE term1_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my @term_children = ();
  while(my $row = $sth->fetchrow_array){
    push @term_children, $row;
  }
  $sth->finish;
  return @term_children;
}

# post: return $ISA, $PARTOF, $DEVELOPSFROM, or ""
sub get_relationship_type {
  my ($self, $parent, $child) = @_;
  die "Invalid input in get_relationship_type2" unless (defined $parent and defined $child);
  my $dbh  = $self->db;
  my $sth = $dbh->prepare("SELECT relationship_type_id FROM term_to_term WHERE term1_id=? and term2_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$parent);
  $sth->bind_param(2,$child);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $relationship_type = $sth->fetchrow_array;
  $sth->finish;
  return $relationship_type;
}

sub get_relationship_symbol {
  my ($self, $parent, $child) = @_;
  my $relationship = get_relationship_type($self, $parent, $child);
 
  if( $relationship){
    if( ( $relationship == ISA ) || ( $relationship == ISA_TWO ) ){ return '[i] '; }
    elsif( ( $relationship == PARTOF ) || ( $relationship == PARTOF_TWO ) ){ return '[p] ';}
  elsif($relationship == DEVELOPSFROM){ return  '[d] ';}
  else{ return ''; }
  }
   
  return '';
  
}

sub get_accession {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die "Invalid input in GetAccessionID" unless defined $id;
  my $sth = $dbh->prepare("SELECT term_accession FROM term WHERE term_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my ($accession) = $sth->fetchrow_array;
  $sth->finish;
  return $accession;
}

sub get_gene_association_count {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die "Invalid input in get_gene_association_count" unless defined $id;
  my $sth = $dbh->prepare("SELECT all_association_count FROM gene_association_count WHERE term_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $count = $sth->fetchrow_array;
  $sth->finish; 
  return $count ? $count :0;  # liya
}

sub get_term_children_count {
  my $self = shift;
  my $id = shift;
  my $dbh  = $self->db;
  die "Invalid input in get_term_children_count" unless defined $id;
  my $sth = $dbh->prepare("
     SELECT count(*)
     FROM   term_to_term 
     WHERE  term1_id=?") || die $dbh->errstr;
  $sth->bind_param(1,$id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $count = $sth->fetchrow_array;
  $sth->finish; 
  return $count;
}
sub get_total_associated_object_count{
  my ($self, $id) = @_;
  my $dbh = $self->db;
  die "Invalid input in get_total_associated_objects_count" unless defined $id;
  my $sql = q[
    SELECT sum(object_count) 
    FROM  object_association_count
    WHERE term_id = ? 
  ];
  
  my $sth = $dbh->prepare($sql);

  $sth->bind_param(1,$id);
  $sth->execute() or die $dbh->errstr;

  my ($obj_count) = $sth->fetchrow_array();
  $sth->finish;

  return $obj_count ? $obj_count: 0;
}

sub get_associated_objects_count_by_type{
  my ($self, $id, $obj_type) = @_;
  my $dbh = $self->db;
  die "Invalid input in get_associated_objects_count_by_type" unless defined $id;
  my $sql = q[
    SELECT object_count,object_species 
    FROM  object_association_count
    WHERE term_id = ? 
    AND   object_type = ?
  ];
  
  my $sth = $dbh->prepare($sql);

  $sth->bind_param(1,$id);
  $sth->bind_param(2,$obj_type);
  $sth->execute() or die $dbh->errstr;

  my $species_to_obj_counts;
  while( my ($obj_count,$object_species) = $sth->fetchrow_array()){
    $species_to_obj_counts->{$object_species}=$obj_count;
  }

  $sth->finish;

  return $species_to_obj_counts;
}

sub get_association_count_by_type{
  my ($self, $id,$obj_type ) = @_;
  my $dbh = $self->db;
  die "Invalid input in get_association_count_by_type" unless defined $id;
  my $sql = q[
    SELECT sum(association_count )
    FROM   object_association_count 
    WHERE  term_id = ?
    ]; 
  
  my @params = ($id);
  if($obj_type){
    $sql.= q[ AND object_type = ?];
    push @params, $obj_type;
  }


  my $sth = $dbh->prepare($sql);

  $sth->execute(@params) or die $dbh->errstr;
  my ($count) = $sth->fetchrow_array;
  $sth->finish;
  return $count ? $count : 0;
}


sub get_associated_object_types{
  my ($self,$id) = @_;
  my $dbh  = $self->db;
  die "Invalid input in get_associated_object_types" unless defined $id;
  my $sth = $dbh->prepare(
    q[
      SELECT DISTINCT object_type
      FROM   object_association_count 
      WHERE  term_id = ? 
    ]);

  $sth->execute(($id)) or die $dbh->errstr;
  my @obj_types;
  while (my ($obj_type) = $sth->fetchrow_array){
    push @obj_types, $obj_type;
  }

  $sth->finish;

  return @obj_types;

}

sub get_term_associations{
  my ($self,$id,$obj_type,$obj_species,$sort_by) = @_;
  my $dbh  = $self->db;
  die "Invalid input in get_associated_object_types" unless defined $id;

  $sort_by ||='object_symbol';
  
  my $sql = q[
      SELECT term_accession,
             term_name,
	     term_type,
	     object_type,
	     object_accession_id,
	     object_symbol,
	     object_name,
	     object_synonyms,
	     object_species,
	     evidences
      FROM   graph_path GP, quick_term_to_object QT
      WHERE  GP.term2_id = QT.term_id
      AND    GP.term1_id = ?
  ];
  my @params = ($id); 
  if($obj_type){
    $sql .= " AND object_type = ?";
    push @params, $obj_type;
  }

  if($obj_species){
    $sql .= " AND object_species =?";
    push @params, $obj_species;
  }
 
  $sql .=' GROUP BY term_accession, object_accession_id,object_type '; 
  $sql .= ' ORDER BY ?';
  push @params,$sort_by;

  my $sth = $dbh->prepare($sql);
  $sth->execute(@params) or die $dbh->errstr;

  my @associations;
  while(my $assocRef = $sth->fetchrow_hashref){
    push @associations, $assocRef;
  }

  $sth->finish;
  return @associations;

}

sub get_all_xref_db_urls{
    my ($self ) = @_;
    my ($url_templates) = $self->{'db'}->selectall_hashref(
	q[
	    SELECT upper(name) as db_name, url_syntax FROM data_base
	],('db_name') );

    return $url_templates;

}


#--------------------------------------------------
sub get_term_by_xref{

=pod

=head2 get_term_by_xref 

  my $term = $odb->get_term_by_xref('NCBI_taxid','4530') or 
       die $odb->error;

Returns the term data corresponding to a given xref_dbname and xref_key

=cut

  my $self = shift;
  my $xref_dbname = shift || die( "Need an xref_dbname" );
  my $xref_key    = shift || die( "Need an xref_key" );

  my $db = $self->db;
  my $sth = $db->prepare( q[
SELECT t.term_id, 
       t.term_name,
       tt.term_type,
       t.term_accession,
       t.is_obsolete,
       t.is_root
FROM   term t,
       term_type tt,
       term_dbxref td,
       dbxref dx
WHERE  t.term_type_id = tt.term_type_id
AND    t.term_id = td.term_id
AND    td.dbxref_id = dx.dbxref_id
AND    dx.xref_dbname = ?
AND    dx.xref_key = ? ] );

  $sth->execute( $xref_dbname, $xref_key );
  my $term = $sth->fetchrow_hashref or (
       warn "Xref $xref_key from $xref_dbname not found in OntologyDB" and
       return {}
  );

return $term;
}

#--------------------------------------------------


=pod

=head2 get_term_family_by_name

  my $term_family = $odb->get_term_family_by_name('plant height','Trait');

Get the whole term family with term accessions and term names by name search

=cut

sub get_term_family_by_name{

    my ($self, $name, $term_type ) = @_;
    die "No name defined" unless $name;
    
    my $db = $self->db;
   
    # get the candidate term by name case insensetively (but need add wild card if
    # wild card search
    my $sql = q[
	SELECT T.term_id 
	FROM   term T
	INNER JOIN term_type TT
	ON     T.term_type_id = TT.term_type_id
	LEFT JOIN term_synonym TS
	ON     T.term_id = TS.term_id
	WHERE 
	(      UPPER(term_name) like ?
	OR     UPPER(synonym_name) like ? )
    ];

    $sql.= 'AND term_type = ?' if $term_type;
    
    my $sth = $db->prepare($sql);

    my @params;
    push @params, uc($name);
    push @params, uc($name);
    push @params, $term_type if $term_type;

    $sth->execute(@params) or die $db->errstr;

    my $candidate_terms = $sth->fetchall_hashref('term_id');
    my @terms = keys %$candidate_terms;

    return unless scalar(@terms)>0;

   # get the descendant terms for the candidate terms (The annotations with
   # descendant terms with belong to candidate terms automatically)

    my $sql_family = q[
	SELECT term_accession, term_name, term_type
	FROM   term T
	INNER JOIN term_type TT
	ON     T.term_type_id = TT.term_type_id
	INNER JOIN graph_path G
	ON     T.term_id = G.term2_id

    ];

    my @param_holders;
    push(@param_holders, '?') for 1 .. scalar(@terms);

    $sql_family = $sql_family.' WHERE G.term1_id in ( '.join(', ', @param_holders).' )';
    
    
    my $sth_family = $db->prepare($sql_family);

    $sth_family->execute(@terms) or die $db->errstr;

    my $term_family = $sth_family->fetchall_hashref('term_accession');

    $sth->finish;
    $sth_family->finish;

    return $term_family;

}


#--------------------------------------------------

=pod

=head2 get_term_name_by_term_accession

  my $term_name = $odb->get_term_name_by_accession('TO:0000303');

Get the term name by term_accession

=cut

sub get_term_name_by_term_accession{
    my ($self, $term_acc) = @_;
    die "No term accession defined." unless $term_acc;

    my $db= $self->db;

    my ($term_name) = $db->selectrow_array(
	q[
	    SELECT term_name FROM term 
	    WHERE  term_accession = ?
	], {}, ($term_acc) );

    return $term_name;

}


1;

