##########################################################################
# Programmer: Kuan Chang
# Date: 04/18/2001
# Version: 1.1
# Description: Mutant Database Perl Module (MutantDB.pm)
#              Design for Complex Mutant
##########################################################################
package Gramene::Mutant::MutantSearchDB;

use strict;
use DBI;
use Gramene::DB;

##########################################################################
use constant STUDY_TYPE_0 => 'Not Recorded';   # liya
use constant STUDY_TYPE_I => 'Field Study';
use constant STUDY_TYPE_II => 'Green House';
use constant STUDY_TYPE_III => 'Growth Chamber';
use constant STUDY_TYPE_IV => 'Laboratory';

use constant MAP_TYPE_I => 'Cornell RFLP 2001';
use constant MAP_TYPE_II => 'JRGP RFLP 2000';
use constant MAP_TYPE_III => 'Rice Morphological 2000';
use constant MAP_TYPE_IV => 'Integrated Morphological_RFLP Map';
use constant MAP_TYPE_V => 'Cornell SSR 2001';

# constants
my $TRAIT = 5;
my $PO_TYPE = 6;
my $GRO_TYPE = 7;  # growth stage
my $GO_PREFIX = "GO:";
my $TO_PREFIX = "TO:";





##########################################################################
# constructor
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = { };
  bless($self, $class);
  return $self;
}

# destructor
sub DESTROY {
  my $self = shift;
 
  # obsolete, it's for CSHL:Config. The  Gramene::DB 's destructor will call terminate_database
  # $self->terminate_database if $self->{'db'};
}

sub connect_db {
  my $self = shift;
  unless ( $self->{'db'} ) {

     my ($dbh, $onto_dbh, $liter_dbh);
     eval{ 
       $dbh = Gramene::DB->new('mutant'); 
       $onto_dbh = Gramene::DB->new('ontology'); 
       $liter_dbh = Gramene::DB->new('literature'); 
     }; 
 
     if($@){  
        die "DB connection failed: $@\n";  
     }  
     $self->{'db'} = $dbh;
     $self->{'ontology_db'} = $onto_dbh;
     $self->{'literature_db'} = $liter_dbh;

    }
}


sub terminate_database{
  my $self = shift;
  $self->{'db'}->disconnect() if $self->{'db'};
  $self->{'ontology_db'}->disconnect() if $self->{'ontology_db'};
  $self->{'literature_db'}->disconnect() if $self->{'literature_db'};
 
}



##########################################################################
#functions for accessing mutant database
##########################################################################
sub get_mutantlist_ids {
  my ($self) = @_;
  my $dbh = $self->{'db'};
  my $sth = $dbh->prepare
    ("SELECT mutant_id
      FROM   mutant
      ORDER BY LOWER(mutant_symbol)") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;

  my @mutant_ids = ();
  while(my ($row) = $sth->fetchrow_array){
    push @mutant_ids, $row;
  }

  $sth->finish;
  return @mutant_ids;
};

sub get_mutant_list_by_order {
  my ($self, $index, $sort_by) = @_;
  my $dbh = $self->{'db'};
  my $sth;

  $sort_by =~ s/^\s+|\s+$//g if $sort_by;
  $sort_by ='mutant_symbol'  unless ($sort_by) ;
    $sth = $dbh->prepare
    ("SELECT *
      FROM   mutant_list
      WHERE  UPPER(mutant_symbol) like '$index%' or  UPPER(mutant_symbol) like '[$index%'
      ORDER BY LOWER($sort_by)") || die $dbh->errstr;
 
  $sth->execute || die "Execute: ", $sth->errstr;

  my %sort_bys;  # store the val of $sort_by -> mutant_ids
  my %id_to_vals; # store the mutant_id -> mutant infos
  my %cols=( 
              mutant_id=>0,
              mutant_name=>1,
              mutant_symbol=>2,
              mutant_synonym=>3,
              chromosome_number=>4
            );

  while(my $row_ref = $sth->fetchrow_arrayref()){
    my $id = $row_ref->[$cols{'mutant_id'}];
    my $sort = $row_ref->[$cols{"$sort_by"}];
    $sort ='' unless $sort;
    my $tmp = $sort_bys{$sort};  # more than one mutant
    if($tmp){
       $tmp="$tmp\t$id";
    }else{
       $tmp = $id;
    }
    
    $sort_bys{$sort}=$tmp;

    $id_to_vals{$id} = join("\t",map{defined $_ ? $_ :''} @$row_ref);
   
  }
  
  $sth->finish;

  my @mutants;
  # sort the mutants by the sort by and only the a-z letters
  foreach my $sort (sort {  my $first = $a;
                                  my $second = $b;
                                  $first =~ s/\W// if $first;             # sort only a-z letters, exclude the non-letter
                                  $second  =~ s/\W// if $second;
                                  $first =~ tr/A-Z/a-z/ if $first;  
                                  $second =~ tr/A-Z/a-z/ if $second;
                                  return $first cmp $second;
			  } keys %sort_bys ){

        my $ids = $sort_bys{$sort};
        my @mutant_ids = split(/\t/,$ids);
        foreach my $id (@mutant_ids){
             push @mutants, $id_to_vals{$id};

	}

  }
  return @mutants;

};

sub get_mutant_name {
  my ($self, $id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_mutant_name" unless defined $id;
 
  my ($name) = $dbh->selectrow_array("SELECT mutant_name FROM mutant WHERE mutant_id=$id");
  return $name;
};

sub get_first_letters {
  my ($self) = @_;
  my $dbh = $self->{'db'};  
  my $sth = $dbh->prepare("SELECT mutant_symbol FROM mutant order by mutant_name") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $first_letters='';

   while( my ($name) = $sth->fetchrow_array ) {  
      my $temp = substr( $name, 0, 1 );
      #@first_letters = ( @first_letters, $temp );
    
      unless( $first_letters =~ /\Q$temp/i ) { 
          $first_letters = $first_letters.$temp; # Quote metacharacters
      }
      
  }


  #print ( "$first_letters\n" );
  my @exist_letters;
  for my $letter ('A' .. 'Z'){
      if( $first_letters =~ /$letter/i ) {
	  @exist_letters = ( @exist_letters, $letter );
      }
  }
  $sth->finish;
  return @exist_letters;
};

sub get_mutant_symbol {
  my ($self, $id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_mutant_symbol" unless defined $id;
  my $sth = $dbh->prepare("SELECT mutant_symbol FROM mutant WHERE mutant_id=$id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $symbol = $sth->fetchrow_array;
  # look nice (to display canonical mutant on the web page)
  #$symbol =~ s/^(\D+)/<i>$1\<\/i>/;
  $sth->finish;
  return $symbol;
};

sub get_mutant_synonym {
  my ($self, $id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_mutant_synonym" unless defined $id;
  my $sth = $dbh->prepare("SELECT mutant_synonym FROM mutant_synonym WHERE mutant_id=$id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $synonym = $sth->fetchrow_array;
  $sth->finish;
  return $synonym;  
};

sub get_image_path {
  my ($self, $id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_image_path" unless defined $id;
  my $sth = $dbh->prepare("SELECT image_path FROM core_mutant WHERE mutant_id=$id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $path = $sth->fetchrow_array;
  $sth->finish;
  return $path; 
};

sub get_phenotypic_description {
  my ($self, $id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_phenotypic_description" unless defined $id;
  my $sth = $dbh->prepare("SELECT mutant_description FROM mutant WHERE mutant_id=$id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $phenotype = $sth->fetchrow_array;
  $sth->finish;
  return $phenotype; 
};

sub get_general_mutant_info {
  my ($self, $query_table, $query_column, $object_table, $object_id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_general_mutant_info" unless defined $object_id;
  my $sth = $dbh->prepare("SELECT $query_column FROM $query_table WHERE object_id = $object_id AND object_table = '$object_table'") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $info = $sth->fetchrow_array;
  $sth->finish;
  return $info; 
};

sub get_general_object_info {
  my ($self, $query_table, $query_column, $object_table, $object_id) = @_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_general_object_info" unless defined $object_id;
  my $sth = $dbh->prepare("SELECT $query_column FROM $query_table WHERE object_id = $object_id AND object_table = '$object_table'") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $info = $sth->fetchrow_array;
  $sth->finish;
  return $info; 
};

#########################################################################
# return image names
sub get_mutant_images {
  my ($self, $mutant_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_images" unless defined $mutant_id;
#  my $sth = $dbh->prepare
#    ("SELECT file_name 
#      FROM image 
#      WHERE image_id IN (SELECT image_id 
#                         FROM object_to_image 
#                         WHERE object_id = $mutant_id AND 
#                               object_table = 'mutant')") || die $dbh->errstr;
#

  my $sth = $dbh->prepare
     ("SELECT file_name 
       FROM image,object_to_image 
       WHERE image.image_id = object_to_image.image_id 
         AND object_table = 'mutant'
         AND object_id = ?
       ORDER BY image.image_id 
     ")|| die $dbh->errstr;

  $sth->bind_param(1,$mutant_id);

  $sth->execute || die "Execute: ", $sth->errstr;
  my @image_names = ();
  while(my ($row) = $sth->fetchrow_array){
    push @image_names, $row;
  }
  $sth->finish;
  return @image_names; 
};

sub get_mutant_images_and_image_comments {
  my ($self, $mutant_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_images" unless defined $mutant_id;
#  my $sth = $dbh->prepare
#    ("SELECT file_name, image_comment 
#      FROM image 
#      WHERE image_id IN (SELECT image_id 
#                         FROM object_to_image 
#                         WHERE object_id = $mutant_id AND 
#                               object_table = 'mutant')") || die $dbh->errstr;

  my $sth = $dbh->prepare
       ("SELECT file_name, image_comment 
         FROM image,object_to_image
         WHERE image.image_id = object_to_image.image_id
          AND  object_table = 'mutant'
          AND  object_id = ?
          ORDER BY image.image_id
        ") || die $dbh->errstr;

  $sth->bind_param(1,$mutant_id);   

  $sth->execute || die "Execute: ", $sth->errstr;
  my @images = ();
  while(my ($file_name, $image_comment) = $sth->fetchrow_array){
    my $row = join("\t", $file_name, $image_comment);
    push @images, $row;
  }
  $sth->finish;
  return @images; 
};


# return an array with corresponding phenotype expressions (composite ids)
sub get_mutant_phenotypes {
  my ($self, $mutant_id, $sort_by ) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_phenotypes" unless defined $mutant_id;
  my $sth;

  if( $sort_by eq "dev_stage" ) {
#      $sth = $dbh->prepare ("SELECT * 
#                             FROM phenotype_expression 
#                             WHERE phenotype_expression_id IN 
#                                   (SELECT phenotype_expression_id 
#                                    FROM object_to_phenotype_expression 
#                                    WHERE object_id = $mutant_id AND 
#                                    object_table = 'mutant')
#                             order by DEVELOPMENTAL_STAGE_ID, TO_ID, ANATOMY_ID, GO_ID") || die $dbh->errstr;

   $sth = $dbh->prepare ("SELECT PH.phenotype_expression_id, 
                                 to_id,
                                 developmental_stage_id,
                                 anatomy_id,
                                 go_id,
                                 trait_score
                           FROM phenotype_expression PH,object_to_phenotype_expression OPH
                           WHERE PH.phenotype_expression_id = OPH.phenotype_expression_id
                             AND  object_table = 'mutant'
                             AND  object_id = ?
                             ORDER BY developmental_stage_id,to_id,anatomy_id,go_id
                         ") || die $dbh->errstr;

  } else {
#      $sth = $dbh->prepare ("SELECT * 
#                             FROM phenotype_expression 
#                             WHERE phenotype_expression_id IN 
#                                   (SELECT phenotype_expression_id 
#                                   FROM object_to_phenotype_expression 
#                                   WHERE object_id = $mutant_id AND 
#                                   object_table = 'mutant')
#                                   order by TO_ID, DEVELOPMENTAL_STAGE_ID, ANATOMY_ID, GO_ID") || die $dbh->errstr;
 
    $sth = $dbh->prepare ("SELECT PH.phenotype_expression_id, 
                                 to_id,
                                 developmental_stage_id,
                                 anatomy_id,
                                 go_id,
                                 trait_score
                           FROM phenotype_expression PH,object_to_phenotype_expression OPH
                           WHERE PH.phenotype_expression_id = OPH.phenotype_expression_id
                             AND  object_table = 'mutant'
                             AND  object_id = ? 
                             ORDER BY to_id, developmental_stage_id,anatomy_id,go_id
                         ") || die $dbh->errstr;

 }

  $sth->bind_param(1, $mutant_id);

  $sth->execute || die "Execute: ", $sth->errstr;
 
   my @phenotypes = ();
  #($phenotype_id, $to_id, $growth_stage, $anatomy, $go_id, $trait_score)
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t",map{defined $_ ? $_ :''} @fields);  
    push @phenotypes, $row;
  }

  $sth->finish;
  return @phenotypes; 
};


sub get_object_phenotypes {
  my ($self, $obj_type, $obj_id ) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_object_phenotypes" unless (defined $obj_id && $obj_type);
  my $sth;
  $sth = $dbh->prepare(q[
    SELECT term_accession, term_type FROM ontology_association
    WHERE  object_id = ? 
    AND    object_table = ?
  ]);

  $sth->execute(($obj_id, $obj_type)) or die $sth->errstr;
  my %phenotypes;
  while(my ($term_acc, $term_type) = $sth->fetchrow_array){
    push @{$phenotypes{$term_type}}, $term_acc;
  }
  $sth->finish;

  return \%phenotypes;


}

sub get_allele_phenotypes{
  my($self,$allele_id)=@_;
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_phenotype" unless defined $allele_id;
#  my $sth = $dbh->prepare("SELECT * 
#                             FROM phenotype_expression 
#                             WHERE phenotype_expression_id IN 
#                                   (SELECT phenotype_expression_id 
#                                   FROM allele_to_phenotype_expression 
#                                   WHERE allele_id = ? 
#                                   )
#                           ")  || die $dbh->errstr;

  my $sth = $dbh->prepare ("SELECT PH.phenotype_expression_id, 
                                 to_id,
                                 developmental_stage_id,
                                 anatomy_id,
                                 go_id,
                                 trait_score
                           FROM phenotype_expression PH,allele_to_phenotype_expression APH                           
                           WHERE PH.phenotype_expression_id = APH.phenotype_expression_id
                           AND   allele_id = ?
                          ")  || die $dbh->errstr;



  $sth->bind_param(1,$allele_id);
  $sth->execute || die "Execute: ", $sth->errstr;
 
 
  my @phenotypes = ();
  #($phenotype_id, $to_id, $growth_stage, $anatomy, $go_id, $trait_score)
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t",map{defined $_ ? $_ :''} @fields);  
    push @phenotypes, $row;
  }
 
  $sth->finish;
  return @phenotypes; 

}

#######################################################################
# sub functions for study
#######################################################################
sub get_study_name{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_name" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT study_name
      FROM study 
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $study_name = $sth->fetchrow_array;    
  $sth->finish;
  return $study_name;
};

sub get_study_year{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_year" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT year
      FROM study 
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $year = $sth->fetchrow_array;    
  $sth->finish;
  return $year;
};

sub get_study_season{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_season" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT season
      FROM study 
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $season = $sth->fetchrow_array;    
  $sth->finish;
  return $season;
};

sub get_study_location{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_location" unless defined $study_id;
#  my $sth = $dbh->prepare
#    ("SELECT location_name
#      FROM geographical_location 
#      WHERE geographical_location_id = (SELECT geographical_location_id
#                                         FROM study
#                                         WHERE study_id = $study_id)")
#                                         || die $dbh->errstr;
  
  my $sth = $dbh->prepare
       ("SELECT location_name
         FROM geographical_location GL,study
         WHERE GL.geographical_location_id = study.geographical_location_id
           and study_id = ?
        ")  || die $dbh->errstr;
 
  $sth->bind_param(1,$study_id); 


  $sth->execute || die "Execute: ", $sth->errstr;
  my $location_name = $sth->fetchrow_array;    
  $sth->finish;
  return $location_name;
};

sub get_study_type{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_type" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT study_type
      FROM study 
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $study_type = $sth->fetchrow_array;    
  $sth->finish;
  if($study_type == 0) { 
    return STUDY_TYPE_0; 
  } 

  if($study_type == 1) {
    return STUDY_TYPE_I;
  }
  elsif($study_type == 2){
    return STUDY_TYPE_II;
  }
  elsif($study_type == 3){
    return STUDY_TYPE_III;
  }
  elsif($study_type == 4){
    return STUDY_TYPE_IV;
  }
  
  return $study_type;
};

sub get_study_environment{
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_environment" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT environmental_factors
      FROM study
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $environment = $sth->fetchrow_array;    
  $sth->finish;
  return $environment;
};

# return an array with corresponding allele studies
sub get_study_to_mutants {
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study_to_mutants" unless defined $study_id;
#  my $sth = $dbh->prepare
#    ("SELECT object_id 
#      FROM object_to_allele 
#      WHERE object_table = 'mutant' 
#       AND allele_id IN (SELECT object_id 
#                         FROM object_to_study 
#                         WHERE study_id = $study_id AND 
#                               object_table = 'allele')") || die $dbh->errstr;
 

 my $sth = $dbh->prepare
      ("SELECT OA.object_id
        FROM object_to_allele OA,object_to_study OS 
        WHERE allele_id = OS.object_id
          AND OA.object_table = 'mutant' 
          AND OS.object_table = 'allele'
          AND study_id = ?
       ");

 $sth->bind_param(1,$study_id); 


 $sth->execute || die "Execute: ", $sth->errstr;
  
  my %mutant_ids;
  while(my ($row) = $sth->fetchrow_array){
    $mutant_ids{$row} = 1;
  }

  $sth->finish;
  return keys %mutant_ids;
};


# return an array with corresponding mutant studies
sub get_study {
  my ($self, $study_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_study" unless defined $study_id;
  my $sth = $dbh->prepare
    ("SELECT * 
      FROM study 
      WHERE study_id = $study_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
 

   my $study = '';
   #($study_id,$study_name, $year, $season, $geographical_location_id, $study_type, $environmental_factors)
  my @fields = $sth->fetchrow_array;  
  $study = join("\t",map {defined $_? $_:''} @fields);

   
  $sth->finish;
  return $study; 
};

# return an array with corresponding allele studies
sub get_allele_studies {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_studies" unless defined $allele_id;

#  my $sth = $dbh->prepare
#    ("SELECT * 
#      FROM study 
#      WHERE study_id IN (SELECT study_id 
#                         FROM object_to_study 
#                         WHERE object_id = $allele_id AND 
#                               object_table = 'allele')") || die $dbh->errstr;
#
   
  my $sth = $dbh->prepare
    ("SELECT S.study_id,
             study_name,
             year, 
             season, 
             geographical_location_id, 
             study_type, 
             environmental_factors
      FROM   study S,  object_to_study OS
      WHERE  S.study_id =OS.study_id 
       AND   object_table = 'allele'
       AND   object_id = ?              
     ") || die $dbh->errstr;
 
   $sth->bind_param(1,$allele_id);

  $sth->execute || die "Execute: ", $sth->errstr;
   my @studies = ();
  # $study_id, $study_name, $year, $season, $geographical_location_id, $study_type, $environmental_factors
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t",map {defined $_?$_:'' }@fields); 
    push @studies, $row;
  }


  $sth->finish;
  return @studies; 
};


# return an array with corresponding mutant studies
sub get_mutant_studies {
  my ($self, $mutant_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_mutant_studies" unless defined $mutant_id;
#  my $sth = $dbh->prepare
#    ("SELECT * 
#      FROM study 
#      WHERE study_id IN (SELECT study_id 
#                         FROM object_to_study 
#                         WHERE object_id = $mutant_id AND 
#                               object_table = 'mutant')") || die $dbh->errstr;



   my $sth = $dbh->prepare
    ("SELECT S.study_id,
             study_name,
             year, 
             season, 
             geographical_location_id, 
             study_type, 
             environmental_factors
      FROM   study S,  object_to_study OS
      WHERE  S.study_id =OS.study_id 
       AND   object_table = 'mutant'
       AND   object_id = ?              
     ") || die $dbh->errstr;


  $sth->bind_param(1, $mutant_id);


  $sth->execute || die "Execute: ", $sth->errstr;
 my @studies = ();
  #($study_id, $study_name, $year, $season, $geographical_location_id, $study_type, $environmental_factors)
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t", map {defined $_?$_:''} @fields); 
    push @studies, $row;
  }

 
  $sth->finish;
  return @studies; 
};

#########################################################################
# return an array with corresponding alleles
#########################################################################
sub get_alleles {
  my ($self, $table_name, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_alleles" unless defined $object_id;
#  my $sth = $dbh->prepare
#    ("SELECT * 
#      FROM allele 
#      WHERE allele_id IN (SELECT allele_id 
#                          FROM object_to_allele 
#                          WHERE object_id = $object_id AND 
#                                object_table = '$table_name')") || die $dbh->errstr;

 my $sth = $dbh->prepare
    ("SELECT  A.allele_id,
              allele_designation,
              allele_description,
              allele_interaction_description
        FROM allele A, object_to_allele OA
        WHERE A.allele_id = OA.allele_id
        AND   object_id = ?
        AND   object_table = ?
      ") || die $dbh->errstr;
  
 $sth->bind_param(1,$object_id);
 $sth->bind_param(2,$table_name);


  $sth->execute || die "Execute: ", $sth->errstr;
   my @alleles = ();
  #($allele_id, $allele_designation, $allele_description, $allele_interaction_description)
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t", map {defined $_?$_:''} @fields); 
    push @alleles, $row;
  }  

  $sth->finish;
  return @alleles; 
};

#########################################################################
# functions for allele
#########################################################################
sub get_allele {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele" unless defined $allele_id;
  my $sth = $dbh->prepare
    ("SELECT * 
      FROM allele 
      WHERE allele_id = ?") || die $dbh->errstr;
  $sth->execute(($allele_id)) || die "Execute: ", $sth->errstr;
   my $allele = '';
  #($db_allele_id, $allele_designation, $allele_description, $allele_interaction_description)
  my @fields = $sth->fetchrow_array;
  $allele = join("\t",map {defined $_?$_:''} @fields);   

 
  $sth->finish;

  return $allele;
};

sub get_allele_designation {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_designation" unless defined $allele_id;
  my $sth = $dbh->prepare
    ("SELECT allele_designation 
      FROM allele 
      WHERE allele_id = $allele_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $allele_designation = $sth->fetchrow_array;
  $sth->finish;
  return $allele_designation; 
};

sub get_allele_description {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_description" unless defined $allele_id;
  my $sth = $dbh->prepare
    ("SELECT allele_description 
      FROM allele 
      WHERE allele_id = $allele_id") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $allele_description = $sth->fetchrow_array;
  $sth->finish;
  return $allele_description; 
};

#######for allele interaction, added by liya
sub get_allele_interaction_description {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele" unless defined $allele_id;
  my $sth = $dbh->prepare
    ("SELECT allele_interaction_description 
      FROM allele 
      WHERE allele_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$allele_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $allele_interaction_description = $sth->fetchrow_array;
  $sth->finish;
  return $allele_interaction_description;
};

sub get_allele_to_mutants {
  my ($self, $allele_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_to_mutants" unless defined $allele_id;
  my $sth = $dbh->prepare
    ("SELECT object_id 
      FROM object_to_allele
      WHERE allele_id = $allele_id
            AND object_table = 'mutant'") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  
  my @mutant_ids;
  while(my ($row) = $sth->fetchrow_array){
    push @mutant_ids, $row;
  }

  $sth->finish;
  return @mutant_ids;
};

sub get_allele_synonym { 
  my ($self, $allele_id) = @_;  
  my $dbh = $self->{'db'}; 
  die "Invalid input in get_allele_synonym" unless defined $allele_id; 
  my $sth = $dbh->prepare("SELECT allele_synonym FROM allele_synonym WHERE allele_id = $allele_id") || die $dbh->errstr; 
  $sth->execute || die "Execute: ", $sth->errstr; 
  my $allele_synonym = $sth->fetchrow_array; 
  $sth->finish; 
  return $allele_synonym;  
};

sub get_allele_name { 
  my ($self, $allele_id) = @_;  
  my $dbh = $self->{'db'}; 
  die "Invalid input in get_allele_name" unless defined $allele_id; 
  my $sth = $dbh->prepare("SELECT allele_name FROM allele WHERE allele_id = $allele_id") || die $dbh->errstr; 
  $sth->execute || die "Execute: ", $sth->errstr; 
  my $allele_name = $sth->fetchrow_array; 
  $sth->finish; 
  return $allele_name;  
};

sub get_allele_interaction { 
  my ($self, $allele_id) = @_;  
  my $dbh = $self->{'db'}; 
  die "Invalid input in get_allele_interaction" unless defined $allele_id; 
 
  my $sth = $dbh->prepare("SELECT interacting_allele_id,interaction_type,interaction_description  FROM allele_interaction WHERE allele_id = $allele_id") || die $dbh->errstr; 
  $sth->execute || die "Execute: ", $sth->errstr; 
  my @allele_interactions; 
 while(my ($inter_allele_id, $interaction_type, $interaction_description)= $sth->fetchrow_array){ 
    my $row = join("\t",$inter_allele_id, $interaction_type, $interaction_description );  
    push @allele_interactions, $row; 
  } 
 $sth->finish; 
  return @allele_interactions;  
}; 
#########################################################################
# functions for germplasms
#########################################################################
sub get_germplasms {
  my ($self, $table_name, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasms" unless defined $object_id;

#  my $sth=$dbh->prepare("SELECT * 
#      FROM germplasm_info 
#      WHERE germplasm_info_id IN
#                         (SELECT germplasm_info_id 
#                          FROM object_to_germplasm_info 
#                          WHERE object_id = ? AND 
#                                object_table = ?)") || die $dbh->errstr;

  
  my $sth=$dbh->prepare("SELECT G.germplasm_info_id,
                           accession,
                           geographical_location_id,
                           mutagenesis_id,
                           wild_type,
                           genus,
                           species_id,
                           subspecies_id
                         FROM germplasm_info G,object_to_germplasm_info OG
                         WHERE G.germplasm_info_id = OG.germplasm_info_id
                           AND object_id = ?
                           AND object_table = ?
                        ") || die $dbh->errstr;


  $sth->bind_param(1,$object_id);
  $sth->bind_param(2,$table_name);

  $sth->execute || die "Execute: ", $sth->errstr;
  my @germplasms = ();
  
  #($germplasm_info_id, $accession, $geographical_location_id, $mutagenesis_id, $wild_type, $genus, $species_id, $subspecies_id,$germplasm_name,$study_type,$genetic_background)
  while(my @fields = $sth->fetchrow_array){
     #for null field, return '' to avoid warning message
    my $row = join("\t",map{defined $_ ? $_ :''} @fields);
    push @germplasms, $row;
  }
  $sth->finish;
  return @germplasms;  
};

## on allele level
sub al_get_germplasms {
  my ($self, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasms" unless defined $object_id;

  my $sth=$dbh->prepare("SELECT * 
      FROM germplasm_info 
      WHERE germplasm_info_id IN
                         (SELECT germplasm_info_id 
                          FROM allele_to_germplasm_info 
                          WHERE allele_id = ? )") || die $dbh->errstr;



  $sth->bind_param(1,$object_id);
 
  $sth->execute || die "Execute: ", $sth->errstr;
   my @germplasms = ();

  #($germplasm_info_id, $accession, $geographical_location_id, $mutagenesis_id, $wild_type, $genus, $species_id, $subspecies_id,$germplasm_name,$study_type,$genetic_background) = $sth->fetchrow_array)
 
  while(my @fields = $sth->fetchrow_array){
    my $row = join("\t",map{defined $_ ? $_ :''} @fields ); 
    push @germplasms, $row;
  }

 
  $sth->finish;
  return @germplasms;  
};

sub get_germplasm {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT * 
      FROM germplasm_info 
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  #($germplasm_id, $accession, $geographical_location_id, $mutagenesis_id, $wild_type, $genus, $species_id, $subspecies_id,$germplasm_name,$study_type,$genetic_background)
  my @fields = $sth->fetchrow_array;
  my $germplasm = join("\t", map {defined $_? $_:''} @fields); 
  
  $sth->finish;
  return $germplasm;  
};

sub get_germplasm_accession {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_accession" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT accession
      FROM  germplasm_info
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $accession = $sth->fetchrow_array;    
  $sth->finish;
  return $accession;
};

sub get_germplasm_name {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_name" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT germplasm_name
      FROM  germplasm_info
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $name = $sth->fetchrow_array;    
  $sth->finish;
  return $name;
};

sub get_germplasm_study_type {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_study_type" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT study_type
      FROM  germplasm_info
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $study_type = $sth->fetchrow_array;    
  $sth->finish;
  if(defined $study_type){
  if($study_type == 0) { 
    return STUDY_TYPE_0; 
  } 

  if($study_type == 1) {
    return STUDY_TYPE_I;
  }
  elsif($study_type == 2){
    return STUDY_TYPE_II;
  }
  elsif($study_type == 3){
    return STUDY_TYPE_III;
  }
  elsif($study_type == 4){
    return STUDY_TYPE_IV;
  }
  return $study_type;
}
};

sub get_germplasm_background {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_background" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT genetic_background
      FROM  germplasm_info
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $name = $sth->fetchrow_array;    
  $sth->finish;
  return $name;
};

sub get_germplasm_location {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_location" unless defined $germplasm_info_id;

  #comment out because it works only for oracle
  #my $sth = $dbh->prepare
  #  ("SELECT location_name
  #    FROM geographical_location
  #    WHERE geographical_location_id = (SELECT geographical_location_id
  #                                       FROM  germplasm_info
  #                                       WHERE germplasm_info_id = ?)")
  #                                       || die $dbh->errstr;
  
  my $sth = $dbh->prepare
         ("SELECT location_name
           FROM geographical_location,germplasm_info
           WHERE  geographical_location.geographical_location_id
                  = germplasm_info.geographical_location_id
                 AND germplasm_info_id = ? "
          )           
          || die $dbh->errstr;

  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $location_name = $sth->fetchrow_array;    
  $sth->finish;
  return $location_name;
};

sub get_germplasm_wild_type {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_wild_type" unless defined $germplasm_info_id;
  my $sth = $dbh->prepare
    ("SELECT wild_type
      FROM  germplasm_info
      WHERE germplasm_info_id = ?") || die $dbh->errstr;
   $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my $wild_type = $sth->fetchrow_array;    
  $sth->finish;
  return $wild_type;
};

sub get_germplasm_mutagenesis {
  my ($self, $germplasm_info_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_mutagenesis" unless defined $germplasm_info_id;
#comment out sql with subquery 
#  my $sth = $dbh->prepare
    ("SELECT mutagen, mutagenesis_method
      FROM  mutagenesis
      WHERE mutagenesis_id = (SELECT mutagenesis_id
                                         FROM  germplasm_info
                                         WHERE germplasm_info_id = ?)")
                                         || die $dbh->errstr;
  my $sth = $dbh->prepare
       ("SELECT mutagen, mutagenesis_method
         FROM  mutagenesis,germplasm_info
         WHERE mutagenesis.mutagenesis_id = germplasm_info.mutagenesis_id
            AND germplasm_info_id = ?
        ") || die $dbh->errstr;


  $sth->bind_param(1,$germplasm_info_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my ($mutagen, $mutagenesis_method) = $sth->fetchrow_array;    
  $sth->finish;
  return ($mutagen, $mutagenesis_method);  
};


# return an array with corresponding allele studies
sub get_germplasm_to_objects {
  my ($self, $germplasm_id, $object_table) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_germplasm_to_mutants"
  unless (defined $germplasm_id  && defined $object_table);
  my $sth = $dbh->prepare
    ("SELECT object_id 
      FROM object_to_germplasm_info 
      WHERE object_table = ? 
       AND germplasm_info_id=?") || die $dbh->errstr;
  
  $sth->bind_param(1,$object_table);
  $sth->bind_param(2,$germplasm_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  
  my @obj_ids;
  while(my ($row) = $sth->fetchrow_array){
    push @obj_ids, $row;
  }

  $sth->finish;
  return @obj_ids;
};




#################################################################################
#
#################################################################################
# return developmental stage name
sub get_developmental_stage {
  my ($self, $stage_id) = @_;
  my $name;
  if( $stage_id eq "NA" ) {
      $name = "NA";
  } else {
      my $dbh = $self->{'db'};
      die "Invalid input in get_developmental_stage" unless defined $stage_id;
      my $sth = $dbh->prepare
	        ("SELECT description 
                  FROM developmental_stage 
                  WHERE developmental_stage_id = ?") || die $dbh->errstr;
      $sth->bind_param( 1, $stage_id );
      $sth->execute || die "Execute: ", $sth->errstr;
      $name = $sth->fetchrow_array;
      $sth->finish;
  }
  return $name;  
};


# return its term name
sub get_term_name {
  my ($self, $accession) = @_;
  $accession =~s/^\s+|\s+$//g; 
  die "Invalid input in get_trait_name" unless defined $accession;
  my $ot_dbh = $self->{'ontology_db'};
 
  return 'NA' unless $accession =~/\d+/;

  my $sth = $ot_dbh->prepare
    ("SELECT term_name 
      FROM term 
      WHERE term_accession  = ?") || die $ot_dbh->errstr;
  $sth->bind_param( 1, $accession );
  $sth->execute || die "Execute: ", $sth->errstr;
  my $name = $sth->fetchrow_array;
  $sth->finish;
  return $name;  
};

sub get_oryza_linkout {
  my ($self, $mutant_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_oryza_linkout" unless defined $mutant_id;
  my $sth = $dbh->prepare
    ("SELECT url_link
      FROM object_to_linkout
      WHERE object_id = $mutant_id AND
            object_table = 'mutant'") || die $dbh->errstr;


  $sth->execute || die "Execute: ", $sth->errstr;
  my $linkout = $sth->fetchrow_array;
  $sth->finish;
  return $linkout; 
};

### ???? change gene_identity_accesion to gene_identity_accession
# gene identities
sub get_gene_identities {
  my ($self, $table_name, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_gene_identities" unless defined $object_id;
  my $sth = $dbh->prepare
    ("SELECT gene_identity_accession
      FROM object_to_gene_identity
      WHERE object_id = $object_id AND
            object_table = '$table_name'
      ORDER BY gene_identity_accession") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my @gene_identities = ();
  while(my ($row) = $sth->fetchrow_array){
    push @gene_identities, $row;
  }
  $sth->finish;
  return @gene_identities; 
};

# gene products
sub get_gene_products {
  my ($self, $table_name, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_gene_products" unless defined $object_id;
  my $sth = $dbh->prepare
    ("SELECT gene_product_accession
      FROM object_to_gene_product
      WHERE object_id = $object_id AND
            object_table = '$table_name'
      ORDER BY gene_product_accession") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my @gene_products = ();
  while(my ($row) = $sth->fetchrow_array){
    push @gene_products, $row;
  }
  $sth->finish;
  return @gene_products; 
};

### ???? can it be more than one 
# evidence codes
sub get_evidence_codes {
  my ($self, $table_name, $object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_evidence_codes" unless defined $object_id;
  my $sth = $dbh->prepare
    ("SELECT evidence_code
      FROM object_to_evidence_code
      WHERE object_id = $object_id AND
            object_table = '$table_name'") || die $dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my @evidence_codes = ();
  while(my ($row) = $sth->fetchrow_array){
    push @evidence_codes, $row;
  }
  $sth->finish;
  return @evidence_codes; 
};

sub get_allele_evidence_codes{
 my ($self,$object_id) = @_; 
  my $dbh = $self->{'db'};
  die "Invalid input in get_allele_evidence_codes" unless defined $object_id;
  my $sth = $dbh->prepare
    ("SELECT evidence_code
      FROM allele_to_evidence_code
      WHERE allele_id = ?") || die $dbh->errstr;
  $sth->bind_param(1,$object_id);
  $sth->execute || die "Execute: ", $sth->errstr;
  my @evidence_codes = ();
  while(my ($row) = $sth->fetchrow_array){
    push @evidence_codes, $row;
  }
  $sth->finish;
  return @evidence_codes; 

}


# references
# get the reference ids for this $id in this $table_name
sub get_reference_ids {
  my ($self, $xref_dbname, $id) = @_;
 
  die "Invalid input in get_reference_ids" unless defined $id;
  #my $dbh = $self->{'db'};
 
  my $liter_dbh = $self->{'literature_db'};

  my $sth = $liter_dbh->prepare("
	      SELECT distinct row_id 
	      FROM dbxref DX, objectxref OX
	      WHERE DX.dbxref_id = OX.dbxref_id
	      AND  xref_dbname = ? AND xref_key = ?
	     ") || die $liter_dbh->errstr;
  $sth->bind_param( 1, $xref_dbname );
  $sth->bind_param( 2, $id );
  $sth->execute || die "Execute: ", $sth->errstr;
 
  my $reference_ids_ref = $sth->fetchall_arrayref();
  $sth->finish();
  return $reference_ids_ref;
 
  
}


# one function fits all
sub get_reference_info {
  my ($self, $column_name, $ref_id) = @_;
 
  die "Invalid input in get_reference_info" unless defined $ref_id;
  #my $dbh = $self->{'db'};
    my $liter_dbh = $self->{'literature_db'};

  my $sth = $liter_dbh->prepare("SELECT $column_name FROM reference WHERE reference_id = $ref_id") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $info = $sth->fetchrow_array;
  $sth->finish;
  return $info; 
}

sub get_reference_source {
  my ($self, $ref_id) = @_;
  die "Invalid input in get_reference_source" unless defined $ref_id;
  #my $dbh = $self->{'db'};

  #my $sth = $liter_dbh->prepare("SELECT source_name FROM source WHERE source_id IN (SELECT source_id FROM reference WHERE reference_id = $ref_id)") || die $liter_dbh->errstr;
    my $liter_dbh = $self->{'literature_db'};


   my $sth = $liter_dbh->prepare("SELECT source.source_name FROM reference, source WHERE reference.reference_id = $ref_id AND reference.source_id = source.source_id") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $source_name = $sth->fetchrow_array;
  $sth->finish;
  return $source_name; 
}

sub get_reference_source_trivia {
  my ($self, $ref_id) = @_;
  #my $dbh = $self->{'db'};
  die "Invalid input in get_reference_source_trivia" unless defined $ref_id;

    my $liter_dbh = $self->{'literature_db'};

   my $sth = $liter_dbh->prepare("SELECT year FROM reference WHERE reference_id = $ref_id") || die $liter_dbh->errstr;

  $sth->execute || die "Execute: ", $sth->errstr;
  my $source_trivia;
  my $tmp = $sth->fetchrow_array;
  if($tmp ne ""){
    $source_trivia = $tmp . "; ";
  }
  
  $sth = $liter_dbh->prepare("SELECT volume FROM reference WHERE reference_id = $ref_id") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  $tmp = $sth->fetchrow_array;
  if($tmp && $tmp ne ""){
    $source_trivia .= $tmp . ": ";
  }
  
  $sth = $liter_dbh->prepare("SELECT start_page FROM reference WHERE reference_id = $ref_id") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  $tmp = $sth->fetchrow_array;
  if($tmp && $tmp ne ""){
    $source_trivia .= $tmp . "-";
  }
  
 $sth = $liter_dbh->prepare("SELECT end_page FROM reference WHERE reference_id = $ref_id") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  $tmp = $sth->fetchrow_array;
  if($tmp && $tmp ne ""){
    $source_trivia .= $tmp . ".";
  }

  $sth->finish;
  return $source_trivia; 
}

sub get_reference_author {
  my ($self, $ref_id) = @_;
  #my $dbh = $self->{'db'};
  die "Invalid input in get_reference_author" unless defined $ref_id;
  # my $sth = $dbh->prepare("SELECT contributor_name FROM contributor WHERE contributor_id IN (SELECT contributor_id FROM author WHERE reference_id = $ref_id order by authorship_position)") || die $dbh->errstr;

  my $liter_dbh = $self->{'literature_db'};

  my $sth = $liter_dbh->prepare("SELECT contributor.contributor_name FROM contributor, author WHERE author.reference_id = $ref_id AND contributor.contributor_id = author.contributor_id order by authorship_position") || die $liter_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  
  #my $author_name = $sth->fetchrow_array;
  my $author_name;

  while(my ($row) = $sth->fetchrow_array){
    $author_name .= $row . ", ";
  }
  $author_name =~ s/,$/./;
  $sth->finish;
  return $author_name; 
}
### 

# post: return exact accesion id.  For example, GO:0005635
sub get_ontology_term_accession {
  my $self = shift;
  my $id = shift;
  #my $dbh  = $self->{'db'};
  die "Invalid input in GetAccessionID" unless defined $id;

  my $ot_dbh = $self->{'ontology_db'};
                                                                                            
  

  my $sth = $ot_dbh->prepare("SELECT term_accession FROM term WHERE term_id=$id") || die $ot_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $accession = $sth->fetchrow_array;
  $sth->finish;
  $sth = $ot_dbh->prepare("SELECT term_type_id FROM term WHERE term_id=$id") || die $ot_dbh->errstr;
  $sth->execute || die "Execute: ", $sth->errstr;
  my $exact_accession;
  if($sth->fetchrow_array == $TRAIT) {
     $exact_accession = $TO_PREFIX;
  }
  else {
    $exact_accession = $GO_PREFIX;
  }
  my $length = 7 - length($accession);
  for my $x (1 .. $length){
      $exact_accession .= "0";
  }
  $exact_accession .= $accession;
  $sth->finish;
  return $exact_accession;
}

# For the url using gene_symbol
sub get_id_by_symbol {
  my ( $self, $symbol ) = @_;
  $symbol =~ tr/a-z/A-Z/;
  my $dbh = $self->{'db'};
  die "Invalid input in get_id_by_symbol" unless defined $symbol;
  my $sth = $dbh->prepare("SELECT mutant_id FROM mutant WHERE upper(mutant_symbol) = ?") || die $dbh->errstr;
  $sth->bind_param( 1, $symbol );
  $sth->execute || die "Execute: ", $sth->errstr;
  my $id = $sth->fetchrow_array;
  
  $sth->finish;
  return $id;
};


sub get_cmap_feature{

  my($self, $mutant_id)=@_;
       die "Invalid input in get_cmap_feature" unless defined $mutant_id;
  my $dbh = $self->{'db'};
  my $sth = $dbh->prepare("SELECT cmap_feature_acc,cmap_map_set,cmap_map_name,start_position,stop_position,map_units FROM map_position where mutant_id= ?") or die $dbh->errstr;

   $sth->bind_param(1,$mutant_id);
   $sth->execute || die "Execute: ", $sth->errstr;
   my $featureRef = $sth->fetchall_arrayref();
   return $featureRef;
}


sub get_gis_on_ensembl_genome{
    my ($self) = @_;
    my $dbh = $self->{'db'};
    my %gis_on_embl;
    my $sth = $dbh->prepare(q[
	SELECT distinct gene_identity_accession
	FROM   object_to_gene_identity
	WHERE  is_on_ensembl_genome = 1
    ]);

    $sth->execute();

    while(my ($gi) = $sth->fetchrow_array){
	$gis_on_embl{$gi} = 1;
    }

    return \%gis_on_embl;
}

  
  
1;
