package Gramene::Mutant::MutantDB; 

=head1 NAME

Mutant::MutantDB

=head1 SYNOPSIS
  

=head1 DESCRIPTION

A module for querying mutant database  

=cut

# ----------------------------------------------------------------

use strict;
use DBI;

use Gramene::DB;


my $dbh;
my $ot_dbh;
my $liter_dbh;

my %term_types=(
                     2=>'GO Biological process',
                     3=>'GO Cellular component',
                     4=>'GO Molecular function',
                     5=>'Trait',
                     6=>'Anatomy',
                     7=>'Growth stage',
		     9=>'Environment'
       	        );


sub new {

    my $class = shift;
    my $self = { };
    bless($self, $class);
    return $self;

}


# destructor
sub DESTROY {
  my $self = shift;
 # $self->terminate_database;
}


sub terminate_database{
  my $self = shift;
  $dbh ->disconnect() if $dbh;
#  $ot_dbh ->disconnect() if $ot_dbh;

}


sub connect_to_db {

  eval{
    $dbh = Gramene::DB->new('mutant_phase1');
    $ot_dbh = Gramene::DB->new('ontology');
    $liter_dbh = Gramene::DB->new('literature');
  };
  if($@){
        die "DB connection failed:$@\n";
  }


}



sub get_all_species{

    my @species;
    my $sth = $dbh->prepare("SELECT species_id,common_name FROM mutant_species");
    $sth->execute or die $dbh->errstr;
    my($species_id,$common_name);
    $sth->bind_columns(\$species_id,\$common_name);

    while($sth->fetch){
      push @species,{'species_id'=>$species_id,'species_name'=>$common_name};
    }
   
    return \@species;
}


sub get_gene_id{

  my $acc = $_[1];
  
  $acc ="GR:00".$acc unless($acc =~/GR:00/);

  my ($gene_id) = $dbh->selectrow_array(
                                        q[
                                           SELECT gene_id FROM mutant_gene
                                           WHERE accession = ?
                                         ],
                           		 {},
                                         ($acc)
					);
 
  return $gene_id;

}

# -------------------------------------------------------------------

=pod

=head2 get_gene_general_info

Get the mutant gene's general information (accession,symbol,name,chromosome,description,curator_comment,species). The species is a species object.

Return a hash object

=cut

sub get_gene_general_info{
    my $gene_id= $_[1];
    

    my $sth = $dbh->prepare(
                            q[
                                          SELECT accession,
                                                 SY.synonym_name as symbol,
                                                 SN.synonym_name as name,
                                                 mutant_gene.species_id,
                                                 chromosome,
                                                 description,
                                                 curator_comment 
			                  FROM   mutant_gene
					  INNER JOIN mutant_synonym SY
					  ON     mutant_gene.symbol_id = SY.synonym_id
					  INNER JOIN mutant_synonym SN
					  ON     mutant_gene.name_id = SN.synonym_id
					  LEFT JOIN  mutant_chromosome
					  ON     mutant_gene.chromosome_id = mutant_chromosome.chromosome_id
                                          WHERE  gene_id =?
                                        ]
			   );



  
    my $sth1 = $dbh->prepare(
                             q[
                               SELECT * FROM mutant_species
                               WHERE species_id = ?
                              ]
			     );



    $sth->execute(($gene_id)) or die $dbh->errstr;


    my $gene_obj = $sth->fetchrow_hashref();

    my $species_id = $gene_obj->{'species_id'};
    
    $sth1->execute(($species_id)) or die $dbh->errstr;


    my $species_obj = $sth1->fetchrow_hashref();

    #delete the species_id
    delete $gene_obj->{'species_id'};
    
    #add the species obj to gene obj
    $gene_obj->{'species'}=$species_obj;


    $sth->finish;
    $sth1->finish;

    return $gene_obj;
}



# ----------------------------------------------------------------------
=pod

=head2 get_gene_synonyms

return an arrayRef of gene synonyms

=cut

sub get_gene_synonyms{
    my $gene_id = $_[1];
 
    my $sth = $dbh->prepare(
                             q[
                                SELECT synonym_name
                                FROM mutant_gene_synonym MS,mutant_synonym S 
                                WHERE MS.synonym_id = S.synonym_id
                                AND gene_id =?
			      ]
			    );
       
    $sth->execute(($gene_id)) or die $dbh->errstr;

    my @synonyms;

    while(my ($syn) = $sth->fetchrow_array){
	push @synonyms,$syn;
    }

    $sth->finish;
 
    return \@synonyms;
      
}



# ------------------------------------------------------------------------------
=pod

=head2 get_gene_ontology_association

Get the mutant gene's ontology associations.

Return a hash object, term_type as the keys and an array of term objects(term_accession,term_name) as value. 

=cut

sub get_gene_ontology_association{
    my $gene_id = $_[1];
    
    my $sth = $dbh->prepare(
                            q[
                               SELECT term_type,term_accession
                               FROM   mutant_gene_term_association
                               WHERE  gene_id =?
                               GROUP BY term_type, term_accession 
                             ]
			    );

    $sth->execute(($gene_id)) or die $dbh->errstr;


    my %terms;
     
    while(my ($type,$acc) = $sth->fetchrow_array){
	my $term_type = $term_types{$type}; 
        my $term_name = _get_term_name($acc,$type);
        $term_name ||= 'NA';   # for term not in ontology DB yet
        push( @{$terms{$term_type}},{'term_accession'=>$acc,'term_name'=>$term_name});
    } 

    return \%terms;
}



sub _get_term_name{

    my ($acc,$type) = @_;

    #$acc =~s/^\D+://;     # remove the prefix;

    #$type = 6 if $type == 7 ; # we don't have type 7(growth stage) 
                              # in ontology database yet
                              # count it as type 6 (PO) in ontology db   


    my ($term_name) = $ot_dbh->selectrow_array(
                                   q[
                                      SELECT term_name FROM term
                                      WHERE  term_accession = ?
                                       AND   term_type_id = ?
                                    ],
				    {},
                                    ($acc,$type)
                         	       );
                            
 
    return $term_name;


}


# --------------------------------------------------------------------------------
=pod

=head2 get_gene_references

Get the mutant gene's literature references.

Return an array of reference object.

=cut

sub get_gene_references{

   my $gene_id = $_[1]; 
   
   my $dbxref_name = 'Gramene.Literature';
   my $table_name = 'mutant.gene';

   my $dbxref_id = _get_dbxref_id($dbxref_name);
   
   my @references;

   my $sth = $dbh->prepare(
                     q[
                        SELECT dbxref_value
                        FROM   mutant_dbxref_to_object
                        WHERE  table_name = ?
                          AND  mutant_dbxref_id = ?
                          AND  record_id = ?
                        ORDER BY dbxref_value DESC
                      ]
			);

   $sth->execute(($table_name,$dbxref_id,$gene_id)) or die $dbh->errstr;

   while(my ($ref_id) = $sth->fetchrow_array()){
       my $ref_obj = _get_reference_detail($ref_id);
       push @references,$ref_obj;
   }                        
 
   $sth->finish;

   return \@references;

}


sub _get_dbxref_id {

   my $dbxref_name = shift;
       
   my ($dbxref_id) = $dbh->selectrow_array(
                                  q[
                                     SELECT dbxref_id FROM mutant_dbxref
                                     WHERE  dbxref_name = ?
				   ],
				   {},
                                   ($dbxref_name)
				 );

   return $dbxref_id;

} 


#
# get the reference's title,source_name,volume,start_page,end_page and author
#
sub  _get_reference_detail{

    my $ref_id = shift;
    
    my $ref_obj;    

    my $sth = $liter_dbh->prepare(
                        q[
                           SELECT reference_id, title,source_name,volume,
                                  start_page,end_page
                           FROM   reference, source
                           WHERE  reference.source_id = source.source_id
                           AND    reference_id = ?
                         ]
                     );    
    $sth->execute(($ref_id)) or die $liter_dbh->errstr;

    $ref_obj = $sth->fetchrow_hashref;

    my $sth2 = $liter_dbh->prepare(
                           q[
                              SELECT contributor.contributor_name 
                              FROM contributor, author 
                              WHERE author.reference_id = ? 
                              AND contributor.contributor_id = author.contributor_id 
                              order by authorship_position
                            ]
			       );

    $sth2->execute(($ref_id)) or die $liter_dbh->errstr;

    my $author_name;

    while(my ($row) = $sth2->fetchrow_array){
      $author_name .= $row . ", ";
    }

    $author_name =~ s/,$/./;       

    $ref_obj->{'author'} = $author_name;  

    $sth->finish;
 
    $sth2->finish;

    return $ref_obj;
   
}


#--------------------------------------------------------------------------------

=pod

=head2 get_gene_sequence_association

Get the mutant gene's sequence_associations('Nucleotide (DNA)','Protein')
The nucleotide sequence may come from genbank or other dbs. The protein association may come from Gramene protein db or other dbs.

Return a hash object with ('Nucleotide (DNA)','Protein') as keys and the values are also hash objects with dbname as key and an array of hype link values.

=cut


sub get_gene_sequence_association{
  
    my $gene_id = $_[1];

   
    my %sequence_association=();

   
    my $dbxref_name ='GenBank.Nucleotide';
    my $table_name = 'mutant.gene';    

    my @nucleotides = _get_dbxref_to_object($dbxref_name,$table_name,$gene_id);    

    
    if(@nucleotides){
       $dbxref_name = 'GenBank';
       $sequence_association{'Nucleotide (DNA)'}->{$dbxref_name}= \@nucleotides;
    } 
    
    $dbxref_name ='Gramene.Protein';
    my @proteins =  _get_dbxref_to_object($dbxref_name,$table_name,$gene_id);
    
    if(@proteins){
       $sequence_association{'Protein'}->{$dbxref_name} = \@proteins;   
    }

    return \%sequence_association;
   

}

#
#get an array of dbxref objects with hyper linked value
#
sub _get_dbxref_to_object{
  my ($dbxref_name,$table_name,$record_id) = @_;    
  
  my $sth = $dbh->prepare(
                         q[
                           SELECT url,dbxref_value
                           FROM   mutant_dbxref_to_object MDO, mutant_dbxref DBX
                           WHERE  MDO.mutant_dbxref_id = DBX.dbxref_id                                               AND  table_name = ?
                             AND  record_id = ?
                             AND  dbxref_name = ?
                           ]
                          );

  $sth->execute(($table_name,$record_id,$dbxref_name));

  my @dbxrefs;

  while(my ($url,$val) = $sth->fetchrow_array){
     $url =~ s/\Q[%?%]/$val/;
     my $link_url =  qq[<a href="$url">$val</a>];     

     push @dbxrefs,$link_url;

  }

  return @dbxrefs;

}

# -----------------------------------------------------------------------------------
=pod

=head2 get_gene_by_simple_fiels

Get the candidate gene objects by searching chromosome,accession fields,optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut

sub get_genes_by_simple_fields{

  my ($self,$query,$search_field,$species,$order_by)=@_;
  
  $query = uc($query);
  $query =~s/\*/%/g;

  my $sql=qq[
             SELECT gene_id, 
                    accession,
                    SY.synonym_name AS symbol,
                    SN.synonym_name AS name,
                    chromosome
               FROM mutant_gene 
	       INNER JOIN mutant_synonym SY
	       ON   mutant_gene.symbol_id = SY.synonym_id
	       INNER JOIN mutant_synonym SN
	       ON   mutant_gene.name_id = SN.synonym_id
	       LEFT JOIN mutant_chromosome
	       ON   mutant_chromosome.chromosome_id = mutant_gene.chromosome_id
            ];
 

  my $condition='';

  if($search_field eq 'chromosome' || $search_field eq 'accession'){
      $condition = " WHERE $search_field like ? ";
  }

  if($species){
       if($condition){	
	  $condition .= ' AND mutant_gene.species_id = ? ';
	}else{
	  $condition .= ' WHERE mutant_gene.species_id = ? ';

	}
  }

  if($order_by eq 'gene_symbol'){   
      $condition .= ' order by SY.synonym_name';
  }elsif($order_by eq 'gene_name'){
      $condition .= ' order by SN.synonym_name'; 
  }elsif($order_by eq 'chromosome'){
      $condition .= ' order by chromosome';
  }elsif($order_by eq 'accession'){
      $condition .= ' order by accession';
  }

 
  my $sth = $dbh->prepare("$sql"."$condition");

  $sth->bind_param(1,$query); 
  $sth->bind_param(2,$species) if $species;

  $sth->execute() or die $dbh->errstr;

  my @candidates;

  while(my $gene_obj = $sth->fetchrow_hashref){
      push @candidates, $gene_obj;
  }

  _get_gene_objs_synonyms(@candidates);

  _get_gene_objs_term_associations(@candidates);
 
  return @candidates;  

}

sub _get_species_id{

    my $species_name = $_[1];
    my ($species_id) = $dbh->selectrow_array(
                                             q[
                                                SELECT species_id FROM mutant_species
                                                WHERE common_name = ?
                                              ],
					     {},
                                             ($species_name)
					     );
    return $species_id;
    
}


sub _get_gene_objs_synonyms{

    my @gene_objs = @_;

    my $sth = $dbh->prepare(
                            q[
                                SELECT synonym_name 
                                FROM mutant_gene_synonym MGS, mutant_synonym MS
                                WHERE MGS.synonym_id = MS.synonym_id
                                AND gene_id = ?
                             ]
			    );
    
    foreach my $gene_obj (@gene_objs){
	my $gene_id = $gene_obj->{'gene_id'};
        $sth->execute(($gene_id)) or die $dbh->errstr;
        my @synonym_names;
        while(my ($name) = $sth->fetchrow_array){
	    push @synonym_names,$name;
        }
        my $synonym_name = join(", ",@synonym_names);
        $gene_obj->{'synonyms'} = $synonym_name;

    }

    $sth->finish;
}

# get gene objs' 'to','po','go' 
sub _get_gene_objs_term_associations{

    my @gene_objs = @_;
   
    my $sth_term = $ot_dbh->prepare(
                          q[
                             SELECT term_name FROM term
                             WHERE  term_accession = ?
                              AND   term_type_id = ?
                           ]
                      );

    my $sth = $dbh->prepare(
                            q[
                                SELECT term_accession,term_type
                                FROM   mutant_gene_term_association
                                WHERE  gene_id = ?
                             ]
			    );
    my %term_type_to_cat;
    $term_type_to_cat{2}='go';
    $term_type_to_cat{3}='go';
    $term_type_to_cat{4}='go';
    $term_type_to_cat{5}='to';
    $term_type_to_cat{6}='po';
    $term_type_to_cat{7}='gro';
    $term_type_to_cat{9}='eo';

    foreach my $gene_obj (@gene_objs){
	my $gene_id = $gene_obj->{'gene_id'};
        $sth->execute(($gene_id)) or die $dbh->errstr;
       
        my %term_cat_to_terms;
        while(my ($term_acc,$term_type) = $sth->fetchrow_array){
           #  $term_type = 6 if $term_type == 7; # for growth stage,no 7 in ontology db now 
             
              my $db_term_acc= $term_acc; 

            # $db_term_acc=~s/^\D+://; #remove the prefix before search ontology db

             $sth_term->execute(($db_term_acc,$term_type)) or die $dbh->errstr;
             my ($term_name) = $sth_term->fetchrow_array();
           
             $term_name ||= $term_acc; #those term not exist in DB    
        
             
             my $cat = $term_type_to_cat{$term_type};            
             push @{$term_cat_to_terms{$cat}},$term_name;
 
        }
        foreach my $cat (keys %term_cat_to_terms){
	    my $terms = join(", ",@{$term_cat_to_terms{$cat}});
            $gene_obj->{$cat} = $terms;
	}
        
        
    }

}

# -----------------------------------------------------------------------------------
=pod

=head2 get_gene_by_synonym

Get the candidate gene objects by searching symbol,name,synonym  fields,optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut

sub get_genes_by_synonym {
  my ($self,$query,$species,$order_by)=@_;

  $query = uc($query);  
  $query =~s/\*/%/g;
   
  my $sql=qq[
             SELECT mutant_gene.gene_id, 
                    accession,
                    SY.synonym_name AS symbol,
                    SN.synonym_name AS name,
                    chromosome
               FROM mutant_gene
	       INNER JOIN mutant_chromosome
	          ON      mutant_gene.chromosome_id = mutant_chromosome.chromosome_id
               INNER JOIN mutant_synonym SY
                  ON      mutant_gene.symbol_id = SY.synonym_id
               INNER JOIN mutant_synonym SN
                  ON      mutant_gene.name_id = SN.synonym_id
               LEFT  JOIN mutant_gene_synonym MGS 
                  ON      mutant_gene.gene_id = MGS.gene_id    
               INNER JOIN mutant_synonym SYN
                  ON      MGS.synonym_id = SYN.synonym_id
               WHERE      UPPER(SY.synonym_name) like ?
                  OR      UPPER(SN.synonym_name) like ?
                  OR      UPPER(SYN.synonym_name) like ?
           ];
	    

 
  my $condition = '';
  $condition .= ' AND mutant_gene.species_id = ? ' if $species;

  if($order_by eq 'gene_symbol'){   
      $condition .= ' order by SY.synonym_name';
  }elsif($order_by eq 'gene_name'){
      $condition .= ' order by SN.synonym_name'; 
  }elsif($order_by eq 'chromosome'){
      $condition .= ' order by chromosome';
  }elsif($order_by eq 'accession'){
      $condition .= ' order by accession';
  }

  my $sth = $dbh->prepare("$sql"."$condition");

  
  $sth->bind_param(1,$query); 
  $sth->bind_param(2,$query);
  $sth->bind_param(3,$query);
  $sth->bind_param(4,$species) if $species;

  $sth->execute() or die $dbh->errstr;

  my @candidates;
  my %uniq_gene_objs; #

  # since gene and synononym are 1 to many relationship, need get the uniqure gene obj
  while(my $gene_obj = $sth->fetchrow_hashref){
      my $gene_id = $gene_obj->{'gene_id'};
      push @candidates, $gene_obj unless $uniq_gene_objs{$gene_id}++;
  }
  

  _get_gene_objs_synonyms(@candidates);
 
  _get_gene_objs_term_associations(@candidates);
 
  return @candidates;  


}

sub get_genes_by_sequence{
  my ($self,$query,$species,$order_by,$dbxref_name)=@_;  
 
  $query = uc($query);  
  $query =~s/\*/%/g;

  
  my $sql = qq[
               SELECT gene_id,
                      accession,
                      SY.synonym_name AS symbol,
                      SN.synonym_name AS name,
                      chromosome
               FROM   mutant_gene 
	       INNER  JOIN mutant_synonym SY
	       ON     mutant_gene.symbol_id = SY.synonym_id
	       INNER  JOIN mutant_synonym SN
	       ON     mutant_gene.name_id = SN.synonym_id 
	       INNER  JOIN mutant_dbxref_to_object DTO
	       ON     gene_id = DTO.record_id
	       INNER  JOIN mutant_dbxref
	       ON     DTO.mutant_dbxref_id = dbxref_id
	       LEFT   JOIN mutant_chromosome
	       ON     mutant_gene.chromosome_id = mutant_chromosome.chromosome_id
	        WHERE dbxref_name = ? 
                AND   table_name = ?
                AND   UPPER(dbxref_value) like ?
           ];


    my $condition = '';
    $condition .= ' AND mutant_gene.species_id = ? ' if $species;

    if($order_by eq 'gene_symbol'){   
      $condition .= ' order by SY.synonym_name';
    }elsif($order_by eq 'gene_name'){
      $condition .= ' order by SN.synonym_name'; 
    }elsif($order_by eq 'chromosome'){
      $condition .= ' order by chromosome';
    }elsif($order_by eq 'accession'){
      $condition .= ' order by accession';
    }

    my $sth = $dbh->prepare("$sql"."$condition");
 
    $sth->bind_param(1,$dbxref_name); 
    $sth->bind_param(2,'mutant.gene'); 
    $sth->bind_param(3,$query); 
    $sth->bind_param(4,$species) if $species;

    $sth->execute() or die $dbh->errstr;

    my @candidates;
    my %uniq_gene_objs; #

    #since gene and synononym are 1 to many relationship,need get the uniqure gene obj
    while(my $gene_obj = $sth->fetchrow_hashref){
      my $gene_id = $gene_obj->{'gene_id'};
      push @candidates, $gene_obj unless $uniq_gene_objs{$gene_id}++;
   }
  

   _get_gene_objs_synonyms(@candidates);
 
   _get_gene_objs_term_associations(@candidates);
 
   return @candidates;  

};

sub get_candidate_term_associations{
   my ($self,$query,$term_type,$species)=@_;
   $query = uc ($query);
   $query =~s/\*/%/g;
  # $term_type = 6 if $term_type == 7; # for term type growth stage, no type in ontology yet

   my $sql= (qq[
                             SELECT    term_accession,term_name 
                             FROM      term
                             LEFT JOIN term_synonym
                             ON        term.term_id = term_synonym.term_id
                             WHERE     (UPPER(term_name) like ?
                                OR     UPPER(synonym_name) like ? )
				 ]);
  		  

   if($term_type <= 4){
       $sql.=" AND term_type_id <=4";
   }else{
       $sql .=" AND term_type_id = $term_type";
   }

   my $sth = $ot_dbh->prepare($sql);
   $sth->execute(($query,$query));

   my %uniq_terms;        
   while(my ($term_acc,$term_name)= $sth->fetchrow_array){
       $uniq_terms{$term_acc} = $term_name;
   }

   
   my $sth2;
   if($species){

     $sth2 = $dbh->prepare(qq[
                              SELECT COUNT(root_term_acc) 
                              FROM   mutant_quick_term_association ASSOC,
                                     mutant_gene MG
                              WHERE  ASSOC.object_id = MG.gene_id
                               AND   object_table = ?
                               AND   root_term_acc = ?
                               AND   species_id = ?
			       ]);

  }else{
    $sth2 = $dbh->prepare(qq[
                              SELECT COUNT(root_term_acc)
                              FROM   mutant_quick_term_association
                              WHERE  object_table = ?
                               AND   root_term_acc = ?
    			     ]);
  
  }  



   my @terms;

   foreach my $term_acc (keys %uniq_terms){
       if($species){
            $sth2->execute(('mutant.gene',$term_acc,$species)) or die $dbh->errstr;
       }else{
            $sth2->execute(('mutant.gene',$term_acc)) or die $dbh->errstr;
       }       
     
       my ($count) = $sth2->fetchrow_array();
       if($count){
          push @terms, {'term_accession'=>$term_acc,
                        'term_name' =>$uniq_terms{$term_acc},
                        'total_association' => $count
                       };   
          
       }
   }
      
  return @terms; 

}

sub get_genes_by_term_accession{
    my ($self,$term_acc,$species,$order_by)=@_;


    my $sql=qq[
             SELECT mutant_gene.gene_id,
                    accession,
                    SY.synonym_name AS symbol,
                    SN.synonym_name AS name,
                    chromosome
               FROM mutant_gene
	       INNER JOIN mutant_synonym SY
	       ON    mutant_gene.symbol_id = SY.synonym_id
	       INNER JOIN mutant_synonym SN
	       ON    mutant_gene.name_id = SN.synonym_id
	       INNER JOIN mutant_quick_term_association QTA
	       ON    QTA.object_id = mutant_gene.gene_id
	       LEFT JOIN mutant_chromosome
	       ON   mutant_gene.chromosome_id = mutant_chromosome.chromosome_id
              WHERE root_term_acc = ?
                AND object_table = ?
	     ];



    my $condition = '';
    $condition .= ' AND mutant_gene.species_id = ? ' if $species;

    if($order_by eq 'gene_symbol'){
	$condition .= ' order by SY.synonym_name';
    }elsif($order_by eq 'gene_name'){
	$condition .= ' order by SN.synonym_name';
    }elsif($order_by eq 'chromosome'){
	$condition .= ' order by chromosome';
    }elsif($order_by eq 'accession'){
	$condition .= ' order by accession';
    }

    my $sth = $dbh->prepare("$sql"."$condition");

    $sth->bind_param(1,$term_acc);
    $sth->bind_param(2,'mutant.gene');
    $sth->bind_param(3,$species) if $species;

    $sth->execute() or die $dbh->errstr;

    my @candidates;
    my %uniq_gene_objs; #                                                                            

  # need get the uniqure gene obj             
    while(my $gene_obj = $sth->fetchrow_hashref){
	my $gene_id = $gene_obj->{'gene_id'};
	push @candidates, $gene_obj unless $uniq_gene_objs{$gene_id}++;
    }


    _get_gene_objs_synonyms(@candidates);

    _get_gene_objs_term_associations(@candidates);

    return @candidates;

}

sub get_term_search_association{
    my ($self,$term_acc,$term_name) = @_;
    my $term_search;

    my ($total_assoc_count)= $dbh->selectrow_array( 
                                 q[
                                   SELECT COUNT(*) FROM mutant_quick_term_association
                                   WHERE root_term_acc = ?
                                   AND object_table = ?
				   ],{},($term_acc,'mutant.gene')
						    );

    my ($direct_assoc_count)= $dbh->selectrow_array(
                                 q[
                                   SELECT COUNT(*) FROM mutant_quick_term_association
                                   WHERE root_term_acc = ? 
                                   AND term_acc= ?
                                   AND object_table = ?
                                   ],{},($term_acc,$term_acc,'mutant.gene')
						     );

    my ($total_object)= $dbh->selectrow_array( 
                                 q[
                                   SELECT COUNT(distinct object_id)
                                     FROM mutant_quick_term_association
                                   WHERE  root_term_acc = ?
                                   AND    object_table = ?
				   ],{},($term_acc,'mutant.gene')
					       );
	



    $term_search->{'term_name'}=$term_name;
    $term_search->{'term_acc'} = $term_acc;
    $term_search->{'direct_association'} = $direct_assoc_count;
    $term_search->{'indirect_association'} = $total_assoc_count-$direct_assoc_count;
    $term_search->{'object_count'} = $total_object;
    

    return $term_search;

}


sub get_genes_by_all_fields{
  my ($self,$query,$species,$order_by)=@_;

  $query = uc($query);  
  $query =~s/\*/%/g;
   
  my $sql=qq[
             SELECT mutant_gene.gene_id, 
                    accession,
                    SY.synonym_name AS symbol,
                    SN.synonym_name AS name,
                    chromosome
               FROM mutant_gene
	       INNER JOIN mutant_chromosome
	          ON      mutant_gene.chromosome_id = mutant_chromosome.chromosome_id
               INNER JOIN mutant_synonym SY
                  ON      mutant_gene.symbol_id = SY.synonym_id
               INNER JOIN mutant_synonym SN
                  ON      mutant_gene.name_id = SN.synonym_id
               LEFT  JOIN mutant_gene_synonym MGS 
                  ON      mutant_gene.gene_id = MGS.gene_id    
               INNER JOIN mutant_synonym SYN
                  ON      MGS.synonym_id = SYN.synonym_id
               LEFT  JOIN mutant_dbxref_to_object DBO
                  ON      mutant_gene.gene_id = DBO.record_id
               LEFT  JOIN mutant_quick_term_association QTA
                  ON      mutant_gene.gene_id = QTA.object_id   
               WHERE
                          UPPER(accession) like ?
                  OR      UPPER(SY.synonym_name) like ?
                  OR      UPPER(SN.synonym_name) like ?
                  OR      UPPER(SYN.synonym_name) like ?
                  OR      UPPER(DBO.dbxref_value) like ?
                  OR      UPPER(QTA.root_term_name) like ?
           ];
	    

 
  my $condition = '';
  $condition .= ' AND mutant_gene.species_id = ? ' if $species;

  if($order_by eq 'gene_symbol'){   
      $condition .= ' order by SY.synonym_name';
  }elsif($order_by eq 'gene_name'){
      $condition .= ' order by SN.synonym_name'; 
  }elsif($order_by eq 'chromosome'){
      $condition .= ' order by chromosome';
  }elsif($order_by eq 'accession'){
      $condition .= ' order by accession';
  }

  my $sth = $dbh->prepare("$sql"."$condition");

  
  $sth->bind_param(1,$query); 
  $sth->bind_param(2,$query);
  $sth->bind_param(3,$query);
  $sth->bind_param(4,$query);
  $sth->bind_param(5,$query);
  $sth->bind_param(6,$query);
  $sth->bind_param(7,$species) if $species;

  $sth->execute() or die $dbh->errstr;

  my @candidates;
  my %uniq_gene_objs; #

  # since gene and synononym are 1 to many relationship, need get the uniqure gene obj
  while(my $gene_obj = $sth->fetchrow_hashref){
      my $gene_id = $gene_obj->{'gene_id'};
      push @candidates, $gene_obj unless $uniq_gene_objs{$gene_id}++;
  }
  

  _get_gene_objs_synonyms(@candidates);
 
  _get_gene_objs_term_associations(@candidates);
 
  return @candidates;  
   
}





1;
