package Gramene::Marker::Marker2CMap;

=pod

=head1 NAME

Gramene::Marker::Marker2CMap - bridge marker db to CMap

=head1 SYNOPSIS

In a data-source-specific CMap config file (i.e., not "global.conf"),
specify:

   <object_plugin>
     feature Gramene::Marker::Marker2CMap::new
   </object_plugin>

=head1 DESCRIPTION

This module is used by CMap's object plugin system to lookup a CMap
feature's entry in the marker db.  This is done by name, but perhaps
it should be done by the CMap feature accession ID given that this is
stored in the marker db.

The code takes as it's only argument to "new" a hashref representing 
a CMap feature.  It adds new hashrefs to the feature's "xrefs" with 
values for "xref_name" and "xref_url" so that xref links will be 
created on the CMap feature details page.

=head1 SEE ALSO

Bio::GMOD::CMap, Gramene::Marker::DB.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2005 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

For good measure, it also uses the marker API to find genome positions
for the feature.  These have to be found by name.

=cut

# ----------------------------------------------------

use strict;
use Gramene::Marker::DB;
use Readonly;
use Exporter qw( import );

use vars qw( @EXPORT_OK );
@EXPORT_OK = qw( new );

Readonly my %XREF_URL => (
    gene   => '/db/genes/search_gene?acc=',
    marker => '/db/markers/marker_view?marker_id=',
    qtl    => '/db/qtl/qtl_display?qtl_accession_id=',
);

sub new {
    my $feature = shift;
    my $mdb     = Gramene::Marker::DB->new;

    my @markers = @{ $mdb->marker_search( 
        feature_acc => $feature->{'feature_acc'} 
    ) };

    my %seen;
    for my $m ( @markers ) {
        next if $seen{ $m->{'marker_id'} }++;

        my $marker_type = $m->{'marker_type'};
        my $xref_url;
        if ( lc $marker_type eq 'qtl' ) {
            $xref_url = $XREF_URL{'qtl'} . $m->{'marker_name'};
        }
        elsif ( lc $marker_type eq 'gene' && $m->{'marker_name'} =~ /^GR:/ ) {
            $xref_url = $XREF_URL{'gene'} . $m->{'marker_name'};
        }
        else {
            $xref_url = $XREF_URL{'marker'} . $m->{'marker_id'};
        }

        push @{ $feature->{'xrefs'} }, {
            xref_name => 
                "View $m->{'species'} $m->{'marker_type'} " .
                "&quot;$m->{'marker_name'}&quot; details",
            xref_url  => $xref_url
        };
    }

    return $feature;
}

1;
