package Gramene::Marker::DB;

# vim: set ts=4 sw=4 et:

# $Id: DB.pm,v 1.96 2007/06/05 19:25:53 kclark Exp $

=head1 NAME

Gramene::Marker::DB - Interface to markers db

=head1 SYNOPSIS

  use Gramene::Marker::DB;
  my $mdb = Gramene::Marker::DB->new;

=head1 DESCRIPTION

Interface to markers db.  Uses Gramene::CDBI::Markers.  All methods throw
exceptions via Carp::croak, so be prepared to catch!

=head1 METHODS

=head2 new

  my $mdb = Gramene::Marker::DB->new;

Takes no arguments.  Returns the object for calling all the below
methods.

=cut

# ----------------------------------------------------

use strict;
use Carp qw( croak carp );
use Config::General;
use Class::Base;
use Data::Dumper;
use Gramene::DB;
use Gramene::CDBI::Literature;
use Gramene::CDBI::Markers;
use Gramene::CDBI::Ontology;
use Gramene::Utils 
    qw( commify get_logger parse_words table_name_to_gramene_cdbi_class);
use List::MoreUtils qw( uniq );
use Readonly;

use base 'Class::Base';

our %EXPANDED_SPECIES;

Readonly my $CDBI                      => 'Gramene::CDBI::Markers';
Readonly my $CDBI_MODULE               => 'Markers';
Readonly my $DOUBLE_COLON              => '::';
Readonly my $EMPTY_STR                 => q{};
Readonly my $UNKNOWN                   => 'UNKNOWN';
Readonly my $MARKER_NAME_WITH_SYN_TYPE => qr/
    \A                 # start of line
    ["]?               # double-quote (optional)
    (.+)               # anything (captured)
    \s+                # one or more spaces
    [[]{2}             # two left square brackets
    (?:synonym_type=)? # optional "synonym_type="
    (\w+)              # something (captured)
    []]{2}             # two right square brackets
    ["]?               # double-quote (optional)
    \z                 # end of line
/xms;
Readonly my $VERSION 
    => sprintf '%d.%02d', qq$Revision: 1.96 $ =~ /(\d+)\.(\d+)/;

# ----------------------------------------------------
sub init {
    my ( $self, $config ) = @_;
    $self->params( $config, qw[ admin db_name ] );
    return $self;
}

#--------------------------------------------------
sub _args_to_Object {

=pod

=head2 _args_to_Object

  my $cdbi_marker_type = $self->_args_to_Object(
    'MarkerType',
    { MarkerType     => $cdbi_marker, # or
      marker_type_id => 12, # or
      marker_type    => 'GSS' }
  );

Returns a CDBI object of the given class based on the following
argumants; the object itself, its ID, or its name. Used by other
methods in this module for argument parsing/validation.

=cut

    my $self       = shift;
    my $class      = shift;
    my $args       = ref $_[0] eq 'HASH' ? shift : { @_ };
    my $name_key   = $class;
    $name_key      =~ s/([a-z])([A-Z])/$1_$2/g;
    $name_key      = lc( $name_key );    # TODO: fix for e.g. library_name
    my $cdbi_class = join( $DOUBLE_COLON, $CDBI, $class );
    my $id_key     = $cdbi_class->columns( 'Primary' ) || "${name_key}_id";

    my $object;
    if ( $object = $args->{ $class } ) {
        if ( !UNIVERSAL::isa($object, "${CDBI}::${class}") ) {
            croak("$class must be a ${CDBI}::${class}");
        }
    }
    elsif ( my $object_id = $args->{ $id_key } ) {
        my $retrieval_method = "retrieve_" . $class;
        $object = $self->$retrieval_method( $object_id )
            or croak( "No $class in DB with ID $object_id" );
    }
    elsif ( my $object_name = $args->{ $name_key } ) {
        my $searching_method = "search_" . $class;
        my $object_iterator
            = $self->$searching_method( { $name_key => $object_name } )
            or croak( "No $class in DB with $name_key $object_name" );
        $object = $object_iterator->next;
    }

    return $object;
}

# ----------------------------------------------------
=pod

=head2 create_marker, create_Marker, update_marker, update_Marker

  my $marker_id  = $mdb->create_marker( # or
  my $marker_obj = $mdb->create_Marker( # or
  my $marker_id  = $mdb->update_marker( # or
  my $marker_obj = $mdb->update_Marker( # or

      marker_name      => 'foo',   # create_marker and create_Marker only  #or
      marker_name      => { marker_name=>'foo',SynonymType=>syn_type_obj },#or
      marker_name      => { marker_name=>'foo',synonym_type_id=>1 }        #or
      marker_name      => { marker_name=>'foo',synonym_type=>'GENBANK'},   #or
      marker_id        => 1,        # update_marker and update_Marker only #or
      Marker           => $marker_obj, # ditto 

      synonyms         => [ 'bar', 'baz' ], #or
      synonyms         => [ { marker_name=>'bar', SynonymType=>syn_type_obj },
                            { marker_name=>'baz', synonym_type_id=>1 },
                            { marker_name=>'wib', synonym_type=>'GENBANK'} ],

      MarkerType      => Gramene::CDBI::Markers::MarkerType, # or
      marker_type_id  => 1, #or
      marker_type     => 'RFLP',

      Species         => Gramene::CDBI::Markers::Species, # or
      species_id      => 1, #or
      species         => 'Oryza sativa',

      Germplasm       => Gramene::CDBI::Markers::Germplasm, # or
      germplasm_id    => 1, #or
      germplasm       => 'ABC123',

      Analysis        => Gramene::CDBI::Markers::Analysis, # or
      analysis_id     => 1, #or
      analysis        => 'foobar',

      description      => '...',
      details          => {'foo'=>'bar'}
  );

For create_Marker and update_Marker, returns the CDBI::Markers::Marker obj
For create_marker and update_marker, returns the marker_id

=cut

# ----------------------------------------------------
sub create_Marker {

    my ( $self, %args )  = @_;

    # Validate params
    unless( $args{ marker_name } ){
        croak( "Need a marker_name" );
    }

    unless( grep{$args{$_}} qw( MarkerType marker_type_id marker_type ) ){
        croak( "Need a MarkerType" );
    }

    unless( grep{$args{$_}} qw( Species species_id species ) ){
        croak( "Need a Species" );
    }

    # Delegate work to update_marker
    my $marker = $self->insert_Marker({});
    return $self->update_Marker( %args, Marker=>$marker );
}

# ----------------------------------------------------
sub create_marker {
    my $self       = shift;
    my $marker_obj = $self->create_Marker( @_ );
    return $marker_obj->id;
}

# ----------------------------------------------------
sub update_Marker {
    my ( $self, %args ) = @_;


    # Get the Marker
    my $cdbi_marker = $self->_args_to_Object( 'Marker', \%args )
        or croak( "Need a Marker or a marker_id" );

    # Process the MarkerType
    if ( my $cdbi_obj = $self->_args_to_Object( 'MarkerType', \%args ) ) {
        $cdbi_marker->marker_type_id( $cdbi_obj->id );
    }

    # Process the Species
    if ( my $cdbi_obj = $self->_args_to_Object( 'Species', \%args ) ) {
        $cdbi_marker->source_species_id( $cdbi_obj->id );
    }

    # Process the germplasm
    if ( my $cdbi_obj = $self->_args_to_Object( 'Germplasm', \%args ) ) {
        $cdbi_marker->germplasm_id( $cdbi_obj->id );
    }

    # Process analysis
    if ( my $cdbi_obj = $self->_args_to_Object( 'Analysis', \%args ) ) {
        $cdbi_marker->analysis_id( $cdbi_obj->id );
    }

    # Process library
    if ( my $cdbi_obj = $self->_args_to_Object( 'Library', \%args ) ) {
        $cdbi_marker->library_id( $cdbi_obj->id );
    }

    # Process description, date_fields
    for my $field ( 'description', 'date_created', 'date_updated' ) {
        if ( $args{$field} ) {
            $cdbi_marker->$field( $args{$field} );
        }
    }

    my $seq = $args{'sequence'}
        || $args{'seq'}
        || $args{'details'}{'sequence'}
        || $args{'details'}{'seq'}
        || '';

    if ( $seq ) {
        my $marker_seq_obj
            = Gramene::CDBI::Markers::MarkerSequence->find_or_create(
                { marker_id => $cdbi_marker->id } 
            );

        $marker_seq_obj->seq( $seq );
        $marker_seq_obj->update;
    }

    # Process synonyms
    my $display_syn_obj;
    if ( $args{'marker_name'} or $args{'synonyms'} ) { 
        # The synonyms arg to add_synonyms_to_Marker is a listref of hashrefs.
        $display_syn_obj = $self->add_synonyms_to_Marker(
            Marker => $cdbi_marker,
            synonyms =>
                [ $args{'marker_name'} || (), @{ $args{'synonyms'} || [] } ]
        );
    }

    # Update marker_name
    if ( $display_syn_obj ) {
        $cdbi_marker->display_synonym_id( $display_syn_obj->id );
    }

    # Update the marker
    $cdbi_marker->update;

    # Process details
    if ( $args{'details'} ) {
        $self->set_marker_details(
            Marker => $cdbi_marker,
            %{ $args{'details'} }
        ) || return;
    }

    return $cdbi_marker;
}

# ----------------------------------------------------
sub update_marker {
    my $self       = shift;
    my $marker_obj = $self->update_Marker( @_ );
    return $marker_obj->id;
}

# ----------------------------------------------------
sub db {

=pod

=head2 db

  my $db = $mdb->db;

Returns a database handle to the marker database (as defined in the
C<config> method).

=cut

    return Gramene::CDBI::Markers->db_Main;
}

# --------------------------------------------------------
sub delete_analytical_correspondence {

=pod

=head2 delete_analytical_correspondence 

  $mdb->delete_analytical_correspondence( 
      analytical_correspondence_id => 42 
  );

Deletes a correspondence.

=cut

    my ( $self, %args ) = @_;
    my $analytical_correspondence_id = $args{'analytical_correspondence_id'}
        or croak('No analytical correspondence id');

    my $db = $self->db or return;

    $db->do(
        qq[
            delete 
            from   analytical_correspondence
            where  analytical_correspondence_id=?
        ],
        {},
        ( $analytical_correspondence_id )
    );

    return 1;
}

# --------------------------------------------------------
sub delete_germplasm {

=pod

=head2 delete_germplasm 

  $mdb->delete_germplasm( germplasm_id => 42 );

Deletes a germplasm.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object('Germplasm',{%args} );
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub delete_mapping {

=pod

=head2 delete_mapping 

  $mdb->delete_mapping( mapping_id => 42 );

Deletes a mapping.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object('Mapping',{%args} );
    my $marker_id = $cdbi_obj->marker_id;
    $cdbi_obj->delete;
    return $marker_id;
}

# --------------------------------------------------------
sub delete_marker {

=pod

=head2 delete_marker 

  $mdb->delete_marker( marker_id => 1984 );

Deletes a marker.

=cut

    my ( $self, %args ) = @_;

    my $marker_id = $args{'marker_id'} or croak('No marker id');
    my $db        = $self->db;

    my $exists = $db->selectrow_array(
        q[
            select count(*) 
            from   marker
            where  marker_id=?
        ],
        {},
        ( $marker_id )
    );

    return 1 if !$exists;

    #
    # Correspondences
    #
    for my $dir ( qw[ from to ] ) {
        $db->do(
            qq[
                delete
                from   analytical_correspondence
                where  ${dir}_marker_id=?
            ],
            {},
            ( $marker_id )
        );
    }

    #
    # Details
    #
    my $marker_type = $db->selectrow_array( 
        qq[
            select mt.marker_type
            from   marker m, marker_type mt
            where  m.marker_id=?
            and    m.marker_type_id=mt.marker_type_id
        ],
        {},
        ( $marker_id )
    );

    #
    # Cross-references
    #
    $db->do(
        q[
            delete
            from   xref
            where  record_id=?
            and    table_name=?
        ],
        {},
        ( $marker_id, 'marker' )
    );

    #
    # Have to NULL field to get around FK dependency
    #
    $db->do(
        q[
            update marker
            set    display_synonym_id=NULL
            where  marker_id=?
        ],
        {},
        ( $marker_id )
    );

    my ( $details_table_name ) = $self->marker_type_to_table_name(
        $marker_type
    );

    #
    # Mappings, etc.
    #
    for my $table ( 
        $details_table_name,
        qw[ mapping marker_image marker_sequence marker_synonym marker ],
    ) {
        next unless $table;

        $db->do(
            qq[
                delete
                from   $table
                where  marker_id=?
            ],
            {},
            ( $marker_id )
        );
    }

    return 1;
}

# --------------------------------------------------------
sub delete_marker_synonym {

=pod

=head2 delete_marker_synonym 

  my $marker_id = $mdb->delete_marker_synonym( 
      marker_synonym_id => 42 
  );

Deletes a marker synonym.  Won't delete a synonym currently being used
as the display name.  Returns the marker id on success.

=cut

    my ( $self, %args )   = @_;
    my $cdbi_syn = $self->_args_to_Object( 'MarkerSynonym', \%args )
        or croak('No MarkerSynonym or id');
    my $marker_id = $cdbi_syn->marker_id;
    $cdbi_syn->delete;
    return $marker_id;
}

# --------------------------------------------------------
sub delete_map_type {

=pod

=head2 delete_map_type 

  $mdb->delete_map_type( map_type_id => 42 );

Deletes a map_type.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object( 'MapType', {%args} )
        or croak('No MapType or id');
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub delete_marker_type {

=pod

=head2 delete_marker_type 

  $mdb->delete_marker_type( marker_type_id => 42 );

Deletes a marker_type.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object( 'MarkerType', {%args} )
        or croak('No MarkerType or id');
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub delete_species {

=pod

=head2 delete_species 

  $mdb->delete_species( species_id => 42 );

Deletes a species.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object( 'Species', {%args} )
        or croak('No Species or id');
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub delete_xref {

=pod

=head2 delete_xref 

  $mdb->delete_xref( xref_id => 42 );

Deletes an xref.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object( 'Xref', {%args} )
        or croak('No Xref or id');
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub delete_xref_type {

=pod

=head2 delete_xref_type 

  $mdb->delete_xref_type( xref_type_id => 42 );

Deletes an xref_type.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_obj = $self->_args_to_Object( 'XrefType', {%args} )
        or croak('No XrefType or id');
    return $cdbi_obj->delete;
}

# --------------------------------------------------------
sub find_or_create_analysis {

=pod

=head2 find_or_create_analysis 

  my $analysis_id = $mdb->find_or_create_analysis('GenBank');

Finds or creates an analysis of the given name.  Returns its ID.

=cut

    my $self          = shift;
    my $analysis_name = shift or croak('No analysis name');
    my $obj = $self->find_or_create_Analysis({analysis_name=>$analysis_name});
    return $obj->id;
}

# ----------------------------------------------------
sub find_or_create_germplasm {

=pod

=head2 find_or_create_germplasm

  my $germplasm_id = $mdb->find_or_create_germplasm('IR64', 1);

Finds the germplasm for a given species or creates it.  Returns its ID.

=cut

    my $self         = shift;
    my $germplasm    = shift or croak('No germplasm name');
    my $species_id   = shift or croak('No species id');

    my $cdbi_species = $self->_args_to_Object('Species',
                                              {species_id=>$species_id});

    my $obj = $self->find_or_create_Germplasm
        ({species_id=>$cdbi_species, germplasm_name=>$germplasm});

    return $obj->id;
}

# ----------------------------------------------------
sub find_or_create_map {

=pod

=head2 find_or_create_map

  my $map_id = $mdb->find_or_create_map(
      map_set_id         => 42,
      map_name           => 'Chr. 1',
      start              => 0,
      end                => 100,
      cmap_map_accession => 'cu-dh-93-1',
  );

Finds or creates a map matching the given args.  Return the map's ID.

=cut

    my ( $self, %args )    = @_;
    my $db                 = $self->db;
    my $cmap_map_accession = $args{'cmap_map_accession'} || '';
    my $map_set_id         = $args{'map_set_id'}         || '';
    my $map_name           = $args{'map_name'}           || '';
    my $start              = $args{'start'}              || '';
    my $end                = $args{'end'}                || '';

    my $cdbi_map;
    if( $cmap_map_accession ){
      ($cdbi_map) = $self->search_Map
          ({cmap_map_accession=>$cmap_map_accession});
    }
    elsif ( $map_set_id && $map_name ) {
      ($cdbi_map) = $self->search_Map
          ({map_set_id=>$map_set_id, map_name=>$map_name});
    }
    else{
      croak("Insufficient info to find or create map");
    }

    unless ( $cdbi_map ) {
      $cdbi_map = $self->insert_Map
          ({ map_set_id => $map_set_id,
             map_name   => $map_name,
             start      => $start,
             end        => $end,
             cmap_map_accession => $cmap_map_accession});
    }
    return $cdbi_map->id;
}

# ----------------------------------------------------
sub find_or_create_map_set {

=pod

=head2 find_or_create_map_set

  my $db = $mdb->find_or_create_map_set(
      map_set_name           => 'Cornell RFLP 93',
      map_type_id            => 1,
      species_id             => 1,
      germplasm_id           => 1,
      cmap_map_set_accession => 'cu-dh-93',
  );

Finds or creates a map set for the given args.  Returns the map set's
ID.

=cut

    my ( $self, %args )        = @_;
    my $map_set_name           = $args{'map_set_name'}  || '';
    my $map_type_id            = $args{'map_type_id'}   || '';
    my $species_id             = $args{'species_id'}    || '';
    my $germplasm_id           = $args{'germplasm_id1'} || '';
    my $cmap_map_set_accession = $args{'cmap_map_set_accession'} || '';
    my $db                     = $self->db;

    my ( $sql, @args );
    if ( $cmap_map_set_accession ) {
        $sql = q[
            select map_set_id
            from   map_set
            where  cmap_map_set_accession=?
        ];
        @args = ( $cmap_map_set_accession );
    }
    elsif ( $map_set_name && $map_type_id && $species_id && $germplasm_id ) {
        $sql = q[
            select map_set_id
            from   map_set
            where  map_set_name=?
            and    map_type_id=?
            and    species_id=?
            and    germplasm_id1=?
        ];
        @args = ( $map_set_name, $map_type_id, $species_id, $germplasm_id );
    }

    croak("Insufficient info to find or create map")
        unless $sql && @args;

    my $map_set_id = $db->selectrow_array( $sql, {}, @args );

    unless ( $map_set_id ) {
        $db->do(
            q[
                insert
                into   map_set 
                       (map_set_name, map_type_id, species_id, germplasm_id1,
                       cmap_map_set_accession)
                values ( ?, ?, ?, ?, ? )
            ],
            {},
            ( $map_set_name, $map_type_id, $species_id, $germplasm_id,
              $cmap_map_set_accession )
        );

        $map_set_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $map_set_id;
}

# ----------------------------------------------------
sub find_or_create_map_type {

=pod

=head2 find_or_create_map_type

  my $map_type_id = $mdb->find_or_create_map_type('Genetic');

Finds or creates the given map type.  Returns its ID.

=cut

    my ( $self, $map_type ) = @_;
    my $db = $self->db;

    my $map_type_id = $db->selectrow_array(
        q[
            select map_type_id
            from   map_type
            where  map_type=?
        ],
        {},
        ( $map_type )
    );

    unless ( $map_type_id ) {
        $db->do(
            'insert into map_type (map_type) values (?)', {}, 
            ( $map_type )
        );

        $map_type_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $map_type_id;
}

# --------------------------------------------------------
sub transfer_mapping {

=pod

=head2 transfer_mapping

  $mdb->transfer_mapping(
      to_marker_id   => 44, # REQUIRED AND
      mapping_id     => 42, # OR
      from_marker_id => 43,
  );

Transfers a mapping to a new marker id.  If "from_marker_id" is present, 
then all the mappings for it will be transferred.  If "mapping_id" is 
present, then just that one will be moved.  An error will be thrown if
neither or both are present.

=cut

    my ( $self, %args ) = @_;
    my $from_marker_id  = $args{'from_marker_id'} || 0;
    my $mapping_id      = $args{'mapping_id'}     || 0;
    my $to_marker_id    = $args{'to_marker_id'} or croak('No to_marker_id');

    if ( !$from_marker_id && !$mapping_id ) {
        croak('Need either from from marker id or mapping id');
    }

    if ( $from_marker_id && $mapping_id ) {
        croak('Need only one of from marker id or mapping id');
    }

    my @mapping_ids;
    if ( $from_marker_id ) {
        @mapping_ids = map { $_->id } Gramene::CDBI::Markers::Mapping->search( 
            marker_id => $from_marker_id
        );
    }
    else {
        @mapping_ids = ( $mapping_id );
    }

    my $to_marker = Gramene::CDBI::Markers::Marker->retrieve( $to_marker_id )
        or croak("Bad marker id ($to_marker_id)");

    for my $id ( @mapping_ids ) {
        my $mapping = Gramene::CDBI::Markers::Mapping->retrieve( $id )
            or croak("Bad mapping id ($id)");

        my $display_syn = $mapping->display_synonym;

        my ($existing_name) = Gramene::CDBI::Markers::MarkerSynonym->search(
            marker_id   => $to_marker->id,
            marker_name => $display_syn->marker_name
        );

        if ( $existing_name ) {
            $mapping->display_synonym_id( $existing_name->marker_synonym_id );
        }
        else {
            $display_syn->marker_id( $to_marker->id );
            $display_syn->update;
        }

        $mapping->marker_id( $to_marker->id );
        $mapping->update;
    }

    return 1;
}

# --------------------------------------------------------
sub update_or_create_marker {

=pod

=head2 update_or_create_marker

  my $marker_id = $mdb->find_or_create_marker(
      marker_name     => 'Foo',
      marker_type     => 'RFLP',
      species_id      => 1,
      synonyms        => [ 'Bar', 'Baz' ]
      # optional
      germplasm_id    => 1,
      description     => '...',
      details         => {'foo'=>'bar'}
  );

Finds a marker and update it or creates a marker for the given args.  Returns its ID.

=cut

    my ( $self, %args ) = @_;
    my $marker_name     = $args{'marker_name'};
    my $marker_type     = $args{'marker_type'}    || '';
    my $marker_type_id  = $args{'marker_type_id'} ||  0;
    my $species         = $args{'species'}        || '';
    my $species_id      = $args{'species_id'}     ||  0;
    my $germplasm_id    = $args{'germplasm_id'}   || '';
    my $description     = $args{'description'}    || '';
    my $synonyms        = ref $args{'synonyms'} eq 'ARRAY'
                          ? $args{'synonyms'} 
                          : [ $args{'synonyms'} ];

    my $analysis        = $args{analysis} || '';
    my $analysis_id     = $args{analysis_id} || '';

    croak('No marker name') unless $marker_name;

    if ( ! $marker_type_id && $marker_type ) {
        $marker_type_id = $self->find_or_create_marker_type( $marker_type );
    }
    croak('No marker type') unless $marker_type_id;

    if ( ! $species_id && $species ) {
        $species_id = $self->find_or_create_species( $species );
    }
    croak('No species') unless $species_id;

    my $db = $self->db;

    #
    # Find or create the marker
    #

    my $marker_id;

    if($analysis || $analysis_id){
      $analysis_id =  $self->search_Analysis( {analysis_name => $analysis} ) unless $analysis_id;
      if( $analysis_id ){
	for my $name ( $marker_name, @$synonyms ) {
	  $marker_id = $db->selectrow_array(
	    q[
                select m.marker_id
                from   marker m, marker_synonym s
                where  s.marker_name=?
                and    s.marker_id=m.marker_id
                and    m.marker_type_id=?
                and    m.source_species_id=?
                and    m.analysis_id=?
             ],
	     {},
	     ( $name, $marker_type_id, $species_id, $analysis_id )
	   );

	  last if $marker_id;
	}
      }else{
	croak("Cannot find analysis for $analysis");
      }
    }else{
      for my $name ( $marker_name, @$synonyms ) {
        $marker_id = $db->selectrow_array(
            q[
                select m.marker_id
                from   marker m, marker_synonym s
                where  s.marker_name=?
                and    s.marker_id=m.marker_id
                and    m.marker_type_id=?
                and    m.source_species_id=?
            ],
            {},
            ( $name, $marker_type_id, $species_id )
        );

        last if $marker_id;
      }
    }

    my $marker_existed = $marker_id ? 1 : 0;
    my $marker;

    if($marker_id){
      $marker = $self->retrieve_Marker( $marker_id );
    }else{
      $marker = $self->insert_Marker({});
      $marker_id = $marker->marker_id();
      
    }

    $self->update_Marker( %args, Marker=>$marker );				  
    

    return wantarray ? ( $marker_id, $marker_existed ) : $marker_id;
}


# --------------------------------------------------------
sub find_or_create_marker {

=pod

=head2 find_or_create_marker

  my $marker_id = $mdb->find_or_create_marker(
      marker_name     => 'Foo',
      marker_type     => 'RFLP',
      species_id      => 1,
      synonyms        => [ 'Bar', 'Baz' ]
      # optional
      germplasm_id    => 1,
      description     => '...',
      details         => {'foo'=>'bar'}
  );

Finds or creates a marker for the given args.  Returns its ID.

=cut

    my ( $self, %args ) = @_;
    my $marker_name     = $args{'marker_name'};
    my $marker_type     = $args{'marker_type'}    || '';
    my $marker_type_id  = $args{'marker_type_id'} ||  0;
    my $species         = $args{'species'}        || '';
    my $species_id      = $args{'species_id'}     ||  0;
    my $germplasm_id    = $args{'germplasm_id'}   || '';
    my $description     = $args{'description'}    || '';
    my $synonyms        = ref $args{'synonyms'} eq 'ARRAY'
                          ? $args{'synonyms'} 
                          : [ $args{'synonyms'} ];

    my $analysis        = $args{'analysis'}       || '';
    my $analysis_id     = $args{'analysis_id'}    || '';
    my $library         = $args{'library'}        || '';
    my $library_id      = $args{'library_id'}     || '';
    my $search_only_primary_name = $args{'search_only_primary_name'} || 0;

    croak('No marker name') unless $marker_name;

    if ( !$marker_type_id && $marker_type ) {
        $marker_type_id = $self->find_or_create_marker_type( $marker_type );
    }
    croak('No marker type') unless $marker_type_id;

    if ( !$species_id && $species ) {
        $species_id = $self->find_or_create_species( $species );
    }
    croak('No species') unless $species_id;

    my $db = $self->db;

    if ( $analysis && !$analysis_id ) {
        $analysis_id =  $self->search_Analysis({ analysis_name => $analysis });
        croak("Cannot find analysis for $analysis") unless $analysis_id;
        $args{'analysis_id'} = $analysis_id;
    }

    if ( $library && !$library_id ) {
        $library_id =  $self->search_Library({ library_name => $library });
        croak("Cannot find library for $library") unless $library_id;
        $args{'library_id'} = $library_id;
    }
     
    my @markers;
    my @search_names = ( $marker_name );
    if ( !$search_only_primary_name ) {
        push @search_names, @$synonyms;
    }

    SEARCH:
    for my $name ( @search_names ) {
        $name =~ s/\*/\\\*/g;
        @markers = $self->marker_search(
            marker_name              => $name,
            marker_type_id           => $marker_type_id,
            species_id               => $species_id,
            analysis_id              => $analysis_id,
            library_id               => $library_id,
            search_only_primary_name => $search_only_primary_name,
        );

        if ( scalar @markers == 1 ) {
            last SEARCH;
        }
    }

    my $marker_id;
    if ( scalar @markers > 1 ) {
        my $Species    = $self->retrieve_Species( $species_id );
        my $MarkerType = $self->retrieve_MarkerType( $marker_type_id );
        croak(
            sprintf('More than one marker matches %s %s %s',
                $Species->species, $MarkerType->marker_type,
                $marker_name
            )
        );
    }
    elsif ( scalar @markers == 1 ) {
        $marker_id = $markers[0]->{'marker_id'};
    }

    my $marker_existed = $marker_id ? 1 : 0;

    my $Marker;
    if ( $marker_id ) {
        $Marker = Gramene::CDBI::Markers::Marker->retrieve( $marker_id );
    }
    else {
        # not found marker, create one
        $Marker    = $self->insert_Marker({});
        $marker_id = $Marker->id();
    }

    $self->update_Marker( %args, Marker => $Marker );

    if ( ref $args{'details'} eq 'HASH' ) {
        $self->set_marker_details(
            %{ $args{'details'} },
            Marker => $Marker
        );
    }

    return wantarray ? ( $marker_id, $marker_existed ) : $marker_id;
}


# ----------------------------------------------------
sub find_or_create_marker_type {

=pod

=head2 find_or_create_marker_type

  my $marker_type_id = $mdb->find_or_create_marker_type('RFLP');

Finds or creates a given marker type.  Returns its ID.

=cut

    my ( $self, $marker_type ) = @_;

    my $db = $self->db;
    my $marker_type_id = $db->selectrow_array(
        q[
            select marker_type_id
            from   marker_type
            where  marker_type=?
        ],
        {},
        ( $marker_type )
    );

    unless ( $marker_type_id ) {
        $db->do(
            'insert into marker_type (marker_type) values (?)', {},
            ( $marker_type )
        );

        $marker_type_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $marker_type_id;
}

# ----------------------------------------------------
sub find_or_create_marker_image {

=pod

=head2 find_or_create_marker_image

  my $marker_image_id = $mdb->find_or_create_marker_image(
      marker_id => 1990,
      file_name => 'foo.png',
      caption   => 'foo',
      width     => 100,
      height    => 100,
  );

Finds the marker image or creates it.  Returns the ID.

=cut

    my ( $self, %args ) = @_; 
    my $marker_id       = $args{'marker_id'} or 
                          croak('No marker id');
    my $file_name       = $args{'file_name'} or 
                          croak('No file name');
    my $caption         = $args{'caption'} || '';
    my $width           = $args{'width'};
    my $height          = $args{'height'};
    my $db              = $self->db;
    my $marker_image_id = $db->selectrow_array(
        q[
            select marker_image_id
            from   marker_image
            where  marker_id=?
            and    file_name=?
        ],
        {},
        ( $marker_id, $file_name )
    );

    unless ( $marker_image_id ) {
        $db->do(
            q[
                insert 
                into   marker_image 
                       (marker_id, file_name, caption, width, height) 
                values (?, ?, ?, ?, ?)
            ], 
            {}, 
            ( $marker_id, $file_name, $caption, $width, $height ) 
        );

        $marker_image_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $marker_image_id;
}

# ----------------------------------------------------
sub find_or_create_species {

=pod

=head2 find_or_create_species

  my $species_id = $mdb->find_or_create_species('rice');

Finds the species or creates it.  Returns its ID.

=cut

    my ( $self, $species ) = @_; 
    my $db = $self->db;

    my $species_id = $db->selectrow_array(
        q[
            select species_id
            from   species
            where  species=?
        ],
        {},
        ( $species )
    );

    unless ( $species_id ) {
        $db->do(
            'insert into species (species) values (?)', {}, 
            ( $species )
        );

        $species_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $species_id;
}

#----------------------------------------------------------------------
sub expand_Species {

=pod

=head2 expand_Species

  my @cdbi_species = $mdb->expand_species( $cdbi_species )

Takes a CDBI Species object, and returns a list of all Species in the
database with the same genus/species. I.e. Oryza sativa (indica
cultivar-group) will return Oryza sativa, Oryza sativa (indica
cultivar-group), Oryza sativa (japonica cultivar-group), Oryza sativa
x Oryza eichingeri ...

=cut

    my $self         = shift;
    my $cdbi_species = shift or croak('Need a CDBI Species');
    my $sp_name      = $cdbi_species->species;
    my @species      = @{ $EXPANDED_SPECIES{$sp_name} || [] };    # cache
    unless ( @species ) {
        my $base_sp = join( " ", ( split( /\s+/, $sp_name ) )[ 0 .. 1 ] );
        @species = $self->search_like_Species( species => "$base_sp%" );
        $EXPANDED_SPECIES{$sp_name} = [@species];
    }
    return @species;
}


#----------------------------------------------------------------------

=pod

=head2 find_or_create_xref

  my $xref_id = $mdb->find_or_create_xref(
      table_name   => 'marker',
      record_id    => 1990,
      xref_value   => 'foo',
      xref_type_id => 3, # OR
      xref_type    => 'Gramene Literature',
  );

Finds the xref or creates it.

=cut

sub find_or_create_xref{
    my ( $self, %args ) = @_; 
    my $table_name      = $args{'table_name'}   or croak('No table name');
    my $record_id       = $args{'record_id'}    or croak('No record id');
    my $xref_value      = $args{'xref_value'}   or croak('No xref value');
    my $xref_type       = $args{'xref_type'}    || '';
    my $xref_type_id    = $args{'xref_type_id'} || '';

    my $XrefType;
    if ( $xref_type_id ) {
        $XrefType = Gramene::CDBI::Markers::XrefType->retrieve( $xref_type_id )
            or croak("Bad xref type id ($xref_type_id)");
    }
    elsif ( $xref_type ) {
        $XrefType = Gramene::CDBI::Markers::XrefType->find_or_create({
            xref_type => $xref_type
        });
    }
    else {
        croak('No xref type info to create xref');
    }

    my $class = table_name_to_gramene_cdbi_class( $CDBI_MODULE, $table_name )
        or croak("Bad table name '$table_name'");
    my $object = $class->retrieve( $record_id )
        or croak("No '$table_name' with ID '$record_id'");

    my $Xref = Gramene::CDBI::Markers::Xref->find_or_create({
        table_name   => $table_name,
        record_id    => $record_id,
        xref_type_id => $XrefType->id,
        xref_value   => $xref_value,
    });
    
    return $Xref->id;
}

# ----------------------------------------------------
sub find_or_create_xref_type {

=pod

=head2 find_or_create_xref_type

  my $xref_type_id = $mdb->find_or_create_xref_type(
      xref_type    => 'Gramene Literature',
      url_template => '/db/literature/pub_search?ref_id=',
  );

Finds the xref type or creates it.

=cut

    my ( $self, %args ) = @_; 
    my $xref_type       = $args{'xref_type'}    or 
                          croak('No xref type');
    my $url_template    = $args{'url_template'} || '';
    my $db              = $self->db;
    my $sql             = sprintf(
        q[
            select xref_type_id
            from   xref_type
            where  xref_type=?
            %s
        ],
        $url_template ? 'and url_template=?' : ''
    );
    my @args = map { $_ || () } ( $xref_type, $url_template );

    my $xref_type_id = $db->selectrow_array( $sql, {}, @args );

    unless ( $xref_type_id ) {
        $db->do(
            'insert into xref_type (xref_type, url_template) values (?,?)', {}, 
            ( $xref_type, $url_template )
        );

        $xref_type_id = $db->selectrow_array( 'select last_insert_id()' );
    }

    return $xref_type_id;
}


# ----------------------------------------------------
sub get_marker_correspondences {

=pod

=head2 get_marker_correspondences 

  my $name = $mdb->get_marker_correspondences( marker_id => $marker_id );

Returns a marker's analytical correspondences.

=cut

    my ( $self, %args ) = @_;
    my $marker_id       = $args{'marker_id'} or 
                          croak('No marker id');
    my $db              = $self->db;
    my $sql             = q[
        select   c.analytical_correspondence_id,
                 c.from_marker_id,
                 c.to_marker_id,
                 a.analysis_id,
                 a.analysis_name,
                 ct.analytical_correspondence_type_id,
                 ct.type,
                 s1.marker_name as from_marker_name,
                 mt1.marker_type_id as from_marker_type_id,
                 mt1.marker_type as from_marker_type,
                 sp1.species_id as from_species_id,
                 sp1.species as from_species,
                 s2.marker_name as to_marker_name,
                 mt2.marker_type_id as to_marker_type_id,
                 mt2.marker_type as to_marker_type,
                 sp2.species_id as to_species_id,
                 sp2.species as to_species,
                 '%s' as direction
        from     analytical_correspondence c,
                 analysis a,
                 analytical_correspondence_type ct,
                 marker m1,
                 marker m2,
                 marker_synonym s1,
                 marker_synonym s2,
                 marker_type mt1,
                 marker_type mt2,
                 species sp1,
                 species sp2
        where    c.%s_marker_id=?
        and      c.analysis_id=a.analysis_id
        and      c.analytical_correspondence_type_id=
                 ct.analytical_correspondence_type_id
        and      c.from_marker_id=m1.marker_id
        and      m1.display_synonym_id=s1.marker_synonym_id
        and      m1.marker_type_id=mt1.marker_type_id
        and      m1.source_species_id=sp1.species_id
        and      c.to_marker_id=m2.marker_id
        and      m2.display_synonym_id=s2.marker_synonym_id
        and      m2.marker_type_id=mt2.marker_type_id
        and      m2.source_species_id=sp2.species_id
    ];

    my @correspondences;
    for my $dirs ( ['from','to'], ['to','from'] ) {
        push @correspondences, @{ 
            $db->selectall_arrayref(
                sprintf( $sql, @$dirs ), { Columns => {} }, ( $marker_id )
            )
        };
    }

    return wantarray ? @correspondences : \@correspondences;
}

# ----------------------------------------------------
sub get_marker_display_name {

=pod

=head2 get_marker_display_name 

  my $name = $mdb->get_marker_display_name( marker_id => $marker_id );

Returns a marker's primary name.

=cut

    my ( $self, %args ) = @_;

    my $marker_id   = $args{'marker_id'} or croak('No marker id');
    my $db          = $self->db;
    my $marker_name = $db->selectrow_array(
        q[
            select s.marker_name
            from   marker_synonym s, marker m
            where  m.marker_id=?
            and    m.display_synonym_id=s.marker_synonym_id
        ],
        {},
        ( $marker_id )
    );

    return $marker_name;
}

# ----------------------------------------------------
sub get_marker_mappings {

=pod

=head2 get_marker_mappings 

  my $mappings = $mdb->get_marker_mappings( marker_id => $marker_id );

Returns all the mappings associated with a marker.

=cut

    my ( $self, %args ) = @_;

    my $marker_id = $args{'marker_id'} or croak('No marker id');
    my $order_by  = $args{'order_by'} || 
                    q[
                        s.display_order, s.species, 
                        mpt.display_order, mpt.map_type, 
                        ms.map_set_name, 
                        mp.score desc,
                        mp.percent_identity desc,
                        map.display_order, map.map_name, 
                        mp.start, mp.end
                    ];

    my $db        = $self->db;
    my $mappings  = $db->selectall_arrayref(
        qq[
            select   mp.mapping_id,
                     mp.start,
                     mp.end,
                     mp.marker_start,
                     mp.marker_end,
                     mp.percent_identity,
                     mp.score,
                     mp.evalue,
                     mp.remark,
                     mp.cmap_feature_accession,
                     mk.marker_id,
                     mks.marker_name,
                     map.map_id,
                     map.map_name,
                     ms.map_set_id,
                     ms.map_set_name,
                     ms.distance_unit,
                     ms.cmap_map_set_accession,
                     ms.ensembl_species_name,
                     mpt.map_type_id,
                     mpt.map_type,
                     s.species_id,
                     s.species,
                     s.common_name as species_common_name
            from     marker mk,
                     marker_synonym mks,
                     mapping mp,
                     map,
                     map_set ms,
                     map_type mpt,
                     species s
            where    mp.marker_id=?
            and      mp.display_synonym_id=mks.marker_synonym_id
            and      mp.marker_id=mk.marker_id
            and      mp.map_id=map.map_id
            and      map.map_set_id=ms.map_set_id
            and      ms.map_type_id=mpt.map_type_id
            and      ms.species_id=s.species_id
            order by $order_by
        ],
        { Columns => {} },
        ( $marker_id )
    );

    return wantarray ? @$mappings : $mappings;
}

# ----------------------------------------------------
sub get_marker_sequence {

=pod

=head2 get_marker_sequence 

  my $seq = $mdb->get_marker_sequence( marker_id => $marker_id );

Returns the sequence (if any) associated with a marker.

=cut

    my ( $self, %args ) = @_;

    my $marker_id = $args{'marker_id'} or croak('No marker id');
    my $db        = $self->db;
    my $seq       = $db->selectrow_array(
        q[
            select   seq 
            from     marker_sequence
            where    marker_id=?
        ],
        {},
        ( $marker_id )
    );

    return $seq || $EMPTY_STR;
}

# ----------------------------------------------------
sub get_marker_synonyms {

=pod

=head2 get_marker_synonyms 

  my $synonyms = $mdb->get_marker_synonyms( marker_id => $marker_id );

Returns all the synonyms associated with a marker.

=cut

    my ( $self, %args ) = @_;

    my $marker_id = $args{'marker_id'} or croak('No marker id');
    my $db        = $self->db;
    my $synonyms  = $db->selectall_arrayref(
        q[
            select   s.marker_synonym_id, 
                     s.marker_name,
                     st.synonym_type_id,
                     st.synonym_type,
                     st.description,
                     st.url_template
            from     marker_synonym s, 
                     marker m,
                     synonym_type st
            where    s.marker_id=?
            and      s.marker_id=m.marker_id
            and      s.marker_synonym_id!=m.display_synonym_id
            and      s.synonym_type_id=st.synonym_type_id
            order by st.synonym_type, s.marker_name
        ],
        { Columns => {} },
        ( $marker_id )
    );

    if ( $args{'process_url'} ) {
        for my $syn ( @$synonyms ) {
            my $url_template = $syn->{'url_template'} or next;
            $syn->{'url'} = sprintf( $url_template, $syn->{'marker_name'} );
        }
    }

    return wantarray ? @$synonyms : $synonyms;
}

# ----------------------------------------------------
sub get_marker_images {

=pod

=head2 get_marker_images 

  my $images = $mdb->get_marker_images( marker_id => $marker_id );

Returns all the images associated with a marker.

=cut

    my ( $self, %args ) = @_;

    my $marker_id = $args{'marker_id'} or croak('No marker id');
    my $db        = $self->db;
    my $images    = $db->selectall_arrayref(
        q[
            select marker_image_id, marker_id, file_name, 
                   caption, width, height
            from   marker_image
            where  marker_id=?
        ],
        { Columns => {} },
        ( $marker_id )
    );

    return $images;
}

# ----------------------------------------------------
sub get_marker_type_id {

=pod

=head2 get_marker_type_id 

  my $marker_type_id = $mdb->get_marker_type_id( 'SSR' );

Returns the marker type ID for a given marker type if it exists.

=cut

    my $self           = shift;
    my $marker_type    = shift or croak('No marker type');
    my $db             = $self->db or return;
    my $marker_type_id = $db->selectrow_array(
        q[
            select marker_type_id
            from   marker_type
            where  marker_type=?
        ],
        {},
        ( $marker_type )
    );

    return $marker_type_id;
}

# ----------------------------------------------------
sub get_species_id {

=pod

=head2 get_species_id 

  my $species_id = $mdb->get_species_id( 'rice' );

Returns the species ID for a given species if it exists.

=cut

    my $self       = shift;
    my $species    = shift or croak('No species');
    my $db         = $self->db or return;
    my $species_id = $db->selectrow_array(
        q[
            select species_id
            from   species
            where  species=?
        ],
        {},
        ( $species )
    );

    return $species_id;
}

# ----------------------------------------------------
sub get_synonym_type_id {

=pod

=head2 get_synonym_type_id 

  my $synonym_type_id = $mdb->get_synonym_type_id('GENBANK_ACCESSION');

Returns the synonym_type ID for a given synonym_type if it exists.

=cut

    my $self            = shift;
    my $synonym_type    = shift or croak('No synonym type');
    my $db              = $self->db or return;
    my $synonym_type_id = $db->selectrow_array(
        q[
            select synonym_type_id
            from   synonym_type
            where  synonym_type=?
        ],
        {},
        ( $synonym_type )
    );

    return $synonym_type_id;
}

# ----------------------------------------------------
sub get_marker_details {

=pod

=head2 get_marker_details

  $marker->{'details'} = $mdb->get_marker_details(
      marker_id => $marker_id
  );

Selects all the fields from the "marker_details_" table.

=cut

    my ( $self, %args ) = @_;

    # Get the Marker
    my $cdbi_marker = $self->_args_to_Object( 'Marker', \%args )
        or croak( "Need a Marker or a valid marker_id" );
    my $marker_id = $cdbi_marker->id;

    my $marker_type = $args{'marker_type'} || '';
    my $db          = $self->db;

    if ( !$marker_type ) {
        $marker_type = $db->selectrow_array(
            q[
                select mt.marker_type
                from   marker m, marker_type mt
                where  m.marker_id=?
                and    m.marker_type_id=mt.marker_type_id
            ],
            {},
            ( $marker_id )
        );
    }

    my ( $table_name, @fields ) 
        = $self->marker_type_to_table_name( $marker_type );

    return {} if !$table_name;

    my $sql = 'select ' . join(', ', @fields) . 
        " from $table_name where marker_id=?";

    my $sth = $db->prepare( $sql );

    my $details = {};
    if ( $sth->execute( $marker_id ) ) {
        $details = $sth->fetchrow_hashref;
        $details->{'ordered_field_names'} = $sth->{'NAME'};
    }

    return $details;
}

# --------------------------------------------------------
sub get_ontology_term {

=pod

=head2 get_ontology_term 

  my $term = $mdb->get_ontology_term('EO:0000453');

Retrieves an ontology term by accession.

=cut

    my $self       = shift;
    my $accession  = shift or return;
    my ($ont_term) = Gramene::CDBI::Ontology::Term->search(
        { term_accession => $accession }
    );

    return $ont_term;
}

# ----------------------------------------------------
sub get_xref_type_id {

=pod

=head2 get_xref_type_id 

  my $xref_type_id = $mdb->get_xref_type_id( 'Gramene Literature' );

Returns the cross-reference type ID for a given xref if it exists.

=cut

    my $self         = shift;
    my $xref_type    = shift or croak('No xref type');
    my $db           = $self->db or return;
    my $xref_type_id = $db->selectrow_array(
        q[
            select xref_type_id
            from   xref_type
            where  xref_type=?
        ],
        {},
        ( $xref_type )
    );

    return $xref_type_id;
}


# ----------------------------------------------------
sub get_xrefs {

=pod

=head2 get_xrefs 

  my $xrefs = $mdb->get_xrefs( 
      table_name  => 'marker',
      record_id   => $marker_id,
      xref_type   => 'Gramene Literature', # optional
      process_url => 1, # optional, whether to process the "url_template"
  );

Returns all the xrefs associated with an object.

=cut

    my ( $self, %args ) = @_;
    my $table_name      = $args{'table_name'} or croak('No table name');
    my $record_id       = $args{'record_id'}  or croak('No record id');
    my $xref_type       = $args{'xref_type'}  || '';

    my $db  = $self->db;
    my $sql = q[
        select x.xref_id, 
               x.xref_value, 
               x.comments,
               xt.xref_type_id, 
               xt.xref_type, 
               xt.url_template
        from   xref x, xref_type xt
        where  x.table_name=?
        and    x.record_id=?
        and    x.xref_type_id=xt.xref_type_id
    ];

    my @args = ( $table_name, $record_id );

    if ( $xref_type ) {
        $sql .= 'and xt.xref_type=?';
        push @args, $xref_type;
    }

    $sql .= 'order by xt.xref_type, x.xref_value';
    my $xrefs = $db->selectall_arrayref( $sql, { Columns => {} }, @args );

    if ( $args{'process_url'} ) {
        for my $xref ( @$xrefs ) {
            my $url_template = $xref->{'url_template'} or next;
            $xref->{'url'} = sprintf $url_template, $xref->{'xref_value'};

            #
            # This expands the Gramene lit refs to get citation info.
            #
            if ( $xref->{'xref_type'} eq 'Gramene Literature' ) {
                my $ref = Gramene::CDBI::Literature::Reference->retrieve(
                    $xref->{'xref_value'}
                );

                if ( $ref ) {
                    my @re = $ref->reference_extras;
                    $xref->{'alternate_display'} = join( ', ',
                        $re[0]->author,
                        '<em>' . $ref->title . '</em>',
                        '<u>' . $ref->source->source_name . '</u>', 
                        $ref->volume,
                        $ref->year,        
                        'pp. ' . $ref->start_page . '-' . $ref->end_page
                    );
                }
                else {
                    $xref->{'alternate_display'} 
                        = "Bad Gramene literature ID ($xref->{'xref_value'})";
                }

            }
            elsif ( $xref->{'xref_type'} =~ /ontology/i ) {
                my ($ont_term) = Gramene::CDBI::Ontology::Term->search(
                    { term_accession => $xref->{'xref_value'} }
                );

                $xref->{'alternate_display'} = $ont_term->term_name;
            }
        }
    }

    return wantarray ? @$xrefs : $xrefs;
}

# ----------------------------------------------------
sub get_all_analyses {

=pod

=head2 get_all_analyses

  my $analyses = $mdb->get_all_analyses( 
      order_by => 'num_correspondences' 
  );

Returns all the analyses and the number of correspondences based on each.

=cut

    my ( $self, %args ) = @_;

    my @analyses;
    if ( my $analysis_name = $args{'analysis_name'} ) {
        my $method = $analysis_name =~ s/\*/%/g ? 'search_like' : 'search';

        @analyses = Gramene::CDBI::Markers::Analysis->$method(
            analysis_name => $analysis_name
        );

    }
    else {
        @analyses = Gramene::CDBI::Markers::Analysis->retrieve_all_sorted_by(
            'analysis_name'
        );
    }

    return wantarray ? @analyses : \@analyses;
}

# ----------------------------------------------------
sub get_all_analytical_correspondence_types {

=pod

=head2 get_all_analytical_correspondence_types

  my $types = $mdb->get_all_analytical_correspondence_types();

Returns all the analytical_correspondence_types.

=cut

    my ( $self, %args ) = @_;

    my $class = 'Gramene::CDBI::Markers::AnalyticalCorrespondenceType';
    my @types;

    if ( my $type = $args{'type'} ) {
        my $method = $type =~ s/\*/%/g ? 'search_like' : 'search';

        @types = $class->$method( type => $type );
    }
    else {
        @types = $class->retrieve_all_sorted_by( 'type' );
    }

    return wantarray ? @types : \@types;
}

# ----------------------------------------------------
sub get_all_germplasm {

=pod

=head2 get_all_germplasm

  my $germplasm = $mdb->get_all_germplasm( 
      order_by => 'num_markers' 
  );

Returns all the germplasm and the number of related populations, 
markers, and maps.

=cut

    my ( $self, %args ) = @_;
    my $search_value    = $args{'search_value'} || '';
    my $species_id      = $args{'species_id'}   || '';
    my $order_by        = $args{'order_by'}     || 'species';
    $order_by          .= ' desc' if $order_by eq 'num_markers';
    $order_by          .= ',germplasm_name' if $order_by eq 'species';

    my $extra_where;
    if ( $search_value ) {
        my $cmp = $search_value =~ s/\*/%/g ? 'like' : '=';
        $extra_where = "and g.germplasm_name $cmp '$search_value'";
    }

    my $db  = $self->db;
    my $sql = sprintf(
        qq[
            select    g.germplasm_id, 
                      g.germplasm_name,
                      sp.species_id,
                      sp.species,
                      count(m.marker_id) as num_markers
            from      species sp,
                      germplasm g left join 
                        marker m on g.germplasm_id=m.germplasm_id
            where     g.species_id=sp.species_id
            %s
            %s
            group by  germplasm_id, germplasm_name, species_id, species
            order by  %s
        ],
        $species_id ? "and g.species_id=$species_id" : '',
        $extra_where,
        $order_by
    );

    my $germplasm = $db->selectall_arrayref( $sql, { Columns => {} } );

    return $germplasm;
}

# ----------------------------------------------------
sub get_all_libraries {

=pod

=head2 get_all_libraries

  my $library = $mdb->get_all_libraries( 
      species_id => 42,
  );

Returns all the libraries matching criteria (or none).

=cut

    my ( $self, %args ) = @_;
    my $library_name    = $args{'library_name'} || '';
    my $species_id      = $args{'species_id'}   || '';
    my $order_by        = $args{'order_by'}     || 'species';
    $order_by          .= ',library_name' if $order_by eq 'species';

    my $extra_where;
    if ( $library_name ) {
        my $cmp = $library_name =~ s/\*/%/g ? 'like' : '=';
        $extra_where = "and l.library_name $cmp '$library_name'";
    }

    my $db  = $self->db;
    my $sql = sprintf(
        qq[
            select    l.library_id, 
                      l.library_name,
                      sp.species_id,
                      sp.species
            from      species sp,
                      library l
            where     l.species_id=sp.species_id
            %s
            %s
            order by  %s
        ],
        $species_id ? "and l.species_id=$species_id" : '',
        $extra_where,
        $order_by
    );

    my $libraries = $db->selectall_arrayref( $sql, { Columns => {} } );

    return $libraries;
}

#-----------------------------------------------------
sub get_all_maps {

=pod

=head2 get_all_maps

  my $maps = $mdb->get_all_maps( 
      map_set_id  => 18, 
  );

Returns all the map sets corresponding to an [optional] map_set_id.

=cut

    my ( $self, %args ) = @_;
    my $map_set_id      = $args{'map_set_id'}  || 0;

    my $sql = qq[
SELECT m.map_id,
       m.map_set_id,
       m.map_name,
       m.display_order,
       m.location_index,
       m.start,
       m.end,
       m.cmap_map_accession
FROM   map m ];

   if( $map_set_id ){
     $sql .= qq[
WHERE  m.map_set_id = $map_set_id ];     
   }

   return $self->db->selectall_arrayref( $sql, { Columns => {} } );

}

# ----------------------------------------------------
sub get_all_map_sets {

=pod

=head2 get_all_map_sets

  my $maps = $mdb->get_all_map_sets( 
      map_set_acc = 'foo,bar',
      species_id  => 18, 
      map_type_id => 3,
      order_by    => 'num_maps' 
  );

Returns all the map sets and the number of maps in each.

=cut

    my ( $self, %args ) = @_;
    my @map_set_accs    = map { split /,/, $_ } $args{'map_set_acc'};
    my $species_id      = $args{'species_id'}  || 0;
    my $map_type_id     = $args{'map_type_id'} || 0;
    my $order_by        = $args{'order_by'}    || 'map_type';
    $order_by          .= ' desc' if $order_by eq 'num_maps';
    my $db              = $self->db;
    my $sql             = sprintf(
        qq[
            select    ms.map_set_id, 
                      ms.map_set_name,
                      sp.species_id,
                      sp.species,
                      mt.map_type_id,
                      mt.map_type,
                      count(map.map_id) as num_maps
            from      map_type mt, 
                      species sp, 
                      map_set ms 
            left join map 
            on        ms.map_set_id=map.map_set_id
            where     ms.map_type_id=mt.map_type_id
            and       ms.species_id=sp.species_id
            %s
            %s
            %s
            group by  map_set_id, map_set_name, species_id, species, 
                      map_type_id, map_type
            order by  %s
        ],
        @map_set_accs
            ? 'and ms.cmap_map_set_accession in ('.
              join(',', map { qq['$_'] } @map_set_accs).')'
            : '',
        $species_id  ? "and ms.species_id=$species_id"   : '',
        $map_type_id ? "and ms.map_type_id=$map_type_id" : '',
        $order_by
    );

    my $map_sets = $db->selectall_arrayref( $sql, { Columns => {} } );

    return $map_sets;
}

# ----------------------------------------------------
sub get_all_map_types {

=pod

=head2 get_all_map_types

  my $map_types = $mdb->get_all_map_types( 
    order_by => 'num_maps' 
  );

Returns all the map types and the number of maps associated with 
each type. 

=cut

    my ( $self, %args ) = @_;
    my $order_by = $args{'order_by'} || 'map_type';
    $order_by   .= ' desc' if $order_by eq 'num_maps';

    my $db   = $self->db;
    my $map_types = $db->selectall_arrayref(
        qq[
            select    mt.map_type_id, 
                      mt.map_type,
                      mt.display_order,
                      count(ms.map_set_id) as num_map_sets
            from      map_type mt 
            left join map_set ms 
            on        mt.map_type_id=ms.map_type_id
            group by  map_type_id, map_type
            order by  $order_by
        ],
        { Columns => {} }
    );

    return $map_types;
}

# ----------------------------------------------------
sub get_all_marker_types {

=pod

=head2 get_all_marker_types

  my $marker_types = $mdb->get_all_marker_types( order_by => 'num_markers' );

Returns all the marker types and the number of markers associated with 
each type. 

=cut

    my ( $self, %args ) = @_;
    my $order_by = $args{'order_by'} || 'marker_type';
    $order_by   .= ' desc' if $order_by eq 'num_markers';

    my $db   = $self->db;
    my $marker_types = $db->selectall_arrayref(
        qq[
            select   marker_type_id, marker_type, is_sequence, description
            from     marker_type
            order by $order_by
        ],
        { Columns => {} }
    );

    return $marker_types;
}

# ----------------------------------------------------
sub get_all_populations {

=pod

=head2 get_all_populations

  my $populations = $mdb->get_all_populations( 
      order_by => 'population_name' 
  );

Returns all the marker types and the number of markers associated with 
each type. 

=cut

    my ( $self, %args ) = @_;
    my $order_by        = $args{'order_by'} || 'population_name';
    my $db              = $self->db;
    my $populations     = $db->selectall_arrayref(
        qq[
            select    p.population_id, 
                      p.population_name,
                      sp.species_id,
                      sp.species,
                      g1.germplasm_id as male_germplasm_id,
                      g1.germplasm_name as male_germplasm,
                      g2.germplasm_id as female_germplasm_id,
                      g2.germplasm_name as female_germplasm
            from      population p, 
                      species sp,
                      germplasm g1,
                      germplasm g2
            where     p.species_id=sp.species_id
            and       p.male_germplasm_id=g1.germplasm_id
            and       p.female_germplasm_id=g2.germplasm_id
            order by  $order_by
        ],
        { Columns => {} }
    );

    return $populations;
}

# ----------------------------------------------------
sub get_all_species {

=pod

=head2 get_all_species

  my $species = $mdb->get_all_species( order_by => 'num_markers' );

Returns all the species and the number of maps associated with each.

=cut

    my ( $self, %args ) = @_;
    my $search_for      = $args{'search_for'} || '';
    my $order_by        = $args{'order_by'}   || 'species';
    $order_by          .= ' desc' if $order_by eq 'num_markers';

    my $sql = qq[
        select   species_id, 
                 species,
                 common_name,
                 display_order,
                 gramene_taxonomy_id,
                 description
        from     species 
    ];

    if ( $search_for ) {
        my $cmp = $search_for =~ s/\*/%/g ? 'like' : '=';
        $sql   .= qq[
            where (species $cmp '$search_for'
            or     common_name $cmp '$search_for' 
            or     gramene_taxonomy_id $cmp '$search_for' 
            )
        ];
    }

    if ( $order_by ) {
        $sql .= " order by $order_by";
    }

    my $db = $self->db;
    my $species = $db->selectall_arrayref( $sql, { Columns => {} } );

    return $species;
}

# ----------------------------------------------------
sub get_all_xref_types {

=pod

=head2 get_all_xref_types

  my $xref_type = $mdb->get_all_xref_types( order_by => 'xref_type' );

Returns all the xref_types.

=cut

    my ( $self, %args ) = @_;
    my $order_by        = $args{'order_by'} || 'xref_type';
    my $db              = $self->db;
    my $xref_types      = $db->selectall_arrayref(
        qq[
            select   x.xref_type_id, 
                     x.xref_type,
                     x.url_template
            from     xref_type x
            order by $order_by
        ],
        { Columns => {} }
    );

    return $xref_types;
}

# --------------------------------------------------------
sub marker_search {

=pod

=head2 marker_search 

  my $markers = $mdb->marker_search( 
      marker_id        => 8832,    # takes ultimate precedence
      marker_name      => 'RG*, RZ*',
      marker_type      => 'RFLP',  # or ['RFLP', 'AFLP']
      marker_type_id   => 18,      # or [18,19]
      synonym_type     => 'GENBANK_ACCESSION', # or ['...', '...']
      synonym_type_id  => 18,      # or [18,19]
      library_id       => 42,      # or [18,19]
      germplasm_id     => 4,       # or [3,4]
      analysis_id      => 4,       # or [3,4]
      species_id       => 3,       # or [3,4]
      map_set_id       => 4,       # or [3,4]
      species          => 'rice',  # or ['rice', 'sorghum']
      genus            => 'Oryza', # or ['Oryza', 'Zea']
      feature_acc      => 'cu-dh-2001-12-27',
      is_sequence      => 1,       # for "marker_type.is_sequence"
      xref_type        => 'GenBank',
      xref_value       => 'AA231754',
      order_by         => 'type',
      search_only_primary_name => 1, 

      max_returned     => 500,    # will cause an error if results exceed
      limit_no         => 25,     # page size
      current_page     => 1,      # used with "limit_no"
  );

Returns all the markers matching the parameters given.

=cut

    my ( $self, %args )  = @_;
    my $marker_id        = $args{'marker_id'}        ||  0;
    my $marker_name      = $args{'marker_name'}      || '';
    my $xref_type        = $args{'xref_type'}        || '';
    my $xref_value       = $args{'xref_value'}       || '';
    my $cmap_feature_acc = $args{'feature_acc'} 
                        || $args{'cmap_feature_acc'}
                        || $args{'cmap_feature_aid'} || '';
    my $is_sequence      = $args{'is_sequence'}      ||  0;
    my $max_returned     = $args{'max_returned'}     ||  0;
    my $order_by         = $args{'order_by'}         || ''; # 'marker_name';
    my $current_page     = $args{'current_page'}     ||  1;
    my $limit_no         = $args{'limit_no'}         ||  0;
    my $timeout          = $args{'timeout'}          ||  0;
    my $search_only_primary_name = $args{'search_only_primary_name'} || 0;

    my @marker_types = map { $_ || () } uniq(
        ref $args{'marker_type'} eq 'ARRAY'
        ? @{ $args{'marker_type'} } 
        : split( /,/, $args{'marker_type'} || '')
    );

    my @species = map { $_ || () } uniq(
        ref $args{'species'} eq 'ARRAY'
        ? @{ $args{'species'} } 
        : split( /,/, $args{'species'} || '' )
    );

    my @marker_type_ids  = map { $_ || () } uniq(
        ref $args{'marker_type_id'} eq 'ARRAY' 
        ? @{ $args{'marker_type_id'} } 
        : split( /,/, $args{'marker_type_id'} || '' )
    );

    my @synonym_types = map { $_ || () } uniq(
        ref $args{'synonym_type'} eq 'ARRAY'
        ? @{ $args{'synonym_type'} } 
        : split( /,/, $args{'synonym_type'} || '' )
    );

    my @synonym_type_ids  = map { $_ || () } uniq(
        ref $args{'synonym_type_id'} eq 'ARRAY' 
        ? @{ $args{'synonym_type_id'} } 
        : split( /,/, $args{'synonym_type_id'} || '' )
    );

    my @library_ids = map { $_ || () } uniq(
        ref $args{'library_id'} eq 'ARRAY' 
        ? @{ $args{'library_id'} } 
        : split( /,/, $args{'library_id'} || '' )
    );

    my @species_ids = map { $_ || () } uniq(
        ref $args{'species_id'} eq 'ARRAY' 
        ? @{ $args{'species_id'} } 
        : split( /,/, $args{'species_id'} || '' )
    );

    my @genera = map { $_ || () } uniq(
        ref $args{'genus'} eq 'ARRAY' 
        ? @{ $args{'genus'} } 
        : split( /,/, $args{'genus'} || '' )
    );

    my @germplasm_ids = map { $_ || () } uniq(
        ref $args{'germplasm_id'} eq 'ARRAY' 
        ? @{ $args{'germplasm_id'} } 
        : split( /,/, $args{'germplasm_id'} || '' )
    );

    my @analysis_ids = map { $_ || () } uniq(
        ref $args{'analysis_id'} eq 'ARRAY' 
        ? @{ $args{'analysis_id'} } 
        : split( /,/, $args{'analysis_id'} || '' )
    );

    my @map_set_ids = ref $args{'map_set_id'} eq 'ARRAY' 
        ? @{ $args{'map_set_id'} } 
        : split( /,/, $args{'map_set_id'} || '' )
    ;

    my @map_set_species_ids = map { $_ || () } uniq(
        ref $args{'map_set_species_id'} eq 'ARRAY' 
        ? @{ $args{'map_set_species_id'} } 
        : split( /,/, $args{'map_set_species_id'} || '' )
    );

    for my $ms_species_id ( @map_set_species_ids ) {
        for my $ms ( 
            Gramene::CDBI::Markers::MapSet->search(
                { species_id => $ms_species_id }
            ) 
        ) {
            push @map_set_ids, $ms->id;
        }
    }

    @map_set_ids = map { $_ || () } uniq( @map_set_ids );

    my $db = $self->db or return;

    if ( @marker_types && !@marker_type_ids ) {
        for my $marker_type ( @marker_types ) {
            next unless $marker_type;
            my $id = $self->get_marker_type_id( $marker_type ) or next;
            push @marker_type_ids, $id;
        }
    }

    if ( @species && !@species_ids ) {
        for my $species ( @species ) {
            next unless $species;
            my $id = $self->get_species_id( $species ) or next;
            push @species_ids, $id;
        }
    }

    if ( @synonym_types && !@synonym_type_ids ) {
        for my $synonym_type ( @synonym_types ) {
            next unless $synonym_type;
            my $id = $self->get_synonym_type_id( $synonym_type ) or next;
            push @synonym_type_ids, $id;
        }
    }

    for my $genus ( @genera ) {
        for my $species (
            Gramene::CDBI::Markers::Species->search_like(
                { species => "$genus%" }
            )
        ) {
            push @species_ids, $species->id;
        }
    }

    my @marker_names;
    if ( ref $marker_name eq 'ARRAY' ) {
        @marker_names = uniq( @$marker_name );
    }
    else {
        @marker_names = uniq(
            map { 
                s/,//g;         # remove commas
                s/^\s+|\s+$//g; # remove leading/trailing whitespace
                s/"//g;         # remove double quotes (")
                s/'/\\'/g;      # backslash escape single quotes
                $_ || ()
            }
            parse_words( $marker_name )
        );
    }

    push @marker_names, '*' unless @marker_names;

    my ( $markers, $count );
    if ( $marker_id ) {
        my $sth = $db->prepare(
            qq[
                select    m.marker_id, 
                          syn.marker_name, 
                          sp.species_id,
                          sp.species,
                          t.marker_type_id,
                          t.marker_type
                from      marker m, 
                          marker_synonym syn,
                          species sp,
                          marker_type t
                where     m.marker_id=?
                and       m.display_synonym_id=syn.marker_synonym_id
                and       m.marker_type_id=t.marker_type_id
                and       m.source_species_id=sp.species_id
            ]
        );

        $sth->execute( $marker_id );
        my $marker = $sth->fetchrow_hashref 
                     or croak("Bad marker id ($marker_id)");

        @$markers = ( $marker );
    }
    elsif ( $cmap_feature_acc ) {
        $markers = $db->selectall_arrayref(
            sprintf(
                qq[
                    select    m.marker_id, 
                              syn.marker_name, 
                              sp.species_id,
                              sp.species,
                              t.marker_type_id,
                              t.marker_type
                    from      mapping mp,
                              marker_synonym syn,
                              species sp,
                              marker_type t,
                              marker m 
                    left join mapping mp2 
                    on        m.marker_id=mp2.marker_id
                    where     mp.cmap_feature_accession %s
                    and       mp.marker_id=m.marker_id
                    and       m.display_synonym_id=syn.marker_synonym_id
                    and       m.marker_type_id=t.marker_type_id
                    and       m.source_species_id=sp.species_id
                    %s
                ],
                $cmap_feature_acc =~ s/\*/%/g 
                    ? "like '$cmap_feature_acc'"
                    : "='$cmap_feature_acc'",
                $order_by ? "order by $order_by " : ''
            ),
            { Columns => {} }
        );
    }
    elsif ( $xref_type ) {
        my $sql = sprintf(
             q[
                select   m.marker_id, 
                         syn.marker_name, 
                         sp.species_id,
                         sp.species,
                         t.marker_type_id,
                         t.marker_type
                from     xref_type xt,
                         xref x,
                         marker m, 
                         marker_synonym syn,
                         species sp,
                         marker_type t
                where    xt.xref_type=?
                and      xt.xref_type_id=x.xref_type_id
                and      x.table_name=?
                %s
                and      x.record_id=m.marker_id
                and      m.display_synonym_id=syn.marker_synonym_id
                and      m.marker_type_id=t.marker_type_id
                and      m.source_species_id=sp.species_id
                %s
            ],
            $xref_value ? 'and x.xref_value=?' : '',
            $order_by ? "order by $order_by " : ''
        );
        my @args = ( $xref_type, 'marker' );
        push @args, $xref_value if $xref_value;

        $markers = $db->selectall_arrayref( $sql, { Columns => {} }, @args );
    }
    elsif ( @marker_names ) {
        my $sql;
        my $fields = q[ 
            distinct m.marker_id, 
            syn2.marker_name, 
            sp.species_id,
            sp.species,
            t.marker_type_id,
            t.marker_type
        ];

        if ( scalar @marker_names == 1 && $marker_names[0] eq '*' ) {
            $sql = q[
                select    %s
                from      marker m, 
                          marker_synonym syn2,
                          species sp,
                          marker_type t
                where     m.display_synonym_id=syn2.marker_synonym_id
                and       m.marker_type_id=t.marker_type_id
                and       m.source_species_id=sp.species_id
            ];
        }
        else {
            my @conditions;
            for my $name ( @marker_names ) {
                my $cmp = q{=};

                if ( $name =~ $MARKER_NAME_WITH_SYN_TYPE ) {
                    $name = $1;
                }

                if ( $name =~ s/\\\*/*/g ) {
                    # this allows actual asterisks in names
                    # to be backslash-escaped, e.g., "ad\*-N377B"
                    ;
                }
                elsif ( $name =~ s/\*/%/g ) {
                    $cmp = 'like';
                }

                push @conditions, "syn1.marker_name $cmp '$name'";
            }

            my $where = ' where (' . shift @conditions;
            $where   .= " or $_ " for @conditions;
            $where   .= ') ';

            $sql = qq[
                select    %s
                from      marker m, 
                          marker_synonym syn1,
                          marker_synonym syn2,
                          species sp,
                          marker_type t
                $where   
                and       syn1.marker_id=m.marker_id
                and       m.display_synonym_id=syn2.marker_synonym_id
                and       m.marker_type_id=t.marker_type_id
                and       m.source_species_id=sp.species_id
            ];
        }

        if ( @map_set_ids ) {
            $sql =~ s/\bwhere/
                , mapping, map
                where m.marker_id=mapping.marker_id
                and   mapping.map_id=map.map_id
                and 
            /;
        }

        if ( scalar @map_set_ids == 1 ) {
            $sql .= 'and map.map_set_id=' . $map_set_ids[0] . ' ';
        }
        elsif ( @map_set_ids ) {
            $sql .= 'and map.map_set_id in (' .
                    join( ', ', @map_set_ids ) . 
            ') ';
        }

        if ( scalar @marker_type_ids == 1 ) {
            $sql .= 'and m.marker_type_id=' . $marker_type_ids[0] . ' ';
        }
        elsif ( @marker_type_ids ) {
            $sql .= 'and m.marker_type_id in (' .
                    join( ', ', @marker_type_ids ) . ') ';
        }

        if ( scalar @library_ids == 1 ) {
            $sql .= 'and m.library_id=' . $library_ids[0] . ' ';
        }
        elsif ( @library_ids ) {
            $sql .= 'and m.library_id in (' .
                    join( ', ', @library_ids ) . ') ';
        }

        if ( scalar @species_ids == 1 ) {
            $sql .= 'and m.source_species_id=' . $species_ids[0] . ' ';
        }
        elsif ( @species_ids ) {
            $sql .= 'and m.source_species_id in (' .
                    join( ', ', @species_ids ) . 
            ') ';
        }

        if ( scalar @synonym_type_ids == 1 ) {
            $sql .= 'and syn2.synonym_type_id='.$synonym_type_ids[0].' ';
        }
        elsif ( @synonym_type_ids ) {
            $sql .= 'and syn2.synonym_type_id in (' .
                    join( ', ', @synonym_type_ids ) . 
            ') ';
        }

        if ( scalar @germplasm_ids == 1 ) {
            $sql .= 'and m.germplasm_id=' . $germplasm_ids[0] . ' ';
        }
        elsif ( @germplasm_ids ) {
            $sql .= 'and m.germplasm_id in (' .
                    join( ', ', @germplasm_ids ) . 
            ') ';
        }

        if ( scalar @analysis_ids == 1 ) {
            $sql .= 'and m.analysis_id=' . $analysis_ids[0] . ' ';
        }
        elsif ( @analysis_ids ) {
            $sql .= 'and m.analysis_id in (' .
                    join( ', ', @analysis_ids ) . 
            ') ';
        }

        if ( $search_only_primary_name ) {
            $sql .= 'and m.display_synonym_id=syn1.marker_synonym_id ';
        }

        if ( $is_sequence ) {
            $sql .= ' and t.is_sequence=1 ';
        }

        my $count_sql = sprintf( $sql, 'count(distinct m.marker_id) as count' );
        $count        = $db->selectrow_array( $count_sql );

        if ( $max_returned ) {
            croak(
                'Found ' . commify($count) . ' markers; maximum of ' .
                commify($max_returned) . ' allowed. Please narrow your search.'
            ) if $count > $max_returned;
        }

        $sql .= " order by $order_by " if $order_by;

        if ( $limit_no > 0 && $count > $limit_no ) {
            my $start = $current_page == 1 
                ? 0 
                : ( ( $current_page - 1 ) * $limit_no ) - 1;
            $sql .= " limit $start, $limit_no";
        } 

        my $select_sql = sprintf( $sql, $fields );

        eval {
            local $SIG{'ALRM'} = sub { 
                die "Marker search timed out after $timeout seconds\n" 
            };
            alarm $timeout if $timeout && $^O !~ /Win32/i;
            $markers = $db->selectall_arrayref($select_sql, { Columns => {} });
            alarm 0 unless $^O =~ /Win32/i;
        }; 

        if ( my $err = $@ ) {
            print STDERR __PACKAGE__, "::marker_search error: ",
                "$err\nSQL=\n$select_sql";
            croak( $err );
            return wantarray ? () : undef;
        }
    }

    return $limit_no 
        ? ( $count || scalar @{ $markers }, $markers ) 
        : wantarray ? @$markers : $markers;
}


# --------------------------------------------------------
=pod

=head2 search_marker_id_by_query

given a query, return an array of marker_ids 

=cut

sub search_marker_ids_by_query {

    my ( $self, $select_sql )  = @_;

    unless ($select_sql =~ m/select \s+ \S* marker_id/xms){
      print STDERR "[*ERR] Invalid query $select_sql\n";
      return; 
    }

    my $db = $self->db or return;

    my $marker_ids_ref = $db->selectcol_arrayref( $select_sql );

    return @$marker_ids_ref;

}

#----------------------------------------------------------------------
sub search_marker_synonyms {

=pod

=head2 search_marker_synonyms

  my @synonyms = $mdb->search_marker_synonyms( 
      synonyms => ['CK986212', '45568753', 'CK986212.1'], REQUIRED
      wildcard        => '1',               # optional, def 0
      MarkerType      => $cdbi_marker_type, # optional
      Species         => $cdbi_spcies,      # optional, or
      SynonymType     => $cdbi_synonym_type # optional
  );

Queries the database for a list of synonyms, can be wildcarded (* or %).
A hashref is returned for each match, consisting of:

  marker_id
  marker_synonym_id
  marker_name
  synonym_type_id,
  synonym_type
  marker_type_id
  marker_type.

=cut

    my ( $self, %args ) = @_;

    my $cdbi_species      = $self->_args_to_Object( 'Species',     \%args );
    my $cdbi_marker_type  = $self->_args_to_Object( 'MarkerType',  \%args );
    my $cdbi_synonym_type = $self->_args_to_Object( 'SynonymType', \%args );
    my @synonyms          = uniq(
        map { defined $_ && $_ ne $EMPTY_STR ? $_ : () }
        ref $args{'synonyms'} eq 'ARRAY'
        ? @{ $args{'synonyms'} }
        : ( $args{'synonyms'} )
    );

    if ( !@synonyms ) {
        croak('No synonyms');
    }

    # Fuzzy-match the species; Oryza sativa == Oryza sativa (indica) etc
    my @species_ids = ();
    if ( $cdbi_species ) {
        @species_ids = map { $_->id } $self->expand_Species( $cdbi_species );
    }

    my $db = $self->db or return;
    my $q = q[ 
         SELECT m.marker_id,
                m.display_synonym_id,
                s.marker_synonym_id,
                s.marker_name,
                st.synonym_type_id,
                st.synonym_type,
                mt.marker_type_id,
                mt.marker_type
         FROM   marker m, 
                marker_synonym s,
                synonym_type st,
                marker_type mt
         WHERE  m.marker_id=s.marker_id 
         AND    s.synonym_type_id=st.synonym_type_id
         AND    m.marker_type_id=mt.marker_type_id %s 
    ];

    my $extra = '';
    if ( grep { m/\*|\%/ } @synonyms ) {
        # Wildcarded
        my @syn_clauses = ();
        for my $syn ( @synonyms ) {
            $syn =~ s/\*/\%/g;
            $syn = $db->quote( $syn );
            push @syn_clauses, qq[ marker_name like $syn ];
        }

        $extra .= sprintf qq[ AND    ( %s )], join( ' or ', @syn_clauses );
    }
    else {
        # No wildcards; use 'IN' list
        $extra .= sprintf qq[ AND    marker_name IN (%s)],
            join( ', ', map { $db->quote( $_ ) } @synonyms );
    }

    if ( $cdbi_marker_type ) {
        my $marker_type_id = $cdbi_marker_type->id;
        $extra .= qq[
         AND    m.marker_type_id=$marker_type_id ];
    }

    if ( $cdbi_synonym_type ) {
        my $synonym_type_id = $cdbi_synonym_type->id;
        $extra .= qq[
         AND    ms.synonym_type_id=$synonym_type_id ];
    }

    if ( @species_ids ) {
        my $in_list = join( ", ", @species_ids );
        $extra .= qq[
         AND    m.source_species_id IN ( $in_list ) ];
    }

    $q = sprintf( $q, $extra );

    my $data = $db->selectall_arrayref( $q, { Columns => {} } );

    return wantarray ? @$data : $data;
}

# --------------------------------------------------------
# search mappings to TIGR assembly for this marker_id
# input: marker_id
# returned: arrayref to mapping_ids for this marker mapped to TIGR assembly

sub search_tigr_mappings_for_markerid {

    my $self = shift;
    my $marker_id = shift || return;

    #get tigr map_set_id
    $self->{TIGR_MAP_SET_NAME} ||= 'Gramene Annotated Nipponbare Sequence%';

    unless ( $self->{TIGR_MAPSET_IDS} ) {
        my @MapSets = Gramene::CDBI::Markers::MapSet->search_like(
            map_set_name => $self->{TIGR_MAP_SET_NAME} );
        $self->{TIGR_MAPSET_IDS} = [ map { $_->map_set_id() } @MapSets ];
    }

    #my $mapset_ids = join ",", @{ $self->{TIGR_MAPSET_IDS} };

    my $db  = $self->db;
    my $sth = $db->prepare(
        qq[
            select mp.mapping_id
            from   mapping mp, map
            where  mp.map_id = map.map_id
            and    mp.marker_id = ?
            and    map.map_set_id = ? 
        ]
    ) or die "Cannot prepare sth for select mappings";

    my @mapping_ids;

    for my $mapset_id ( @{ $self->{TIGR_MAPSET_IDS} } ) {

        $sth->execute( $marker_id, $mapset_id );

        my $mappings = $sth->fetchall_arrayref or croak(
              "Cannot fetchall_arrayref for mappings "
            . "[marker_id=$marker_id, mapset_id=$mapset_id]"
        );

        push @mapping_ids, map { $_->[0] } @{$mappings};
    }

    return \@mapping_ids;

}

# --------------------------------------------------------
sub marker_synonym_search {

=pod

=head2 marker_synonym_search 

  my @markers = $mdb->marker_synonym_search( 
      synonyms => ['CK986212', '45568753', 'CK986212.1'],
      marker_type_id  => 26,                # optional, or
      MarkerType      => $cdbi_marker_type, # optional
      species_id      => 12,                # optional, or
      Species         => $cdbi_spcies,      # optional, or
      species_ids     => [12,13,14]         # optional,
      synonym_type_id => 5,                 # optional, or
      SynonymType     => $cdbi_synonym_type # optional
  );

Returns IDs of all markers matching a list of synonyms. Used as quick
check of whether marer already exists by e.g. load scripts. 
Can specify an optional MarkerType or marker_type_id, Species or species_id,
SynonymType or synonym_type_id.

=cut

    my ( $self, %args ) = @_;
    my @synonyms       = @{ $args{'synonyms'} || [] } or 
                   croak('Need some synonyms [listref]');

    
    my $marker_type_id;
    my $species_id;
    my $synonym_type_id;
    if   ( my $v = $args{'marker_type_id' } ){ $marker_type_id  = $v }
    elsif( my $o = $args{'MarkerType'     } ){ $marker_type_id  = $o->id }
    if   ( my $v = $args{'species_id'     } ){ $species_id      = $v }
    elsif( my $o = $args{'Species'        } ){ $species_id      = $o->id }
    if   ( my $v = $args{'synonym_type_id'} ){ $synonym_type_id = $v }
    elsif( my $o = $args{'SynonymType'    } ){ $synonym_type_id = $o->id }

    my $db = $self->db or return;
    my $q = qq[ 
        SELECT DISTINCT( m.marker_id )
        FROM   marker m,
               marker_synonym ms
        WHERE  m.marker_id=ms.marker_id %s
        AND    marker_name IN(%s) ];

    my $extra = '';
    if( $marker_type_id ){
      $extra .= qq[
        AND    m.marker_type_id=$marker_type_id ];
    }
    if( $species_id ){
      $extra .= qq[
        AND    m.source_species_id=$species_id ];
    }
    if( $synonym_type_id ){
      $extra .= qq[
        AND    ms.synonym_type_id=$synonym_type_id ];
    }

    $q = sprintf( $q, $extra, join( ", ", ( map{"'$_'"} @synonyms ) ) );
    my $markers = $db->selectcol_arrayref( $q );
    return wantarray ? @$markers : $markers;
}

# --------------------------------------------------------
sub marker_type_to_table_name {

=pod

=head2 marker_type_to_table_name

  my ( $table_name, @field_names ) = $mdb->marker_type_to_table_name('RFLP');


=cut

    my $self              = shift;
    my $orig_marker_type  = shift;
    my $marker_type       = lc $orig_marker_type;
    $marker_type          =~ s/\s+/_/g;
    my $table_name        = 'marker_details_' . $marker_type;
    my $class             = 'Gramene::CDBI::Markers::' . 
                             join '', map { ucfirst $_ } split /_/, $table_name;
    my %valid_marker_type = map { lc $_->marker_type, $_->id }
                            Gramene::CDBI::Markers::MarkerType->retrieve_all;

    if ( lc $marker_type eq 'undefined' ) {
        return;
    }

    my %valid = map { $_, 1 } Gramene::CDBI::Markers->represented_tables;

    if ( !$valid{ $table_name } ) {
        if ( !$valid_marker_type{ lc $orig_marker_type } ) {
            croak("Invalid marker type ($orig_marker_type)");
        }
        else {
            return; # valid marker type w/o a details table
        }
    }

    if ( my @fields = $class->columns('Ordered') ) {
        return wantarray ? ( $table_name, @fields ) : $table_name;
    }
    else {
        croak("Class $class for marker type $orig_marker_type has no columns!");
    }
}

# --------------------------------------------------------                      
sub add_synonyms_to_Marker {

=pod

=head2 add_synonyms_to_Marker

  my $display_synonym = $mdb->add_synonyms_to_Marker(
     Marker   => $Gramene_CDBI_Markers_Marker,
     synonyms => [ 
       'C22468',                       # will be of synonym type 'UNKNOWN'
       '"C22468 [GENBANK_ACCESSION]"', # will be parsed for syn type
       { marker_name => 'Foo', synonym_type    => 'GENBANK' },
       { marker_name => 'Bar', synonym_type_id => 3 },
       { marker_name => 'Wib', 
         SynonymType => $Gramene_CDBI_Markers_SynonymType } 
     ] 
  );

Takes a Gramene::CDBI::Markers::Marker object and a list of hashrefs
representing marker_synonyms. Makes sure each name exists as a synonym
of the marker ID.  Returns the "synonym_obj" of the first name as this
one is considered the primary name and should be used for the 
"display_synonym_id" in the "marker" and/or "mapping" table.

=cut

    my ( $self, %args ) = @_;
    my $marker          = $args{'Marker'};
    my @synonyms        = uniq( @{ $args{'synonyms'} || [] } );

    unless ( UNIVERSAL::isa( $marker, "${CDBI}::Marker" ) ) {
        croak( "No ${CDBI}::Marker" );
    }

    unless ( @synonyms ) {
        croak( 'No synonym data' );
    }

    my %existing_syns = map { $_->marker_name => $_ } $marker->marker_synonyms;

    my $display_synonym;
    SYNONYM:
    for my $synref ( @synonyms ) {
        unless ( ref $synref eq 'HASH' ) {
            if ( $synref =~ $MARKER_NAME_WITH_SYN_TYPE ) {
                my $s  = $1;
                my $st = $2;
                $s =~ s/\s+$//g;

                $synref = {
                    marker_name  => $s,
                    synonym_type => $st,
                };
            }
            elsif ( defined $synref && $synref ne $EMPTY_STR ) {
                # 'plain' marker name
                $synref = { marker_name => $synref };
            }
            else {
                $synref = undef;
            }
        }

        next SYNONYM if !$synref;

        my $name = $synref->{'marker_name'}
            || croak( 'Synonym has no marker_name' );
        $name    =~ s/^"|"$//g; # remove quotes

        my $type_obj;
        my $type_id;
        if ( $type_obj = $synref->{'SynonymType'} ) {
            unless ( UNIVERSAL::isa( $type_obj, "${CDBI}::SynonymType" ) ) {
                croak( "No ${CDBI}::SynonymType" );
            }

            $type_id = $type_obj->id;
        }

        if ( !$type_obj and $type_id = $synref->{'synonym_type_id'} ) {
            $type_obj = $self->retrieve_SynonymType( $type_id )
                || croak( "No SynonymType with ID $type_id" );
        }

        my $syn_type = $synref->{'synonym_type'};
        if ( !$type_id && $syn_type ) {
            $type_obj = $self->find_or_create_SynonymType(
                synonym_type => $syn_type 
            );
            $type_id = $type_obj->id;
        }

        my $regex;
        if ( $type_obj and ( $regex = $type_obj->validation ) ) {
            if ( $name !~ /$regex/ ) {
                croak( "'$name' does not appear to be a valid $syn_type" );
            }
        }

        # Is this synonym already part of the object?
        my $syn_obj;
        if ( my $old_syn = $existing_syns{ $name } ) {
            my $old_type = $old_syn->synonym_type;

            if (   !$type_id
                or $type_id == $old_type->id
                or uc( $type_obj->synonym_type ) eq $UNKNOWN 
            ) {    
                # Skip this syn
                $syn_obj = $old_syn;
            }
            elsif ( uc( $old_type->synonym_type ) eq $UNKNOWN ) {
                # Update this type
                $old_syn->synonym_type_id( $type_id );
                $old_syn->update;
                $syn_obj = $old_syn;
            }
        }

        if ( !$syn_obj ) {
            # Add the synonym
            my %create = ( marker_id => $marker->id, marker_name => $name );
            if ( $type_id ) { $create{synonym_type_id} = $type_id }
            $syn_obj = $self->find_or_create_MarkerSynonym( {%create} );
            $existing_syns{$name} = $syn_obj;    # Used to screen duplicates
        }

        $display_synonym ||= $syn_obj;
    }

    return $display_synonym;
}

# --------------------------------------------------------
sub make_marker_synonyms {

=pod

=head2  make_marker_synonyms

  my $display_synonym_id = $mdb->make_marker_synonyms( 
      $marker_id, 'Foo', 'Bar', 'Baz'
  );

DEPRECATED: use $mdb->add_synonyms_to_Marker instead

=cut

#    carp( "make_marker_synonyms is DEPRECATED; use add_synonyms_to_Marker" );
    my ( $self, $marker_id, @names ) = @_;
    return unless $marker_id && @names;

    my $marker_obj = Gramene::CDBI::Markers::Marker->retrieve($marker_id)
        || croak("No marker with id $marker_id");

    # create array of hashrefs
    my @synonyms;
    for my $name ( @names ) {
        if ( ref $name eq 'HASH' ) {
            push @synonyms, $name;
        }
        else {
            push @synonyms, { marker_name => $name };
        }
    }

    my $display_syn = $self->add_synonyms_to_Marker(
         Marker     => $marker_obj,
         synonyms   => \@synonyms,
    );

    return $display_syn->id;
}

# --------------------------------------------------------
sub merge_markers {

=pod

=head2 merge_markers 

  $mdb->merge_markers(
      target_marker_id     => 1984,
      deprecated_marker_id => 42,
  );

Merges the information from "deprecated_marker_id" into "target_marker_id."
Deletes the deprecated marker.

=cut

    my ( $self, %args )      = @_;
    my $target_marker_id     = $args{'target_marker_id'} or 
        croak('No target marker id');
    my $deprecated_marker_id = $args{'deprecated_marker_id'} or 
        croak('No deprecated marker id');

    if ( $target_marker_id == $deprecated_marker_id ) {
        croak("Cannot merge a marker with itself!");
    }

    my $db = $self->db;

    for my $marker_id ( $target_marker_id, $deprecated_marker_id ) {
        my $exists = $db->selectrow_array(
            'select count(*) from marker where marker_id=?', {}, ( $marker_id )
        );

        # Bail if one doesn't exist
        return 1 if !$exists;
    }

    my $orig_re = $db->{'RaiseError'};
    my $orig_pe = $db->{'PrintError'};
    my $orig_ac = $db->{'AutoCommit'};

    $db->{'RaiseError'} = 1;
    $db->{'PrintError'} = 0;
    $db->{'AutoCommit'} = 0;

    eval {
        my $marker_types = $db->selectall_hashref( 
            qq[
                select m.marker_id, mt.marker_type_id, mt.marker_type
                from   marker m, marker_type mt
                where  m.marker_id in ($target_marker_id, $deprecated_marker_id)
                and    m.marker_type_id=mt.marker_type_id
            ],
            'marker_id'
        );

        if ( 
            $marker_types->{$target_marker_id}{'marker_type'} ne 'Undefined'
            &&
            $marker_types->{$deprecated_marker_id}{'marker_type'} ne 'Undefined'
            && (
                $marker_types->{ $target_marker_id }{'marker_type_id'} !=
                $marker_types->{ $deprecated_marker_id }{'marker_type_id'}
            )
        ) {
            die "Can't merge markers of different types (",
                $marker_types->{ $target_marker_id }{'marker_type'}, ', ',
                $marker_types->{ $deprecated_marker_id }{'marker_type'}, ')'
            ;
        }
        my $marker_type = $marker_types->{ $target_marker_id }{'marker_type'};

        my $tseq = $self->get_marker_sequence( marker_id => $target_marker_id );
        my $dseq 
            = $self->get_marker_sequence( marker_id => $deprecated_marker_id );

        if ( $tseq && $dseq && ( $tseq ne $dseq ) ) {
            die "Both markers have sequence and are different.\n";
        }
        elsif ( $dseq ) {
            $db->do(
                'delete from marker_sequence where marker_id=?', {},
                ( $deprecated_marker_id )
            );
        }

        #
        # Names
        #
        my $deprecated_marker_names = $db->selectall_arrayref(
            q[
                select s.marker_synonym_id, s.marker_name, 
                       st.synonym_type_id, st.synonym_type
                from   marker_synonym s, synonym_type st
                where  s.marker_id=?
                and    s.synonym_type_id=st.synonym_type_id
            ],
            { Columns => {} },
            ( $deprecated_marker_id )
        );

        my $TargetMarker 
            = Gramene::CDBI::Markers::Marker->retrieve($target_marker_id);
        my $DeprecatedMarker 
            = Gramene::CDBI::Markers::Marker->retrieve($deprecated_marker_id);

        $self->add_synonyms_to_Marker(
            Marker   => $TargetMarker,
            synonyms => $deprecated_marker_names,
        );

        #
        # Mappings
        #
        my $mappings = $self->get_marker_mappings( 
            marker_id => $deprecated_marker_id 
        );

        for my $mapping ( @$mappings ) {
            $db->do(
                q[
                    update mapping
                    set    marker_id=?
                    where  mapping_id=?
                ],
                {},
                ( $target_marker_id, $mapping->{'mapping_id'} )
            );
        }

        #
        # Marker Images
        #
        my $images = $self->get_marker_images( 
            marker_id => $deprecated_marker_id 
        );

        for my $image ( @$images ) {
            $db->do(
                q[
                    update marker_image
                    set    marker_id=?
                    where  marker_image_id=?
                ],
                {},
                ( $target_marker_id, $image->{'marker_image_id'} )
            );
        }

        #
        # Correspondences
        #
        my @correspondences = $self->get_marker_correspondences(
            marker_id => $deprecated_marker_id
        );

        for my $c ( @correspondences ) {
            my $from_or_to
            = $c->{'from_marker_id'} == $deprecated_marker_id ? 'from_marker_id'
            : $c->{'to_marker_id'}   == $deprecated_marker_id ? 'to_marker_id'
            : '';

            next unless $from_or_to;

            $db->do(
                qq[
                    update analytical_correspondence
                    set    $from_or_to=?
                    where  analytical_correspondence_id=?
                ],
                {},
                ( $target_marker_id, $c->{'analytical_correspondence_id'} )
            );
        }

        #
        # Details
        #
        my $target_details = $self->get_marker_details(
            marker_id => $target_marker_id
        );

        my $deprecated_details = $self->get_marker_details(
            marker_id => $deprecated_marker_id
        );

        if ( %{ $deprecated_details || {} } ) {
            for my $fld ( @{ $deprecated_details->{'ordered_field_names'} } ) {
                my $cur_val = $target_details->{ $fld };
                my $dep_val = $deprecated_details->{ $fld };
                if ( 
                    defined $dep_val   && 
                    $dep_val ne ''     && 
                    ( ! defined $cur_val || $cur_val eq '' )
                ) {
                    $target_details->{ $fld } = $dep_val;
                }
            }

            $target_details->{'marker_id'} = $target_marker_id;
            $self->set_marker_details( $target_details );
        }

        if ( 
            $TargetMarker->analysis->analysis_name =~ /UNKNOWN/i 
            &&
            $DeprecatedMarker->analysis->analysis_name !~ /UNKNOWN/i 
        ) {
            $TargetMarker->analysis_id( $DeprecatedMarker->analysis_id );
            $TargetMarker->update;
        }

        if ( 
            $TargetMarker->library->library_name =~ /UNKNOWN/i 
            &&
            $DeprecatedMarker->library->library_name !~ /UNKNOWN/i 
        ) {
            $TargetMarker->library_id( $DeprecatedMarker->library_id );
            $TargetMarker->update;
        }

        if ( 
            (
            $TargetMarker->germplasm->germplasm_name =~ /UNKNOWN/i 
            &&
            $DeprecatedMarker->germplasm->germplasm_name !~ /UNKNOWN/i 
            )
            ||
            (
            $TargetMarker->germplasm->germplasm_name =~ /UNKNOWN/i 
            &&
            $DeprecatedMarker->germplasm->germplasm_name =~ /UNKNOWN/i 
            &&
            $DeprecatedMarker->germplasm->species_id > 1 # not unknown
            )
        ) {
            $TargetMarker->germplasm_id( $DeprecatedMarker->germplasm_id );
            $TargetMarker->update;
        }

        if ( 
            !$TargetMarker->description && $DeprecatedMarker->description
        ) {
            $TargetMarker->description( $DeprecatedMarker->description );
            $TargetMarker->update;
        }

        #
        # Cross-references
        #
        $db->do(
            q[
                update xref
                set    record_id=?
                where  table_name=?
                and    record_id=?
            ],
            {},
            ( $target_marker_id, 'marker', $deprecated_marker_id )
        );

        my ( $details_table_name ) = $self->marker_type_to_table_name(
            $marker_type
        );

        if ( $details_table_name ) {
            $db->do(
                qq[
                    delete
                    from   $details_table_name
                    where  marker_id=?
                ],
                {},
                ( $deprecated_marker_id )
            );
        }

        #
        # Have to NULL field to get around FK dependency
        #
        $db->do(
            q[
                update marker
                set    display_synonym_id=NULL
                where  marker_id=?
            ],
            {},
            ( $deprecated_marker_id )
        );

        for my $syn ( @$deprecated_marker_names ) {
            my $is_mapped = $db->selectrow_array(
                q[
                    select count(s.marker_synonym_id)
                    from   marker_synonym s, mapping mp
                    where  s.marker_synonym_id=?
                    and    mp.display_synonym_id=s.marker_synonym_id
                ],
                {},
                ( $syn->{'marker_synonym_id'} )
            );

            if ( $is_mapped ) {
                my $new_disp_syn_id = $db->selectrow_array(
                    q[
                        select marker_synonym_id
                        from   marker_synonym
                        where  marker_id=?
                        and    marker_name=?
                    ],
                    {},
                    ( $target_marker_id, $syn->{'marker_name'} )
                );

                $db->do(
                    q[
                        update mapping
                        set    display_synonym_id=?
                        where  display_synonym_id=?
                    ],
                    {},
                    ( $new_disp_syn_id, $syn->{'marker_synonym_id'} )
                );
            }

            $db->do(
                q[
                    delete
                    from   marker_synonym
                    where  marker_synonym_id=?
                ],
                {},
                ( $syn->{'marker_synonym_id'} )
            );
        }

        $self->delete_marker( marker_id => $deprecated_marker_id );

        $db->commit;
    };

    $db->{'RaiseError'} = $orig_re;
    $db->{'PrintError'} = $orig_pe;
    $db->{'AutoCommit'} = $orig_ac;

    my $return = 1;
    if ( my $err = $@ ) {
        eval { $db->rollback };
        croak( $err );
    }

    return $return; 
}

# --------------------------------------------------------
sub set_correspondence {

=pod

=head2 set_correspondence

  my $correspondence_id = $mdb->set_correspondence(
      from_Marker                  => $cdbi_marker1, #or
      from_marker_id               => 1998,
      to_Marker                    => $cdbi_marker2, #or
      to_marker_id                 => 443,
      Analysis                     => $cdbi_analysis, #or
      analysis_id                  => 2,
      AnalyticalCorrespondenceType => $cdbi_analytical_correspondence_type, #or
      analytical_correspondence_type_id => 8,
  );

Creates a correspondence from one marker to another based on some
analysis.

=cut

    my ( $self, %args ) = @_;

    my %from_args = ( 'Marker' => $args{from_Marker}, 
                      'marker_id'=>$args{from_marker_id} );

    my %to_args = ( 'Marker' => $args{to_Marker}, 
                    'marker_id'=>$args{to_marker_id} );

    my $from_cdbi_marker = $self->_args_to_Object( 'Marker', {%from_args} )
        or croak( "Need a from_Marker or a from_marker_id" );

    my $to_cdbi_marker = $self->_args_to_Object( 'Marker', {%to_args} )
        or croak( "Need a to_Marker or a to_marker_id" );

    my $cdbi_analysis = $self->_args_to_Object( 'Analysis', {%args} )
        or croak( "Need an Analysis or an analysis_id" );

    my $cdbi_corr_type = $self->_args_to_Object('AnalyticalCorrespondenceType',
                                                {%args})
        or croak( "Need an AnalyticalCorrespondenceType or id" );


    my $from_marker_id  = $from_cdbi_marker->id;
    my $to_marker_id    = $to_cdbi_marker->id;
    my $analysis_id     = $cdbi_analysis->id;
    my $corr_type_id    = $cdbi_corr_type->id;

    my $db = $self->db;
    my $corr_id = $db->selectrow_array( # Find existing corr
        q[
            select analytical_correspondence_id
            from   analytical_correspondence
            where  from_marker_id=?
            and    to_marker_id=?
            and    analytical_correspondence_type_id = ?
        ],
        {},
        ( 
          $from_marker_id, 
          $to_marker_id, 
          $corr_type_id 
        )
    );

    $corr_id ||= $db->selectrow_array( #... and in reverse direction
        q[
            select analytical_correspondence_id
            from   analytical_correspondence
            where  from_marker_id=?
            and    to_marker_id=?
            and    analytical_correspondence_type_id = ?
        ],
        {},
        (
          $to_marker_id,
          $from_marker_id,
          $corr_type_id
        )
    );


    unless ( $corr_id ) {
        $db->do(
            q[
                insert
                into   analytical_correspondence 
                       ( from_marker_id, 
                         to_marker_id, 
                         analysis_id ,
                         analytical_correspondence_type_id )
                values ( ?, ?, ?, ? )
            ],
            {},
            ( $from_marker_id, 
              $to_marker_id, 
              $analysis_id,
              $corr_type_id )
        );

        $corr_id = $db->selectrow_array('select last_insert_id()');
    }

    return $corr_id;
}

# --------------------------------------------------------
sub set_marker_display_synonym {

=pod

=head2 set_marker_display_synonym

  $mdb->set_marker_display_synonym(
      marker_id         => 1998,
      marker_synonym_id => 13,
  );

Sets the primary ("display") synonym for the marker.

=cut

    my ( $self, %args ) = @_;
    my $marker_id       = $args{'marker_id'} or 
                          croak('No marker id');
    my $display_syn_id  = $args{'marker_synonym_id'} or 
                          croak('No marker synonym id');

    my $Marker = Gramene::CDBI::Markers::Marker->retrieve($marker_id)
        or croak("Bad marker id '$marker_id'");
    my $Syn = Gramene::CDBI::Markers::MarkerSynonym->retrieve($display_syn_id)
        or croak("Bad marker synonym id '$display_syn_id'");

    if ( $Marker->id == $Syn->marker_id ) {
        $Marker->display_synonym_id( $display_syn_id );
        $Marker->update;
    }
    else {
        croak("The marker synonym '$display_syn_id' does not belong to the "
            . "marker '$marker_id'"
        );
    }

    return 1;
}

# --------------------------------------------------------
sub set_marker_details {

=pod

=head2 set_marker_details

  $mdb->set_marker_details(
      marker_id => 1998, #or
      Marker    => $cdbi_marker_obj,
      fld1      => '...',
      fld2      => '...',
  );

Set the fields in the "marker_details_*" table for a given marker_id.

=cut

    my $self            = shift;
    my %args            = ref $_[0] eq 'HASH' ? %{ $_[0] } : @_;
    my $marker_obj      = $args{'Marker'};
    my $marker_id       = $args{'marker_id'};
    my $marker_type_obj = $args{'MarkerType'};
    my $marker_type_id  = $args{'marker_type_id'} || 0;
    my $marker_type     = $args{'marker_type'}    || '';
    $marker_type        = lc $marker_type;
    $marker_type        =~ s/[\s-]/_/g;
    my $db              = $self->db;

    for my $arg ( 
        qw[ Marker marker_id MarkerType marker_type_id marker_type ]
    ) {
        delete $args{ $arg };
    }

    # Process marker
    if ( !$marker_id && !$marker_obj ) {
        croak( "Need a Marker or a marker_id" );
    }

    if ( $marker_obj ) {
        UNIVERSAL::isa( $marker_obj, "${CDBI}::Marker" )
            or croak( "Marker must be a ${CDBI}::Marker" );
        $marker_id = $marker_obj->id;
    }
    elsif ( $marker_id ) {
        $marker_obj = $self->retrieve_Marker( $marker_id )
            or croak( "No marker in DB with ID $marker_id" );
    }

    # Process marker type
    if ( $marker_type_obj ) {
        UNIVERSAL::isa( $marker_type_obj, "${CDBI}::MarkerType" )
            or croak( "MarkerType must be a ${CDBI}::MarkerType" );
        $marker_type_id = $marker_type_obj->id;
    }
    elsif ( $marker_type_id ) {
        $marker_type_obj = $self->retrieve_MarkerType( $marker_type_id )
            or croak( "No MarkerType in DB with ID $marker_type_id" );
    }
    elsif ( $marker_type ) {
        ( $marker_type_obj )
            = $self->search_MarkerType( { marker_type => $marker_type } )
            or croak( "No MarkerType in DB with type $marker_type" );
        $marker_type_id = $marker_type_obj->id;
    }
    else {
        $marker_type_obj = $marker_obj->marker_type_id
            or croak( "No MarkerType in DB with ID $marker_type_id" );
    }

    $marker_type = $marker_type_obj->marker_type;
    $marker_id   = $marker_obj->id;

    if ( lc $marker_type eq 'undefined' ) {
        return 1;
    }

    if ( !$marker_type ) {
        croak('No marker type');
    }

    my $table_name = $self->marker_type_to_table_name( $marker_type );
    if ( $table_name ) {
        my $class 
            = table_name_to_gramene_cdbi_class( $CDBI_MODULE, $table_name );
        my $obj = $class->find_or_create({ marker_id => $marker_id });

        $args{'marker_id'} = $marker_id;

        for my $field ( $class->columns('All') ) {
            next if $field eq 'marker_id';
            my $value = $args{ $field };
            next unless defined $value;
            $obj->$field( $value );
        }

        $obj->update;
    }

    if ( my $species_id = $args{'species_id'} ) {
        $marker_obj->source_species_id( $species_id );
    }

    if ( my $germplasm_id = $args{'germplasm_id'} ) {
        $marker_obj->germplasm_id( $germplasm_id );
    }

    if ( my $analysis_id = $args{'analysis_id'} ) {
        $marker_obj->analysis_id( $analysis_id );
    }

    if ( my $desc = $args{'description'} ) {
        $marker_obj->description( $desc );
    }

    my $seq = $args{'sequence'} || $args{'seq'} || '';

    if ( $seq ) {
        $self->set_marker_sequence(
            Marker => $marker_obj, 
            seq    => $seq,
        );
    }

    $marker_obj->update;

    return 1;
}

# --------------------------------------------------------
sub set_marker_mapping {

=pod

=head2 set_marker_mapping

  my $mapping_id = $mdb->set_marker_mapping(
    Marker                 => $cdbi_marker,        # REQUIRED OR
    marker_id              => 1998,
    Map                    => $cdbi_map,           # REQUIRED OR
    map_id                 => 42,
    Analysis               => $cdbi_analysis,      # REQUIRED OR
    analysis_id            => 15,
    MarkerSynonym          => $cdbi_marker_synonyn # OPTIONAL OR
    marker_synonym_id      => 123456
    start                  => 2000000,             # REQUIRED
    end                    => 2000100,
    strand                 => 1,
    marker_start           => 55,
    marker_end             => 157,
    marker_strand          => -1,
    cigar_line             => '60M2D40M',
    score                  => 113,
    evalue                 => 1.7723E-3
    percent_identity       => 97.35
    remark                 => 'whatever',
    cmap_feature_accession => 'cu-91-2-453',
#    marker_name            => 'Foo',            # Needed?
#    synonyms               => [ 'Bar', 'Baz' ], # Needed?
  );

Creates or updates a marker's mapping on a given map.
Needs the Map, Marker and a start coordinate.
Optional mapping data include coordinates and scores.
A MarkerSynonym to use for the mapping display can be provided, 
 else the Marker's display synonym will be assumed.

=cut

    my ( $self, %args ) = @_;

    # Get the Marker
    my $cdbi_marker 
        = $self->_args_to_Object( 'Marker', {%args} )
        or croak( "Need a Marker or a marker_id" );

    # Get the Map
    my $cdbi_map 
        = $self->_args_to_Object( 'Map', {%args} )
        or croak( "Need a Map or a map_id" );

    # Get the Analysis
    my $cdbi_analysis 
        = $self->_args_to_Object( 'Analysis', {%args} )
        || ($self->search_Analysis({analysis_name=>'unknown'}))[0];

    # Get the MarkerSynonym
    my $cdbi_marker_synonym 
        = $self->_args_to_Object( 'MarkerSynonym', {%args} )
        || $cdbi_marker->display_synonym;

    # Process the foreign keys
    my $marker_id      = $cdbi_marker->id;
    my $map_id         = $cdbi_map->id;
    my $analysis_id    = $cdbi_analysis->id;
    my $display_syn_id = $cdbi_marker_synonym->id;

    # Gather the rest of the mapping data
    my $start         = $args{'start'};
    defined( $start ) || croak('No start');
    my $end           = $args{'end'};
       $end           = undef if $end && $end < $start;
    my $strand        = $args{'strand'};
    my $marker_start  = $args{'marker_start'};
    my $marker_end    = $args{'marker_end'};
    my $marker_strand = $args{'marker_strand'};
    my $cigar_line    = $args{'cigar_line'};
    my $score         = $args{'score'};
    my $evalue        = $args{'evalue'};
    my $percent_identity = $args{'percent_identity'};
    my $remark        = $args{'remark'};
    my $cmap_feature_accession = $args{'cmap_feature_accession'};

    # Do we need these? I would prefer to depend on the supplied marker.
    #my $marker_name   = $args{'marker_name'};
    #my $synonyms      = ref $args{'synonyms'} eq 'ARRAY'
    #                    ? $args{'synonyms'} : [ $args{'synonyms'}||() ];
    #my @syns = ($marker_name||(), @{$synonyms} );
    #my $display_syn_id = $cdbi_marker->display_synonym_id;
    #if( @syns ){
    #  my $dsyn = $self->add_synonyms_to_Marker
    #      ( Marker => $cdbi_marker, synonyms => [@syns] );
    #  $display_syn_id = $dsyn->id;
    #}


    my $cdbi_mapping;

    if ( $cmap_feature_accession ) { # Can we find mapping by cmap acc'n?
      ($cdbi_mapping) = $self->search_Mapping
          ({marker_id             => $marker_id,
            cmap_feature_accession=> $cmap_feature_accession});
    }
    else { # Can we find mapping by Marker and map locus?
      ($cdbi_mapping) = $self->search_Mapping
          ({marker_id             => $marker_id,
            map_id                => $map_id,
            display_synonym_id    => $display_syn_id,
            start                 => $start });
    }


    my $date_created = $args{'date_created'}  if $args{'date_created'};

    if ( $cdbi_mapping ) { # Update existing mapping
      $cdbi_mapping->display_synonym_id( $display_syn_id );
      $cdbi_mapping->start( $start );
      $cdbi_mapping->end( $end );
      $cdbi_mapping->strand( $strand );
      $cdbi_mapping->marker_start( $marker_start );
      $cdbi_mapping->marker_end( $marker_end );
      $cdbi_mapping->marker_strand( $marker_strand );
      $cdbi_mapping->cigar_line( $cigar_line );
      $cdbi_mapping->score( $score );
      $cdbi_mapping->evalue( $evalue );
      $cdbi_mapping->percent_identity( $percent_identity );
      $cdbi_mapping->remark( $remark );
      $cdbi_mapping->cmap_feature_accession( $cmap_feature_accession );
      $cdbi_mapping->analysis_id( $analysis_id );
      $cdbi_mapping->date_created( $date_created );    
      $cdbi_mapping->update;
    }
    else { # Create new mapping
      $cdbi_mapping = $self->insert_Mapping
          ({ marker_id => $marker_id,
             map_id => $map_id,
             display_synonym_id => $display_syn_id,
             start => $start,
             end => $end,
             strand => $strand,
             marker_start => $marker_start,
             marker_end => $marker_end,
             marker_strand => $marker_strand,
             cigar_line => $cigar_line,
             score => $score,
             evalue => $evalue,
             percent_identity => $percent_identity,
             remark => $remark,
             cmap_feature_accession => $cmap_feature_accession,
             analysis_id => $analysis_id,
             date_created => $date_created,
           });
    }
    return $cdbi_mapping->id;
}

# --------------------------------------------------------
sub set_marker_sequence {

=pod

=head2 set_marker_sequence

  my $mapping_id = $mdb->set_marker_sequence(
    Marker       => $cdbi_marker, # REQUIRED OR
    marker_id    => 42,
    seq          => 'ACTGACCCA',
  );

Creates or updates a marker's sequence.

=cut

    my ( $self, %args ) = @_;
    my $seq = $args{'seq'} || $args{'sequence'} || '';

    my $marker = $self->_args_to_Object( 'Marker', \%args )
        or croak('Need a Marker or a marker_id');

    if ( $seq ) {
        # Can't use find_or_create as seq can be too long
        #Gramene::CDBI::Markers::MarkerSequence->find_or_create(
        #    { marker_id => $marker->id, seq => $seq }
        #);
        if( my $ms = $self->retrieve_MarkerSequence($marker->id) ){
          # Already in DB - update seq
          $ms->seq($seq);
          $ms->update;
        }
        else{
          # New - insert record
          $self->insert_MarkerSequence(
               { marker_id => $marker->id, seq => $seq }
          );
        }
    }
    else {
        croak('No sequence to set');
    }

    return 1;
}

# --------------------------------------------------------
sub update_analysis {

=pod

=head2 update_analysis

  $mdb->update_analysis(
      analysis_id   => 1,
      analysis_name => 'Foo',
      type          => '...',
      description   => '...',
      last_run      => '2005-07-21'
  );

Updates the "analysis."

=cut

    my ( $self, %args ) = @_;
    my $id = $args{'analysis_id'} || croak('No analysis id');

    my $analysis_obj = $self->retrieve_Analysis( $id ) 
        || croak( "No analysis in DB with ID $id" );

    for my $field ( $analysis_obj->columns('All') ) {
        next if $field eq $analysis_obj->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $analysis_obj->$field( $value );
    }  

    return $analysis_obj->update();
}

# --------------------------------------------------------
sub update_analytical_correspondence_type {

=pod

=head2 update_analytical_correspondence_type

  $mdb->update_analytical_correspondence_type(
      analytical_correspondence_type_id   => 1,
      type        => 'Foo',
      description => '...',
  );

Updates the "analytical correspondence type."

=cut

    my ( $self, %args ) = @_;
    my $id = $args{'id'} || $args{'analytical_correspondence_type_id'} 
        || croak('No analytical_correspondence_type id');

    my $Type = $self->retrieve_AnalyticalCorrespondenceType( $id ) 
        || croak("Invalide analytical correspondence type ($id)");

    for my $field ( $Type->columns('All') ) {
        next if $field eq $Type->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $Type->$field( $value );
    }  

    return $Type->update();
}

# --------------------------------------------------------
sub update_germplasm {

=pod

=head2 update_germplasm

  $mdb->update_germplasm(
      germplasm_id   => 1,
      germplasm_name => 'Foo',
      species_id     => 13,
      description    => '...',
  );

Updates the "germplasm."

=cut

    my ( $self, %args ) = @_;
    my $germplasm_id    = $args{'germplasm_id'} or 
                          croak('No germplasm id');
    my $species_id      = $args{'species_id'} or 
                          croak('No species id');
    my $germplasm_name  = $args{'germplasm_name'} or 
                          croak('No germplasm name');
    my $description     = $args{'description'} || '';

    my $db = $self->db;
    $db->do(
        q[
            update germplasm
            set    germplasm_name=?, species_id=?, description=?
            where  germplasm_id=?
        ],
        {},
        ( $germplasm_name, $species_id, $description, $germplasm_id )
    );

    return 1;
}

# --------------------------------------------------------
sub update_library {

=pod

=head2 update_library

  $mdb->update_library(
      library_id   => 1,
      library_name => 'Foo',
  );

Updates the "library."

=cut

    my ( $self, %args ) = @_;
    my $id = $args{'library_id'} || croak('No library id');

    my $library_obj = Gramene::CDBI::Markers::Library->retrieve( $id ) 
        || croak( "No library in DB with ID $id" );

    for my $field ( $library_obj->columns('All') ) {
        next if $field eq $library_obj->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $library_obj->$field( $value );
    }  

    return $library_obj->update();
}

# --------------------------------------------------------
sub update_map {

=pod

=head2 update_map

  $mdb->update_map(
      map_id   => 1,
      map_name => 'foo',
      ...
  );

Updates the "map" table.

=cut

    my ( $self, %args ) = @_;
    my $map_id          = $args{'map_id'} or croak('No map id');

    my $map = Gramene::CDBI::Markers::Map->retrieve( $map_id )
        or croak("Bad map set id ($map_id)");

    my %valid_field = map { $_, 1 } $map->columns('All');

    for my $arg ( keys %args ) {
        next if $arg eq 'map_id' || !$valid_field{ $arg };
        my $value = $args{ $arg };

        next unless defined $value;# && $value ne '';

        $map->$arg( $value );
    }

    $map->update;

    return 1;
}

# --------------------------------------------------------
sub update_map_set {

=pod

=head2 update_map_set

  $mdb->update_map_set(
      map_set_id   => 1,
      map_set_name => 'foo',
      ...
  );

Updates the "map_set" table.

=cut

    my ( $self, %args ) = @_;
    my $map_set_id      = $args{'map_set_id'} or croak('No map_set id');

    my $map_set = Gramene::CDBI::Markers::MapSet->retrieve( $map_set_id )
        or croak("Bad map set id ($map_set_id)");

    for my $field ( $map_set->columns('All') ) {
        next if $field eq $map_set->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        next unless defined $value && $value ne $EMPTY_STR;
        $map_set->$field( $value );
    }
    $map_set->update;

    return 1;
}

# --------------------------------------------------------
sub update_map_type {

=pod

=head2 update_map_type

  $mdb->update_map_type(
      map_type_id => 1,
      map_type    => 'Genetic',
      description => '...',
  );

Updates the "map_type" and "description" fields in the map type table
for a given "map_type_id."

=cut

    my ( $self, %args ) = @_;
    my $map_type_id     = $args{'map_type_id'} or 
                          croak('No map_type id');
    my $map_type        = $args{'map_type'} or 
                          croak('No map_type');

    my $mt = Gramene::CDBI::Markers::MapType->retrieve( $map_type_id )
        or croak("Bad map type id ($map_type_id)");

    for my $field ( qw[ map_type display_order description ] ) {
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $mt->$field( $value );
    }

    $mt->update;

    return 1;
}


# --------------------------------------------------------
sub update_marker_synonym {

=pod

=head2 update_marker_synonym

  $mdb->update_marker_synonym(
      marker_synonym_id => 1,
      marker_synonym    => 'foo',
  );

Updates the "marker_synonym" for a given "marker_synonym_id."

=cut

    my ( $self, %args ) = @_;
    my $marker_syn_id   = $args{'marker_synonym_id'} or 
                          croak('No marker synonym id');

    my $Syn = Gramene::CDBI::Markers::MarkerSynonym->retrieve( $marker_syn_id )
        or croak("Bad marker synonym id ($marker_syn_id)");

    for my $field ( $Syn->columns('All') ) {
        next if $field eq $Syn->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $Syn->$field( $value );
    }
    $Syn->update;

    return $Syn->marker_id;
}

# --------------------------------------------------------
sub update_marker_type {

=pod

=head2 update_marker_type

  $mdb->update_marker_type(
      marker_type_id => 1,
      marker_type    => 'AFLP',
      description    => 'Amplified Fragment Length Polymorphism',
  );

Updates the "marker_type" and "description" fields in the marker type table
for a given "marker_type_id."

=cut

    my ( $self, %args ) = @_;
    my $marker_type_id  = $args{'marker_type_id'} or 
                          croak('No marker_type id');

    my $mt = Gramene::CDBI::Markers::MarkerType->retrieve( $marker_type_id )
        or croak("Bad marker type id ($marker_type_id)");

    for my $field ( $mt->columns('All') ) {
        next if $field eq $mt->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $mt->$field( $value );
    }
    $mt->update;

    return 1;
}

# --------------------------------------------------------
sub update_species {

=pod

=head2 update_species

  $mdb->update_species(
      species_id  => 1,
      species     => 'Rice',
      description => '...',
  );

Updates the "species" and "description" fields in the species table
for a given "species_id."

=cut

    my ( $self, %args ) = @_;
    my $species_id      = $args{'species_id'} or 
                          croak('No species id');

    my $s = Gramene::CDBI::Markers::Species->retrieve( $species_id )
        or croak("Bad species id ($species_id");

    for my $field ( $s->columns('All') ) {
        next if $field eq $s->columns('Primary');
        next unless defined $args{ $field };
        my $value = $args{ $field };
        $s->$field( $value );
    }
    $s->update;

    return $s->id;
}

# --------------------------------------------------------
sub update_synonym_type {

=pod

=head2 update_synonym_type

  $mdb->update_synonym_type(
      synonym_type_id => 1,
      synonym_type    => 'GENBANK_ACCESSION',
      ...
  );

Updates "synonym_type".

=cut

    my ( $self, %args ) = @_;
    my $syn_type_id     = $args{'synonym_type_id'} or 
                          croak('No synonym_type id');

    my $st = Gramene::CDBI::Markers::SynonymType->retrieve( $syn_type_id )
        or croak("Bad synonym type ID ($syn_type_id)"); 

    for my $field ( $st->columns('All') ) {
        next if $field eq $st->columns('Primary');
        my $value = $args{ $field };
        next unless defined $value;
        $st->$field( $value );
    }

    $st->update;

    return 1;
}


# --------------------------------------------------------
sub update_xref {

=pod

=head2 update_xref

  $mdb->update_xref(
      xref_id      => 1,
      xref_type_id => 14,
      xref_value   => 'foo',
  );

Updates the "xref".

=cut

    my ( $self, %args ) = @_;
    my $xref_id         = $args{'xref_id'} or 
                          croak('No xref id');
    my $xref_type_id    = $args{'xref_type_id'} or 
                          croak('No xref type id');
    my $xref_value      = $args{'xref_value'} or 
                          croak('No xref value');

    my $db = $self->db;
    $db->do(
        q[
            update xref
            set    xref_value=?, xref_type_id=?
            where  xref_id=?
        ],
        {},
        ( $xref_value, $xref_type_id, $xref_id )
    );

    return 1;
}

# --------------------------------------------------------
sub update_xref_type {

=pod

=head2 update_xref_type

  $mdb->update_xref_type(
      xref_type_id  => 1,
      xref_type     => 'Rice',
      description => '...',
  );

Updates the "xref_type" and "description" fields in the xref_type table
for a given "xref_type_id."

=cut

    my ( $self, %args ) = @_;
    my $xref_type_id    = $args{'xref_type_id'} or 
                          croak('No xref_type id');
    my $xref_type       = $args{'xref_type'} or 
                          croak('No xref_type');
    my $url_template    = $args{'url_template'} || '';

    my $db = $self->db;
    $db->do(
        q[
            update xref_type
            set    xref_type=?, url_template=?
            where  xref_type_id=?
        ],
        {},
        ( $xref_type, $url_template, $xref_type_id )
    );

    return 1;
}

# ----------------------------------------------------
sub view_marker {

=pod

=head2 view_marker

  my $marker = $mdb->view_marker( 
      marker_id => 3,
      order_by  => 'marker_name',
  );

Gets a marker and related mappings, correspondences.

=cut

    my ( $self, %args ) = @_;
    my $marker_id       = $args{'marker_id'} or croak('No marker id');
    my $no_details      = $args{'no_details'} || 0;
    my $order_by        = $args{'order_by'}   || 'marker_name';
    $order_by          .= ' desc' if $order_by eq 'num_mappings';
    my $db              = $self->db or return;
    my $marker = Gramene::CDBI::Markers::Marker->retrieve( $marker_id );

    unless ( $no_details ) {
        $marker->{'mappings'} = $self->get_marker_mappings(
            marker_id => $marker->{'marker_id'} 
        );

        $marker->{'correspondences'} = $self->get_marker_correspondences(
            marker_id => $marker->{'marker_id'} 
        );

        $marker->{'marker_type'} = $self->retrieve_MarkerType(
            marker_type_id => $marker->marker_type_id() 
        )->marker_type();

        $marker->{'details'} = $self->get_marker_details(
            marker_id   => $marker->{'marker_id'},    #$marker_id,
            marker_type => $marker->{marker_type},
        ) or return;

        my $species_obj = $self->retrieve_Species(
            species_id => $marker->source_species_id() 
        );

        $marker->{'species'} = $species_obj->species();
    }

    return $marker;
}

# ----------------------------------------------------
sub view_analysis {

=pod

=head2 view_analysis

  my $analysis = $mdb->view_analysis( 
      analysis_id => 3
  );

  my $analysis = $mdb->view_analysis( 
      analysis_name => 'foo'
  );

Gets a analysis.

=cut

    my ( $self, %args ) = @_;

    my $id = $args{'analysis_id'} || $args{'analysis_name'} ||
        croak('No analysis_id or analysis_name');

    my $id_type = $args{'analysis_id'} ? 'analysis_id' : 'analysis_name';

    my $db          = $self->db;
    my $sth         = $db->prepare(
        qq[
            select analysis_id,
                   analysis_name,
                   type,
                   description,
                   last_run,
                   ok_to_release
            from   analysis
            where  $id_type = ?
        ]
    );
    $sth->execute( $id );

    my $analysis = $sth->fetchrow_hashref or croak(
        "$id_type ($id) not found in MarkersDB"
    );

    return $analysis;
}

# ----------------------------------------------------
sub view_analytical_correspondence_type {

=pod

=head2 view_analytical_correspondence_type

  my $type = $mdb->view_analytical_correspondence_type( 
      analytical_correspondence_type_id => 3
  );

  my $type = $mdb->view_analytical_correspondence_type( 
      type => 'foo'
  );

Gets a analytical correspondence type.

=cut

    my ( $self, %args ) = @_;

    my $id = $args{'analytical_correspondence_type_id'} 
          || $args{'analytical_correspondence_type_name'} 
          || croak('No id or type');

    my $id_type = $args{'analytical_correspondence_type_id'} 
        ? 'analytical_correspondence_type_id' : 'type';

    my $db          = $self->db;
    my $sth         = $db->prepare(
        qq[
            select analytical_correspondence_type_id,
                   type,
                   description
            from   analytical_correspondence_type
            where  $id_type = ?
        ]
    );
    $sth->execute( $id );

    my $type = $sth->fetchrow_hashref or croak("Invalid $id_type ($id)");

    return $type;
}

# ----------------------------------------------------
sub view_germplasm {

=pod

=head2 view_germplasm

  my $germplasm = $mdb->view_germplasm( 
      germplasm_id => 3,
  );

Gets a germplasm.

=cut

    my ( $self, %args ) = @_;
    my $germplasm_id    = $args{'germplasm_id'}   
                          or croak('No germplasm id');

    my $db  = $self->db;
    my $sth = $db->prepare(
        q[
            select    g.germplasm_id,
                      g.germplasm_name,
                      g.description,
                      sp.species_id,
                      sp.species,
                      count(m.marker_id) as num_markers
            from      species sp,
                      germplasm g 
            left join marker m 
            on        g.germplasm_id=m.germplasm_id
            where     g.germplasm_id=?
            and       g.species_id=sp.species_id
            group by  germplasm_id, germplasm_name, description, 
                      species_id, species
        ]
    );
    $sth->execute( $germplasm_id );
    my $germplasm = $sth->fetchrow_hashref or croak(
        "Bad germplasm ID ($germplasm_id)"
    );

    $germplasm->{'libraries'} = $db->selectall_arrayref(
        q[
            select   l.library_id,
                     l.library_name,
                     s.species_id,
                     s.species
            from     library l,
                     species s
            where    l.germplasm_id=?
            and      l.species_id=s.species_id
            order by library_name, species
        ],
        { Columns => {} },
        ( $germplasm_id )
    );

    $germplasm->{'map_sets'} = $db->selectall_arrayref(
        q[
            select   ms.map_set_id,
                     ms.map_set_name,
                     mt.map_type_id,
                     mt.map_type,
                     s.species_id,
                     s.species
            from     germplasm_to_map_set g2ms,
                     map_set ms,
                     map_type mt,
                     species s
            where    g2ms.germplasm_id=? 
            and      g2ms.map_set_id=ms.map_set_id
            and      ms.species_id=s.species_id
            and      ms.map_type_id=mt.map_type_id
            order by mt.display_order, mt.map_type, 
                     s.display_order, s.species,
                     ms.map_set_name
        ],
        { Columns => {} },
        ( $germplasm_id )
    );

    $germplasm->{'populations'} = $db->selectall_arrayref(
        q[
            select   p.population_id,
                     p.population_name as population_name,
                     s.species_id,
                     s.species
            from     population p,
                     species s
            where    ( p.male_germplasm_id=? or p.female_germplasm_id=?)
            and      p.species_id=s.species_id
            order by s.display_order, s.species, p.population_name
        ],
        { Columns => {} },
        ( $germplasm_id, $germplasm_id )
    );

    return $germplasm;
}

# ----------------------------------------------------
sub view_library {

=pod

=head2 view_library

  my $map = $mdb->view_library( 
      library_id => 3,
  );

Gets a map set and related maps.

=cut

    my ( $self, %args ) = @_;
    my $library_id      = $args{'library_id'}   || '';
    my $library_name    = $args{'library_name'} || '';

    my $library;
    if ( $library_id ) {
        $library = Gramene::CDBI::Markers::Library->retrieve($library_id)
            or croak("Bad library id ($library_id)");
    }
    elsif ( $library_name ) {
        ($library) = Gramene::CDBI::Markers::Library->search(
            { library_name => $library_name }
        ) or croak("Bad library name ($library_name)");
    }
    else {
        croak('No library identifier (ID, name)');
    }

    return $library;
}

# ----------------------------------------------------
sub view_map {

=pod

=head2 view_map

  my $map = $mdb->view_map( 
      Map            => $cdbi_map, #OR
      map_id         => 3,
      MarkerType     => $cdbi_marker_type, # Optional, OR 
      marker_type_id => 5, # Optionally filter by marker_type_id
      order_by       => 'map_name',
  );

Gets a map and related mappings.

=cut

    my ( $self, %args ) = @_;
    my $cdbi_map = $self->_args_to_Object( 'Map', \%args )
        or croak( "Need a Map or a map_id" );
    my $cdbi_marker_type = $self->_args_to_Object( 'MarkerType', \%args );

    my $map_id          = $cdbi_map->id;
    my $marker_type_id  = $cdbi_marker_type ? $cdbi_marker_type->id : 0;

    my $order_by        = $args{'order_by'}       || 'start';
    $order_by          .= ' desc' if $order_by eq 'num_mappings';
    my $db              = $self->db;

    my $map = Gramene::CDBI::Markers::Map->retrieve($map_id)
        or croak("Bad map id ($map_id)");

    my $sql = sprintf( 
        q[
            select   m.marker_id,
                     m.source_species_id,
                     syn.marker_name as display_synonym,
                     mkr_type.marker_type_id,
                     mkr_type.marker_type,
                     sp.species_id,
                     sp.species,
                     mp.start,
                     mp.end,
                     mp.cmap_feature_accession,
                     mp.marker_start,
                     mp.marker_end
            from     mapping mp,
                     marker m,
                     marker_type mkr_type,
                     marker_synonym syn,
                     species sp
            where    mp.map_id=?
            and      mp.marker_id=m.marker_id
            and      m.display_synonym_id=syn.marker_synonym_id
            and      m.marker_type_id=mkr_type.marker_type_id
            and      m.source_species_id=sp.species_id
            %s
            %s
        ],
        $marker_type_id ? "and m.marker_type_id=$marker_type_id" : '',
        $order_by ? "order by $order_by" : '',
    );

    $map->{'sorted_mappings'} = $db->selectall_arrayref(
        $sql, { Columns => {} }, ( $map_id )
    );

    $map->{'mappings_by_type'} = $db->selectall_arrayref(
        q[
            select   mt.marker_type_id, mt.marker_type, 
                     count(mapping.marker_id) as num_mappings
            from     mapping, marker m, marker_type mt
            where    mapping.map_id=?
            and      mapping.marker_id=m.marker_id 
            and      m.marker_type_id=mt.marker_type_id 
            group by marker_type_id, marker_type
            order by num_mappings desc, marker_type
        ],
        { Columns => {} },
        ( $map_id )
    );

    return $map;
}

# ----------------------------------------------------
sub view_map_set {

=pod

=head2 view_map_set

  my $map = $mdb->view_map_set( 
      map_set_id              => 3,     # OR
      map_set_acc             => 'foo', # OR
      cmap_map_set_acc        => 'foo', # OR
      cmap_map_set_accession  => 'foo', # OR
      order_by                => 'map_name',
      include_mapping_details => 1,
  );

Gets a map set and related maps.

As the gathering of the mapping details (by marker type, map, etc.)
can be expensive, you must indicate that you want this with the
"include_mapping_details" argument.

=cut

    my ( $self, %args ) = @_;
    my $map_set_id      = $args{'map_set_id'}   || '';
    my $map_set_name    = $args{'map_set_name'} || '';
    my $cmap_ms_acc     = $args{'cmap_map_set_accession'}   
                       || $args{'cmap_map_set_acc'}
                       || $args{'map_set_acc'}
                       || '';
    my $order_by        = $args{'order_by'} || '';

    if ( !$order_by || $order_by eq 'map_name' ) {
        $order_by = 'display_order, map_name';
    }
    elsif ( $order_by eq 'num_mappings' ) {
        $order_by .= ' desc';
    }

    my $db = $self->db;

    my $map_set;
    if ( $map_set_id ) {
        $map_set = Gramene::CDBI::Markers::MapSet->retrieve($map_set_id)
            or croak("Bad map set id ($map_set_id)");
    }
    elsif ( $map_set_name ) {
        ($map_set) = Gramene::CDBI::Markers::MapSet->search(
            { map_set_name => $map_set_name }
        ) or croak("Bad map set name ($map_set_name)");
    }
    elsif ( $cmap_ms_acc ) {
        ($map_set) = Gramene::CDBI::Markers::MapSet->search(
            { cmap_map_set_accession => $cmap_ms_acc }
        ) or croak("Bad CMap map set accession ($cmap_ms_acc)");
    }
    else {
        croak('No map set identifier (ID, name, accession)');
    }

    if ( $args{'include_mapping_details'} ) {
        $map_set->{'sorted_maps'} = $db->selectall_arrayref(
            qq[
                select    map.map_id,
                          map.map_name,
                          map.start,
                          map.end,
                          map.cmap_map_accession,
                          count(mp.mapping_id) as num_mappings
                from      map
                left join mapping mp
                on        map.map_id=mp.map_id
                where     map.map_set_id=?
                group by  map.map_id, map.map_name
                order by  $order_by
            ],
            { Columns => {} },
            ( $map_set->id )
        );

        $map_set->{'mappings_by_map_and_type'} = $db->selectall_arrayref(
            q[
                select   map.map_id, map.map_name,
                         mt.marker_type_id, mt.marker_type, 
                         count(mapping.marker_id) as num_mappings
                from     map, mapping, marker m, marker_type mt
                where    map.map_set_id=?
                and      map.map_id=mapping.map_id
                and      mapping.marker_id=m.marker_id 
                and      m.marker_type_id=mt.marker_type_id 
                group by map.map_id, map.map_name, 
                         mt.marker_type_id, mt.marker_type
                order by map.display_order, map.map_name, 
                         num_mappings desc, marker_type
            ],
            { Columns => {} },
            ( $map_set->id )
        );

        $map_set->{'mappings_by_marker_type'} = $db->selectall_arrayref(
            q[
                select   mt.marker_type_id, mt.marker_type, 
                         count(mapping.marker_id) as num_mappings
                from     map, mapping, marker m, marker_type mt
                where    map.map_set_id=?
                and      map.map_id=mapping.map_id
                and      mapping.marker_id=m.marker_id 
                and      m.marker_type_id=mt.marker_type_id 
                group by mt.marker_type_id, mt.marker_type
                order by num_mappings desc, marker_type
            ],
            { Columns => {} },
            ( $map_set->id )
        );
    }

    return $map_set;
}

# ----------------------------------------------------
sub view_map_type {

=pod

=head2 view_map_type

  my $map_type = $mdb->view_map_type( 
      map_type_id => 3,
      order_by    => 'map_set_name',
  );

Gets a map type and related map sets.

=cut

    my ( $self, %args ) = @_;

    my $map_type_id = $args{'map_type_id'} or 
                      croak('No map_type id');
    my $order_by    = $args{'order_by'} || 'species';
    $order_by      .= ' desc' if $order_by eq 'num_maps';
    $order_by      .= ', map_set_name' if $order_by eq 'species';
    my $db          = $self->db;

    my $map_type = Gramene::CDBI::Markers::MapType->retrieve( $map_type_id )
        or croak("Bad map type ID ($map_type_id)");

    $map_type->{'sorted_map_sets'} = $db->selectall_arrayref(
        qq[
            select   ms.map_set_id,
                     ms.map_set_name,
                     sp.species_id,
                     sp.species,
                     count(map.map_id) as num_maps
            from     map_set ms,
                     species sp,
                     map
            where    ms.map_type_id=?
            and      ms.species_id=sp.species_id
            and      ms.map_set_id=map.map_set_id
            group by map_set_id, map_set_name, species_id, species
            order by $order_by
        ],
        { Columns => {} },
        ( $map_type_id )
    ); 

    return $map_type;
}

# ----------------------------------------------------
sub view_marker_type {

=pod

=head2 view_marker_type

  my $marker_type = $mdb->view_marker_type( 
      marker_type    => 'SSR' # either
      marker_type_id => 3,    # or
      order_by       => 'marker_name',
  );

Gets a marker_type and related markers.

=cut

    my ( $self, %args ) = @_;
    my $marker_type     = $args{'marker_type'}    || '';
    my $marker_type_id  = $args{'marker_type_id'} || '';
    my $order_by        = $args{'order_by'} || 'marker_name';
    $order_by          .= ' desc' if $order_by eq 'num_mappings';

    unless ( $marker_type || $marker_type_id ) {
        croak('No marker type or marker type id');
    }

    my $db = $self->db;

    if ( !$marker_type_id && $marker_type ) {
        $marker_type_id = $db->selectrow_array(
            q[
                select marker_type_id
                from   marker_type
                where  marker_type=?
            ],
            {},
            ( $marker_type )
        ) or croak("Can't determine id for marker type '$marker_type'");
    }

    my $sth = $db->prepare(
        q[
            select mt.marker_type_id,
                   mt.marker_type,
                   mt.is_sequence,
                   mt.description
            from   marker_type mt
            where  mt.marker_type_id=?
        ]
    );
    $sth->execute( $marker_type_id );

    my $mt = $sth->fetchrow_hashref or croak(
        "Bad marker type ID ($marker_type_id)"
    );

    return $mt;
}

# ----------------------------------------------------
sub view_population {

=pod

=head2 view_population

  my $population = $mdb->view_population( 
      population_id => 3,
  );

Gets a population.

=cut

    my ( $self, %args ) = @_;
    my $population_id   = $args{'population_id'} or 
                          croak('No population id');

    my $db  = $self->db;
    my $sth = $db->prepare( 
        q[            
            select p.population_id,
                   p.population_name,
                   p.generation_number,
                   p.generation_type,
                   sp.species_id,
                   sp.species,
                   g1.germplasm_id as male_germplasm_id,
                   g1.germplasm_name as male_germplasm,
                   g2.germplasm_id as female_germplasm_id,
                   g2.germplasm_name as female_germplasm
            from   population p,
                   species sp,
                   germplasm g1,
                   germplasm g2
            where  p.population_id=?
            and    p.species_id=sp.species_id
            and    p.male_germplasm_id=g1.germplasm_id
            and    p.female_germplasm_id=g2.germplasm_id
        ]
    );
    $sth->execute( $population_id );
    my $population = $sth->fetchrow_hashref or croak(
        "Bad population ID ($population_id)"
    );

    return $population;
}

# ----------------------------------------------------
sub view_species {

=pod

=head2 view_species

  my $species = $mdb->view_species( 
      species_id => 3,
      species    => 'Oryza sativa',
      order_by   => 'map_name',
  );

Gets a species and related maps.

=cut

    my ( $self, %args ) = @_;

    my $db  = $self->db or return ;

    my $species    = $args{'species'};
    my $species_id = $args{'species_id'};

    if ( !$species_id && $species ) {
        $species_id = $db->selectrow_array( 
            q[
                select    species_id
                from      species
                where     species = ?
            ],
	    {},
	    ($species)				   
        ) or croak("Species '$species' not found");       
    }

    $species_id || croak('No species id');
    my $order_by   = $args{'order_by'} || 'map_type';
    $order_by     .= ' desc'         if $order_by eq 'num_maps';
    $order_by     .= ',map_set_name' if $order_by eq 'map_type';

    $species = Gramene::CDBI::Markers::Species->retrieve($species_id)
        or croak("Bad species ID (species_id)");

    $species->{'sorted_map_sets'} = $db->selectall_arrayref(
        qq[
            select   ms.map_set_id,
                     ms.map_set_name,
                     mt.map_type_id,
                     mt.map_type,
                     count(map.map_id) as num_maps
            from     map_set ms,
                     map_type mt,
                     map
            where    ms.species_id=?
            and      ms.map_type_id=mt.map_type_id
            and      ms.map_set_id=map.map_set_id
            group by map_set_id, map_set_name, map_type_id, map_type
            order by $order_by
        ],
        { Columns => {} },
        ( $species_id )
    ); 

    return $species;
}

# ----------------------------------------------------
sub view_synonym_type {

=pod

=head2 view_synonym_type

  my $synonym_type = $mdb->view_synonym_type( 
      synonym_type    => 'GENBANK_ACCESSION' # either
      synonym_type_id => 3,                  # or
  );

Gets a synonym_type and related markers.

=cut

    my ( $self, %args ) = @_;
    my $synonym_type    = $args{'synonym_type'}    || '';
    my $synonym_type_id = $args{'synonym_type_id'} || '';

    unless ( $synonym_type || $synonym_type_id ) {
        croak('No marker type or marker type id');
    }

    my $db = $self->db;

    if ( !$synonym_type_id && $synonym_type ) {
        $synonym_type_id = $db->selectrow_array(
            q[
                select synonym_type_id
                from   synonym_type
                where  synonym_type=?
            ],
            {},
            ( $synonym_type )
        ) or croak("Can't determine id for marker type '$synonym_type'");
    }

    my $sth = $db->prepare(
        q[
            select synonym_type_id,
                   synonym_type,
                   url_template,
                   description,
                   validation
            from   synonym_type
            where  synonym_type_id=?
        ]
    );
    $sth->execute( $synonym_type_id );

    my $st = $sth->fetchrow_hashref or croak(
        "Bad synonym type ID ($synonym_type_id)"
    );

    return $st;
}

# ----------------------------------------------------
sub view_xref {

=pod

=head2 view_xref

  my $xref = $mdb->view_xref( 
      xref_id => 3,
  );

Show an xref type.

=cut

    my ( $self, %args ) = @_;
    my $xref_id = $args{'xref_id'} or croak('No xref id');

    my $xref = Gramene::CDBI::Markers::Xref->retrieve( $xref_id )
        or croak("Bad xref type ID ($xref_id)");

    return $xref;
}

# ----------------------------------------------------
sub view_xref_type {

=pod

=head2 view_xref_type

  my $xref_type = $mdb->view_xref_type( 
      xref_type_id => 3,
  );

Show an xref type.

=cut

    my ( $self, %args ) = @_;
    my $xref_type_id    = $args{'xref_type_id'} or 
                          croak('No xref type id');

    my $db  = $self->db;
    my $sth = $db->prepare(
        q[
            select xt.xref_type_id,
                   xt.xref_type,
                   xt.url_template
            from   xref_type xt
            where  xt.xref_type_id=?
        ]
    );
    $sth->execute( $xref_type_id );
    my $xref_type = $sth->fetchrow_hashref or croak(
        "Bad xref type ID ($xref_type_id)"
    );

    return $xref_type;
}

# --------------------------------------------------------
sub _parse_marker_name {
    my $name = shift;
    my $type = '';
    if ( $name =~ $MARKER_NAME_WITH_SYN_TYPE ) {
        ( $name, $type ) = ( $1, $2 );
    }

    return wantarray ? ( $name, $type ) : $name;
}

# --------------------------------------------------------

=pod

=head2 AUTOLOAD

Provides a wrapper around Gramene::CDBI::Markers classes. Any
Class::DBI method, followed by the Gramene::CDBI::Markers object type,
will be proxied. Examples include;

  # Instead of Gramene::CDBI::Markers::Species->retrieve(1);
  my $species = $mdb->retrieve_Species( 1 );

  # Instead of Gramene::CDBI::Markers::Species->search({...});
  my( $species ) = $mdb->search_Species({species=>'Oryza sativa'});

  my $analysis = $mdb->find_or_create_Analysis({
      analysis_name => 'test', 
      type          => 'test'
  });

If the first argument that is passed is a Gramene::CDBI::Markers
object, then the indicated method will be called on the object. E.g;

  $mdb->type_Analysis( $analysis, 't2' ); # same as $analysis->type('t2');
  $mdb->update_Analysis( $analysis );     # same as $analysis->update;
  $mdb->delete_Analysis( $analysis );     # same as $analysis->delete;

=cut

our $AUTOLOAD;

sub AUTOLOAD {
    my $self = shift;
    my @args = @_;

    my $method = $AUTOLOAD;
    $method =~ s/.*:://;
    return if $method eq 'DESTROY';

    if ( $method =~ /^([a-z_]+)_(\w+)$/ ) {
        my ( $cdbi_method, $cdbi_class ) = ( $1, $2 );
        $cdbi_method =~ s/_$//;

        # Look for object methods
        if ( 
            UNIVERSAL::isa( $args[0], $CDBI ) and $args[0]->can($cdbi_method) 
        ) {
            my $obj = shift @args;
            return $obj->$cdbi_method(@args);
        }
        else {
            # Use class method
            $cdbi_class = $CDBI . "::" . $cdbi_class;
            return $cdbi_class->$cdbi_method(@args);
        }
    }

    # Delegate
    my $superior = "SUPER::$method";
    return $self->$superior(@args);
}

# --------------------------------------------------------

1;

=pod

=head1 SEE ALSO

Gramene::DB, Gramene::CDBI::Markers.

=head1 AUTHOR

Will Spooner E<lt>whs@ebi.ac.ukE<gt>,
Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2007 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
