package Gramene::Literature::RefEditDB;

=head1 NAME

Gramene::Literature::RefEditDB

=head1 SYNOPSIS

  

=head1 DESCRIPTION

A module for operating protein database when editing protein 

=cut

# ----------------------------------------------------------------

use strict;
use DBI;

use Gramene::CDBI::Literature;


sub new {

    my $class = shift;
    my $self  = {};
    bless( $self, $class );
    return $self;

}

sub terminate_database {
    my $self = shift;
    $self->{'db'}->disconnect() if $self->{'db'};
}

sub connect_to_db {

    my $self = shift;
    eval { 
	my $cdbi = new Gramene::CDBI::Literature;
	my $cs_reference = new Gramene::CDBI::Literature::Reference;
	my $cs_reference_url = new Gramene::CDBI::Literature::ReferenceUrl;
	my $cs_source = new Gramene::CDBI::Literature::Source;
	my $cs_source_synonym = new Gramene::CDBI::Literature::SourceSynonym;
	my $cs_objectxref = new Gramene::CDBI::Literature::Objectxref;
	my $cs_dbxref = new Gramene::CDBI::Literature::Dbxref;
	
	my $db = $cdbi->db_Main;
	
	$self->{'db'} = $db; 
	$self->{'cdbi'} = $cdbi; 
	$self->{'cs_reference'} = $cs_reference;
	$self->{'cs_reference_url'} = $cs_reference_url;
	$self->{'cs_source'} = $cs_source;
	$self->{'cs_source_synonym'} = $cs_source_synonym;
	$self->{'cs_objectxref'} = $cs_objectxref;
	$self->{'cs_dbxref'} = $cs_dbxref;
    };

    if ($@) {
        die "DB connection to literature DB failed\n";

    }

}

sub get_reference{
    my ( $self, $ref_id ) = @_;
    return if($ref_id =~/\D/);
    my $ref = $self->{'cs_reference'}->retrieve($ref_id);

    return unless $ref;

    my $urls = &get_reference_urls($self, $ref_id);
    $ref->{'urls'} = $urls;    # an array of url objects

    my $abstract = &get_reference_abstract($self, $ref_id);
    $ref->{'abstract'} = $abstract; # a string

    my $authors = &get_reference_authors($self, $ref_id);
    $ref->{'author'} = $authors; # a string

    my $xrefs = &get_reference_all_xrefs($self, $ref_id); # except PubMed
    $ref->{'xrefs'} = $xrefs;

    my $pubmed_ref = &get_reference_pubmed($self, $ref_id);
    $ref->{'PubMed'} = $pubmed_ref;

    return $ref;
}

sub get_reference_urls{
    my ($self, $ref_id ) = @_;
    my @urls = $self->{'cs_reference_url'}->search( reference_id => $ref_id );
    return \@urls;
}

sub get_reference_authors{
    my ($self, $ref_id) = @_;
    my $author_ref = $self->{'db'}->selectcol_arrayref(q[
	SELECT contributor_name
	FROM   contributor C, author A
	WHERE  A.contributor_id = C.contributor_id
	AND    A.reference_id = ?
	ORDER BY A.authorship_position
    ], {}, ($ref_id));

    my $authors = join(', ', @$author_ref);
    return $authors;
}

# for historic reason, abstract is splited by part a and part b
sub get_reference_abstract{
    my ($self, $ref_id ) = @_;

    my ( $abstract, $aa, $ab );
    my $sth5 = $self->{'db'}->prepare( "SELECT abstract_part_a, abstract_part_b
					 FROM reference_abstract
					 WHERE reference_id = ?" );
    $sth5->bind_param( 1, $ref_id );
    $sth5->execute();
    ( $aa, $ab ) = $sth5->fetchrow_array();
    if( $aa ) {
        if( $ab ) {
	      $abstract = $aa.$ab;
	  } else {
	      $abstract = $aa;
        }
    } else {
	  $abstract = '';
    }
    $sth5->finish;

    return $abstract;

}

sub get_reference_pubmed{
    my ($self, $ref_id) = @_;
    return &get_reference_xrefs($self, $ref_id,'PubMed');
}

sub get_reference_xrefs{
    my ($self, $ref_id,$xref_dbname ) = @_;

    my $sth = $self->{'db'}->prepare( 
	"SELECT objectxref_id,
		DX.dbxref_id,
		xref_key,
		xref_dbname
	FROM    objectxref OX, dbxref DX
	WHERE   OX.dbxref_id = DX.dbxref_id
	AND     row_id = ?
	AND     xref_dbname = ?
    ");

    $sth->execute(($ref_id,$xref_dbname));
    my @xrefs;
    while(my $xref = $sth->fetchrow_hashref){
	push @xrefs,$xref;
    }

    $sth->finish;
    return \@xrefs;

}





sub get_reference_all_xrefs{
    my ($self, $ref_id ) = @_;

    my $sth = $self->{'db'}->prepare( 
	"SELECT objectxref_id,
		DX.dbxref_id,
		xref_key,
		xref_dbname
	FROM    objectxref OX, dbxref DX
	WHERE   OX.dbxref_id = DX.dbxref_id
	AND     row_id = ?
    ");

    $sth->execute(($ref_id));
    my %xrefs;
    while(my $xref = $sth->fetchrow_hashref){
	my $xdb= $xref->{'xref_dbname'};
	next if $xdb=~/PubMed/i;       # exclude pubmed
	push @{$xrefs{$xdb}},$xref;
    }

    $sth->finish;
    return \%xrefs;

}

sub update_reference_abstract{
    my ($self, $ref_id, $abstract ) = @_;
    $abstract =~s/^\s+|\s+$//g;

    # only use abstract_part_a since
    $self->{'db'}->do(q[
	DELETE FROM reference_abstract
	WHERE  reference_id = ?
    ], {}, ($ref_id));

    my $sth = $self->{'db'}->do(q[
	INSERT INTO reference_abstract
	(reference_id, abstract_part_a)
	VALUES (?, ?)
    ], {}, ($ref_id, $abstract));
}



sub update_reference_title{
    my ($self, $ref, $title) = @_;
    $title =~s/^\s+|\s+$//g;
    $ref->set('title' => $title );
    $ref->update;
}

sub update_reference_source{
    my ($self, $ref, $source_name) = @_;
    $source_name =~s/^\s+|\s+$//g;

    my $source = $ref->source_id;
    my $old_name = $source->source_name;

    return if $source_name eq $old_name;

    # check the synonym, if it is in synonym table, switch the names 
    my @source_synonyms = $self->{'cs_source_synonym'}->search( source_synonym => $source_name);
    if(@source_synonyms){
	foreach my $src (@source_synonyms){
	    my $src_id = $src->source_id;
	    if($src_id == $source->source_id){
		$source->set('source_name' => $source_name);
		$source->update;
		$src->set('source_synonym' => $old_name);
		$src->update;
		return;
	    }
	}
    }

    # check source table to see if it is another source;
    my ($another_source) = $self->{'cs_source'}->search( source_name => $source_name);
    if($another_source){
	$ref->set('source_id' => $another_source ); # $another_source
	$ref->update;
    }else{
	# create a new source, old one become the synonym of the new
	my $source_id = &_next_id ($self->{'db'},'source','source_id');
	my $new_source = $self->{'cs_source'}->create({
	    'source_id'   => $source_id,
	    'source_name' => $source_name
	});

	my $source_syn_id = &_next_id ($self->{'db'},'source_synonym','source_synonym_id');
	my $new_source_syn = $self->{'cs_source_synonym'}->create({
	    'source_synonym_id' => $source_syn_id,
	    'source_id'         => $source_id,
	    'source_synonym'    => $old_name
	});

	$ref->set( 'source_id' => $new_source );
	$ref->update;
    }
    
}

sub update_reference_volume{
    my ($self, $ref, $year, $vol, $start_page, $end_page) = @_;

    foreach my $p (($year, $vol, $start_page, $end_page)){
	$p =~s/^\s+|\s+$//g;
    }

    $ref->set('year' => $year );
    $ref->set('volume' => $vol );
    $ref->set('start_page' => $start_page );
    $ref->set('end_page' => $end_page );
    $ref->update;
}

sub add_reference_url{
    my ($self, $ref, $urls ) = @_;
    $urls =~s/^\s+|\s+$//g;
    my @ref_urls = split /\s+/, $urls;

    my $ref_id = $ref->reference_id;
    foreach my $url (@ref_urls){
	my $url_id = _next_id($self->{'db'},'reference_url','reference_url_id');
	$self->{'cs_reference_url'}->create({
	    reference_url_id => $url_id,
	    reference_id     => $ref_id,
	    reference_url    => $url
	});
    }
}
    
sub delete_reference_url{
    my ($self, $ref_url_id ) = @_;
    $self->{'db'}->do(q[
	DELETE FROM reference_url
	WHERE  reference_url_id = ?
    ], {}, ($ref_url_id));
}

sub add_reference_pubmed{
    my ($self, $ref, $pubmed_id) = @_;
    $pubmed_id =~s/^\s+|\s+$//g;

    my ($dbxref_obj) = $self->{'cs_dbxref'}->search(
	xref_dbname => 'PubMed',
	xref_key    => $pubmed_id
    );
    unless($dbxref_obj){
    
	my $dbxref_id = &_next_id ($self->{'db'},'dbxref','dbxref_id');
	$dbxref_obj = $self->{'cs_dbxref'}->create({
	    'dbxref_id'   => $dbxref_id,
	    'xref_key'    => $pubmed_id,
	    'xref_dbname' => 'PubMed',
	    'xref_keytype' => 'acc'
	});
    }
    my $dbxref_id = $dbxref_obj->dbxref_id;
    my $ref_id = $ref->reference_id;
    my ($objectxref_obj) = $self->{'cs_objectxref'}->search(
	row_id => $ref_id,
	dbxref_id => $dbxref_id,
    );
    unless($objectxref_obj){
	my $objx_id = &_next_id($self->{'db'},'objectxref','objectxref_id');
	$objectxref_obj = $self->{'cs_objectxref'}->create({
	    'objectxref_id' => $objx_id,
	    'table_name' => 'gramene.literature',
	    'row_id' => $ref_id,
	    'dbxref_id' => $dbxref_id
	});

    }

}

sub delete_objectxref{
    my ($self, $objecxref_id) = @_;
    my $objectxref_obj = $self->{'cs_objectxref'}->retrieve($objecxref_id);
    $objectxref_obj->delete;
}

sub delete_reference{
    my ($self, $ref_id) = @_;
    my $db_attr_ref = _transact_init( $self->{'db'} );
    eval{

	my @del_ref_sqls=(
	    'delete from reference_abstract  where reference_id= ?',
	    'delete from reference_extra where reference_id= ?',
	    'delete from reference_url where reference_id= ?',
	    'delete from author where reference_id= ?',
	    'delete from reference_comment where reference_id= ?',
	    'delete from objectxref where row_id = ?',
	    'delete from reference where reference_id= ?'
	);
	
	foreach my $del_ref_sql (@del_ref_sqls){
	    $self->{'db'}->do($del_ref_sql,{},($ref_id));
	}

	$self->{'db'}->commit();

    };
    _transact_finish ($self->{'db'}, $db_attr_ref,$@);
}

sub _transact_init {
    my $dbh = shift;
    my $attr_ref = {};
    $attr_ref->{AutoCommit} = $dbh->{AutoCommit};

    $dbh->{AutoCommit}= 0;

    return ($attr_ref);
}

sub _transact_finish {
    my ($dbh, $attr_ref, $error) = @_;
    if ( $error ){
	print "Database update failed, rolling back. Error
	was:\n$error\n";
	eval { $dbh->rollback; };
    }
    $dbh->{AutoCommit} = $attr_ref->{AutoCommit};
}



sub _next_id {
    my ( $db, $table_name, $field_name ) = @_;
    my $id = $db->selectrow_array("select max($field_name) from $table_name");
    return ++$id;
}

1;
