package Gramene::Gene::GeneDB;

=head1 NAME

Gramene::Gene::GeneDB

=head1 SYNOPSIS

use Gramene::Gene::GeneDB;

=head1 DESCRIPTION

A module for querying gene database  

=cut

# ----------------------------------------------------------------

use strict;
use DBI;

use Gramene::DB;
use Gramene::Ontology::OntologyDB;


sub new {

    my $class = shift;
    my $self  = {};
    bless( $self, $class );
    return $self;

}

# destructor
sub DESTROY {
    my $self = shift;

    # $self->terminate_database;
}

sub terminate_database {
    my $self = shift;
    $self->{'db'}->disconnect() if $self->{'db'};
    $self->{'ontology_db'}->disconnect() if $self->{'ontology_db'};
    $self->{'literature_db'}->disconnect() if $self->{'literature_db'};


}

sub connect_to_db {
    my $self = shift;

    eval {
	# get gene db connection from configuration file
        $self->{'db'}  = Gramene::DB->new('genes');  
	
	# get literature db connection from configuration file
        $self->{'literature_db'} = Gramene::DB->new('literature'); 
	
	# get ontology db connection from ontology DB module for using its methods
	$self->{'OntologyDB'} = Gramene::Ontology::OntologyDB->new();
    };
    if ($@) {
        die "DB connection failed:$@\n";
    }

}


# -------------------------------------------------------------------

=pod

=head2 get_all_species

Get all species objects(id, common_name) in gene database

=cut


sub get_all_species {

    my $self = shift;
    my @species;
    my $sth = $self->{'db'}->prepare("SELECT * FROM gene_species");
    $sth->execute or die $self->{'db'}->errstr;
    my @all_species; 

    while ( my $spe = $sth->fetchrow_hashref ) {
        push @all_species, $spe;
    }

    return \@all_species;
}



# -------------------------------------------------------------------

=pod

=head2 get_species

Get species object by species_id

=cut


sub get_species {

    my ($self, $species_id) = @_;

    my $sth1 = $self->{'db'}->prepare(
        q[
                               SELECT * FROM gene_species
                               WHERE species_id = ?
                              ]
    );

    $sth1->execute( ($species_id) ) or die $self->{'db'}->errstr;

    my $species_obj = $sth1->fetchrow_hashref();
    $sth1->finish;

    return $species_obj;
}

sub get_gene_id {

    my ($self, $acc) = @_;

    #$acc = "GR:00" . $acc unless ( $acc =~ /GR:00/ );

    my ($gene_id) = $self->{'db'}->selectrow_array(
        q[
             SELECT gene_id FROM gene_gene
             WHERE accession = ?
         ],
        {},
        ($acc)
    );

    unless($gene_id){
	($gene_id) = $self->{'db'}->selectrow_array(
	    q[
		SELECT gene_id FROM gene_gene_synonym
		WHERE  synonym_name = ?
	    ],{},($acc)
	);
    }

    return $gene_id;

}

# -------------------------------------------------------------------

=pod

=head2 get_gene_general_info

Get the gene gene's general information (accession,symbol,name,chromosome,description,curator_comment,species). The species is a species object.

Return a hash object

=cut
sub get_gene_general_info {
    my ($self, $gene_id) = @_;

    my $sth = $self->{'db'}->prepare(
	q[
	      SELECT gene_id,
		     accession,
		     symbol,
		     name,
		     species_id,
		     gene_type,
		     chromosome,
		     description,
		     public_curation_comment,
		     internal_curation_comment,
		     has_phenotype,
		     is_obsolete
	      FROM   gene_gene, gene_gene_type
	      WHERE  gene_gene.gene_type_id = gene_gene_type.gene_type_id 
	      AND    gene_id =?
	]
    );

    $sth->execute( ($gene_id) ) or die $self->{'db'}->errstr;

    my $gene_obj = $sth->fetchrow_hashref();

    my $species_id = $gene_obj->{'species_id'};



    #add the species obj to gene obj
    $gene_obj->{'species'} = get_species($self, $species_id); 

    #delete the species_id
    delete $gene_obj->{'species_id'};

    $sth->finish;

    return $gene_obj;
}


sub get_gene_simple_field {
    my ( $self, $field, $gene_id ) = @_;
    my ($val) = $self->{'db'}->selectrow_array(
        qq[
                          SELECT $field FROM gene_gene
                          WHERE gene_id = ?
			  ], {}, ($gene_id)
    );
    return $val;

}

# ----------------------------------------------------------------------

=pod

=head2 get_gene_synonyms

return an arrayRef of gene synonyms

=cut

sub get_gene_synonyms {
    my ($self, $gene_id) = @_;

    my $sth = $self->{'db'}->prepare(
        q[
                                SELECT synonym_name
                                FROM   gene_gene_synonym
                                WHERE  gene_id =?
			      ]
    );

    $sth->execute( ($gene_id) ) or die $self->{'db'}->errstr;

    my @synonyms;

    while ( my ($syn) = $sth->fetchrow_array ) {
        push @synonyms, $syn;
    }

    $sth->finish;

    return \@synonyms;

}



# -------------------------------------------------------------------

=pod

=head2 get_gene_interactions

Get gene gene interaction data 


=cut

sub get_gene_interactions {
    my ($self, $gene_id) = @_;

    return unless $gene_id ;

    my $sth = $self->{'db'}->prepare(q[
	SELECT G.gene_interaction_id, 
		gene1_id,
		G1.accession as gene1_accession,
		G1.symbol as gene1_symbol,
		gene1_object_class,
		T.interaction_type,
		T.gene_interaction_type_id,
		gene2_id,
		G2.accession as gene2_accession,
		G2.symbol as gene2_symbol,
		gene2_object_class
	FROM   gene_gene_interaction G,
		gene_gene_interaction_type T,
		gene_gene G1,
		gene_gene G2
	WHERE   G.gene_interaction_type_id = T.gene_interaction_type_id
	AND    G.gene1_id = G1.gene_id
	AND    G.gene2_id = G2.gene_id
	AND    (
	        gene1_id = ?
		OR gene2_id = ?
		)
    ]);

    my $sth2 = $self->{'db'}->prepare(q[
	SELECT gene_interaction_evidence_id,
	       gene_interaction_id,
	       evidence_code,
	       comment,
	       D.dbxref_id,
	       O.dbxref_to_object_id,
	       D.dbxref_name,
	       D.url,
	       O.dbxref_value
	FROM   gene_gene_interaction_evidence E,
	       gene_dbxref_to_object O,
	       gene_dbxref D
	WHERE  E.dbxref_to_object_id = O.dbxref_to_object_id
	AND    O.dbxref_id = D.dbxref_id
	AND    E.gene_interaction_id = ?
    ]);
		
    $sth->execute(($gene_id, $gene_id)) or die $self->{'db'}->errstr;

    my @interactions;
    while(my $interaction = $sth->fetchrow_hashref){
	my $interaction_id = $interaction->{'gene_interaction_id'};
	$sth2->execute(($interaction_id));
	while(my $evn = $sth2->fetchrow_hashref){
	    my $url = $evn->{'url'};
	    my $val = $evn->{'dbxref_value'};
	    $url =~ s/\Q[%?%]/$val/;
	    my $link_url = qq[<a href="$url">$val</a>];
	    $evn->{'ref_link'} = $link_url;

	    push @{$interaction->{'evidences'}},$evn;
	};
	push @interactions, $interaction;
    }
    return \@interactions;

}


# -------------------------------------------------------------------

=pod

=head2 get_gene_map_position

Get gene map position

Return a hash of maps( file_name, image_comment)

=cut

sub get_gene_map_position {
    my ($self, $gene_id) = @_;

    return unless $gene_id ;

    my @maps;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT * 
	      FROM   gene_map_position
	      WHERE  gene_id = ?
	  ]
    );

    $sth->execute( ($gene_id) ) or die $self->{'db'}->errstr;
    while(my $map = $sth->fetchrow_hashref()){
	push @maps, $map;
    }

    return \@maps;

}


# -------------------------------------------------------------------

=pod

=head2 get_object_images

Get images

Return a hash of images( file_name, image_comment)

=cut

sub get_object_images {
    my ($self, $obj_type, $obj_id) = @_;

    return unless ($obj_id && $obj_type) ;

    my @images;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT file_name, image_comment 
	      FROM   gene_image I,
		     gene_object_to_image OI
	      WHERE  I.image_id = OI.image_id
	      AND    object_id = ?
	      AND    object_table = ?
	  ]
    );

    $sth->execute( ($obj_id, $obj_type) ) or die $self->{'db'}->errstr;
    while(my $image = $sth->fetchrow_hashref()){
        $image->{'image_comment'} = 'No Caption' unless $image->{'image_comment'};
	push @images, $image;
    }

    return \@images;

}


# ------------------------------------------------------------------------------

=pod

=head2 get_ontology_association

Get the object's ontology associations.

Return a hash object, term_type as the keys and an array of term objects(term_accession,term_name) as value. 

=cut

sub get_ontology_association {
    my ($self, $object_type, $object_id) = @_;

    my $sth = $self->{'db'}->prepare(
        q[
                               SELECT term_type,term_accession
                               FROM   gene_ontology_association
                               WHERE  object_table =?
			       AND    object_id =  ?
                               GROUP BY term_type, term_accession 
                             ]
    );

    $sth->execute( ($object_type, $object_id) ) or die $self->{'db'}->errstr;

    my %terms_name;
    my %terms;

    while ( my ( $term_type, $acc ) = $sth->fetchrow_array ) {
        my $term_name = _get_term_name($self,  $acc );
        $term_name ||= $acc;    # for term not in ontology DB yet
	$terms_name{$term_type}->{$term_name} = $acc;
   }

   # sort the term_names
   foreach my $term_type (keys %terms_name){
	my $term_nameRef = $terms_name{$term_type};
	foreach my $term_name (sort keys %$term_nameRef){
	    my @accs = $term_nameRef->{$term_name};
	    foreach my $acc (@accs){
		push(@{$terms{$term_type}},{'term_accession' => $acc, 'term_name' => $term_name });
	    }
	}

   }

    
    return \%terms;
}

sub _get_term_name {

    my ( $self, $acc ) = @_;

    return $self->{'OntologyDB'}->get_term_name_by_term_accession($acc);

}

# --------------------------------------------------------------------------------

=pod

=head2 get_gene_references

Get the gene gene's or allele's  literature references.

Return an array of reference object.

=cut

sub get_related_references {

    my ($self, $table_name, $obj_id, $dbxref_name) = @_;



    my $dbxref_id = &get_dbxref_id($self, $dbxref_name);

    my %unsorted_references;
    my @references;

    my $sth = $self->{'db'}->prepare(
        q[
                        SELECT dbxref_value
                        FROM   gene_dbxref_to_object
                        WHERE  table_name = ?
                          AND  dbxref_id = ?
                          AND  record_id = ?
                      ]
    );

    $sth->execute( ( $table_name, $dbxref_id, $obj_id ) ) or die $self->{'db'}->errstr;

    while ( my ($ref_id) = $sth->fetchrow_array() ) {
        my $ref_obj = _get_reference_detail($self, $ref_id);
	my $pub_year = $ref_obj->{'year'};
	push @{$unsorted_references{$pub_year}},$ref_obj;
    }

    $sth->finish;

    # sort references by publish year
    foreach my $pub_year (sort {$a<=>$b} keys %unsorted_references){
	my @ref_objs = @{$unsorted_references{$pub_year}};
        push @references, @ref_objs;

    }

    return \@references;

}

sub get_dbxref_id {

    my ($self, $dbxref_name) = @_;

    my ($dbxref_id) = $self->{'db'}->selectrow_array(
        q[
                                     SELECT dbxref_id FROM gene_dbxref
                                     WHERE  dbxref_name = ?
				   ],
        {},
        ($dbxref_name)
    );

    return $dbxref_id;

}

#
# get the reference's title,source_name,volume,start_page,end_page and author
#
sub _get_reference_detail {

    my ($self, $ref_id) = @_;

    my $ref_obj;

    my $sth = $self->{'literature_db'}->prepare(
        q[
                           SELECT reference_id, title,source_name,year,volume,
                                  start_page,end_page
                           FROM   reference
			   LEFT JOIN source
                           ON  reference.source_id = source.source_id
                           WHERE    reference_id = ?
                         ]
    );
    $sth->execute( ($ref_id) ) or die $self->{'literature_db'}->errstr;

    $ref_obj = $sth->fetchrow_hashref;
    $ref_obj->{'title'} =~s/\.$//;

    my $sth2 = $self->{'literature_db'}->prepare(
        q[
                              SELECT contributor.contributor_name 
                              FROM contributor, author 
                              WHERE author.reference_id = ? 
                              AND contributor.contributor_id = author.contributor_id 
                              order by authorship_position
                            ]
    );

    $sth2->execute( ($ref_id) ) or die $self->{'literature_db'}->errstr;

    my $author_name;

    while ( my ($row) = $sth2->fetchrow_array ) {
        $author_name .= $row . ", ";
    }

    $author_name =~s/\s+$//;
    $author_name =~ s/,$/./;

    $ref_obj->{'author'} = $author_name;

    $sth->finish;

    $sth2->finish;

    return $ref_obj;

}


#--------------------------------------------------------------------------------

=pod

=head2 get_gene_xref_vals_by_xdb

Get the  gene's xref values by querying the xref_dbname
return  an array of dbxref values 

=cut


sub get_gene_xref_vals_by_xdb {
    my ($self, $table_name, $record_id, $dbxref_name ) = @_;

    my $sth = $self->{'db'}->prepare(
        q[
	   SELECT dbxref_value
	   FROM   gene_dbxref_to_object MDO, gene_dbxref DBX
	   WHERE  MDO.dbxref_id = DBX.dbxref_id
	   AND  table_name = ?
	   AND  record_id = ?
	   AND  dbxref_name = ?
	   ORDER BY dbxref_value
        ]
    );

    $sth->execute( ( $table_name, $record_id, $dbxref_name ) );

    my @dbxrefs;

    while ( my (  $val ) = $sth->fetchrow_array ) {

        push @dbxrefs, $val;

    }

    return @dbxrefs;

}



#--------------------------------------------------------------------------------

=pod

=head2 get_gene_xrefs

Get the  gene's xref values by querying the xref_dbname
return  an array of dbxref values with hyper link to the xref_db

=cut


sub get_gene_xrefs_by_xdb {
    my ($self, $table_name, $record_id, $dbxref_name ) = @_;

    my $sth = $self->{'db'}->prepare(
        q[
	   SELECT url,dbxref_value
	   FROM   gene_dbxref_to_object MDO, gene_dbxref DBX
	   WHERE  MDO.dbxref_id = DBX.dbxref_id
	   AND  table_name = ?
	   AND  record_id = ?
	   AND  dbxref_name = ?
	   ORDER BY dbxref_value
        ]
    );

    $sth->execute( ( $table_name, $record_id, $dbxref_name ) );

    my @dbxrefs;

    while ( my ( $url, $val ) = $sth->fetchrow_array ) {
        $url =~ s/\Q[%?%]/$val/;
        my $link_url = qq[<a href="$url">$val</a>];

        push @dbxrefs, $link_url;

    }

    return @dbxrefs;

}


# get dbxref with dbxrefs->{db_name}->{$gene_dbxref_to_object_id}=$dbxref_value;
sub get_object_all_dbxrefs {
    my ($self, $table_name, $obj_id ) = @_;
    my $sql = q[
                     SELECT dbxref_to_object_id,dbxref_value,dbxref_name
                     FROM  gene_dbxref_to_object DTO,gene_dbxref DBX
                     WHERE DTO.dbxref_id = DBX.dbxref_id
                     AND   table_name= ?
                     AND   record_id = ?
	      ];

    my $sth = $self->{'db'}->prepare($sql);

    $sth->execute( ( $table_name, $obj_id ) );

    my $dbxrefs;
    while ( my ( $id, $val, $db_name ) = $sth->fetchrow_array ) {
        $dbxrefs->{ $db_name }->{$id} = $val;
    }

    $sth->finish;
    return $dbxrefs;
}

sub get_xref_object{
    my ($self, $dbxref_to_object_id) = @_;
    my ($val) = $self->{'db'}->selectrow_array(q[
	SELECT dbxref_value FROM gene_dbxref_to_object
	WHERE  dbxref_to_object_id = ?
    ],{},($dbxref_to_object_id));

    return $val;
}

sub get_all_dbxrefs {

    my $self = shift;

    my @dbxrefs;
    my $sth = $self->{'db'}->prepare("SELECT dbxref_id,dbxref_name, url FROM gene_dbxref");

    $sth->execute or die $self->{'db'}->errstr;


    while ( my $rowhash = $sth->fetchrow_hashref ) {
        push @dbxrefs, $rowhash;
    }

    return \@dbxrefs;
}


sub get_xref_db {
    my ($self, $dbxref_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from gene_dbxref WHERE dbxref_id = ?");
    $sth->execute(($dbxref_id));
    my ($xref_db) = $sth->fetchrow_hashref();
    $sth->finish;
    return $xref_db;
}

#
#sub get_gene_sequence_association {
#
#    my ($self, $gene_id) = @_;
#
#    my %sequence_association = ();
#
#    my $dbxref_name = 'GenBank.Nucleotide';
#    my $table_name  = 'gene.gene';
#
#    my @nucleotides =
#      _get_dbxref_to_object($self, $dbxref_name, $table_name, $gene_id );
#
#    if (@nucleotides) {
#        $dbxref_name = 'GenBank';
#        $sequence_association{'Nucleotide (DNA)'}->{$dbxref_name} =
#          \@nucleotides;
#    }
#
#    $dbxref_name = 'Gramene.Protein';
#    my @proteins = _get_dbxref_to_object($self, $dbxref_name, $table_name, $gene_id );
#
#    if (@proteins) {
#        $sequence_association{'Protein'}->{$dbxref_name} = \@proteins;
#    }
#
#    return \%sequence_association;
#
#}

# -----------------------------------------------------------------------------------

=pod

=head2 get_genes_by_gene_fiels

Get the candidate gene objects by searching  fields in gene_gene table and gene_gene_synonym table, 
optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut

sub get_genes_by_gene_fields {
    my ( $self, $query, $search_field, $species, $gene_type_id, $has_phenotype, $order_by ) = @_;

    $query = uc($query);
    $query =~ s/\*/%/g;

    my $sql = qq[
             SELECT G.gene_id as gene_id, 
                    accession,
                    symbol,
                    name,
		    gene_type,
		    has_phenotype,
                    chromosome,
		    common_name,
		    synonym_name
               FROM gene_gene G
	       INNER JOIN gene_gene_type GT
	       ON   G.gene_type_id = GT.gene_type_id
	       INNER JOIN gene_species GS
	       ON   G.species_id = GS.species_id
               LEFT  JOIN gene_gene_synonym GSN 
                 ON  G.gene_id = GSN.gene_id    
           ];


    my $condition = '';
    my @params;
    if($search_field eq 'name' ){
	$condition = qq[
	    WHERE   
		    (
		    UPPER(symbol) like ?
	    OR      UPPER(name) like ?
	    OR      UPPER(synonym_name) like ?
		    )
	];
	@params = ($query,$query, $query);
    }elsif($search_field eq 'accession'){
	$condition = qq[
	    WHERE (
		accession like ?
		OR UPPER(synonym_name) like ?
	    )
	];
	@params = ($query,$query);
    
    }else{
	$condition = " WHERE UPPER($search_field) like ? ";
	@params = ($query);
    }


    if($species){
	$condition .= ' AND G.species_id = ? ';
	push @params, $species;
    }


    if($gene_type_id){
	$condition .= ' AND G.gene_type_id = ? ';
	push @params, $gene_type_id;
    }
    
    if($has_phenotype){
	$condition .= ' AND G.has_phenotype = ? ';
	push @params, $has_phenotype;
    } 



    if($order_by){
	# use if ( is null) to set null or empty field displayed at last
	$condition .= " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by" ;
    }

    my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );

    $sth->execute(@params) or die $self->{'db'}->errstr;

    my @candidates; # candidate genes
    my %uniq_gene_objs;    # genes and positions

    while ( my $gene_obj = $sth->fetchrow_hashref ) {
        my $gene_id = $gene_obj->{'gene_id'};
	my $position = $uniq_gene_objs{$gene_id};
	unless(defined $position){
	    $position = $#candidates + 1;
	    $uniq_gene_objs{$gene_id} = $position;
	    $candidates[$position] = $gene_obj;
	}

    }
    
    # can't use join to get all gene synonyms, it only give the synonym
    # matched joined
    _get_gene_objs_synonyms($self, \@candidates);
    _get_gene_objs_term_associations($self, \@candidates);

    return @candidates;

}


# -----------------------------------------------------------------------------------

=pod

=head2 get_genes_by_xref

Get the candidate gene objects by searching  xref, 
optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut
sub get_genes_by_xref {
    my ( $self, $query, $dbxref_name, $species, $gene_type_id, $has_phenotype, $order_by) = @_;

    $query = uc($query);
    $query =~ s/\*/%/g;

    my $sql = qq[
               SELECT G.gene_id as gene_id,
                      accession,
                      symbol,
                      name,
		      gene_type,
		      has_phenotype,
                      chromosome,
		      common_name,
		      synonym_name as synonyms 
               FROM   gene_gene G
	       INNER JOIN gene_gene_type GT
	       ON    G.gene_type_id = GT.gene_type_id
	       INNER JOIN gene_species GS
	       ON    G.species_id = GS.species_id
	       LEFT JOIN gene_gene_synonym GSN
	       ON     G.gene_id = GSN.gene_id 
	       INNER  JOIN gene_dbxref_to_object DTO
	       ON     G.gene_id = DTO.record_id
	       INNER  JOIN gene_dbxref DX
	       ON     DTO.dbxref_id = DX.dbxref_id
	       WHERE  dbxref_name = ? 
               AND    UPPER(dbxref_value) like ?
           ];

    my $condition = '';
    my @params = ($dbxref_name, $query);

    if($species){
	$condition .= ' AND G.species_id = ? ' if $species;
	push @params, $species;
    }


    if($gene_type_id){
	$condition .= ' AND G.gene_type_id = ? ';
	push @params, $gene_type_id;
    }
    
    if($has_phenotype){
	$condition .= ' AND G.has_phenotype = ? ';
	push @params, $has_phenotype;
    } 


    if($order_by){
	# use if ( is null) to set null or empty field displayed at last
	$condition .= " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by" ;
    }

    my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );

    $sth->execute(@params) or die $self->{'db'}->errstr;


    my @candidates; # candidate genes
    my %uniq_gene_objs;    # genes and positions
    my %gene_synonyms;

    while ( my $gene_obj = $sth->fetchrow_hashref ) {
        my $gene_id = $gene_obj->{'gene_id'};
	my $position = $uniq_gene_objs{$gene_id};
	if(defined $position){
	    my $syn = $gene_obj->{'synonyms'};
	    if($syn){
		unless($gene_synonyms{$gene_id}->{$syn}++){
		    my $gene = $candidates[$position];
		    my $syns = $gene->{'synonyms'};
		    $syns = $syns ? "$syns, $syn" : $syn;
		    $gene->{'synonyms'} = $syns;
		}

	    }

	}else{
	    $position = $#candidates + 1;
	    $uniq_gene_objs{$gene_id} = $position;
	    $candidates[$position] = $gene_obj;
	}

    }

    _get_gene_objs_term_associations($self, \@candidates);

    return @candidates;

}




# -----------------------------------------------------------------------------------

=pod

=head2 get_genes_by_ontology_term_name

Get the candidate gene objects by searching  ontology term name, 
optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut
sub get_genes_by_ontology_term_name {
    my ( $self, $query, $term_type, $species, $gene_type_id, $has_phenotype, $order_by) = @_;

    $query = uc($query);
    $query =~ s/\*/%/g;

    my $candidate_terms = $self->{'OntologyDB'}->get_term_family_by_name($query, $term_type);
    my @term_accs = keys %$candidate_terms;


    return unless(scalar(@term_accs))>0;
    my $sql_assoc = q[
	SELECT object_id FROM gene_ontology_association
	WHERE  object_table = 'gene' AND term_accession in 
    ];

    my @param_accs;
    push(@param_accs, '?') for 1 .. scalar(@term_accs);
    $sql_assoc = $sql_assoc.'('.join(', ', @param_accs).')';

    my $sth_assoc = $self->{'db'}->prepare($sql_assoc);

    $sth_assoc->execute(@term_accs) or die $self->{'db'}->errstr;

    my $onto_genes = $sth_assoc->fetchall_hashref('object_id');
    my @params = keys(%$onto_genes);


    return unless(scalar(@params)>0);


    my $sql = qq[
             SELECT G.gene_id as gene_id, 
                    accession,
                    symbol,
                    name,
		    gene_type,
		    has_phenotype,
                    chromosome,
		    common_name,
		    synonym_name as synonyms
               FROM gene_gene G
	       INNER JOIN gene_gene_type GT
	       ON   G.gene_type_id = GT.gene_type_id
	       INNER JOIN gene_species GS
	       ON   G.species_id = GS.species_id
               LEFT  JOIN gene_gene_synonym GSN 
                 ON  G.gene_id = GSN.gene_id    
           ];
    

    my @param_holders;
    push(@param_holders, '?') for 1 .. scalar(@params);

    my $condition = ' WHERE G.gene_id in ('.join(', ', @param_holders).')';

    

    if($species){
	$condition .= ' AND G.species_id = ? ';
	push @params, $species;
    }


    if($gene_type_id){
	$condition .= ' AND G.gene_type_id = ? ';
	push @params, $gene_type_id;
    }
    
    if($has_phenotype){
	$condition .= ' AND G.has_phenotype = ? ';
	push @params, $has_phenotype;
    } 

    if($order_by){
	# use if ( is null) to set null or empty field displayed at last
	$condition .= " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by" ;
    }

    my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );


    $sth->execute(@params) or die $self->{'db'}->errstr;

    my @candidates; # candidate genes
    my %uniq_gene_objs;    # genes and positions

    while ( my $gene_obj = $sth->fetchrow_hashref ) {
        my $gene_id = $gene_obj->{'gene_id'};
	my $position = $uniq_gene_objs{$gene_id};
	if(defined $position){
	    my $syn = $gene_obj->{'synonyms'};
	    if($syn){
		my $gene = $candidates[$position];
		my $syns = $gene->{'synonyms'};
		$syns = $syns ? "$syns, $syn" : $syn;
		$gene->{'synonyms'} = $syns;
	    }

	}else{
	    $position = $#candidates + 1;
	    $uniq_gene_objs{$gene_id} = $position;
	    $candidates[$position] = $gene_obj;
	}

    }

    _get_gene_objs_term_associations($self, \@candidates);

    return @candidates;

}



# -----------------------------------------------------------------------------------

=pod

=head2 get_genes_by_all_fields

Get the candidate gene objects by searching  fields in gene_gene table, gene_gene_synonym table
and xref tables, optionally with species id 
The order of the gene objects will depend on the order by fields (name,symbol,chromosome #, accession)

=cut

sub get_genes_by_all_fields {
    my ( $self, $query,$species, $gene_type_id, $has_phenotype, $order_by) = @_;

    $query = uc($query);
    $query =~ s/\*/%/g;

    # search ontology term
    my $candidate_terms = $self->{'OntologyDB'}->get_term_family_by_name($query);
    my @term_accs = keys %$candidate_terms;

    # search genes with term association
    my @onto_genes;
    if(scalar(@term_accs) > 0 ){

	my $sql_assoc = q[
	    SELECT object_id FROM gene_ontology_association
	    WHERE  object_table = 'gene' AND term_accession in 
	];

	my @param_accs;
	push(@param_accs, '?') for 1 .. scalar(@term_accs);
	$sql_assoc = $sql_assoc.'('.join(', ', @param_accs).')';

	my $sth_assoc = $self->{'db'}->prepare($sql_assoc);

	$sth_assoc->execute(@term_accs) or die $self->{'db'}->errstr;

	my $onto_genesRef = $sth_assoc->fetchall_hashref('object_id');
	@onto_genes = keys(%$onto_genesRef);
    }

    # search gene in gene and synonym table
    my $sql_gene = qq[
               SELECT G.gene_id
               FROM   gene_gene G 
	       LEFT JOIN gene_gene_synonym GSN
	       ON     G.gene_id = GSN.gene_id 
	       WHERE  
	       (
		     UPPER(accession) like  ? 
	       OR    UPPER(symbol) like ?
	       OR    UPPER(name) like ?
	       OR    UPPER(chromosome) like ?
	       OR    UPPER(synonym_name) like ?
	       )
           ];

    my @gene_params;
    push(@gene_params, $query) for (1 .. 5);
    
    my $sth_gene = $self->{'db'}->prepare($sql_gene );
    $sth_gene->execute(@gene_params) or die $self->{'db'}->errstr;
    my $core_genesRef = $sth_gene->fetchall_hashref('gene_id');
    my @core_genes = keys(%$core_genesRef);
 
    # search gene in xref

    my $sql_xref = qq[
               SELECT G.gene_id as gene_id
               FROM   gene_gene G
	       INNER  JOIN gene_dbxref_to_object DTO
	       ON     G.gene_id = DTO.record_id
	       INNER  JOIN gene_dbxref DX
	       ON     DTO.dbxref_id = DX.dbxref_id
	       WHERE  dbxref_name in ( ?, ?)
               AND    UPPER(dbxref_value) like ?
           ];

    my @xref_db = ('Gramene Protein', 'GenBank Nucleotide');
    my $sth_xref = $self->{'db'}->prepare($sql_xref );
    $sth_xref->execute((@xref_db,$query)) or die $self->{'db'}->errstr;
    my $xref_genesRef = $sth_xref->fetchall_hashref('gene_id');
    my @xref_genes = keys(%$xref_genesRef);



    my %uniq_candidates;
    foreach my $gene (@onto_genes,@core_genes,@xref_genes){
	$uniq_candidates{$gene} = 1;
    }

    my @candidates; # candidate genes

    if(%uniq_candidates && scalar(keys %uniq_candidates) > 0){


	my $sql = qq[
             SELECT G.gene_id as gene_id, 
                    accession,
                    symbol,
                    name,
		    gene_type,
		    has_phenotype,
                    chromosome,
		    common_name
               FROM gene_gene G, gene_species GS, gene_gene_type GT
	       WHERE   G.species_id = GS.species_id
	       AND     G.gene_type_id = GT.gene_type_id
           ];


	my $condition = '';
	my @params;

	my @param_holders;
	push(@param_holders, '?') for 1 .. scalar(keys %uniq_candidates);
	
	$condition .= ' AND G.gene_id in ('.join(', ', @param_holders).')';
	push @params, keys(%uniq_candidates);
	if($species){
	    $condition .= ' AND G.species_id = ? ';
	    push @params, $species;
	}



	if($gene_type_id){
	    $condition .= ' AND G.gene_type_id = ? ';
	    push @params, $gene_type_id;
	}
    
	if($has_phenotype){
	    $condition .= ' AND G.has_phenotype = ? ';
	    push @params, $has_phenotype;
	} 


	if($order_by){
	    # use if ( is null) to set null or empty field displayed at last
	    $condition .= " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by" ;
	}

	my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );

	$sth->execute(@params) or die $self->{'db'}->errstr;


	while ( my $gene_obj = $sth->fetchrow_hashref ) {
	    push @candidates, $gene_obj;
	}
    
	_get_gene_objs_synonyms($self, \@candidates);
	_get_gene_objs_term_associations($self, \@candidates);
    }

	return @candidates;
}

# get genes by field in gene_gene and gene_gene_synonym
sub get_genes_by_core_fields {
    my ( $self, $query,$species, $gene_type_id, $has_phenotype, $order_by) = @_;

    $query = uc($query);
    $query =~ s/\*/%/g;

    my $sql = qq[
               SELECT G.gene_id as gene_id,
                      accession,
                      symbol,
                      name,
		      gene_type,
		      has_phenotype,
                      chromosome,
		      common_name,
		      synonym_name as synonyms 
               FROM   gene_gene G
	       INNER JOIN gene_gene_type GT
	       ON    G.gene_type_id = GT.gene_type_id
	       INNER JOIN gene_species GS
	       ON    G.species_id = GS.species_id
	       LEFT JOIN gene_gene_synonym GSN
	       ON     G.gene_id = GSN.gene_id 
	       WHERE  
	       (
		     UPPER(accession) like  ? 
	       OR    UPPER(symbol) like ?
	       OR    UPPER(name) like ?
	       OR    UPPER(chromosome) like ?
	       OR    UPPER(synonym_name) like ?
	       )
           ];

    my $condition = '';
    my @params;
    push(@params, $query) for (1 .. 5);


    if($species){
	$condition .= ' AND G.species_id = ? ' if $species;
	push @params, $species;
    }



    if($gene_type_id){
	$condition .= ' AND G.gene_type_id = ? ';
	push @params, $gene_type_id;
    }
    
    if($has_phenotype){
	$condition .= ' AND G.has_phenotype = ? ';
	push @params, $has_phenotype;
    } 


    if($order_by){
	# use if ( is null) to set null or empty field displayed at last
	$condition .= " order by if ( $order_by is NULL or $order_by='', 1, 0), $order_by" ;
    }

    my $sth = $self->{'db'}->prepare( "$sql" . "$condition" );


    $sth->execute(@params) or die $self->{'db'}->errstr;


    my @candidates; # candidate genes
    my %uniq_gene_objs;    # genes and positions

    while ( my $gene_obj = $sth->fetchrow_hashref ) {
        my $gene_id = $gene_obj->{'gene_id'};
	my $position = $uniq_gene_objs{$gene_id};
	unless(defined $position){
	    $position = $#candidates + 1;
	    $uniq_gene_objs{$gene_id} = $position;
	    $candidates[$position] = $gene_obj;
	}

    }
    
    # can't use join to get all gene synonyms, it only give the synonym
    # matched joined
    _get_gene_objs_synonyms($self, \@candidates);
    _get_gene_objs_term_associations($self, \@candidates);

    return @candidates;
}

# get gene obj 'synonyms' as a string concatednated by comma
sub _get_gene_objs_synonyms {

    my ($self, $gene_objsRef) = @_;
    my @gene_objs = @$gene_objsRef;

    my $sth = $self->{'db'}->prepare(
        q[
	    SELECT synonym_name 
	    FROM   gene_gene_synonym 
	    WHERE  gene_id = ?
	 ]
    );

    foreach my $gene_obj (@gene_objs) {
        my $gene_id = $gene_obj->{'gene_id'};
        $sth->execute( ($gene_id) ) or die $self->{'db'}->errstr;
        my @synonym_names;
        while ( my ($name) = $sth->fetchrow_array ) {
            push @synonym_names, $name;
        }
        my $synonym_name = join ( ", ", @synonym_names );
        $gene_obj->{'synonyms'} = $synonym_name;

    }

    $sth->finish;
}

# get gene objs' 'TO','PO','GO','EO' etc
# The term accession prfix as the  attribute
sub _get_gene_objs_term_associations {

    my ($self, $gene_objsRef) = @_;
    my @gene_objs = @$gene_objsRef;
    
    return unless(scalar(@gene_objs)>0);

    my $sth = $self->{'db'}->prepare(
        q[
	    SELECT term_accession
	    FROM   gene_ontology_association
	    WHERE  object_id = ?
	    AND    object_table = ?
	 ]
    );

    my $onto_db = $self->{'OntologyDB'}->db; 
    my $onto_terms = $onto_db->selectall_hashref(q[
	SELECT term_accession, term_name FROM term
    ], 'term_accession',{});

    foreach my $gene_obj (@gene_objs) {
        my $gene_id = $gene_obj->{'gene_id'};
        $sth->execute( ($gene_id,'gene') ) or die $self->{'db'}->errstr;

	my %term_cat_to_terms;
        while ( my ( $term_acc ) = $sth->fetchrow_array ) {
	    #my $term_name = $self->{'OntologyDB'}->get_term_name_by_term_accession($term_acc);
	    my $term_name = $onto_terms->{$term_acc}->{'term_name'};
            $term_name ||= $term_acc;    #those term not exist in DB

	    if($term_acc =~/(.+):(\d+)/){
		$term_cat_to_terms{lc($1)}->{$term_name} = 1;
	    }

        }
        foreach my $cat ( keys %term_cat_to_terms ) {
            my $terms = join ( ", ", keys(%{$term_cat_to_terms{$cat}}) );
            $gene_obj->{$cat} = $terms;
        }

    }

}


sub get_species {
    my ($self,$spe_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from gene_species WHERE species_id = ?");
    $sth->execute(($spe_id));
    my $spe = $sth->fetchrow_hashref;

    $sth->finish;
    return $spe;
}

sub _get_species_id {

    my ($self, $species_name) = @_;
    my ($species_id) = $self->{'db'}->selectrow_array(
        q[
                                                SELECT species_id FROM gene_species
                                                WHERE common_name = ?
                                              ],
        {},
        ($species_name)
    );
    return $species_id;

}


sub get_germplasm_id {

    my ($self, $acc) = @_;


    my ($germplasm_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT germplasm_id FROM gene_germplasm
	    WHERE accession = ?
	], {}, ($acc)
    );

    return $germplasm_id;

}



# -------------------------------------------------------------------

=pod

=head2 get_germplasm_info

Get the germplasm information (accession,location_name,wild_type,mutagen,
mutagenesis_method, related genes). 

Return a hash object

=cut

sub get_germplasm_info {
    my ($self, $germplasm_id) = @_;

    return unless defined $germplasm_id;

    my $germplasm_obj;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT accession,
		     location_name,
		     mutagen,
		     mutagenesis_method,
		     wild_type
	      FROM   gene_germplasm G,
		     gene_geographical_location GEO,
		     gene_mutagenesis M
	      WHERE  G.geographical_location_id = GEO.geographical_location_id
	      AND    G.mutagenesis_id = M.mutagenesis_id
	      AND    germplasm_id =?
	  ]
    );

    $sth->execute( ($germplasm_id) ) or die $self->{'db'}->errstr;

    $germplasm_obj = $sth->fetchrow_hashref();
    
    if($germplasm_obj){

	$germplasm_obj->{'genes'} = get_object_related_genes($self,
							     'gene_object_to_germplasm',
							     'germplasm_id',
							     $germplasm_id
							     );
	

	$germplasm_obj->{'alleles'} = get_object_related_alleles($self,
								 'gene_object_to_germplasm',
								 'germplasm_id',
								 $germplasm_id
								 );
    }

    $sth->finish;

    return $germplasm_obj;
}




# -------------------------------------------------------------------

=pod

=head2 get_related_germplasms

Get related germplasms

Return an array of germplams( id, accession)

=cut

sub get_related_germplasms {
    my ($self, $obj_type, $obj_id) = @_;

    return unless ($obj_id && $obj_type);

    my @germplasms;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT G.germplasm_id, accession
	      FROM   gene_germplasm G,
		     gene_object_to_germplasm OG
	      WHERE  G.germplasm_id = OG.germplasm_id
	      AND    object_table = ? 
	      AND    object_id =?
	  ]
    );

    $sth->execute( ($obj_type, $obj_id) ) or die $self->{'db'}->errstr;
    while(my $germplasm = $sth->fetchrow_hashref()){
	push @germplasms, $germplasm;
    }

    return \@germplasms;

}



# -------------------------------------------------------------------

=pod

=head2 get_related_studies

Get related studies

Return an array of studies( id, accession)

=cut

sub get_related_studies {
    my ($self, $obj_type, $obj_id) = @_;

    return unless ($obj_id && $obj_type);

    my @studies;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT G.study_id,name 
	      FROM   gene_study G,
		     gene_object_to_study OG
	      WHERE  G.study_id = OG.study_id
	      AND    object_table = ? 
	      AND    object_id =?
	  ]
    );

    $sth->execute( ($obj_type, $obj_id) ) or die $self->{'db'}->errstr;
    while(my $study = $sth->fetchrow_hashref()){
	push @studies, $study;
    }

    return \@studies;

}

# -------------------------------------------------------------------

=pod

=head2 get_study_info

Get the study information 

Return a hash object

=cut

sub get_study_info {
    my ($self, $study_id) = @_;

    return unless defined $study_id;

    my $study_obj;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT study_id,
		     name,
		     year,
		     season,
		     study_type,
		     location_name,
		     environmental_factors
	      FROM   gene_study G,
		     gene_geographical_location GEO,
		     gene_study_type GT 
	      WHERE  G.geographical_location_id = GEO.geographical_location_id
	      AND    G.study_type_id = GT.study_type_id
	      AND    study_id =?
	  ]
    );

    $sth->execute( ($study_id) ) or die $self->{'db'}->errstr;

    $study_obj = $sth->fetchrow_hashref();
    
    if($study_obj){
	$study_obj->{'alleles'} = get_object_related_alleles(
								$self,
								'gene_object_to_study',
								'study_id',
								$study_id
							    );
	
    }

    $sth->finish;

    return $study_obj;
}



sub get_allele_id {

    my ($self, $acc) = @_;

    my ($allele_id) = $self->{'db'}->selectrow_array(
	q[
	    SELECT allele_id FROM gene_allele
	    WHERE accession = ?
	], {}, ($acc)
    );

    return $allele_id;

}


# -------------------------------------------------------------------

=pod

=head2 get_allele_general_info

Get the allele allele's general information (accession,symbol,name,description,curator_comments etc). 

Return a hash object

=cut

sub get_allele_general_info {
    my ($self, $allele_id) = @_;

    my $sth = $self->{'db'}->prepare(
	q[
	    SELECT	allele_id, 
			accession,
			symbol,
			name,
			description,
			allele_interaction_description,
			public_curation_comment 
	      FROM      gene_allele
	      WHERE  	allele_id =?
	]
    );

    $sth->execute( ($allele_id) ) or die $self->{'db'}->errstr;

    my $allele_obj = $sth->fetchrow_hashref();


    $sth->finish;

    return $allele_obj;
}




# -------------------------------------------------------------------

=pod

=head2 get_gene_related_alleles

Get related alleles

Return an array of alleles( id, accession)

=cut

sub get_gene_related_alleles {
    my ($self, $obj_id) = @_;

    return unless $obj_id ;

    my @alleles;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT A.allele_id, symbol 
	      FROM   gene_allele A,
		     gene_gene_to_allele OA
	      WHERE  A.allele_id = OA.allele_id
	      AND    gene_id =?
	  ]
    );

    $sth->execute( ($obj_id) ) or die $self->{'db'}->errstr;
    while(my $allele = $sth->fetchrow_hashref()){
	push @alleles, $allele;
    }

    return \@alleles;

}




# -------------------------------------------------------------------

=pod

=head2 get_allele_related_genes

Get related genes

Return an array of genes( id, accession)

=cut

sub get_allele_related_genes {
    my ($self, $obj_id) = @_;

    return unless $obj_id ;

    my @genes;
    
    my $sth = $self->{'db'}->prepare(
        q[
	      SELECT accession, name 
	      FROM   gene_gene G,
		     gene_gene_to_allele OA
	      WHERE  G.gene_id = OA.gene_id
	      AND    allele_id =?
	  ]
    );

    $sth->execute( ($obj_id) ) or die $self->{'db'}->errstr;
    while(my $gene = $sth->fetchrow_hashref()){
	push @genes, $gene;
    }

    return \@genes;

}



# -------------------------------------------------------------------

=pod

=head2 get_object_related_genes

Get related genes

Return an array of genes( id, accession)

=cut

sub get_object_related_genes {
    my ($self, $table_name, $obj_field_name, $obj_id) = @_;

    my $sth1 = $self->{'db'}->prepare(
	    qq[
		SELECT accession,name 
		FROM   $table_name OG,
		       gene_gene
		WHERE  OG.object_id = gene_gene.gene_id
		AND    object_table = 'gene'
		AND    $obj_field_name = ?
	      ]
    );

    $sth1->execute( ($obj_id) ) or die $self->{'db'}->errstr;
	
    my @genes;
    while(my $gene_ref = $sth1->fetchrow_hashref){
	push @genes, $gene_ref;
    }
    
    return \@genes;

}




# -------------------------------------------------------------------

=pod

=head2 get_object_related_alleles

Get related alleles

Return an array of alleles( id, accession)

=cut

sub get_object_related_alleles {
    my ($self, $table_name, $obj_field_name, $obj_id) = @_;

    my $sth1 = $self->{'db'}->prepare(
	    qq[
		SELECT allele_id, symbol 
		FROM   $table_name OG,
		       gene_allele
		WHERE  OG.object_id = gene_allele.allele_id
		AND    object_table = 'allele'
		AND    $obj_field_name = ?
	      ]
    );

    $sth1->execute( ($obj_id) ) or die $self->{'db'}->errstr;
	
    my @alleles;
    while(my $allele_ref = $sth1->fetchrow_hashref){
	push @alleles, $allele_ref;
    }
    
    return \@alleles;

}

sub get_object_images_and_image_comments {
    my ($self, $obj_type, $obj_id ) = @_;
    my $sth = $self->{'db'}->prepare(
	q[
	    SELECT file_name, image_comment
	    FROM   gene_image GI,
		   object_to_image OI,
	    WHERE  GI.image_id = OI.image_id
	    AND    object_table = ?
	    AND    object_id = ?
	    ORDER BY GI.image_id
	]);

    $sth->execute(($obj_type, $obj_id)) or die $self->{'db'}->errstr;

    my @images;
    while (my $image_ref = $sth->fetchrow_hashref){
	push @images, $image_ref;
    }

    return \@images;
}


sub get_gene_type {
    my ($self, $gene_type_id) = @_;
    my $sth = $self->{'db'}->prepare("SELECT * from gene_gene_type WHERE gene_type_id = ?");
    $sth->execute(($gene_type_id));
    my ($gene_type) = $sth->fetchrow_hashref();
    $sth->finish;
    return $gene_type;
}


sub get_all_gene_types {

    my $self = shift;

    my @gene_types;
    my $sth = $self->{'db'}->prepare("SELECT gene_type_id,gene_type FROM
    gene_gene_type order by gene_type_id");

    $sth->execute or die $self->{'db'}->errstr;


    while ( my $rowhash = $sth->fetchrow_hashref ) {
        push @gene_types, $rowhash;
    }

    return \@gene_types;
}


sub get_all_gene_interaction_types {

    my $self = shift;

    my @gene_interaction_types;
    my $sth = $self->{'db'}->prepare("SELECT gene_interaction_type_id,interaction_type FROM
    gene_gene_interaction_type order by gene_interaction_type_id");

    $sth->execute or die $self->{'db'}->errstr;


    while ( my $rowhash = $sth->fetchrow_hashref ) {
        push @gene_interaction_types, $rowhash;
    }

    return \@gene_interaction_types;
}

1;
