package Gramene::Gene::GeneCurationDB;

=head1 NAME

Gramene::Gene::GeneCurationDB

=head1 SYNOPSIS
  

=head1 DESCRIPTION

A module for querying gene database  

=cut

# ----------------------------------------------------------------

use strict;
use DBI;

use Gramene::DB;
use Gramene::Ontology::OntologyDB;
use Gramene::CDBI::Markers;


sub new {

    my $class = shift;
    my $self  = {};
    bless( $self, $class );
    return $self;

}

# destructor
sub DESTROY {
    my $self = shift;

    #$self->terminate_database;
}

sub terminate_database {
    my $self = shift;
    $self->{'db'}->disconnect() if $self->{'db'};

}

sub connect_to_db {
    my $self = shift;

    eval {
        $self->{'db'} = Gramene::DB->new('genes_edit');
        $self->{'db'}->{AutoCommit} = 0;    # set transaction control
	$self->{'OntologyDB'} = Gramene::Ontology::OntologyDB->new();
    };
    if ($@) {
        die "DB connection failed: $@\n";
    }

}




#########################################################
#########################################################
#   gene curation
#
sub load_gene {

    my ($self, $params, $ontology_fields) = @_;

    # a hashref to hold the errors if error and gene accession if success
    my $result = {};

    #check required fields
    my @required_general_fields = qw[gene_symbol gene_name species  gene_type has_phenotype description];

    foreach my $req_field (@required_general_fields) {
        unless ( $params->{$req_field} ) {
            push ( @{ $result->{'error'} }, "$req_field can't be empty." );
        }

    }



    # at least one ontology association
    my $check_onto = 0;
    foreach my $onto ( keys %$ontology_fields ) {
        if ( $params->{$onto} ) {
            $check_onto = 1;
	    last;
        }
    }
    push (@{ $result->{'error'} }, "Gene must have at least one ontology association.") unless $check_onto;

    # errors before loading
    return $result if ( $result->{'error'} );

    foreach my $key ( keys %$params ) {
        my $val = $params->{$key};
	if($val){
	    $val =~ s/^\s+|\s+$//g;
	    $val =~ s/\s+/ /g;
	    $params->{$key} = $val;
	}
    }

    my $accession = $params->{'acc'};

    eval {
        my $gene_id = _next_id($self,  'gene_gene', 'gene_id' );
	my $obj_type = 'gene';

        _load_gene_general_information( $self, $gene_id, $params, $result );

        if ( $params->{'synonyms'} ) {
            my @synonyms = process_delimited_values($self, ',', $params->{'synonyms'} );
            foreach my $syn (@synonyms) {
                _load_gene_synonym($self, $gene_id, $syn ) if $syn;
            }
        }

	foreach my $onto ( keys %$ontology_fields ) {
	    if ( $params->{$onto} ) {
		my $term_type = $ontology_fields->{$onto};
		my @term_accs = process_delimited_values($self, ',', $params->{$onto} );
		foreach my $term_acc (@term_accs){
		    _load_term_assocs($self, $gene_id, $obj_type, $term_type, $term_acc, $result );
		}
	    }
	}


        if ( $params->{'seq_gene'} ) {
            my $db_xref_id = &get_dbxref_id($self,'GenBank Nucleotide');
	    my @dbxref_vals = process_delimited_values($self, ',', $params->{'seq_gene'} );
	    foreach my $xref_val (@dbxref_vals){
		_load_gene_dbxref($self, $obj_type, $gene_id,$db_xref_id, $xref_val );
	    }
        }

        if ( $params->{'seq_protein'} ) {
            my $db_xref_id = &get_dbxref_id($self,'Gramene Protein');
	    my @dbxref_vals = process_delimited_values($self, ',', $params->{'seq_protein'} );
            foreach my $xref_val (@dbxref_vals){
		_load_gene_dbxref($self, $obj_type, $gene_id,$db_xref_id, $xref_val );
	    }
        }

        my $i = 0;
        while ( $i < 3 ) {    # 3 dbxrefs now
            if ($params->{"dbxref_db_name_$i"} && $params->{"dbxref_db_val_$i"} ) {
		my $db_xref_id = $params->{"dbxref_db_name_$i"};
		my @dbxref_vals = process_delimited_values($self, ',', $params->{"dbxref_db_val_$i"});
		foreach my $xref_val (@dbxref_vals){
		    _load_gene_dbxref($self, $obj_type, $gene_id,$db_xref_id, $xref_val );
		}
	    }
            $i++;
        }

        $self->{'db'}->commit;

    };
    if ($@) {
        push ( @{ $result->{'error'} }, "Unable to save to database: $@" );
        $self->{'db'}->rollback();
    }
    else {
        $result->{'ok'} = $accession;    # loading succefully
    }

    return $result;

}

sub _load_gene_general_information {
    my ($self, $gene_id, $params, $result ) = @_;

    my $is_obsolete = 0;

    my @data = map { defined $_ ? $_ : '' } (
        $gene_id,
	$params->{'acc'},
        $params->{'gene_symbol'},
        $params->{'gene_name'},
        $params->{'species'},
	$params->{'gene_type'},
	$params->{'has_phenotype'},
	$params->{'chromosome'},
        $params->{'description'},
	$params->{'curator_comment'},
	$params->{'curator_note'},
        $is_obsolete
    );


    $self->{'db'}->do(
        q[
                INSERT INTO gene_gene
               (gene_id,accession,symbol,name,species_id, gene_type_id,
	       has_phenotype, chromosome,
                description, public_curation_comment, internal_curation_comment, is_obsolete)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
              ],
        {}, @data
    );

}

sub _load_gene_synonym {
    my ( $self, $gene_id, $synonym ) = @_;

    # use BINARY to let mysql case sensitive
    my ($gene_synonym_id) = $self->{'db'}->selectrow_array(q[
	SELECT gene_synonym_id
	FROM   gene_gene_synonym
	WHERE  gene_id = ?
	AND    BINARY (synonym_name) = ?
    ], {}, ($gene_id, $synonym));

    unless($gene_synonym_id){

	$gene_synonym_id = _next_id( $self, 'gene_gene_synonym', 'gene_synonym_id' );
	$self->{'db'}->do(
	    q[ 
	      INSERT INTO gene_gene_synonym (gene_synonym_id, gene_id, synonym_name)
              VALUES   (?, ?, ?) 
            ], {}, ( $gene_synonym_id, $gene_id, $synonym )
	);
    }
}

sub _load_term_assocs {
    my ($self, $obj_id, $obj_type, $term_type, $term, $result) = @_;

    unless($term){
	return;
    }

    my $gene_term_assoc_id = $self->{'db'}->selectrow_array(
	q[
	    SELECT ontology_association_id 
	    FROM   gene_ontology_association
	    WHERE  object_id = ?
	    AND    object_table = ?
	    AND    term_accession = ?
	], {}, ($obj_id, $obj_type,$term)
    );

    if($gene_term_assoc_id){
	return;
    }

    $gene_term_assoc_id = _next_id($self, 'gene_ontology_association', 'ontology_association_id' );

    $self->{'db'}->do(
	q[
	     INSERT INTO  gene_ontology_association
	     (ontology_association_id, object_id, object_table, term_accession, term_type)
	     VALUES (?,?,?,?,?)
	], {}, ( $gene_term_assoc_id, $obj_id, $obj_type, $term, $term_type )
    );

}

sub get_dbxref_id {
    my ($self, $xref_name) = @_;

    my ($xref_id ) = $self->{'db'}->selectrow_array(
        q[
              SELECT dbxref_id FROM gene_dbxref
              WHERE dbxref_name = ?
             ],
        {}, ( $xref_name )
    );

    return $xref_id;
}

sub _load_gene_dbxref {

    my ($self, $table_name, $obj_id, $dbxref_id, $dbval ) = @_;

    my ($dbxref_to_obj_id) = $self->{'db'}->selectrow_array(
            q[
	     SELECT dbxref_to_object_id
	     FROM   gene_dbxref_to_object
	     WHERE  dbxref_id = ?
	     AND    table_name = ?
	     AND    record_id = ?
	     AND    dbxref_value = ?
      ], {}, ( $dbxref_id, $table_name, $obj_id, $dbval));

    unless($dbxref_to_obj_id){    # already in db

        $dbxref_to_obj_id = _next_id( $self, 'gene_dbxref_to_object', 'dbxref_to_object_id' );

        $self->{'db'}->do( q[
              INSERT INTO gene_dbxref_to_object 
	      (dbxref_to_object_id, table_name, record_id, dbxref_id, dbxref_value )
              VALUES (?,?,?,?,?)
	      ], {}, ( $dbxref_to_obj_id, $table_name, $obj_id, $dbxref_id, $dbval)
        );

    }

    return $dbxref_to_obj_id;
}


sub _find_or_create_species_by_taxonomy{
    my ( $self, $gr_taxon, $result ) = @_;
    
    unless ($gr_taxon) {
        push ( @{ $result->{'error'} }, "Unable to create species: no gramene taxononum defined.");
	return;
    }

    my ($species_id) = $self->{'db'}->selectrow_array(q[
	SELECT species_id
	FROM   gene_species
	WHERE  gramene_taxa_id = ?
    ], {}, ($gr_taxon));

    return $species_id if $species_id;
   
    # create species

    my $onto_db = $self->{'OntologyDB'}->db;
    my ($term_name,$common_name) = $onto_db->selectrow_array(q[
	SELECT term_name, synonym_name
	FROM   term
	LEFT JOIN term_synonym
	ON     term.term_id = term_synonym.term_id
	WHERE  term_accession = ?
    ], {}, ($gr_taxon));

    $common_name ||= '';
    my $genus = '';
    my $species = '';
    if($term_name){
	if($term_name = ~/(\D+)\s+(\D+)/){
	    $genus = $1;
	    $species = $2;
	}
    }


    $species_id = _next_id($self, 'gene_species', 'species_id' );
    my $sth = $self->{'db'}->prepare(q[
	INSERT INTO gene_species
	(species_id,gramene_taxa_id,common_name,genus,species)
	VALUES (?,?,?,?,?)
    ]);

    $sth->execute(($species_id,$gr_taxon,$common_name,$genus,$species));
    $sth->finish;

    return $species_id;

}

## gene update

# update field in gene_gene table
sub update_gene_simple_field {
    my ( $self, $field_name, $field_value, $gene_id, $edit_result ) = @_;

    eval {
        $self->{'db'}->do(
            qq[
                 UPDATE gene_gene
                    SET $field_name = ?
                  WHERE gene_id = ?
                 ], {}, ( $field_value, $gene_id )
        );

        $self->{'db'}->commit;
    };
    if ($@) {
        push (
            @{ $edit_result->{'error'} },
            "Unable to update gene $field_name: $@"
        );
        $self->{'db'}->rollback();
    }

}


sub add_gene_synonyms {

    my ( $self, $gene_id, $syn_val, $edit_result ) = @_;

    eval {
        my @synonyms = process_delimited_values($self, ',', $syn_val );
        foreach my $syn (@synonyms) {
	    _load_gene_synonym($self, $gene_id, $syn ) if $syn;
        }
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add gene synonym: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


sub delete_gene_synonym {
    my ( $self, $gene_id, $syn, $edit_result ) = @_;
    eval {
	    $self->{'db'}->do(
		q[
                    DELETE FROM gene_gene_synonym
                    WHERE gene_id =? 
                    AND BINARY( synonym_name ) = ?
	       ], {}, ( $gene_id, $syn )
	    );

	    $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to delete gene synonym: $@" );
        $self->{'db'}->rollback();
    }

}


sub delete_gene_dbxref {
    my ( $self, $gene_id, $dbx_id, $dbx_val) = @_;
    eval {
	$self->{'db'}->do(
	    q[
		DELETE FROM gene_dbxref_to_object
		 WHERE table_name = ?
		 AND   record_id = ?
		 AND   dbxref_id = ?
		 AND   dbxref_value = ?
	    ], {}, ('gene',$gene_id,$dbx_id,$dbx_val)
	);
        $self->{'db'}->commit;
    };
    if ($@) {
        $self->{'db'}->rollback();
    }

}

sub delete_gene_dbxrefs {
    my ( $self, $dbx_valsRef, $edit_result ) = @_;
    eval {
        foreach my $val ( @$dbx_valsRef ) {
            $self->{'db'}->do(
                q[
                    DELETE FROM gene_dbxref_to_object
                     WHERE dbxref_to_object_id = ?
		    ], {}, ($val)
            );
        }
        $self->{'db'}->commit;
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add gene dbxref: $@" );
        $self->{'db'}->rollback();
    }

}


sub add_gene_dbxrefs {
    my ( $self, $gene_id, $dbx_id, $dbx_vals, $edit_result ) = @_;

    eval {

        my @dbxref_vals = process_delimited_values($self, ',', $dbx_vals );
	foreach my $dbx_val (@dbxref_vals){
	    _load_gene_dbxref($self, 'gene', $gene_id, $dbx_id, $dbx_val );
	}

        $self->{'db'}->commit;
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add gene dbxref: $@" );
        $self->{'db'}->rollback();
    }
}

sub _next_id {
    my ($self, $table_name, $field_name ) = @_;
    my $id = $self->{'db'}->selectrow_array("select max($field_name) from $table_name");
    return $id + 1;
}


sub process_delimited_values {
    my ($self, $delimiter, $val ) = @_;
    my @vals_raw = split ( /$delimiter/, $val );
    my %seen;
    my @vals;
    foreach my $val (@vals_raw) {
        $val =~ s/^\s+|\s+$//g;
        unless ( $seen{$val}++ ) {
            push @vals, $val;
        }
    }

    return @vals;
}



sub delete_gene_ontology_associations {
    my ( $self, $gene_id, $terms, $edit_result ) = @_;
    eval {
        foreach my $term_acc (@$terms)
        {
            $self->{'db'}->do(
                q[
                    DELETE FROM gene_ontology_association
                    WHERE  object_table=?
                    AND    object_id =?
                    AND    term_accession = ?  
                  ],
                {}, ( 'gene', $gene_id, $term_acc )
            );

        }

        $self->{'db'}->commit;
    };
    if ($@) {
        push (
            @{ $edit_result->{'error'} },
            "Unable to delete ontology associaiton: $@"
        );
        $self->{'db'}->rollback();
    }

}



sub add_gene_ontology_associations {
    my ( $self, $gene_id, $term_type, $term_acc_string, $edit_result ) = @_;

    my @term_accs = process_delimited_values($self, ',', $term_acc_string );

    eval {
	foreach my $term_acc (@term_accs){
	    _load_term_assocs($self, $gene_id, 'gene', $term_type, $term_acc, $edit_result );
	}

        $self->{'db'}->commit;
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add ontology associaiton: $@");
        $self->{'db'}->rollback();
    }

}

sub add_dbxref_source {
    my ( $self, $dbx_name, $dbx_url ) = @_;
    eval {
	my $sth0 = $self->{'db'}->prepare(q[
	    SELECT dbxref_id FROM gene_dbxref
	    WHERE  dbxref_name = ?
	]);

	$sth0->execute(($dbx_name));
	my ($dbx_id) = $sth0->fetchrow_array;
	$sth0->finish;

	unless($dbx_id){
	    my $dbx_id = _next_id($self, 'gene_dbxref', 'dbxref_id' );
	    $self->{'db'}->do(
		q[
		    INSERT INTO gene_dbxref VALUES (?,?,?)
		    ], {}, ( $dbx_id, $dbx_name, $dbx_url )
	    );
	}

        $self->{'db'}->commit();
    };
    if ($@) {
	$self->{'db'}->rollback();
    }
}

sub update_dbxref_source {
    my ( $self, $dbx_id, $dbx_name, $dbx_url ) = @_;
    eval{
	my $sth = $self->{'db'}->prepare(q[
	    UPDATE gene_dbxref
	    SET    dbxref_name = ?,
	           url = ?
            WHERE  dbxref_id = ?
	]);
	$sth->execute(($dbx_name, $dbx_url, $dbx_id) ) or die $self->{'db'}->errstr;
	$sth->finish;
	$self->{'db'}->commit();
    };
    if($@){
	$self->{'db'}->rollback();
    }
}


sub update_species {
    my ( $self, $spe_id, $genus, $spe, $name, $taxid, $gr_taxid ) = @_;

    my @spe_vals = map { defined $_ ? $_ : '' } ($genus, $spe, $name, $taxid, $gr_taxid, $spe_id);

    eval {
	my $sth = $self->{'db'}->prepare(
	    qq[
		Update gene_species 
		set genus = ?,
		species = ?,
		common_name = ?, 
		ncbi_taxa_id = ?,
		gramene_taxa_id = ?
		where species_id = ?
	    ]);

	$sth->execute(@spe_vals ) or die $self->{'db'}->errstr;
	$sth->finish;

	$self->{'db'}->commit();
    };
    if ($@) {
	$self->{'db'}->rollback();
    }
}


sub add_species {
    my ( $self, $genus, $spe, $name, $taxid, $gr_taxid ) = @_;

    my @spe_vals = map { defined $_ ? $_ : '' } ($genus, $spe, $name, $taxid, $gr_taxid);

    eval{
	
	my $sth2 = $self->{'db'}->prepare("SELECT MAX(species_id) FROM gene_species");
	$sth2->execute;
	my ($sp_id) = $sth2->fetchrow_array();
	$sp_id++;
	@spe_vals = ($sp_id, @spe_vals);
	my $sth3 = $self->{'db'}->prepare(
	      "INSERT INTO gene_species 
	      (species_id, genus, species, common_name, ncbi_taxa_id,
	      gramene_taxa_id)
	      VALUES(?, ?, ?, ?, ?, ?)");
	$sth3->execute( @spe_vals);

	$sth2->finish;
	$sth3->finish;

	$self->{'db'}->commit();

    };
    if ($@) {
	$self->{'db'}->rollback();
    }
}


sub update_gene_type {
    my ( $self, $gene_type_id, $gene_type ) = @_;
    eval{
	my $sth = $self->{'db'}->prepare(q[
	    UPDATE gene_gene_type
	    SET    gene_type = ?
            WHERE  gene_type_id = ?
	]);
	$sth->execute(($gene_type, $gene_type_id) ) or die $self->{'db'}->errstr;
	$sth->finish;
	$self->{'db'}->commit();
    };
    if($@){
	$self->{'db'}->rollback();
    }
}


sub add_gene_type {
    my ( $self, $gene_type ) = @_;
    eval {
	my $sth0 = $self->{'db'}->prepare(q[
	    SELECT gene_type_id FROM gene_gene_type
	    WHERE  gene_type = ?
	]);

	$sth0->execute(($gene_type));
	my ($type_id) = $sth0->fetchrow_array;
	$sth0->finish;

	unless($type_id){
	    my $type_id = _next_id($self, 'gene_gene_type', 'gene_type_id' );
	    $self->{'db'}->do(
		q[
		    INSERT INTO gene_gene_type VALUES (?,?)
		    ], {}, ( $type_id, $gene_type )
	    );
	}

        $self->{'db'}->commit();
    };
    if ($@) {
	$self->{'db'}->rollback();
    }
}



sub add_map_position {

    my ( $self, $gene_id, $gene_acc, $map_acc, $map_start, $map_stop, $edit_result ) = @_;

    return unless ($map_acc && $map_start);

    eval {
	my ($map) = Gramene::CDBI::Markers::Map->search(
	    { cmap_map_accession => $map_acc }
	);

	if($map){
	    my $map_name = $map->map_name;
	    my $map_set = $map->map_set_id; # a scalar or object
	    my $map_set_name = $map_set->map_set_name;
	    my $unit = $map_set->distance_unit;
	    my $map_pos_id = _next_id($self,  'gene_map_position', 'map_position_id' );
	    my $feature_acc = $map_acc.'_'.$gene_acc;
	    my $sth = $self->{'db'}->prepare(q[
		INSERT INTO gene_map_position
		(map_position_id,
		 gene_id,
		 cmap_feature_accession,
		 cmap_map_set,
		 cmap_map_accession,
		 cmap_map_name,
		 start_position,
		 stop_position,
		 map_units
		)
		VALUES (?,?,?,?,?,?,?,?,?)
	    ]);
	    $sth->execute(($map_pos_id,
			    $gene_id,
			    $feature_acc,
			    $map_set_name,
			    $map_acc,
			    $map_name,
			    $map_start,
			    $map_stop,
			    $unit
			));


        }
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add map position: $@" );
        $self->{'db'}->rollback();
        return;
    }
}




sub update_map_position {

    my ( $self, $map_pos_id, $map_start, $map_stop, $edit_result ) = @_;


    eval {
	my $sth = $self->{'db'}->prepare(q[
		UPDATE gene_map_position
		SET    start_position = ?,
		       stop_position = ?
		WHERE  map_position_id = ?
	    ]);
	$sth->execute((
			$map_start,
			$map_stop,
			$map_pos_id
			));


        
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add map position: $@" );
        $self->{'db'}->rollback();
        return;
    }
}



sub delete_map_position {

    my ( $self, $map_pos_id, $edit_result ) = @_;


    eval {
	my $sth = $self->{'db'}->prepare(q[
		DELETE FROM gene_map_position
		WHERE  map_position_id = ?
	    ]);
	$sth->execute(( $map_pos_id));


        
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add map position: $@" );
        $self->{'db'}->rollback();
        return;
    }
}

# ----------------------
sub add_gene_interaction {

    my ( $self, $g1_obj_id, $g1_obj_class,$interaction_type_id,$g2_obj_id,$g2_obj_class,$evn_code,$comment,$dbx_id,$dbx_val, $edit_result ) = @_;

    eval{
	my ($interact_id) = $self->{'db'}->selectrow_array(q[
	    SELECT gene_interaction_id 
	    FROM   gene_gene_interaction
	    WHERE  gene1_id = ?
	    AND    gene1_object_class = ?
	    AND    gene_interaction_type_id = ?
	    AND    gene2_id = ?
	    AND    gene2_object_class = ?
	], {}, ($g1_obj_id, $g1_obj_class,$interaction_type_id,$g2_obj_id,$g2_obj_class));

	unless($interact_id){
	    $interact_id = _next_id($self,  'gene_gene_interaction', 'gene_interaction_id' );
	    $self->{'db'}->do(q[
		INSERT INTO gene_gene_interaction
		(gene_interaction_id,gene1_id,gene1_object_class,gene_interaction_type_id,gene2_id,gene2_object_class)
		VALUES (?,?,?,?,?,?)
	    ],{},
	    ($interact_id,$g1_obj_id, $g1_obj_class,$interaction_type_id,$g2_obj_id,$g2_obj_class)
	    );
	}

	
	&add_gene_interaction_evidence($self, $interact_id,$evn_code, $comment, $dbx_id, $dbx_val, $edit_result );

       
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add gene interaction: $@" );
        $self->{'db'}->rollback();
        return;
    }
    

}


sub update_gene_interaction {

    my ( $self, $interaction_id,$g1_obj_class,$interaction_type_id,$g2_obj_class,  $edit_result ) = @_;


    eval {
	my $sth = $self->{'db'}->prepare(q[
		UPDATE gene_gene_interaction
		SET    gene1_object_class = ?,
		       gene_interaction_type_id = ?,
		       gene2_object_class = ?
		WHERE  gene_interaction_id = ?
	    ]);
	$sth->execute((
			$g1_obj_class,
			$interaction_type_id,
			$g2_obj_class,
			$interaction_id
			));

        
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to update gene interaction: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


sub delete_gene_interaction {

    my ( $self, $interaction_id, $edit_result ) = @_;


    eval {

	my $evns = $self->{'db'}->selectall_hashref(q[
	    SELECT gene_interaction_evidence_id, dbxref_to_object_id
	    FROM   gene_gene_interaction_evidence
	    WHERE  gene_interaction_id = ?
	],'gene_interaction_evidence_id',{},($interaction_id));


	foreach my $evn_id (keys %$evns){
	    my $dbx_to_obj_id = $evns->{$evn_id}->{'dbxref_to_object_id'};
	    &delete_gene_interaction_evidence($self,$evn_id,$dbx_to_obj_id,$edit_result);
	}

	#$self->{'db'}->do(q[
	#    DELETE FROM gene_gene_interaction_evidence
	#    WHERE  gene_interaction_id = ?
	#],{},($interaction_id));

	$self->{'db'}->do(q[
	    DELETE FROM gene_gene_interaction
	    WHERE  gene_interaction_id = ?
	],{},($interaction_id));


        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to delete gene interaction: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


sub update_gene_interaction_evidence {

    my ( $self, $interaction_evn_id,$evn_code,$comment, $edit_result ) = @_;


    eval {
	my $sth = $self->{'db'}->prepare(q[
		UPDATE gene_gene_interaction_evidence
		SET    evidence_code = ?,
		       comment = ?
		WHERE  gene_interaction_evidence_id = ?
	    ]);
	$sth->execute((
			$evn_code,
			$comment,
			$interaction_evn_id
			));

        
        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to update gene interaction evidence: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


sub delete_gene_interaction_evidence {

    my ( $self, $interaction_evn_id,$dbxref_to_object_id, $edit_result ) = @_;


    eval {

	$self->{'db'}->do(q[
	    DELETE FROM gene_gene_interaction_evidence
	    WHERE  gene_interaction_evidence_id = ?
	],{},($interaction_evn_id));

	$self->{'db'}->do(q[
	    DELETE FROM gene_dbxref_to_object
	    WHERE  dbxref_to_object_id = ?
	],{},($dbxref_to_object_id));

        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to delete gene interaction evidence: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


sub add_gene_interaction_evidence {

    my ( $self, $interaction_id,$evn_code, $comment, $dbx_id, $dbx_val, $edit_result ) = @_;


    eval {
	my $sth = $self->{'db'}->prepare(q[
	    SELECT gene_interaction_evidence_id
	    FROM   gene_gene_interaction_evidence E,
		   gene_dbxref_to_object O,
		   gene_dbxref D
	    WHERE  E.dbxref_to_object_id = O.dbxref_to_object_id
	    AND    O.dbxref_id = D.dbxref_id
	    AND    E.gene_interaction_id = ?
	    AND    evidence_code = ?
	    AND    D.dbxref_id = ?
	    AND    O.dbxref_value = ?
	]);

	$sth->execute(($interaction_id,$evn_code,$dbx_id,$dbx_val));
	my ($evn_id) = $sth->fetchrow_array;

	if($evn_id){
	    return;
	}

	
        $evn_id = _next_id($self,  'gene_gene_interaction_evidence', 'gene_interaction_evidence_id' );

	my $dbx_to_obj_id = &_load_gene_dbxref($self, 'gene_interaction_evidence', $evn_id, $dbx_id, $dbx_val );

	my $sth_evn= $self->{'db'}->prepare(q[
	    INSERT INTO gene_gene_interaction_evidence
	    (gene_interaction_evidence_id,gene_interaction_id,evidence_code,dbxref_to_object_id, comment)
	    VALUES(?,?,?,?,?)
	]);

	$sth_evn->execute(($evn_id,$interaction_id,$evn_code,$dbx_to_obj_id,$comment));


        $self->{'db'}->commit();
    };
    if ($@) {
        push ( @{ $edit_result->{'error'} }, "Unable to add gene interaction evidence: $@" );
        $self->{'db'}->rollback();
        return;
    }
}


1;

