package Gramene::DB;

# $Id: DB.pm,v 1.7 2004/11/11 22:10:13 kclark Exp $

=head1 NAME

Gramene::DB - make a connection to a Gramene database

=head1 SYNOPSIS

  use Gramene::DB;
  my $dbh = Gramene::DB->new('markers');

=head1 DESCRIPTION

This module read information from Gramene::Config (itself an
object-oriented interface to the "gramene.conf" file) for a given
project (e.g., "markers," "mutants," "qtl") and returns a DBI 
database handle.  The argument to "new" must match a section definition
in the Gramene config which defines the following:

=over 4

=item * dsn

The DSN for connecting to the database

=item * db_user

The username for the database

=item * db_pass

The password for connecting (optional, e.g. for Oracle)

=back

=head1 METHODS

=cut

use strict;
use Carp 'croak';
use Gramene::Config;
use DBI;

use vars qw[ $VERSION @ISA ];
$VERSION = sprintf "%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/;

BEGIN { @ISA = qw[ DBI::db DBI ]; }

use constant DB_OPTIONS => {
    AutoCommit          => 1,
    RaiseError          => 1,
    FetchHashKeyName    => 'NAME_lc',
    LongReadLen         => 3000,
    LongTruncOk         => 1,
};

# ----------------------------------------------------
sub new {

=pod

=head2 new

  my $dbh = Gramene::DB->new('section');

This is the only method you can call.  Returns a DBI database handle.

=cut

    my $class   = shift;
    my $section = shift                     or croak 'No section name';
    my $gconfig = Gramene::Config->new      or croak 'No config file object';
    my $config  = $gconfig->get( $section ) or croak "Bad section '$section'";
    my $db      = DBI->connect(
        $config->{'db_dsn'}, 
        $config->{'db_user'}, 
        $config->{'db_pass'}, 
        DB_OPTIONS
    );
    return undef unless defined $db;
    return bless $db, $class;
}

# ----------------------------------------------------
sub DESTROY {

=pod

=head2 DESTROY

When the database handle is destroyed, it will automatically call the
DBI disconnect() method.

=cut

    my $self = shift;
    $self->disconnect;
    $self->SUPER::DESTROY;
}

1;

# ----------------------------------------------------

=pod

=head1 SEE ALSO

Gramene::Config, DBI.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
