package Gramene::Config;

# $Id: Config.pm,v 1.5 2007/04/23 16:01:54 kclark Exp $

use strict;
use Carp qw( croak );
use Class::Base;
use Config::General;
use Readonly;

use base 'Class::Base';

Readonly my $CONFIG_FILE => '/usr/local/gramene/conf/gramene.conf';

# ----------------------------------------------------------------
sub init {
    my ( $self, $config ) = @_;
    $self->filename( $config->{'filename'} || $ENV{'GrameneConfPath'} );
    return $self;
}

# ----------------------------------------------------------------
sub filename { 
    my $self = shift;

    if ( my $filename = shift ) {
        my $curfile = $self->{'filename'} || '';
        if ( $filename eq $curfile ) {
            return $filename; # do nothing
        }
        elsif ( -d $filename ) {
            croak("Cannot use directory '$filename' as input source");
        }
        elsif ( -f _ && -r _ ) {
            $self->{'filename'} = $filename;
            $self->{'config'}   = '';
        }
        else {
            croak(
                "Cannot use '$filename' as input source: ".
                "file does not exist or is not readable."
            );
        }
    }

    return $self->{'filename'} || '';
}

# ----------------------------------------------------------------
sub get {
    my ( $self, $option ) = @_;

    unless ( $self->{'config'} ) {
        my $filename = $self->{'filename'} || $CONFIG_FILE;
        my $conf     = Config::General->new( $filename ) or
                       croak("Error reading config file: '$filename'");
        my %config   = $conf->getall or
                       croak('No configuration options');

        $self->{'config'} = \%config;
    }

    if ( $option ) {
        my $value;

        if ( ref $self->{'config'} eq 'HASH' ) {
            $value = $self->{'config'}{ $option }
                or croak("No config section named '$option'");
        }

#        if ( $value && ref $value ) {
#            bless $value, 'Gramene::Config::Object';
#        }

        if ( $value ) {
            return wantarray && ref $value ? @$value : $value;
        }
        else {
            return wantarray ? () : '';
        }
    }
    else {
        my $c = $self->{'config'};
        bless $c, 'Gramene::Config::Object';
        return wantarray ? @{ $c } : $c;
    }
}

package Gramene::Config::Object;

use base 'Class::Base';

our $AUTOLOAD;

sub AUTOLOAD {
    my $self = shift;
    my $method = $AUTOLOAD;
    $method =~ s/.*:://;

    if ( exists $self->{ $method } ) {
        return wantarray
            ? ref $self->{ $method } eq 'ARRAY'
                ? @{ $self->{ $method } }
                : [ $self->{ $method } ]
            : $self->{ $method }
        ;
    }
    else {
        croak("No defined hash key for '$method'");
    }
}

1;

__END__

# ----------------------------------------------------------------

=pod

=head1 NAME

Gramene::Config - Read local configuration information

=head1 SYNOPSIS

  use Gramene::Config;

  my $config = Gramene::Config->new;
  my $db_info = $config->get('db_info');

=head1 DESCRIPTION

Certain configuration items are specific to the machine they are 
deployed on, and should, therefore, be available locally to the apps 
that need them without any code changing.  With this module, only the
one location of the configuration file needs to be changed, and then
all the calling modules can read the correct info.  

This module relies on Config::General, so perldoc that module for more
information on how the configuration options can be set.  It is hoped
that module-specific information (e.g., for QTL, mutants, etc.) will
be wrapped in specific tags (e.g., "<qtl>...</qtl>") to logically
group like elements.

=head1 METHODS

=head2 new

Constructor for object.  Takes an optional "filename" argument of the 
path to the configuration file.

  my $config   =  Gramene::Config->new(
      filename => '/path/to/config.conf'
  );

=head2 filename

Gets or sets the complete path to the configuration file.  If a file is
already opened, then the handle on it will be closed and a new one
opened on the new file.

  $config->filename('/path/to/config.conf');

=head2 get

Returns one or all options from the config file.

  my $foo = $config->get('foo');

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
