package Gramene::CDBI::Qtl;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'qtl';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        qtl
        qtl_ontology_association
        qtl_ontology_association_evidence
        qtl_trait
        qtl_trait_category
        qtl_trait_synonym
        species
        trait_ontology_association
        xref
        xref_type

    /
}

# ------------------------------------------------

package Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->table('qtl_ontology_association_evidence');


sub qtl_ontology_association {
    my $self = shift;
    return $self->qtl_ontology_association_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->columns( 
    All => qw/ 
        qtl_ontology_association_evidence_id
        qtl_ontology_association_id
        evidence_code
        gramene_reference_id 
    /, @extras
);

Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->columns( 
    Essential => qw/ 
        qtl_ontology_association_evidence_id
        qtl_ontology_association_id
        evidence_code
        gramene_reference_id 
    /
);

Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->columns( 
    Ordered => qw/ 
        qtl_ontology_association_evidence_id
        qtl_ontology_association_id
        evidence_code
        gramene_reference_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_ontology_association_evidence_id' => 'number',
        'qtl_ontology_association_id' => 'number',
        'evidence_code' => 'string',
        'gramene_reference_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence->has_a(
    qtl_ontology_association_id => 'Gramene::CDBI::Qtl::QtlOntologyAssociation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_ontology_association => [ 'qtl_ontology_association_id', 'qtl_ontology_association_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::QtlOntologyAssociation; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::QtlOntologyAssociation->table('qtl_ontology_association');


sub qtl {
    my $self = shift;
    return $self->qtl_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::QtlOntologyAssociation->columns( 
    All => qw/ 
        qtl_ontology_association_id
        qtl_id
        term_type
        term_accession 
    /, @extras
);

Gramene::CDBI::Qtl::QtlOntologyAssociation->columns( 
    Essential => qw/ 
        qtl_ontology_association_id
        qtl_id
        term_type
        term_accession 
    /
);

Gramene::CDBI::Qtl::QtlOntologyAssociation->columns( 
    Ordered => qw/ 
        qtl_ontology_association_id
        qtl_id
        term_type
        term_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_ontology_association_id' => 'number',
        'qtl_id' => 'number',
        'term_type' => 'string',
        'term_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::QtlOntologyAssociation->has_a(
    qtl_id => 'Gramene::CDBI::Qtl::Qtl'
);

Gramene::CDBI::Qtl::QtlOntologyAssociation->has_many( 
    'qtl_ontology_association_evidences',
    'Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence',
    'qtl_ontology_association_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl => [ 'qtl_id', 'qtl_id' ],
            qtl_ontology_association_evidence => [ 'qtl_ontology_association_id', 'qtl_ontology_association_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::Qtl; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::Qtl->table('qtl');


sub qtl_trait {
    my $self = shift;
    return $self->qtl_trait_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::Qtl->columns( 
    All => qw/ 
        qtl_id
        qtl_trait_id
        species_id
        qtl_accession_id
        published_symbol
        linkage_group
        chromosome
        cmap_map_accession
        start_position
        stop_position
        comments 
    /, @extras
);

Gramene::CDBI::Qtl::Qtl->columns( 
    Essential => qw/ 
        qtl_id
        qtl_trait_id
        species_id
        qtl_accession_id
        published_symbol
        linkage_group
        chromosome
        cmap_map_accession
        start_position
        stop_position 
    /
);

Gramene::CDBI::Qtl::Qtl->columns( 
    Ordered => qw/ 
        qtl_id
        qtl_trait_id
        species_id
        qtl_accession_id
        published_symbol
        linkage_group
        chromosome
        cmap_map_accession
        start_position
        stop_position
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->published_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_id' => 'number',
        'qtl_trait_id' => 'number',
        'species_id' => 'number',
        'qtl_accession_id' => 'string',
        'published_symbol' => 'string',
        'linkage_group' => 'string',
        'chromosome' => 'string',
        'cmap_map_accession' => 'string',
        'start_position' => 'number',
        'stop_position' => 'number',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::Qtl->has_a(
    qtl_trait_id => 'Gramene::CDBI::Qtl::QtlTrait'
);

Gramene::CDBI::Qtl::Qtl->has_a(
    species_id => 'Gramene::CDBI::Qtl::Species'
);

Gramene::CDBI::Qtl::Qtl->has_many( 
    'qtl_ontology_associations',
    'Gramene::CDBI::Qtl::QtlOntologyAssociation',
    'qtl_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_ontology_association => [ 'qtl_id', 'qtl_id' ],
            qtl_trait => [ 'qtl_trait_id', 'qtl_trait_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::QtlTraitSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::QtlTraitSynonym->table('qtl_trait_synonym');


sub qtl_trait {
    my $self = shift;
    return $self->qtl_trait_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::QtlTraitSynonym->columns( 
    All => qw/ 
        qtl_trait_synonym_id
        qtl_trait_id
        trait_synonym 
    /, @extras
);

Gramene::CDBI::Qtl::QtlTraitSynonym->columns( 
    Essential => qw/ 
        qtl_trait_synonym_id
        qtl_trait_id
        trait_synonym 
    /
);

Gramene::CDBI::Qtl::QtlTraitSynonym->columns( 
    Ordered => qw/ 
        qtl_trait_synonym_id
        qtl_trait_id
        trait_synonym 
    /
);

sub to_string {
    my $self = shift;
    return $self->trait_synonym || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_trait_synonym_id' => 'number',
        'qtl_trait_id' => 'number',
        'trait_synonym' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::QtlTraitSynonym->has_a(
    qtl_trait_id => 'Gramene::CDBI::Qtl::QtlTrait'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_trait => [ 'qtl_trait_id', 'qtl_trait_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::QtlTrait; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::QtlTrait->table('qtl_trait');


sub qtl_trait_category {
    my $self = shift;
    return $self->qtl_trait_category_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::QtlTrait->columns( 
    All => qw/ 
        qtl_trait_id
        qtl_trait_category_id
        trait_symbol
        trait_name
        to_accession 
    /, @extras
);

Gramene::CDBI::Qtl::QtlTrait->columns( 
    Essential => qw/ 
        qtl_trait_id
        qtl_trait_category_id
        trait_symbol
        trait_name
        to_accession 
    /
);

Gramene::CDBI::Qtl::QtlTrait->columns( 
    Ordered => qw/ 
        qtl_trait_id
        qtl_trait_category_id
        trait_symbol
        trait_name
        to_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->trait_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_trait_id' => 'number',
        'qtl_trait_category_id' => 'number',
        'trait_symbol' => 'string',
        'trait_name' => 'string',
        'to_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::QtlTrait->has_a(
    qtl_trait_category_id => 'Gramene::CDBI::Qtl::QtlTraitCategory'
);

Gramene::CDBI::Qtl::QtlTrait->has_many( 
    'qtls',
    'Gramene::CDBI::Qtl::Qtl',
    'qtl_trait_id' 
);

Gramene::CDBI::Qtl::QtlTrait->has_many( 
    'qtl_trait_synonyms',
    'Gramene::CDBI::Qtl::QtlTraitSynonym',
    'qtl_trait_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl => [ 'qtl_trait_id', 'qtl_trait_id' ],
            qtl_trait_category => [ 'qtl_trait_category_id', 'qtl_trait_category_id' ],
            qtl_trait_synonym => [ 'qtl_trait_id', 'qtl_trait_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::QtlTraitCategory; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::QtlTraitCategory->table('qtl_trait_category');

our @extras = ();
Gramene::CDBI::Qtl::QtlTraitCategory->columns( 
    All => qw/ 
        qtl_trait_category_id
        trait_category
        to_accession 
    /, @extras
);

Gramene::CDBI::Qtl::QtlTraitCategory->columns( 
    Essential => qw/ 
        qtl_trait_category_id
        trait_category
        to_accession 
    /
);

Gramene::CDBI::Qtl::QtlTraitCategory->columns( 
    Ordered => qw/ 
        qtl_trait_category_id
        trait_category
        to_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->trait_category || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_trait_category_id' => 'number',
        'trait_category' => 'string',
        'to_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::QtlTraitCategory->has_many( 
    'qtl_traits',
    'Gramene::CDBI::Qtl::QtlTrait',
    'qtl_trait_category_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_trait => [ 'qtl_trait_category_id', 'qtl_trait_category_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::Species; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::Species->table('species');

our @extras = ();
Gramene::CDBI::Qtl::Species->columns( 
    All => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        ncbi_taxonomy_id
        display_order 
    /, @extras
);

Gramene::CDBI::Qtl::Species->columns( 
    Essential => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        ncbi_taxonomy_id
        display_order 
    /
);

Gramene::CDBI::Qtl::Species->columns( 
    Ordered => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        ncbi_taxonomy_id
        display_order 
    /
);

sub to_string {
    my $self = shift;
    return $self->species || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'species_id' => 'number',
        'species' => 'string',
        'common_name' => 'string',
        'gramene_taxonomy_id' => 'string',
        'ncbi_taxonomy_id' => 'string',
        'display_order' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::Species->has_many( 
    'qtls',
    'Gramene::CDBI::Qtl::Qtl',
    'species_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::TraitOntologyAssociation; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::TraitOntologyAssociation->table('trait_ontology_association');

our @extras = ();
Gramene::CDBI::Qtl::TraitOntologyAssociation->columns( 
    All => qw/ 
        trait_ontology_association_id
        to_accession
        related_accession
        species_id 
    /, @extras
);

Gramene::CDBI::Qtl::TraitOntologyAssociation->columns( 
    Essential => qw/ 
        trait_ontology_association_id
        to_accession
        related_accession
        species_id 
    /
);

Gramene::CDBI::Qtl::TraitOntologyAssociation->columns( 
    Ordered => qw/ 
        trait_ontology_association_id
        to_accession
        related_accession
        species_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->to_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'trait_ontology_association_id' => 'number',
        'to_accession' => 'string',
        'related_accession' => 'string',
        'species_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::Xref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::Xref->table('xref');


sub xref_type {
    my $self = shift;
    return $self->xref_type_id;
}
    
our @extras = ();
Gramene::CDBI::Qtl::Xref->columns( 
    All => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /, @extras
);

Gramene::CDBI::Qtl::Xref->columns( 
    Essential => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value 
    /
);

Gramene::CDBI::Qtl::Xref->columns( 
    Ordered => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'table_name' => 'string',
        'record_id' => 'number',
        'xref_type_id' => 'number',
        'xref_value' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::Xref->has_a(
    xref_type_id => 'Gramene::CDBI::Qtl::XrefType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref_type => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Qtl::XrefType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Qtl ); };
# use base qw( Gramene::CDBI::Qtl );

Gramene::CDBI::Qtl::XrefType->table('xref_type');

our @extras = ();
Gramene::CDBI::Qtl::XrefType->columns( 
    All => qw/ 
        xref_type_id
        xref_type
        url_template 
    /, @extras
);

Gramene::CDBI::Qtl::XrefType->columns( 
    Essential => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

Gramene::CDBI::Qtl::XrefType->columns( 
    Ordered => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_type_id' => 'number',
        'xref_type' => 'string',
        'url_template' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Qtl::XrefType->has_many( 
    'xrefs',
    'Gramene::CDBI::Qtl::Xref',
    'xref_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Qtl - Gramene::CDBI-base classes for Qtl

=head1 VERSION

This documentation refers to version $Revision: 1.16 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Qtl;

  my @data   = Gramene::CDBI::Qtl::Xref->retrieve_all;
  my $object = Gramene::CDBI::Qtl::Xref->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Qtl 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'qtl'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Qtl::Qtl

Corresponds to the table "qtl."  

Stringifies (C<to_name>) to "published_symbol".

Fields:

=over 4

=item * qtl_id

=item * qtl_trait_id

=item * species_id

=item * qtl_accession_id

=item * published_symbol

=item * linkage_group

=item * chromosome

=item * cmap_map_accession

=item * start_position

=item * stop_position

=item * comments

=back


Has A:

=over 4

=item * qtl_trait => Gramene::CDBI::Qtl::QtlTrait

=item * species => Gramene::CDBI::Qtl::Species

=back


Has Many:

=over 4

=item * qtl_ontology_associations => Gramene::CDBI::Qtl::QtlOntologyAssociation

=back


=head2 Gramene::CDBI::Qtl::QtlOntologyAssociation

Corresponds to the table "qtl_ontology_association."  

Stringifies (C<to_name>) to "term_type".

Fields:

=over 4

=item * qtl_ontology_association_id

=item * qtl_id

=item * term_type

=item * term_accession

=back


Has A:

=over 4

=item * qtl => Gramene::CDBI::Qtl::Qtl

=back


Has Many:

=over 4

=item * qtl_ontology_association_evidences => Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence

=back


=head2 Gramene::CDBI::Qtl::QtlOntologyAssociationEvidence

Corresponds to the table "qtl_ontology_association_evidence."  

Stringifies (C<to_name>) to "evidence_code".

Fields:

=over 4

=item * qtl_ontology_association_evidence_id

=item * qtl_ontology_association_id

=item * evidence_code

=item * gramene_reference_id

=back


Has A:

=over 4

=item * qtl_ontology_association => Gramene::CDBI::Qtl::QtlOntologyAssociation

=back



=head2 Gramene::CDBI::Qtl::QtlTrait

Corresponds to the table "qtl_trait."  

Stringifies (C<to_name>) to "trait_symbol".

Fields:

=over 4

=item * qtl_trait_id

=item * qtl_trait_category_id

=item * trait_symbol

=item * trait_name

=item * to_accession

=back


Has A:

=over 4

=item * qtl_trait_category => Gramene::CDBI::Qtl::QtlTraitCategory

=back


Has Many:

=over 4

=item * qtls => Gramene::CDBI::Qtl::Qtl

=item * qtl_trait_synonyms => Gramene::CDBI::Qtl::QtlTraitSynonym

=back


=head2 Gramene::CDBI::Qtl::QtlTraitCategory

Corresponds to the table "qtl_trait_category."  

Stringifies (C<to_name>) to "trait_category".

Fields:

=over 4

=item * qtl_trait_category_id

=item * trait_category

=item * to_accession

=back



Has Many:

=over 4

=item * qtl_traits => Gramene::CDBI::Qtl::QtlTrait

=back


=head2 Gramene::CDBI::Qtl::QtlTraitSynonym

Corresponds to the table "qtl_trait_synonym."  

Stringifies (C<to_name>) to "trait_synonym".

Fields:

=over 4

=item * qtl_trait_synonym_id

=item * qtl_trait_id

=item * trait_synonym

=back


Has A:

=over 4

=item * qtl_trait => Gramene::CDBI::Qtl::QtlTrait

=back



=head2 Gramene::CDBI::Qtl::Species

Corresponds to the table "species."  

Stringifies (C<to_name>) to "species".

Fields:

=over 4

=item * species_id

=item * species

=item * common_name

=item * gramene_taxonomy_id

=item * ncbi_taxonomy_id

=item * display_order

=back



Has Many:

=over 4

=item * qtls => Gramene::CDBI::Qtl::Qtl

=back


=head2 Gramene::CDBI::Qtl::TraitOntologyAssociation

Corresponds to the table "trait_ontology_association."  

Stringifies (C<to_name>) to "to_accession".

Fields:

=over 4

=item * trait_ontology_association_id

=item * to_accession

=item * related_accession

=item * species_id

=back




=head2 Gramene::CDBI::Qtl::Xref

Corresponds to the table "xref."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * xref_id

=item * table_name

=item * record_id

=item * xref_type_id

=item * xref_value

=item * comments

=back


Has A:

=over 4

=item * xref_type => Gramene::CDBI::Qtl::XrefType

=back



=head2 Gramene::CDBI::Qtl::XrefType

Corresponds to the table "xref_type."  

Stringifies (C<to_name>) to "xref_type".

Fields:

=over 4

=item * xref_type_id

=item * xref_type

=item * url_template

=back



Has Many:

=over 4

=item * xrefs => Gramene::CDBI::Qtl::Xref

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon May  7 14:58:34 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
