package Gramene::CDBI::Protein;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'protein';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        association
        association_022707
        association_022807
        cultivar
        cultivar_synonym
        data_base
        dbxref
        evidence
        evidence_022707
        evidence_022807
        evidence_dbxref
        evidence_dbxref_022707
        evidence_dbxref_022807
        expression
        gene_product
        gene_product_comment
        gene_product_feature
        gene_product_feature_type
        gene_product_gene_name
        gene_product_helper
        gene_product_organelle
        gene_product_prosite
        gene_product_seq
        gene_product_synonym
        gene_product_tissue
        gene_product_to_cultivar
        gene_product_to_embl_accession
        gene_product_to_gi_number
        gene_product_to_keyword
        gene_product_to_pid
        keyword
        objectxref
        seq
        species

    /
}

# ------------------------------------------------

package Gramene::CDBI::Protein::EvidenceDbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::EvidenceDbxref->table('evidence_dbxref');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    

sub evidence {
    my $self = shift;
    return $self->evidence_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::EvidenceDbxref->columns( 
    All => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Protein::EvidenceDbxref->columns( 
    Essential => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

Gramene::CDBI::Protein::EvidenceDbxref->columns( 
    Ordered => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_dbxref_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_dbxref_id' => 'number',
        'evidence_id' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::EvidenceDbxref->has_a(
    dbxref_id => 'Gramene::CDBI::Protein::Dbxref'
);

Gramene::CDBI::Protein::EvidenceDbxref->has_a(
    evidence_id => 'Gramene::CDBI::Protein::Evidence'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
            evidence => [ 'evidence_id', 'evidence_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Evidence; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Evidence->table('evidence');


sub association {
    my $self = shift;
    return $self->association_id;
}
    

sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::Evidence->columns( 
    All => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /, @extras
);

Gramene::CDBI::Protein::Evidence->columns( 
    Essential => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

Gramene::CDBI::Protein::Evidence->columns( 
    Ordered => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_id' => 'number',
        'evidence_code' => 'string',
        'association_id' => 'number',
        'dbxref_id' => 'number',
        'seq_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Evidence->has_a(
    association_id => 'Gramene::CDBI::Protein::Association'
);

Gramene::CDBI::Protein::Evidence->has_a(
    dbxref_id => 'Gramene::CDBI::Protein::Dbxref'
);

Gramene::CDBI::Protein::Evidence->has_many( 
    'evidence_dbxrefs',
    'Gramene::CDBI::Protein::EvidenceDbxref',
    'evidence_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            association => [ 'association_id', 'association_id' ],
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
            evidence_dbxref => [ 'evidence_id', 'evidence_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Association; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Association->table('association');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::Association->columns( 
    All => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /, @extras
);

Gramene::CDBI::Protein::Association->columns( 
    Essential => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

Gramene::CDBI::Protein::Association->columns( 
    Ordered => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'association_id' => 'number',
        'term_accession' => 'string',
        'gene_product_id' => 'number',
        'term_type' => 'string',
        'is_not' => 'string',
        'role_group' => 'number',
        'assocdate' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Association->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);

Gramene::CDBI::Protein::Association->has_many( 
    'evidences',
    'Gramene::CDBI::Protein::Evidence',
    'association_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            evidence => [ 'association_id', 'association_id' ],
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Association022707; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Association022707->table('association_022707');

our @extras = ();
Gramene::CDBI::Protein::Association022707->columns( 
    All => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /, @extras
);

Gramene::CDBI::Protein::Association022707->columns( 
    Essential => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

Gramene::CDBI::Protein::Association022707->columns( 
    Ordered => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'association_id' => 'number',
        'term_accession' => 'string',
        'gene_product_id' => 'number',
        'term_type' => 'string',
        'is_not' => 'string',
        'role_group' => 'number',
        'assocdate' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Association022807; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Association022807->table('association_022807');

our @extras = ();
Gramene::CDBI::Protein::Association022807->columns( 
    All => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /, @extras
);

Gramene::CDBI::Protein::Association022807->columns( 
    Essential => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

Gramene::CDBI::Protein::Association022807->columns( 
    Ordered => qw/ 
        association_id
        term_accession
        gene_product_id
        term_type
        is_not
        role_group
        assocdate 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'association_id' => 'number',
        'term_accession' => 'string',
        'gene_product_id' => 'number',
        'term_type' => 'string',
        'is_not' => 'string',
        'role_group' => 'number',
        'assocdate' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::CultivarSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::CultivarSynonym->table('cultivar_synonym');


sub cultivar {
    my $self = shift;
    return $self->cultivar_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::CultivarSynonym->columns( 
    All => qw/ 
        cultivar_synonym_id
        cultivar_id
        cultivar_synonym 
    /, @extras
);

Gramene::CDBI::Protein::CultivarSynonym->columns( 
    Essential => qw/ 
        cultivar_synonym_id
        cultivar_id
        cultivar_synonym 
    /
);

Gramene::CDBI::Protein::CultivarSynonym->columns( 
    Ordered => qw/ 
        cultivar_synonym_id
        cultivar_id
        cultivar_synonym 
    /
);

sub to_string {
    my $self = shift;
    return $self->cultivar_synonym || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cultivar_synonym_id' => 'number',
        'cultivar_id' => 'number',
        'cultivar_synonym' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::CultivarSynonym->has_a(
    cultivar_id => 'Gramene::CDBI::Protein::Cultivar'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cultivar => [ 'cultivar_id', 'cultivar_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductToCultivar; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductToCultivar->table('gene_product_to_cultivar');


sub cultivar {
    my $self = shift;
    return $self->cultivar_id;
}
    

sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductToCultivar->columns( 
    All => qw/ 
        gene_product_to_cultivar_id
        gene_product_id
        cultivar_id 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductToCultivar->columns( 
    Essential => qw/ 
        gene_product_to_cultivar_id
        gene_product_id
        cultivar_id 
    /
);

Gramene::CDBI::Protein::GeneProductToCultivar->columns( 
    Ordered => qw/ 
        gene_product_to_cultivar_id
        gene_product_id
        cultivar_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_to_cultivar_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_to_cultivar_id' => 'number',
        'gene_product_id' => 'number',
        'cultivar_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductToCultivar->has_a(
    cultivar_id => 'Gramene::CDBI::Protein::Cultivar'
);

Gramene::CDBI::Protein::GeneProductToCultivar->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cultivar => [ 'cultivar_id', 'cultivar_id' ],
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Cultivar; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Cultivar->table('cultivar');


sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::Cultivar->columns( 
    All => qw/ 
        cultivar_id
        species_id
        cultivar_name 
    /, @extras
);

Gramene::CDBI::Protein::Cultivar->columns( 
    Essential => qw/ 
        cultivar_id
        species_id
        cultivar_name 
    /
);

Gramene::CDBI::Protein::Cultivar->columns( 
    Ordered => qw/ 
        cultivar_id
        species_id
        cultivar_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->cultivar_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cultivar_id' => 'number',
        'species_id' => 'number',
        'cultivar_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Cultivar->has_a(
    species_id => 'Gramene::CDBI::Protein::Species'
);

Gramene::CDBI::Protein::Cultivar->has_many( 
    'cultivar_synonyms',
    'Gramene::CDBI::Protein::CultivarSynonym',
    'cultivar_id' 
);

Gramene::CDBI::Protein::Cultivar->has_many( 
    'gene_product_to_cultivars',
    'Gramene::CDBI::Protein::GeneProductToCultivar',
    'cultivar_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cultivar_synonym => [ 'cultivar_id', 'cultivar_id' ],
            gene_product_to_cultivar => [ 'cultivar_id', 'cultivar_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::DataBase; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::DataBase->table('data_base');

our @extras = ();
Gramene::CDBI::Protein::DataBase->columns( 
    All => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /, @extras
);

Gramene::CDBI::Protein::DataBase->columns( 
    Essential => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

Gramene::CDBI::Protein::DataBase->columns( 
    Ordered => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'data_base_id' => 'number',
        'name' => 'string',
        'fullname' => 'string',
        'datatype' => 'string',
        'generic_url' => 'string',
        'url_syntax' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Expression; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Expression->table('expression');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::Expression->columns( 
    All => qw/ 
        expression_id
        gene_product_id
        expression_annotation 
    /, @extras
);

Gramene::CDBI::Protein::Expression->columns( 
    Essential => qw/ 
        expression_id
        gene_product_id
        expression_annotation 
    /
);

Gramene::CDBI::Protein::Expression->columns( 
    Ordered => qw/ 
        expression_id
        gene_product_id
        expression_annotation 
    /
);

sub to_string {
    my $self = shift;
    return $self->expression_annotation || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'expression_id' => 'number',
        'gene_product_id' => 'number',
        'expression_annotation' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Expression->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductComment; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductComment->table('gene_product_comment');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductComment->columns( 
    All => qw/ 
        gene_product_id
        public_comment
        curator_comment
        p_comment_curation_date
        c_comment_curation_date 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductComment->columns( 
    Essential => qw/ 
        gene_product_id
        public_comment
        curator_comment
        p_comment_curation_date
        c_comment_curation_date 
    /
);

Gramene::CDBI::Protein::GeneProductComment->columns( 
    Ordered => qw/ 
        gene_product_id
        public_comment
        curator_comment
        p_comment_curation_date
        c_comment_curation_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->public_comment || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_id' => 'number',
        'public_comment' => 'string',
        'curator_comment' => 'string',
        'p_comment_curation_date' => 'string',
        'c_comment_curation_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductComment->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductFeature->table('gene_product_feature');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    

sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    

sub feature_type {
    my $self = shift;
    return $self->feature_type_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductFeature->columns( 
    All => qw/ 
        gene_product_feature_id
        gene_product_id
        feature_type_id
        from_position
        to_position
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductFeature->columns( 
    Essential => qw/ 
        gene_product_feature_id
        gene_product_id
        feature_type_id
        from_position
        to_position
        dbxref_id 
    /
);

Gramene::CDBI::Protein::GeneProductFeature->columns( 
    Ordered => qw/ 
        gene_product_feature_id
        gene_product_id
        feature_type_id
        from_position
        to_position
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->from_position || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_feature_id' => 'number',
        'gene_product_id' => 'number',
        'feature_type_id' => 'number',
        'from_position' => 'number',
        'to_position' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductFeature->has_a(
    dbxref_id => 'Gramene::CDBI::Protein::Dbxref'
);

Gramene::CDBI::Protein::GeneProductFeature->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);

Gramene::CDBI::Protein::GeneProductFeature->has_a(
    feature_type_id => 'Gramene::CDBI::Protein::GeneProductFeatureType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_feature_type => [ 'feature_type_id', 'feature_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductGeneName; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductGeneName->table('gene_product_gene_name');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductGeneName->columns( 
    All => qw/ 
        gene_product_gene_name_id
        gene_product_id
        gene_name 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductGeneName->columns( 
    Essential => qw/ 
        gene_product_gene_name_id
        gene_product_id
        gene_name 
    /
);

Gramene::CDBI::Protein::GeneProductGeneName->columns( 
    Ordered => qw/ 
        gene_product_gene_name_id
        gene_product_id
        gene_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_gene_name_id' => 'number',
        'gene_product_id' => 'number',
        'gene_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductGeneName->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductHelper->table('gene_product_helper');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductHelper->columns( 
    All => qw/ 
        gene_product_id
        swissprot_acc
        swissprot_id
        gene_product_name
        gene_product_gene_name
        organism
        organism_rank
        non_IEA_codes
        IEA_codes
        evidence_codes
        is_obsolete 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductHelper->columns( 
    Essential => qw/ 
        gene_product_id
        swissprot_acc
        swissprot_id
        gene_product_name
        gene_product_gene_name
        organism
        organism_rank
        non_IEA_codes
        IEA_codes
        evidence_codes
        is_obsolete 
    /
);

Gramene::CDBI::Protein::GeneProductHelper->columns( 
    Ordered => qw/ 
        gene_product_id
        swissprot_acc
        swissprot_id
        gene_product_name
        gene_product_gene_name
        organism
        organism_rank
        non_IEA_codes
        IEA_codes
        evidence_codes
        is_obsolete 
    /
);

sub to_string {
    my $self = shift;
    return $self->swissprot_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_id' => 'number',
        'swissprot_acc' => 'string',
        'swissprot_id' => 'string',
        'gene_product_name' => 'string',
        'gene_product_gene_name' => 'string',
        'organism' => 'string',
        'organism_rank' => 'string',
        'non_IEA_codes' => 'number',
        'IEA_codes' => 'number',
        'evidence_codes' => 'string',
        'is_obsolete' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductHelper->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductOrganelle; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductOrganelle->table('gene_product_organelle');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductOrganelle->columns( 
    All => qw/ 
        gene_product_organelle_id
        gene_product_id
        organelle 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductOrganelle->columns( 
    Essential => qw/ 
        gene_product_organelle_id
        gene_product_id
        organelle 
    /
);

Gramene::CDBI::Protein::GeneProductOrganelle->columns( 
    Ordered => qw/ 
        gene_product_organelle_id
        gene_product_id
        organelle 
    /
);

sub to_string {
    my $self = shift;
    return $self->organelle || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_organelle_id' => 'number',
        'gene_product_id' => 'number',
        'organelle' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductOrganelle->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductProsite; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductProsite->table('gene_product_prosite');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductProsite->columns( 
    All => qw/ 
        gene_product_prosite_id
        gene_product_id
        prosite_id
        prosite_desc
        prosite_sequence 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductProsite->columns( 
    Essential => qw/ 
        gene_product_prosite_id
        gene_product_id
        prosite_id
        prosite_desc
        prosite_sequence 
    /
);

Gramene::CDBI::Protein::GeneProductProsite->columns( 
    Ordered => qw/ 
        gene_product_prosite_id
        gene_product_id
        prosite_id
        prosite_desc
        prosite_sequence 
    /
);

sub to_string {
    my $self = shift;
    return $self->prosite_desc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_prosite_id' => 'number',
        'gene_product_id' => 'number',
        'prosite_id' => 'string',
        'prosite_desc' => 'string',
        'prosite_sequence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductProsite->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductSeq; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductSeq->table('gene_product_seq');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    

sub seq {
    my $self = shift;
    return $self->seq_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductSeq->columns( 
    All => qw/ 
        gene_product_seq_id
        gene_product_id
        seq_id
        is_primary_seq 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductSeq->columns( 
    Essential => qw/ 
        gene_product_seq_id
        gene_product_id
        seq_id
        is_primary_seq 
    /
);

Gramene::CDBI::Protein::GeneProductSeq->columns( 
    Ordered => qw/ 
        gene_product_seq_id
        gene_product_id
        seq_id
        is_primary_seq 
    /
);

sub to_string {
    my $self = shift;
    return $self->is_primary_seq || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_seq_id' => 'number',
        'gene_product_id' => 'number',
        'seq_id' => 'number',
        'is_primary_seq' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductSeq->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);

Gramene::CDBI::Protein::GeneProductSeq->has_a(
    seq_id => 'Gramene::CDBI::Protein::Seq'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
            seq => [ 'seq_id', 'id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductSynonym->table('gene_product_synonym');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductSynonym->columns( 
    All => qw/ 
        gene_product_synonym_id
        gene_product_id
        gene_product_synonym_symbol 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductSynonym->columns( 
    Essential => qw/ 
        gene_product_synonym_id
        gene_product_id
        gene_product_synonym_symbol 
    /
);

Gramene::CDBI::Protein::GeneProductSynonym->columns( 
    Ordered => qw/ 
        gene_product_synonym_id
        gene_product_id
        gene_product_synonym_symbol 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_synonym_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_synonym_id' => 'number',
        'gene_product_id' => 'number',
        'gene_product_synonym_symbol' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductSynonym->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductTissue; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductTissue->table('gene_product_tissue');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductTissue->columns( 
    All => qw/ 
        gene_product_tissue_id
        gene_product_id
        tissue 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductTissue->columns( 
    Essential => qw/ 
        gene_product_tissue_id
        gene_product_id
        tissue 
    /
);

Gramene::CDBI::Protein::GeneProductTissue->columns( 
    Ordered => qw/ 
        gene_product_tissue_id
        gene_product_id
        tissue 
    /
);

sub to_string {
    my $self = shift;
    return $self->tissue || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_tissue_id' => 'number',
        'gene_product_id' => 'number',
        'tissue' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductTissue->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductToEmblAccession; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductToEmblAccession->table('gene_product_to_embl_accession');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductToEmblAccession->columns( 
    All => qw/ 
        gene_product_to_embl_accession_id
        gene_product_id
        embl_accession 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductToEmblAccession->columns( 
    Essential => qw/ 
        gene_product_to_embl_accession_id
        gene_product_id
        embl_accession 
    /
);

Gramene::CDBI::Protein::GeneProductToEmblAccession->columns( 
    Ordered => qw/ 
        gene_product_to_embl_accession_id
        gene_product_id
        embl_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_to_embl_accession_id' => 'number',
        'gene_product_id' => 'number',
        'embl_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductToEmblAccession->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductToGiNumber; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductToGiNumber->table('gene_product_to_gi_number');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductToGiNumber->columns( 
    All => qw/ 
        gene_product_to_gi_number_id
        gene_product_id
        gi_number 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductToGiNumber->columns( 
    Essential => qw/ 
        gene_product_to_gi_number_id
        gene_product_id
        gi_number 
    /
);

Gramene::CDBI::Protein::GeneProductToGiNumber->columns( 
    Ordered => qw/ 
        gene_product_to_gi_number_id
        gene_product_id
        gi_number 
    /
);

sub to_string {
    my $self = shift;
    return $self->gi_number || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_to_gi_number_id' => 'number',
        'gene_product_id' => 'number',
        'gi_number' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductToGiNumber->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductToKeyword; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductToKeyword->table('gene_product_to_keyword');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    

sub keyword {
    my $self = shift;
    return $self->keyword_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductToKeyword->columns( 
    All => qw/ 
        gene_product_to_keyword_id
        gene_product_id
        keyword_id 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductToKeyword->columns( 
    Essential => qw/ 
        gene_product_to_keyword_id
        gene_product_id
        keyword_id 
    /
);

Gramene::CDBI::Protein::GeneProductToKeyword->columns( 
    Ordered => qw/ 
        gene_product_to_keyword_id
        gene_product_id
        keyword_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_to_keyword_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_to_keyword_id' => 'number',
        'gene_product_id' => 'number',
        'keyword_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductToKeyword->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);

Gramene::CDBI::Protein::GeneProductToKeyword->has_a(
    keyword_id => 'Gramene::CDBI::Protein::Keyword'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
            keyword => [ 'keyword_id', 'keyword_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductToPid; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductToPid->table('gene_product_to_pid');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProductToPid->columns( 
    All => qw/ 
        gene_product_to_pid_id
        gene_product_id
        pid 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductToPid->columns( 
    Essential => qw/ 
        gene_product_to_pid_id
        gene_product_id
        pid 
    /
);

Gramene::CDBI::Protein::GeneProductToPid->columns( 
    Ordered => qw/ 
        gene_product_to_pid_id
        gene_product_id
        pid 
    /
);

sub to_string {
    my $self = shift;
    return $self->pid || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_to_pid_id' => 'number',
        'gene_product_id' => 'number',
        'pid' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductToPid->has_a(
    gene_product_id => 'Gramene::CDBI::Protein::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProduct; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProduct->table('gene_product');


sub organism_dbxref {
    my $self = shift;
    return $self->organism_dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::GeneProduct->columns( 
    All => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /, @extras
);

Gramene::CDBI::Protein::GeneProduct->columns( 
    Essential => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /
);

Gramene::CDBI::Protein::GeneProduct->columns( 
    Ordered => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_id' => 'number',
        'gene_product_symbol' => 'string',
        'organism_dbxref_id' => 'number',
        'species_id' => 'number',
        'gene_product_full_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProduct->has_a(
    organism_dbxref_id => 'Gramene::CDBI::Protein::Dbxref'
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'associations',
    'Gramene::CDBI::Protein::Association',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'expressions',
    'Gramene::CDBI::Protein::Expression',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_comments',
    'Gramene::CDBI::Protein::GeneProductComment',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_features',
    'Gramene::CDBI::Protein::GeneProductFeature',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_gene_names',
    'Gramene::CDBI::Protein::GeneProductGeneName',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_helpers',
    'Gramene::CDBI::Protein::GeneProductHelper',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_organelles',
    'Gramene::CDBI::Protein::GeneProductOrganelle',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_prosites',
    'Gramene::CDBI::Protein::GeneProductProsite',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_seqs',
    'Gramene::CDBI::Protein::GeneProductSeq',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_synonyms',
    'Gramene::CDBI::Protein::GeneProductSynonym',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_tissues',
    'Gramene::CDBI::Protein::GeneProductTissue',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_to_cultivars',
    'Gramene::CDBI::Protein::GeneProductToCultivar',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_to_embl_accessions',
    'Gramene::CDBI::Protein::GeneProductToEmblAccession',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_to_gi_numbers',
    'Gramene::CDBI::Protein::GeneProductToGiNumber',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_to_keywords',
    'Gramene::CDBI::Protein::GeneProductToKeyword',
    'gene_product_id' 
);

Gramene::CDBI::Protein::GeneProduct->has_many( 
    'gene_product_to_pids',
    'Gramene::CDBI::Protein::GeneProductToPid',
    'gene_product_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            association => [ 'gene_product_id', 'gene_product_id' ],
            dbxref => [ 'organism_dbxref_id', 'dbxref_id' ],
            expression => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_comment => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_feature => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_gene_name => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_helper => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_organelle => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_prosite => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_seq => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_synonym => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_tissue => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_to_cultivar => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_to_embl_accession => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_to_gi_number => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_to_keyword => [ 'gene_product_id', 'gene_product_id' ],
            gene_product_to_pid => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Objectxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Objectxref->table('objectxref');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Protein::Objectxref->columns( 
    All => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Protein::Objectxref->columns( 
    Essential => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /
);

Gramene::CDBI::Protein::Objectxref->columns( 
    Ordered => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'objectxref_id' => 'number',
        'table_name' => 'string',
        'row_id' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Objectxref->has_a(
    dbxref_id => 'Gramene::CDBI::Protein::Dbxref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Dbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Dbxref->table('dbxref');

our @extras = ();
Gramene::CDBI::Protein::Dbxref->columns( 
    All => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /, @extras
);

Gramene::CDBI::Protein::Dbxref->columns( 
    Essential => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

Gramene::CDBI::Protein::Dbxref->columns( 
    Ordered => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_key || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dbxref_id' => 'number',
        'xref_key' => 'string',
        'xref_keytype' => 'string',
        'xref_dbname' => 'string',
        'xref_desc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Dbxref->has_many( 
    'evidences',
    'Gramene::CDBI::Protein::Evidence',
    'dbxref_id' 
);

Gramene::CDBI::Protein::Dbxref->has_many( 
    'evidence_dbxrefs',
    'Gramene::CDBI::Protein::EvidenceDbxref',
    'dbxref_id' 
);

Gramene::CDBI::Protein::Dbxref->has_many( 
    'gene_products',
    'Gramene::CDBI::Protein::GeneProduct',
    'organism_dbxref_id' 
);

Gramene::CDBI::Protein::Dbxref->has_many( 
    'gene_product_features',
    'Gramene::CDBI::Protein::GeneProductFeature',
    'dbxref_id' 
);

Gramene::CDBI::Protein::Dbxref->has_many( 
    'objectxrefs',
    'Gramene::CDBI::Protein::Objectxref',
    'dbxref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            evidence => [ 'dbxref_id', 'dbxref_id' ],
            evidence_dbxref => [ 'dbxref_id', 'dbxref_id' ],
            gene_product => [ 'dbxref_id', 'organism_dbxref_id' ],
            gene_product_feature => [ 'dbxref_id', 'dbxref_id' ],
            objectxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Evidence022707; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Evidence022707->table('evidence_022707');

our @extras = ();
Gramene::CDBI::Protein::Evidence022707->columns( 
    All => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /, @extras
);

Gramene::CDBI::Protein::Evidence022707->columns( 
    Essential => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

Gramene::CDBI::Protein::Evidence022707->columns( 
    Ordered => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_id' => 'number',
        'evidence_code' => 'string',
        'association_id' => 'number',
        'dbxref_id' => 'number',
        'seq_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Evidence022807; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Evidence022807->table('evidence_022807');

our @extras = ();
Gramene::CDBI::Protein::Evidence022807->columns( 
    All => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /, @extras
);

Gramene::CDBI::Protein::Evidence022807->columns( 
    Essential => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

Gramene::CDBI::Protein::Evidence022807->columns( 
    Ordered => qw/ 
        evidence_id
        evidence_code
        association_id
        dbxref_id
        seq_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_id' => 'number',
        'evidence_code' => 'string',
        'association_id' => 'number',
        'dbxref_id' => 'number',
        'seq_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::EvidenceDbxref022707; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::EvidenceDbxref022707->table('evidence_dbxref_022707');

our @extras = ();
Gramene::CDBI::Protein::EvidenceDbxref022707->columns( 
    All => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Protein::EvidenceDbxref022707->columns( 
    Essential => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

Gramene::CDBI::Protein::EvidenceDbxref022707->columns( 
    Ordered => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_dbxref_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_dbxref_id' => 'number',
        'evidence_id' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::EvidenceDbxref022807; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::EvidenceDbxref022807->table('evidence_dbxref_022807');

our @extras = ();
Gramene::CDBI::Protein::EvidenceDbxref022807->columns( 
    All => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Protein::EvidenceDbxref022807->columns( 
    Essential => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

Gramene::CDBI::Protein::EvidenceDbxref022807->columns( 
    Ordered => qw/ 
        evidence_dbxref_id
        evidence_id
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_dbxref_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_dbxref_id' => 'number',
        'evidence_id' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::GeneProductFeatureType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::GeneProductFeatureType->table('gene_product_feature_type');

our @extras = ();
Gramene::CDBI::Protein::GeneProductFeatureType->columns( 
    All => qw/ 
        feature_type_id
        feature_type 
    /, @extras
);

Gramene::CDBI::Protein::GeneProductFeatureType->columns( 
    Essential => qw/ 
        feature_type_id
        feature_type 
    /
);

Gramene::CDBI::Protein::GeneProductFeatureType->columns( 
    Ordered => qw/ 
        feature_type_id
        feature_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->feature_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'feature_type_id' => 'number',
        'feature_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::GeneProductFeatureType->has_many( 
    'gene_product_features',
    'Gramene::CDBI::Protein::GeneProductFeature',
    'feature_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product_feature => [ 'feature_type_id', 'feature_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Keyword; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Keyword->table('keyword');

our @extras = ();
Gramene::CDBI::Protein::Keyword->columns( 
    All => qw/ 
        keyword_id
        keyword 
    /, @extras
);

Gramene::CDBI::Protein::Keyword->columns( 
    Essential => qw/ 
        keyword_id
        keyword 
    /
);

Gramene::CDBI::Protein::Keyword->columns( 
    Ordered => qw/ 
        keyword_id
        keyword 
    /
);

sub to_string {
    my $self = shift;
    return $self->keyword || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'keyword_id' => 'number',
        'keyword' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Keyword->has_many( 
    'gene_product_to_keywords',
    'Gramene::CDBI::Protein::GeneProductToKeyword',
    'keyword_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product_to_keyword => [ 'keyword_id', 'keyword_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Seq; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Seq->table('seq');

our @extras = ();
push @extras, Class::DBI::Column->new( id => { 
    accessor => 'id_field',
    mutator  => 'id_field' 
}); 

Gramene::CDBI::Protein::Seq->columns( 
    All => qw/ 
        id
        display_id
        description
        seq
        seq_len
        md5checksum
        moltype
        timestamp 
    /, @extras
);

Gramene::CDBI::Protein::Seq->columns( 
    Essential => qw/ 
        id
        display_id
        description
        seq
        seq_len
        md5checksum
        moltype
        timestamp 
    /
);

Gramene::CDBI::Protein::Seq->columns( 
    Ordered => qw/ 
        id
        display_id
        description
        seq
        seq_len
        md5checksum
        moltype
        timestamp 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'id' => 'number',
        'display_id' => 'string',
        'description' => 'string',
        'seq' => 'string',
        'seq_len' => 'number',
        'md5checksum' => 'string',
        'moltype' => 'string',
        'timestamp' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Seq->has_many( 
    'gene_product_seqs',
    'Gramene::CDBI::Protein::GeneProductSeq',
    'seq_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product_seq => [ 'id', 'seq_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Protein::Species; {

BEGIN { our @ISA = qw( Gramene::CDBI::Protein ); };
# use base qw( Gramene::CDBI::Protein );

Gramene::CDBI::Protein::Species->table('species');

our @extras = ();
Gramene::CDBI::Protein::Species->columns( 
    All => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /, @extras
);

Gramene::CDBI::Protein::Species->columns( 
    Essential => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /
);

Gramene::CDBI::Protein::Species->columns( 
    Ordered => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /
);

sub to_string {
    my $self = shift;
    return $self->common_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'species_id' => 'number',
        'ncbi_taxa_id' => 'number',
        'common_name' => 'string',
        'lineage_string' => 'string',
        'genus' => 'string',
        'species' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Protein::Species->has_many( 
    'cultivars',
    'Gramene::CDBI::Protein::Cultivar',
    'species_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cultivar => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Protein - Gramene::CDBI-base classes for Protein

=head1 VERSION

This documentation refers to version $Revision: 1.8 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Protein;

  my @data   = Gramene::CDBI::Protein::Cultivar->retrieve_all;
  my $object = Gramene::CDBI::Protein::Cultivar->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Protein 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'protein'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Protein::Association

Corresponds to the table "association."  

Stringifies (C<to_name>) to "term_accession".

Fields:

=over 4

=item * association_id

=item * term_accession

=item * gene_product_id

=item * term_type

=item * is_not

=item * role_group

=item * assocdate

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back


Has Many:

=over 4

=item * evidences => Gramene::CDBI::Protein::Evidence

=back


=head2 Gramene::CDBI::Protein::Association022707

Corresponds to the table "association_022707."  

Stringifies (C<to_name>) to "term_accession".

Fields:

=over 4

=item * association_id

=item * term_accession

=item * gene_product_id

=item * term_type

=item * is_not

=item * role_group

=item * assocdate

=back




=head2 Gramene::CDBI::Protein::Association022807

Corresponds to the table "association_022807."  

Stringifies (C<to_name>) to "term_accession".

Fields:

=over 4

=item * association_id

=item * term_accession

=item * gene_product_id

=item * term_type

=item * is_not

=item * role_group

=item * assocdate

=back




=head2 Gramene::CDBI::Protein::Cultivar

Corresponds to the table "cultivar."  

Stringifies (C<to_name>) to "cultivar_name".

Fields:

=over 4

=item * cultivar_id

=item * species_id

=item * cultivar_name

=back


Has A:

=over 4

=item * species => Gramene::CDBI::Protein::Species

=back


Has Many:

=over 4

=item * cultivar_synonyms => Gramene::CDBI::Protein::CultivarSynonym

=item * gene_product_to_cultivars => Gramene::CDBI::Protein::GeneProductToCultivar

=back


=head2 Gramene::CDBI::Protein::CultivarSynonym

Corresponds to the table "cultivar_synonym."  

Stringifies (C<to_name>) to "cultivar_synonym".

Fields:

=over 4

=item * cultivar_synonym_id

=item * cultivar_id

=item * cultivar_synonym

=back


Has A:

=over 4

=item * cultivar => Gramene::CDBI::Protein::Cultivar

=back



=head2 Gramene::CDBI::Protein::DataBase

Corresponds to the table "data_base."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * data_base_id

=item * name

=item * fullname

=item * datatype

=item * generic_url

=item * url_syntax

=back




=head2 Gramene::CDBI::Protein::Dbxref

Corresponds to the table "dbxref."  

Stringifies (C<to_name>) to "xref_key".

Fields:

=over 4

=item * dbxref_id

=item * xref_key

=item * xref_keytype

=item * xref_dbname

=item * xref_desc

=back



Has Many:

=over 4

=item * evidences => Gramene::CDBI::Protein::Evidence

=item * evidence_dbxrefs => Gramene::CDBI::Protein::EvidenceDbxref

=item * gene_products => Gramene::CDBI::Protein::GeneProduct

=item * gene_product_features => Gramene::CDBI::Protein::GeneProductFeature

=item * objectxrefs => Gramene::CDBI::Protein::Objectxref

=back


=head2 Gramene::CDBI::Protein::Evidence

Corresponds to the table "evidence."  

Stringifies (C<to_name>) to "evidence_code".

Fields:

=over 4

=item * evidence_id

=item * evidence_code

=item * association_id

=item * dbxref_id

=item * seq_acc

=back


Has A:

=over 4

=item * association => Gramene::CDBI::Protein::Association

=item * dbxref => Gramene::CDBI::Protein::Dbxref

=back


Has Many:

=over 4

=item * evidence_dbxrefs => Gramene::CDBI::Protein::EvidenceDbxref

=back


=head2 Gramene::CDBI::Protein::Evidence022707

Corresponds to the table "evidence_022707."  

Stringifies (C<to_name>) to "evidence_code".

Fields:

=over 4

=item * evidence_id

=item * evidence_code

=item * association_id

=item * dbxref_id

=item * seq_acc

=back




=head2 Gramene::CDBI::Protein::Evidence022807

Corresponds to the table "evidence_022807."  

Stringifies (C<to_name>) to "evidence_code".

Fields:

=over 4

=item * evidence_id

=item * evidence_code

=item * association_id

=item * dbxref_id

=item * seq_acc

=back




=head2 Gramene::CDBI::Protein::EvidenceDbxref

Corresponds to the table "evidence_dbxref."  

Stringifies (C<to_name>) to "evidence_dbxref_id".

Fields:

=over 4

=item * evidence_dbxref_id

=item * evidence_id

=item * dbxref_id

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Protein::Dbxref

=item * evidence => Gramene::CDBI::Protein::Evidence

=back



=head2 Gramene::CDBI::Protein::EvidenceDbxref022707

Corresponds to the table "evidence_dbxref_022707."  

Stringifies (C<to_name>) to "evidence_dbxref_id".

Fields:

=over 4

=item * evidence_dbxref_id

=item * evidence_id

=item * dbxref_id

=back




=head2 Gramene::CDBI::Protein::EvidenceDbxref022807

Corresponds to the table "evidence_dbxref_022807."  

Stringifies (C<to_name>) to "evidence_dbxref_id".

Fields:

=over 4

=item * evidence_dbxref_id

=item * evidence_id

=item * dbxref_id

=back




=head2 Gramene::CDBI::Protein::Expression

Corresponds to the table "expression."  

Stringifies (C<to_name>) to "expression_annotation".

Fields:

=over 4

=item * expression_id

=item * gene_product_id

=item * expression_annotation

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProduct

Corresponds to the table "gene_product."  

Stringifies (C<to_name>) to "gene_product_symbol".

Fields:

=over 4

=item * gene_product_id

=item * gene_product_symbol

=item * organism_dbxref_id

=item * species_id

=item * gene_product_full_name

=back


Has A:

=over 4

=item * organism_dbxref => Gramene::CDBI::Protein::Dbxref

=back


Has Many:

=over 4

=item * associations => Gramene::CDBI::Protein::Association

=item * expressions => Gramene::CDBI::Protein::Expression

=item * gene_product_comments => Gramene::CDBI::Protein::GeneProductComment

=item * gene_product_features => Gramene::CDBI::Protein::GeneProductFeature

=item * gene_product_gene_names => Gramene::CDBI::Protein::GeneProductGeneName

=item * gene_product_helpers => Gramene::CDBI::Protein::GeneProductHelper

=item * gene_product_organelles => Gramene::CDBI::Protein::GeneProductOrganelle

=item * gene_product_prosites => Gramene::CDBI::Protein::GeneProductProsite

=item * gene_product_seqs => Gramene::CDBI::Protein::GeneProductSeq

=item * gene_product_synonyms => Gramene::CDBI::Protein::GeneProductSynonym

=item * gene_product_tissues => Gramene::CDBI::Protein::GeneProductTissue

=item * gene_product_to_cultivars => Gramene::CDBI::Protein::GeneProductToCultivar

=item * gene_product_to_embl_accessions => Gramene::CDBI::Protein::GeneProductToEmblAccession

=item * gene_product_to_gi_numbers => Gramene::CDBI::Protein::GeneProductToGiNumber

=item * gene_product_to_keywords => Gramene::CDBI::Protein::GeneProductToKeyword

=item * gene_product_to_pids => Gramene::CDBI::Protein::GeneProductToPid

=back


=head2 Gramene::CDBI::Protein::GeneProductComment

Corresponds to the table "gene_product_comment."  

Stringifies (C<to_name>) to "public_comment".

Fields:

=over 4

=item * gene_product_id

=item * public_comment

=item * curator_comment

=item * p_comment_curation_date

=item * c_comment_curation_date

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductFeature

Corresponds to the table "gene_product_feature."  

Stringifies (C<to_name>) to "from_position".

Fields:

=over 4

=item * gene_product_feature_id

=item * gene_product_id

=item * feature_type_id

=item * from_position

=item * to_position

=item * dbxref_id

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Protein::Dbxref

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=item * feature_type => Gramene::CDBI::Protein::GeneProductFeatureType

=back



=head2 Gramene::CDBI::Protein::GeneProductFeatureType

Corresponds to the table "gene_product_feature_type."  

Stringifies (C<to_name>) to "feature_type".

Fields:

=over 4

=item * feature_type_id

=item * feature_type

=back



Has Many:

=over 4

=item * gene_product_features => Gramene::CDBI::Protein::GeneProductFeature

=back


=head2 Gramene::CDBI::Protein::GeneProductGeneName

Corresponds to the table "gene_product_gene_name."  

Stringifies (C<to_name>) to "gene_name".

Fields:

=over 4

=item * gene_product_gene_name_id

=item * gene_product_id

=item * gene_name

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductHelper

Corresponds to the table "gene_product_helper."  

Stringifies (C<to_name>) to "swissprot_acc".

Fields:

=over 4

=item * gene_product_id

=item * swissprot_acc

=item * swissprot_id

=item * gene_product_name

=item * gene_product_gene_name

=item * organism

=item * organism_rank

=item * non_IEA_codes

=item * IEA_codes

=item * evidence_codes

=item * is_obsolete

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductOrganelle

Corresponds to the table "gene_product_organelle."  

Stringifies (C<to_name>) to "organelle".

Fields:

=over 4

=item * gene_product_organelle_id

=item * gene_product_id

=item * organelle

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductProsite

Corresponds to the table "gene_product_prosite."  

Stringifies (C<to_name>) to "prosite_desc".

Fields:

=over 4

=item * gene_product_prosite_id

=item * gene_product_id

=item * prosite_id

=item * prosite_desc

=item * prosite_sequence

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductSeq

Corresponds to the table "gene_product_seq."  

Stringifies (C<to_name>) to "is_primary_seq".

Fields:

=over 4

=item * gene_product_seq_id

=item * gene_product_id

=item * seq_id

=item * is_primary_seq

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=item * seq => Gramene::CDBI::Protein::Seq

=back



=head2 Gramene::CDBI::Protein::GeneProductSynonym

Corresponds to the table "gene_product_synonym."  

Stringifies (C<to_name>) to "gene_product_synonym_symbol".

Fields:

=over 4

=item * gene_product_synonym_id

=item * gene_product_id

=item * gene_product_synonym_symbol

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductTissue

Corresponds to the table "gene_product_tissue."  

Stringifies (C<to_name>) to "tissue".

Fields:

=over 4

=item * gene_product_tissue_id

=item * gene_product_id

=item * tissue

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductToCultivar

Corresponds to the table "gene_product_to_cultivar."  

Stringifies (C<to_name>) to "gene_product_to_cultivar_id".

Fields:

=over 4

=item * gene_product_to_cultivar_id

=item * gene_product_id

=item * cultivar_id

=back


Has A:

=over 4

=item * cultivar => Gramene::CDBI::Protein::Cultivar

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductToEmblAccession

Corresponds to the table "gene_product_to_embl_accession."  

Stringifies (C<to_name>) to "embl_accession".

Fields:

=over 4

=item * gene_product_to_embl_accession_id

=item * gene_product_id

=item * embl_accession

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductToGiNumber

Corresponds to the table "gene_product_to_gi_number."  

Stringifies (C<to_name>) to "gi_number".

Fields:

=over 4

=item * gene_product_to_gi_number_id

=item * gene_product_id

=item * gi_number

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::GeneProductToKeyword

Corresponds to the table "gene_product_to_keyword."  

Stringifies (C<to_name>) to "gene_product_to_keyword_id".

Fields:

=over 4

=item * gene_product_to_keyword_id

=item * gene_product_id

=item * keyword_id

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=item * keyword => Gramene::CDBI::Protein::Keyword

=back



=head2 Gramene::CDBI::Protein::GeneProductToPid

Corresponds to the table "gene_product_to_pid."  

Stringifies (C<to_name>) to "pid".

Fields:

=over 4

=item * gene_product_to_pid_id

=item * gene_product_id

=item * pid

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Protein::GeneProduct

=back



=head2 Gramene::CDBI::Protein::Keyword

Corresponds to the table "keyword."  

Stringifies (C<to_name>) to "keyword".

Fields:

=over 4

=item * keyword_id

=item * keyword

=back



Has Many:

=over 4

=item * gene_product_to_keywords => Gramene::CDBI::Protein::GeneProductToKeyword

=back


=head2 Gramene::CDBI::Protein::Objectxref

Corresponds to the table "objectxref."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * objectxref_id

=item * table_name

=item * row_id

=item * dbxref_id

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Protein::Dbxref

=back



=head2 Gramene::CDBI::Protein::Seq

Corresponds to the table "seq."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * id

=item * display_id

=item * description

=item * seq

=item * seq_len

=item * md5checksum

=item * moltype

=item * timestamp

=back



Has Many:

=over 4

=item * gene_product_seqs => Gramene::CDBI::Protein::GeneProductSeq

=back


=head2 Gramene::CDBI::Protein::Species

Corresponds to the table "species."  

Stringifies (C<to_name>) to "common_name".

Fields:

=over 4

=item * species_id

=item * ncbi_taxa_id

=item * common_name

=item * lineage_string

=item * genus

=item * species

=back



Has Many:

=over 4

=item * cultivars => Gramene::CDBI::Protein::Cultivar

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:22:25 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
