package Gramene::CDBI::PathwayArabidopsis;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'pathway_arabidopsis';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        pathway_gramene_search

    /
}

# ------------------------------------------------

package Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch; {

BEGIN { our @ISA = qw( Gramene::CDBI::PathwayArabidopsis ); };
# use base qw( Gramene::CDBI::PathwayArabidopsis );

Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->table('pathway_gramene_search');

our @extras = ();
Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->columns( 
    All => qw/ 
        pathway_gramene_search_id
        species
        gene_name
        enzyme_name
        reaction_id
        reaction_name
        ec
        pathway_id
        pathway_name 
    /, @extras
);

Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->columns( 
    Essential => qw/ 
        pathway_gramene_search_id
        species
        gene_name
        enzyme_name
        reaction_id
        reaction_name
        ec
        pathway_id
        pathway_name 
    /
);

Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->columns( 
    Ordered => qw/ 
        pathway_gramene_search_id
        species
        gene_name
        enzyme_name
        reaction_id
        reaction_name
        ec
        pathway_id
        pathway_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->species || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'pathway_gramene_search_id' => 'number',
        'species' => 'string',
        'gene_name' => 'string',
        'enzyme_name' => 'string',
        'reaction_id' => 'string',
        'reaction_name' => 'string',
        'ec' => 'string',
        'pathway_id' => 'string',
        'pathway_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::PathwayArabidopsis - Gramene::CDBI-base classes for PathwayArabidopsis

=head1 VERSION

This documentation refers to version $Revision: 1.2 $.

=head1 SYNOPSIS

  use Gramene::CDBI::PathwayArabidopsis;

  my @data   = Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->retrieve_all;
  my $object = Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the PathwayArabidopsis 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'pathway_arabidopsis'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::PathwayArabidopsis::PathwayGrameneSearch

Corresponds to the table "pathway_gramene_search."  

Stringifies (C<to_name>) to "species".

Fields:

=over 4

=item * pathway_gramene_search_id

=item * species

=item * gene_name

=item * enzyme_name

=item * reaction_id

=item * reaction_name

=item * ec

=item * pathway_id

=item * pathway_name

=back




=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Tue Mar 27 14:34:25 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
