package Gramene::CDBI::Ontology;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'ontology';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        association
        data_base
        dbxref
        evidence
        gene_product
        gene_product_synonym
        graph_path
        graph_path_to_term
        object_association_count
        quick_term_to_object
        relationship_type
        species
        term
        term_dbxref
        term_definition
        term_synonym
        term_to_term
        term_type

    /
}

# ------------------------------------------------

package Gramene::CDBI::Ontology::Association; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::Association->table('association');


sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::Association->columns( 
    All => qw/ 
        association_id
        term_id
        gene_product_id
        object_class
        is_not
        role_group 
    /, @extras
);

Gramene::CDBI::Ontology::Association->columns( 
    Essential => qw/ 
        association_id
        term_id
        gene_product_id
        object_class
        is_not
        role_group 
    /
);

Gramene::CDBI::Ontology::Association->columns( 
    Ordered => qw/ 
        association_id
        term_id
        gene_product_id
        object_class
        is_not
        role_group 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_class || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'association_id' => 'number',
        'term_id' => 'number',
        'gene_product_id' => 'number',
        'object_class' => 'string',
        'is_not' => 'string',
        'role_group' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::Association->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::DataBase; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::DataBase->table('data_base');

our @extras = ();
Gramene::CDBI::Ontology::DataBase->columns( 
    All => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /, @extras
);

Gramene::CDBI::Ontology::DataBase->columns( 
    Essential => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

Gramene::CDBI::Ontology::DataBase->columns( 
    Ordered => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'data_base_id' => 'number',
        'name' => 'string',
        'fullname' => 'string',
        'datatype' => 'string',
        'generic_url' => 'string',
        'url_syntax' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::Evidence; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::Evidence->table('evidence');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::Evidence->columns( 
    All => qw/ 
        evidence_id
        evidence_code
        association_id
        object_class
        dbxref_id
        seq_acc 
    /, @extras
);

Gramene::CDBI::Ontology::Evidence->columns( 
    Essential => qw/ 
        evidence_id
        evidence_code
        association_id
        object_class
        dbxref_id
        seq_acc 
    /
);

Gramene::CDBI::Ontology::Evidence->columns( 
    Ordered => qw/ 
        evidence_id
        evidence_code
        association_id
        object_class
        dbxref_id
        seq_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->evidence_code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'evidence_id' => 'number',
        'evidence_code' => 'string',
        'association_id' => 'number',
        'object_class' => 'string',
        'dbxref_id' => 'number',
        'seq_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::Evidence->has_a(
    dbxref_id => 'Gramene::CDBI::Ontology::Dbxref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::GeneProductSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::GeneProductSynonym->table('gene_product_synonym');


sub gene_product {
    my $self = shift;
    return $self->gene_product_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::GeneProductSynonym->columns( 
    All => qw/ 
        gene_product_id
        gene_product_synonym_symbol 
    /, @extras
);

Gramene::CDBI::Ontology::GeneProductSynonym->columns( 
    Essential => qw/ 
        gene_product_id
        gene_product_synonym_symbol 
    /
);

Gramene::CDBI::Ontology::GeneProductSynonym->columns( 
    Ordered => qw/ 
        gene_product_id
        gene_product_synonym_symbol 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_synonym_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_id' => 'number',
        'gene_product_synonym_symbol' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::GeneProductSynonym->has_a(
    gene_product_id => 'Gramene::CDBI::Ontology::GeneProduct'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'gene_product_id', 'gene_product_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::GeneProduct; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::GeneProduct->table('gene_product');


sub organism_dbxref {
    my $self = shift;
    return $self->organism_dbxref_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::GeneProduct->columns( 
    All => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /, @extras
);

Gramene::CDBI::Ontology::GeneProduct->columns( 
    Essential => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /
);

Gramene::CDBI::Ontology::GeneProduct->columns( 
    Ordered => qw/ 
        gene_product_id
        gene_product_symbol
        organism_dbxref_id
        species_id
        gene_product_full_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_product_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_product_id' => 'number',
        'gene_product_symbol' => 'string',
        'organism_dbxref_id' => 'number',
        'species_id' => 'number',
        'gene_product_full_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::GeneProduct->has_a(
    organism_dbxref_id => 'Gramene::CDBI::Ontology::Dbxref'
);

Gramene::CDBI::Ontology::GeneProduct->has_a(
    species_id => 'Gramene::CDBI::Ontology::Species'
);

Gramene::CDBI::Ontology::GeneProduct->has_many( 
    'gene_product_synonyms',
    'Gramene::CDBI::Ontology::GeneProductSynonym',
    'gene_product_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'organism_dbxref_id', 'dbxref_id' ],
            gene_product_synonym => [ 'gene_product_id', 'gene_product_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::TermDbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::TermDbxref->table('term_dbxref');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    

sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::TermDbxref->columns( 
    All => qw/ 
        term_dbxref_id
        term_id
        dbxref_id
        is_for_definition 
    /, @extras
);

Gramene::CDBI::Ontology::TermDbxref->columns( 
    Essential => qw/ 
        term_dbxref_id
        term_id
        dbxref_id
        is_for_definition 
    /
);

Gramene::CDBI::Ontology::TermDbxref->columns( 
    Ordered => qw/ 
        term_dbxref_id
        term_id
        dbxref_id
        is_for_definition 
    /
);

sub to_string {
    my $self = shift;
    return $self->is_for_definition || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_dbxref_id' => 'number',
        'term_id' => 'number',
        'dbxref_id' => 'number',
        'is_for_definition' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::TermDbxref->has_a(
    dbxref_id => 'Gramene::CDBI::Ontology::Dbxref'
);

Gramene::CDBI::Ontology::TermDbxref->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::Dbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::Dbxref->table('dbxref');

our @extras = ();
Gramene::CDBI::Ontology::Dbxref->columns( 
    All => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /, @extras
);

Gramene::CDBI::Ontology::Dbxref->columns( 
    Essential => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

Gramene::CDBI::Ontology::Dbxref->columns( 
    Ordered => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_key || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dbxref_id' => 'number',
        'xref_key' => 'string',
        'xref_keytype' => 'string',
        'xref_dbname' => 'string',
        'xref_desc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::Dbxref->has_many( 
    'evidences',
    'Gramene::CDBI::Ontology::Evidence',
    'dbxref_id' 
);

Gramene::CDBI::Ontology::Dbxref->has_many( 
    'gene_products',
    'Gramene::CDBI::Ontology::GeneProduct',
    'organism_dbxref_id' 
);

Gramene::CDBI::Ontology::Dbxref->has_many( 
    'term_dbxrefs',
    'Gramene::CDBI::Ontology::TermDbxref',
    'dbxref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            evidence => [ 'dbxref_id', 'dbxref_id' ],
            gene_product => [ 'dbxref_id', 'organism_dbxref_id' ],
            term_dbxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::GraphPathToTerm; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::GraphPathToTerm->table('graph_path_to_term');


sub graph_path {
    my $self = shift;
    return $self->graph_path_id;
}
    

sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::GraphPathToTerm->columns( 
    All => qw/ 
        graph_path_to_term_id
        graph_path_id
        term_id
        rank 
    /, @extras
);

Gramene::CDBI::Ontology::GraphPathToTerm->columns( 
    Essential => qw/ 
        graph_path_to_term_id
        graph_path_id
        term_id
        rank 
    /
);

Gramene::CDBI::Ontology::GraphPathToTerm->columns( 
    Ordered => qw/ 
        graph_path_to_term_id
        graph_path_id
        term_id
        rank 
    /
);

sub to_string {
    my $self = shift;
    return $self->rank || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'graph_path_to_term_id' => 'number',
        'graph_path_id' => 'number',
        'term_id' => 'number',
        'rank' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::GraphPathToTerm->has_a(
    graph_path_id => 'Gramene::CDBI::Ontology::GraphPath'
);

Gramene::CDBI::Ontology::GraphPathToTerm->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            graph_path => [ 'graph_path_id', 'graph_path_id' ],
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::GraphPath; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::GraphPath->table('graph_path');


sub term1 {
    my $self = shift;
    return $self->term1_id;
}
    
sub term2 {
    my $self = shift;
    return $self->term2_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::GraphPath->columns( 
    All => qw/ 
        graph_path_id
        term1_id
        term2_id
        distance 
    /, @extras
);

Gramene::CDBI::Ontology::GraphPath->columns( 
    Essential => qw/ 
        graph_path_id
        term1_id
        term2_id
        distance 
    /
);

Gramene::CDBI::Ontology::GraphPath->columns( 
    Ordered => qw/ 
        graph_path_id
        term1_id
        term2_id
        distance 
    /
);

sub to_string {
    my $self = shift;
    return $self->distance || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'graph_path_id' => 'number',
        'term1_id' => 'number',
        'term2_id' => 'number',
        'distance' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::GraphPath->has_a(
    term1_id => 'Gramene::CDBI::Ontology::Term'
);

Gramene::CDBI::Ontology::GraphPath->has_a(
    term2_id => 'Gramene::CDBI::Ontology::Term'
);

Gramene::CDBI::Ontology::GraphPath->has_many( 
    'graph_path_to_terms',
    'Gramene::CDBI::Ontology::GraphPathToTerm',
    'graph_path_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            graph_path_to_term => [ 'graph_path_id', 'graph_path_id' ],
            term => [ 'term2_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::ObjectAssociationCount; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::ObjectAssociationCount->table('object_association_count');


sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::ObjectAssociationCount->columns( 
    All => qw/ 
        object_association_count_id
        term_id
        object_type
        object_species
        object_count
        association_count 
    /, @extras
);

Gramene::CDBI::Ontology::ObjectAssociationCount->columns( 
    Essential => qw/ 
        object_association_count_id
        term_id
        object_type
        object_species
        object_count
        association_count 
    /
);

Gramene::CDBI::Ontology::ObjectAssociationCount->columns( 
    Ordered => qw/ 
        object_association_count_id
        term_id
        object_type
        object_species
        object_count
        association_count 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_association_count_id' => 'number',
        'term_id' => 'number',
        'object_type' => 'string',
        'object_species' => 'string',
        'object_count' => 'number',
        'association_count' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::ObjectAssociationCount->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::QuickTermToObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::QuickTermToObject->table('quick_term_to_object');


sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::QuickTermToObject->columns( 
    All => qw/ 
        quick_term_to_object_id
        term_id
        term_accession
        term_name
        term_type
        object_database
        object_type
        object_accession_id
        object_symbol
        object_name
        object_synonyms
        object_species
        evidences 
    /, @extras
);

Gramene::CDBI::Ontology::QuickTermToObject->columns( 
    Essential => qw/ 
        quick_term_to_object_id
        term_id
        term_accession
        term_name
        term_type
        object_database
        object_type
        object_accession_id
        object_symbol
        object_name
        object_synonyms
        object_species
        evidences 
    /
);

Gramene::CDBI::Ontology::QuickTermToObject->columns( 
    Ordered => qw/ 
        quick_term_to_object_id
        term_id
        term_accession
        term_name
        term_type
        object_database
        object_type
        object_accession_id
        object_symbol
        object_name
        object_synonyms
        object_species
        evidences 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'quick_term_to_object_id' => 'number',
        'term_id' => 'number',
        'term_accession' => 'string',
        'term_name' => 'string',
        'term_type' => 'string',
        'object_database' => 'string',
        'object_type' => 'string',
        'object_accession_id' => 'string',
        'object_symbol' => 'string',
        'object_name' => 'string',
        'object_synonyms' => 'string',
        'object_species' => 'string',
        'evidences' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::QuickTermToObject->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::TermToTerm; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::TermToTerm->table('term_to_term');


sub relationship_type {
    my $self = shift;
    return $self->relationship_type_id;
}
    

sub term1 {
    my $self = shift;
    return $self->term1_id;
}
    
sub term2 {
    my $self = shift;
    return $self->term2_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::TermToTerm->columns( 
    All => qw/ 
        term_to_term_id
        relationship_type_id
        term1_id
        term2_id 
    /, @extras
);

Gramene::CDBI::Ontology::TermToTerm->columns( 
    Essential => qw/ 
        term_to_term_id
        relationship_type_id
        term1_id
        term2_id 
    /
);

Gramene::CDBI::Ontology::TermToTerm->columns( 
    Ordered => qw/ 
        term_to_term_id
        relationship_type_id
        term1_id
        term2_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_to_term_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_to_term_id' => 'number',
        'relationship_type_id' => 'number',
        'term1_id' => 'number',
        'term2_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::TermToTerm->has_a(
    relationship_type_id => 'Gramene::CDBI::Ontology::RelationshipType'
);

Gramene::CDBI::Ontology::TermToTerm->has_a(
    term1_id => 'Gramene::CDBI::Ontology::Term'
);

Gramene::CDBI::Ontology::TermToTerm->has_a(
    term2_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            relationship_type => [ 'relationship_type_id', 'relationship_type_id' ],
            term => [ 'term2_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::RelationshipType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::RelationshipType->table('relationship_type');

our @extras = ();
Gramene::CDBI::Ontology::RelationshipType->columns( 
    All => qw/ 
        relationship_type_id
        type_name
        type_desc 
    /, @extras
);

Gramene::CDBI::Ontology::RelationshipType->columns( 
    Essential => qw/ 
        relationship_type_id
        type_name
        type_desc 
    /
);

Gramene::CDBI::Ontology::RelationshipType->columns( 
    Ordered => qw/ 
        relationship_type_id
        type_name
        type_desc 
    /
);

sub to_string {
    my $self = shift;
    return $self->type_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'relationship_type_id' => 'number',
        'type_name' => 'string',
        'type_desc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::RelationshipType->has_many( 
    'term_to_terms',
    'Gramene::CDBI::Ontology::TermToTerm',
    'relationship_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term_to_term => [ 'relationship_type_id', 'relationship_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::Species; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::Species->table('species');

our @extras = ();
Gramene::CDBI::Ontology::Species->columns( 
    All => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /, @extras
);

Gramene::CDBI::Ontology::Species->columns( 
    Essential => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /
);

Gramene::CDBI::Ontology::Species->columns( 
    Ordered => qw/ 
        species_id
        ncbi_taxa_id
        common_name
        lineage_string
        genus
        species 
    /
);

sub to_string {
    my $self = shift;
    return $self->common_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'species_id' => 'number',
        'ncbi_taxa_id' => 'number',
        'common_name' => 'string',
        'lineage_string' => 'string',
        'genus' => 'string',
        'species' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::Species->has_many( 
    'gene_products',
    'Gramene::CDBI::Ontology::GeneProduct',
    'species_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_product => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::TermDefinition; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::TermDefinition->table('term_definition');


sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::TermDefinition->columns( 
    All => qw/ 
        term_definition_id
        term_id
        definition
        dbxref_id
        term_def_comment
        term_def_reference 
    /, @extras
);

Gramene::CDBI::Ontology::TermDefinition->columns( 
    Essential => qw/ 
        term_definition_id
        term_id
        definition
        dbxref_id
        term_def_comment
        term_def_reference 
    /
);

Gramene::CDBI::Ontology::TermDefinition->columns( 
    Ordered => qw/ 
        term_definition_id
        term_id
        definition
        dbxref_id
        term_def_comment
        term_def_reference 
    /
);

sub to_string {
    my $self = shift;
    return $self->definition || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_definition_id' => 'number',
        'term_id' => 'number',
        'definition' => 'string',
        'dbxref_id' => 'number',
        'term_def_comment' => 'string',
        'term_def_reference' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::TermDefinition->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::TermSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::TermSynonym->table('term_synonym');


sub term {
    my $self = shift;
    return $self->term_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::TermSynonym->columns( 
    All => qw/ 
        term_synonym_id
        term_id
        synonym_name
        synonym_acc 
    /, @extras
);

Gramene::CDBI::Ontology::TermSynonym->columns( 
    Essential => qw/ 
        term_synonym_id
        term_id
        synonym_name
        synonym_acc 
    /
);

Gramene::CDBI::Ontology::TermSynonym->columns( 
    Ordered => qw/ 
        term_synonym_id
        term_id
        synonym_name
        synonym_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->synonym_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_synonym_id' => 'number',
        'term_id' => 'number',
        'synonym_name' => 'string',
        'synonym_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::TermSynonym->has_a(
    term_id => 'Gramene::CDBI::Ontology::Term'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_id', 'term_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::Term; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::Term->table('term');


sub term_type {
    my $self = shift;
    return $self->term_type_id;
}
    
our @extras = ();
Gramene::CDBI::Ontology::Term->columns( 
    All => qw/ 
        term_id
        term_name
        term_type_id
        term_accession
        is_obsolete
        is_root 
    /, @extras
);

Gramene::CDBI::Ontology::Term->columns( 
    Essential => qw/ 
        term_id
        term_name
        term_type_id
        term_accession
        is_obsolete
        is_root 
    /
);

Gramene::CDBI::Ontology::Term->columns( 
    Ordered => qw/ 
        term_id
        term_name
        term_type_id
        term_accession
        is_obsolete
        is_root 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_id' => 'number',
        'term_name' => 'string',
        'term_type_id' => 'number',
        'term_accession' => 'string',
        'is_obsolete' => 'string',
        'is_root' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::Term->has_a(
    term_type_id => 'Gramene::CDBI::Ontology::TermType'
);

Gramene::CDBI::Ontology::Term->has_many( 
    'associations',
    'Gramene::CDBI::Ontology::Association',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'graph_paths_term1',
    'Gramene::CDBI::Ontology::GraphPath',
    'term1_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'graph_paths_term2',
    'Gramene::CDBI::Ontology::GraphPath',
    'term2_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'graph_path_to_terms',
    'Gramene::CDBI::Ontology::GraphPathToTerm',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'object_association_counts',
    'Gramene::CDBI::Ontology::ObjectAssociationCount',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'quick_term_to_objects',
    'Gramene::CDBI::Ontology::QuickTermToObject',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'term_dbxrefs',
    'Gramene::CDBI::Ontology::TermDbxref',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'term_definitions',
    'Gramene::CDBI::Ontology::TermDefinition',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'term_synonyms',
    'Gramene::CDBI::Ontology::TermSynonym',
    'term_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'term_to_terms_term1',
    'Gramene::CDBI::Ontology::TermToTerm',
    'term1_id' 
);

Gramene::CDBI::Ontology::Term->has_many( 
    'term_to_terms_term2',
    'Gramene::CDBI::Ontology::TermToTerm',
    'term2_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            association => [ 'term_id', 'term_id' ],
            graph_path => [ 'term_id', 'term2_id' ],
            graph_path_to_term => [ 'term_id', 'term_id' ],
            object_association_count => [ 'term_id', 'term_id' ],
            quick_term_to_object => [ 'term_id', 'term_id' ],
            term_dbxref => [ 'term_id', 'term_id' ],
            term_definition => [ 'term_id', 'term_id' ],
            term_synonym => [ 'term_id', 'term_id' ],
            term_to_term => [ 'term_id', 'term2_id' ],
            term_type => [ 'term_type_id', 'term_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Ontology::TermType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Ontology ); };
# use base qw( Gramene::CDBI::Ontology );

Gramene::CDBI::Ontology::TermType->table('term_type');

our @extras = ();
Gramene::CDBI::Ontology::TermType->columns( 
    All => qw/ 
        term_type_id
        term_type 
    /, @extras
);

Gramene::CDBI::Ontology::TermType->columns( 
    Essential => qw/ 
        term_type_id
        term_type 
    /
);

Gramene::CDBI::Ontology::TermType->columns( 
    Ordered => qw/ 
        term_type_id
        term_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->term_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'term_type_id' => 'number',
        'term_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Ontology::TermType->has_many( 
    'terms',
    'Gramene::CDBI::Ontology::Term',
    'term_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            term => [ 'term_type_id', 'term_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Ontology - Gramene::CDBI-base classes for Ontology

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Ontology;

  my @data   = Gramene::CDBI::Ontology::DataBase->retrieve_all;
  my $object = Gramene::CDBI::Ontology::DataBase->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Ontology 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'ontology'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Ontology::Association

Corresponds to the table "association."  

Stringifies (C<to_name>) to "object_class".

Fields:

=over 4

=item * association_id

=item * term_id

=item * gene_product_id

=item * object_class

=item * is_not

=item * role_group

=back


Has A:

=over 4

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::DataBase

Corresponds to the table "data_base."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * data_base_id

=item * name

=item * fullname

=item * datatype

=item * generic_url

=item * url_syntax

=back




=head2 Gramene::CDBI::Ontology::Dbxref

Corresponds to the table "dbxref."  

Stringifies (C<to_name>) to "xref_key".

Fields:

=over 4

=item * dbxref_id

=item * xref_key

=item * xref_keytype

=item * xref_dbname

=item * xref_desc

=back



Has Many:

=over 4

=item * evidences => Gramene::CDBI::Ontology::Evidence

=item * gene_products => Gramene::CDBI::Ontology::GeneProduct

=item * term_dbxrefs => Gramene::CDBI::Ontology::TermDbxref

=back


=head2 Gramene::CDBI::Ontology::Evidence

Corresponds to the table "evidence."  

Stringifies (C<to_name>) to "evidence_code".

Fields:

=over 4

=item * evidence_id

=item * evidence_code

=item * association_id

=item * object_class

=item * dbxref_id

=item * seq_acc

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Ontology::Dbxref

=back



=head2 Gramene::CDBI::Ontology::GeneProduct

Corresponds to the table "gene_product."  

Stringifies (C<to_name>) to "gene_product_symbol".

Fields:

=over 4

=item * gene_product_id

=item * gene_product_symbol

=item * organism_dbxref_id

=item * species_id

=item * gene_product_full_name

=back


Has A:

=over 4

=item * organism_dbxref => Gramene::CDBI::Ontology::Dbxref

=item * species => Gramene::CDBI::Ontology::Species

=back


Has Many:

=over 4

=item * gene_product_synonyms => Gramene::CDBI::Ontology::GeneProductSynonym

=back


=head2 Gramene::CDBI::Ontology::GeneProductSynonym

Corresponds to the table "gene_product_synonym."  

Stringifies (C<to_name>) to "gene_product_synonym_symbol".

Fields:

=over 4

=item * gene_product_id

=item * gene_product_synonym_symbol

=back


Has A:

=over 4

=item * gene_product => Gramene::CDBI::Ontology::GeneProduct

=back



=head2 Gramene::CDBI::Ontology::GraphPath

Corresponds to the table "graph_path."  

Stringifies (C<to_name>) to "distance".

Fields:

=over 4

=item * graph_path_id

=item * term1_id

=item * term2_id

=item * distance

=back


Has A:

=over 4

=item * term1 => Gramene::CDBI::Ontology::Term

=item * term2 => Gramene::CDBI::Ontology::Term

=back


Has Many:

=over 4

=item * graph_path_to_terms => Gramene::CDBI::Ontology::GraphPathToTerm

=back


=head2 Gramene::CDBI::Ontology::GraphPathToTerm

Corresponds to the table "graph_path_to_term."  

Stringifies (C<to_name>) to "rank".

Fields:

=over 4

=item * graph_path_to_term_id

=item * graph_path_id

=item * term_id

=item * rank

=back


Has A:

=over 4

=item * graph_path => Gramene::CDBI::Ontology::GraphPath

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::ObjectAssociationCount

Corresponds to the table "object_association_count."  

Stringifies (C<to_name>) to "object_type".

Fields:

=over 4

=item * object_association_count_id

=item * term_id

=item * object_type

=item * object_species

=item * object_count

=item * association_count

=back


Has A:

=over 4

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::QuickTermToObject

Corresponds to the table "quick_term_to_object."  

Stringifies (C<to_name>) to "term_accession".

Fields:

=over 4

=item * quick_term_to_object_id

=item * term_id

=item * term_accession

=item * term_name

=item * term_type

=item * object_database

=item * object_type

=item * object_accession_id

=item * object_symbol

=item * object_name

=item * object_synonyms

=item * object_species

=item * evidences

=back


Has A:

=over 4

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::RelationshipType

Corresponds to the table "relationship_type."  

Stringifies (C<to_name>) to "type_name".

Fields:

=over 4

=item * relationship_type_id

=item * type_name

=item * type_desc

=back



Has Many:

=over 4

=item * term_to_terms => Gramene::CDBI::Ontology::TermToTerm

=back


=head2 Gramene::CDBI::Ontology::Species

Corresponds to the table "species."  

Stringifies (C<to_name>) to "common_name".

Fields:

=over 4

=item * species_id

=item * ncbi_taxa_id

=item * common_name

=item * lineage_string

=item * genus

=item * species

=back



Has Many:

=over 4

=item * gene_products => Gramene::CDBI::Ontology::GeneProduct

=back


=head2 Gramene::CDBI::Ontology::Term

Corresponds to the table "term."  

Stringifies (C<to_name>) to "term_name".

Fields:

=over 4

=item * term_id

=item * term_name

=item * term_type_id

=item * term_accession

=item * is_obsolete

=item * is_root

=back


Has A:

=over 4

=item * term_type => Gramene::CDBI::Ontology::TermType

=back


Has Many:

=over 4

=item * associations => Gramene::CDBI::Ontology::Association

=item * graph_paths_term1 => Gramene::CDBI::Ontology::GraphPath

=item * graph_paths_term2 => Gramene::CDBI::Ontology::GraphPath

=item * graph_path_to_terms => Gramene::CDBI::Ontology::GraphPathToTerm

=item * object_association_counts => Gramene::CDBI::Ontology::ObjectAssociationCount

=item * quick_term_to_objects => Gramene::CDBI::Ontology::QuickTermToObject

=item * term_dbxrefs => Gramene::CDBI::Ontology::TermDbxref

=item * term_definitions => Gramene::CDBI::Ontology::TermDefinition

=item * term_synonyms => Gramene::CDBI::Ontology::TermSynonym

=item * term_to_terms_term1 => Gramene::CDBI::Ontology::TermToTerm

=item * term_to_terms_term2 => Gramene::CDBI::Ontology::TermToTerm

=back


=head2 Gramene::CDBI::Ontology::TermDbxref

Corresponds to the table "term_dbxref."  

Stringifies (C<to_name>) to "is_for_definition".

Fields:

=over 4

=item * term_dbxref_id

=item * term_id

=item * dbxref_id

=item * is_for_definition

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Ontology::Dbxref

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::TermDefinition

Corresponds to the table "term_definition."  

Stringifies (C<to_name>) to "definition".

Fields:

=over 4

=item * term_definition_id

=item * term_id

=item * definition

=item * dbxref_id

=item * term_def_comment

=item * term_def_reference

=back


Has A:

=over 4

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::TermSynonym

Corresponds to the table "term_synonym."  

Stringifies (C<to_name>) to "synonym_name".

Fields:

=over 4

=item * term_synonym_id

=item * term_id

=item * synonym_name

=item * synonym_acc

=back


Has A:

=over 4

=item * term => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::TermToTerm

Corresponds to the table "term_to_term."  

Stringifies (C<to_name>) to "term_to_term_id".

Fields:

=over 4

=item * term_to_term_id

=item * relationship_type_id

=item * term1_id

=item * term2_id

=back


Has A:

=over 4

=item * relationship_type => Gramene::CDBI::Ontology::RelationshipType

=item * term1 => Gramene::CDBI::Ontology::Term

=item * term2 => Gramene::CDBI::Ontology::Term

=back



=head2 Gramene::CDBI::Ontology::TermType

Corresponds to the table "term_type."  

Stringifies (C<to_name>) to "term_type".

Fields:

=over 4

=item * term_type_id

=item * term_type

=back



Has Many:

=over 4

=item * terms => Gramene::CDBI::Ontology::Term

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:22:21 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
