package Gramene::CDBI::Markers;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'markers';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        analysis
        analytical_correspondence
        analytical_correspondence_type
        germplasm
        germplasm_to_map_set
        library
        map
        map_set
        map_type
        mapping
        marker
        marker_details_aflp
        marker_details_caps
        marker_details_clone
        marker_details_est
        marker_details_est_cluster
        marker_details_fpc
        marker_details_gene
        marker_details_gene_prediction
        marker_details_gene_primer
        marker_details_gss
        marker_details_mrna
        marker_details_primer
        marker_details_qtl
        marker_details_rapd
        marker_details_rflp
        marker_details_snp
        marker_details_sscp
        marker_details_ssr
        marker_details_ssr_primer_pair
        marker_details_sts
        marker_details_unclassified_dna
        marker_image
        marker_sequence
        marker_synonym
        marker_type
        population
        species
        synonym_type
        xref
        xref_type

    /
}

# ------------------------------------------------

package Gramene::CDBI::Markers::AnalyticalCorrespondence; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::AnalyticalCorrespondence->table('analytical_correspondence');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub analytical_correspondence_type {
    my $self = shift;
    return $self->analytical_correspondence_type_id;
}
    

sub from_marker {
    my $self = shift;
    return $self->from_marker_id;
}
    
sub to_marker {
    my $self = shift;
    return $self->to_marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::AnalyticalCorrespondence->columns( 
    All => qw/ 
        analytical_correspondence_id
        from_marker_id
        to_marker_id
        analysis_id
        analytical_correspondence_type_id
        location
        location_index
        description 
    /, @extras
);

Gramene::CDBI::Markers::AnalyticalCorrespondence->columns( 
    Essential => qw/ 
        analytical_correspondence_id
        from_marker_id
        to_marker_id
        analysis_id
        analytical_correspondence_type_id
        location
        location_index
        description 
    /
);

Gramene::CDBI::Markers::AnalyticalCorrespondence->columns( 
    Ordered => qw/ 
        analytical_correspondence_id
        from_marker_id
        to_marker_id
        analysis_id
        analytical_correspondence_type_id
        location
        location_index
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->location || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analytical_correspondence_id' => 'number',
        'from_marker_id' => 'number',
        'to_marker_id' => 'number',
        'analysis_id' => 'number',
        'analytical_correspondence_type_id' => 'number',
        'location' => 'number',
        'location_index' => 'string',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::AnalyticalCorrespondence->has_a(
    analysis_id => 'Gramene::CDBI::Markers::Analysis'
);

Gramene::CDBI::Markers::AnalyticalCorrespondence->has_a(
    analytical_correspondence_type_id => 'Gramene::CDBI::Markers::AnalyticalCorrespondenceType'
);

Gramene::CDBI::Markers::AnalyticalCorrespondence->has_a(
    from_marker_id => 'Gramene::CDBI::Markers::Marker'
);

Gramene::CDBI::Markers::AnalyticalCorrespondence->has_a(
    to_marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            analytical_correspondence_type => [ 'analytical_correspondence_type_id', 'analytical_correspondence_type_id' ],
            marker => [ 'to_marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Mapping; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Mapping->table('mapping');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub map {
    my $self = shift;
    return $self->map_id;
}
    

sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub display_synonym {
    my $self = shift;
    return $self->display_synonym_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Mapping->columns( 
    All => qw/ 
        mapping_id
        marker_id
        map_id
        display_synonym_id
        analysis_id
        start
        end
        strand
        marker_start
        marker_end
        marker_strand
        cigar_line
        score
        evalue
        percent_identity
        percent_coverage
        alignment_length
        remark
        cmap_feature_accession
        date_created 
    /, @extras
);

Gramene::CDBI::Markers::Mapping->columns( 
    Essential => qw/ 
        mapping_id
        marker_id
        map_id
        display_synonym_id
        analysis_id
        start
        end
        strand
        marker_start
        marker_end
        marker_strand
        cigar_line
        score
        evalue
        percent_identity
        percent_coverage
        alignment_length
        remark
        cmap_feature_accession
        date_created 
    /
);

Gramene::CDBI::Markers::Mapping->columns( 
    Ordered => qw/ 
        mapping_id
        marker_id
        map_id
        display_synonym_id
        analysis_id
        start
        end
        strand
        marker_start
        marker_end
        marker_strand
        cigar_line
        score
        evalue
        percent_identity
        percent_coverage
        alignment_length
        remark
        cmap_feature_accession
        date_created 
    /
);

sub to_string {
    my $self = shift;
    return $self->start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'mapping_id' => 'number',
        'marker_id' => 'number',
        'map_id' => 'number',
        'display_synonym_id' => 'number',
        'analysis_id' => 'number',
        'start' => 'number',
        'end' => 'number',
        'strand' => 'string',
        'marker_start' => 'number',
        'marker_end' => 'number',
        'marker_strand' => 'string',
        'cigar_line' => 'string',
        'score' => 'number',
        'evalue' => 'number',
        'percent_identity' => 'number',
        'percent_coverage' => 'number',
        'alignment_length' => 'number',
        'remark' => 'string',
        'cmap_feature_accession' => 'string',
        'date_created' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Mapping->has_a(
    analysis_id => 'Gramene::CDBI::Markers::Analysis'
);

Gramene::CDBI::Markers::Mapping->has_a(
    map_id => 'Gramene::CDBI::Markers::Map'
);

Gramene::CDBI::Markers::Mapping->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);

Gramene::CDBI::Markers::Mapping->has_a(
    display_synonym_id => 'Gramene::CDBI::Markers::MarkerSynonym'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            map => [ 'map_id', 'map_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'display_synonym_id', 'marker_synonym_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsAflp; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsAflp->table('marker_details_aflp');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsAflp->columns( 
    All => qw/ 
        marker_id
        molecular_weight
        adapter1_name
        adapter1_restriction
        adapter1_sequence
        adapter1_complement
        adapter2_name
        adapter2_restriction
        adapter2_sequence
        adapter2_complement
        primer1_name
        primer1_common_seq
        primer1_overhang
        primer2_name
        primer2_common_seq
        primer2_overhang 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsAflp->columns( 
    Essential => qw/ 
        marker_id
        molecular_weight
        adapter1_name
        adapter1_restriction
        adapter1_sequence
        adapter1_complement
        adapter2_name
        adapter2_restriction
        adapter2_sequence
        adapter2_complement
        primer1_name
        primer1_common_seq
        primer1_overhang
        primer2_name
        primer2_common_seq
        primer2_overhang 
    /
);

Gramene::CDBI::Markers::MarkerDetailsAflp->columns( 
    Ordered => qw/ 
        marker_id
        molecular_weight
        adapter1_name
        adapter1_restriction
        adapter1_sequence
        adapter1_complement
        adapter2_name
        adapter2_restriction
        adapter2_sequence
        adapter2_complement
        primer1_name
        primer1_common_seq
        primer1_overhang
        primer2_name
        primer2_common_seq
        primer2_overhang 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'molecular_weight' => 'number',
        'adapter1_name' => 'string',
        'adapter1_restriction' => 'string',
        'adapter1_sequence' => 'string',
        'adapter1_complement' => 'string',
        'adapter2_name' => 'string',
        'adapter2_restriction' => 'string',
        'adapter2_sequence' => 'string',
        'adapter2_complement' => 'string',
        'primer1_name' => 'string',
        'primer1_common_seq' => 'string',
        'primer1_overhang' => 'string',
        'primer2_name' => 'string',
        'primer2_common_seq' => 'string',
        'primer2_overhang' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsAflp->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsCaps; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsCaps->table('marker_details_caps');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsCaps->columns( 
    All => qw/ 
        marker_id
        primer1
        primer2 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsCaps->columns( 
    Essential => qw/ 
        marker_id
        primer1
        primer2 
    /
);

Gramene::CDBI::Markers::MarkerDetailsCaps->columns( 
    Ordered => qw/ 
        marker_id
        primer1
        primer2 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'primer1' => 'string',
        'primer2' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsCaps->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsClone; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsClone->table('marker_details_clone');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsClone->columns( 
    All => qw/ 
        marker_id
        clone_id
        forward_primer
        reverse_primer
        seq_primer
        clone_insert_length
        lab_host
        vector
        site1
        site2
        tissue_type
        clone
        date_created
        date_updated
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsClone->columns( 
    Essential => qw/ 
        marker_id
        clone_id
        forward_primer
        reverse_primer
        seq_primer
        clone_insert_length
        lab_host
        vector
        site1
        site2
        tissue_type
        clone
        date_created
        date_updated
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /
);

Gramene::CDBI::Markers::MarkerDetailsClone->columns( 
    Ordered => qw/ 
        marker_id
        clone_id
        forward_primer
        reverse_primer
        seq_primer
        clone_insert_length
        lab_host
        vector
        site1
        site2
        tissue_type
        clone
        date_created
        date_updated
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'clone_id' => 'string',
        'forward_primer' => 'string',
        'reverse_primer' => 'string',
        'seq_primer' => 'string',
        'clone_insert_length' => 'number',
        'lab_host' => 'string',
        'vector' => 'string',
        'site1' => 'string',
        'site2' => 'string',
        'tissue_type' => 'string',
        'clone' => 'string',
        'date_created' => 'string',
        'date_updated' => 'string',
        'origin' => 'string',
        'ref_authors' => 'string',
        'ref_location' => 'string',
        'ref_medline' => 'number',
        'ref_pubmed' => 'number',
        'ref_title' => 'string',
        'ref_year' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsClone->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsEst; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsEst->table('marker_details_est');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsEst->columns( 
    All => qw/ 
        marker_id
        clone
        comment
        date_created
        date_updated
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsEst->columns( 
    Essential => qw/ 
        marker_id
        clone
        comment
        date_created
        date_updated
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /
);

Gramene::CDBI::Markers::MarkerDetailsEst->columns( 
    Ordered => qw/ 
        marker_id
        clone
        comment
        date_created
        date_updated
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'clone' => 'string',
        'comment' => 'string',
        'date_created' => 'string',
        'date_updated' => 'string',
        'lab_host' => 'string',
        'map' => 'string',
        'note' => 'string',
        'origin' => 'string',
        'ref_authors' => 'string',
        'ref_location' => 'string',
        'ref_medline' => 'number',
        'ref_pubmed' => 'number',
        'ref_title' => 'string',
        'ref_year' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsEst->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsEstCluster; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsEstCluster->table('marker_details_est_cluster');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsEstCluster->columns( 
    All => qw/ 
        marker_id
        date_updated
        seqence_count
        est_count
        htc_count
        mrna_count 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsEstCluster->columns( 
    Essential => qw/ 
        marker_id
        date_updated
        seqence_count
        est_count
        htc_count
        mrna_count 
    /
);

Gramene::CDBI::Markers::MarkerDetailsEstCluster->columns( 
    Ordered => qw/ 
        marker_id
        date_updated
        seqence_count
        est_count
        htc_count
        mrna_count 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'date_updated' => 'string',
        'seqence_count' => 'number',
        'est_count' => 'number',
        'htc_count' => 'number',
        'mrna_count' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsEstCluster->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsFpc; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsFpc->table('marker_details_fpc');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsFpc->columns( 
    All => qw/ 
        marker_id
        length_band
        length_basepair
        clone_count 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsFpc->columns( 
    Essential => qw/ 
        marker_id
        length_band
        length_basepair
        clone_count 
    /
);

Gramene::CDBI::Markers::MarkerDetailsFpc->columns( 
    Ordered => qw/ 
        marker_id
        length_band
        length_basepair
        clone_count 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'length_band' => 'number',
        'length_basepair' => 'number',
        'clone_count' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsFpc->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsGene; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsGene->table('marker_details_gene');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsGene->columns( 
    All => qw/ 
        marker_id
        chromosome 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsGene->columns( 
    Essential => qw/ 
        marker_id
        chromosome 
    /
);

Gramene::CDBI::Markers::MarkerDetailsGene->columns( 
    Ordered => qw/ 
        marker_id
        chromosome 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'chromosome' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsGene->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsGenePrediction; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsGenePrediction->table('marker_details_gene_prediction');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsGenePrediction->columns( 
    All => qw/ 
        marker_id
        chromosome 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsGenePrediction->columns( 
    Essential => qw/ 
        marker_id
        chromosome 
    /
);

Gramene::CDBI::Markers::MarkerDetailsGenePrediction->columns( 
    Ordered => qw/ 
        marker_id
        chromosome 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'chromosome' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsGenePrediction->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsGenePrimer; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsGenePrimer->table('marker_details_gene_primer');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsGenePrimer->columns( 
    All => qw/ 
        marker_id
        marker_accession
        primer1
        primer2
        gene
        gene_accession
        product_size
        anneal_temperature 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsGenePrimer->columns( 
    Essential => qw/ 
        marker_id
        marker_accession
        primer1
        primer2
        gene
        gene_accession
        product_size
        anneal_temperature 
    /
);

Gramene::CDBI::Markers::MarkerDetailsGenePrimer->columns( 
    Ordered => qw/ 
        marker_id
        marker_accession
        primer1
        primer2
        gene
        gene_accession
        product_size
        anneal_temperature 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'marker_accession' => 'string',
        'primer1' => 'string',
        'primer2' => 'string',
        'gene' => 'string',
        'gene_accession' => 'string',
        'product_size' => 'string',
        'anneal_temperature' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsGenePrimer->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsGss; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsGss->table('marker_details_gss');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsGss->columns( 
    All => qw/ 
        marker_id
        read_direction
        insert_size
        template
        cell_line
        cell_type
        clone
        clone_lib
        comment
        date_created
        date_updated
        dev_stage
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year
        sex
        tissue_type 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsGss->columns( 
    Essential => qw/ 
        marker_id
        read_direction
        insert_size
        template
        cell_line
        cell_type
        clone
        clone_lib
        comment
        date_created
        date_updated
        dev_stage
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year
        sex
        tissue_type 
    /
);

Gramene::CDBI::Markers::MarkerDetailsGss->columns( 
    Ordered => qw/ 
        marker_id
        read_direction
        insert_size
        template
        cell_line
        cell_type
        clone
        clone_lib
        comment
        date_created
        date_updated
        dev_stage
        lab_host
        map
        note
        origin
        ref_authors
        ref_location
        ref_medline
        ref_pubmed
        ref_title
        ref_year
        sex
        tissue_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'read_direction' => 'string',
        'insert_size' => 'string',
        'template' => 'string',
        'cell_line' => 'string',
        'cell_type' => 'string',
        'clone' => 'string',
        'clone_lib' => 'string',
        'comment' => 'string',
        'date_created' => 'string',
        'date_updated' => 'string',
        'dev_stage' => 'string',
        'lab_host' => 'string',
        'map' => 'string',
        'note' => 'string',
        'origin' => 'string',
        'ref_authors' => 'string',
        'ref_location' => 'string',
        'ref_medline' => 'number',
        'ref_pubmed' => 'number',
        'ref_title' => 'string',
        'ref_year' => 'number',
        'sex' => 'string',
        'tissue_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsGss->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsMrna; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsMrna->table('marker_details_mrna');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsMrna->columns( 
    All => qw/ 
        marker_id
        date_created
        date_updated
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsMrna->columns( 
    Essential => qw/ 
        marker_id
        date_created
        date_updated
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /
);

Gramene::CDBI::Markers::MarkerDetailsMrna->columns( 
    Ordered => qw/ 
        marker_id
        date_created
        date_updated
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'date_created' => 'string',
        'date_updated' => 'string',
        'EC_number' => 'string',
        'allele' => 'string',
        'bound_moiety' => 'string',
        'cell_line' => 'string',
        'cell_type' => 'string',
        'chromosome' => 'string',
        'citation' => 'string',
        'clone' => 'string',
        'clone_lib' => 'string',
        'codon_start' => 'string',
        'comment' => 'string',
        'compare' => 'string',
        'cons_splice' => 'string',
        'country' => 'string',
        'dev_stage' => 'string',
        'ecotype' => 'string',
        'evidence' => 'string',
        'exception' => 'string',
        'function' => 'string',
        'gene' => 'string',
        'germline' => 'string',
        'haplotype' => 'string',
        'insertion_seq' => 'string',
        'isolate' => 'string',
        'isolation_source' => 'string',
        'keyword' => 'string',
        'lab_host' => 'string',
        'label' => 'string',
        'locus_tag' => 'string',
        'macronuclear' => 'string',
        'map' => 'string',
        'mod_base' => 'string',
        'mol_type' => 'string',
        'note' => 'string',
        'number' => 'string',
        'origin' => 'string',
        'phenotype' => 'string',
        'plasmid' => 'string',
        'pop_variant' => 'string',
        'product' => 'string',
        'protein_id' => 'string',
        'pseudo' => 'string',
        'rearranged' => 'string',
        'ref_authors' => 'string',
        'ref_location' => 'string',
        'ref_pubmed' => 'string',
        'ref_title' => 'string',
        'ref_year' => 'number',
        'rpt_family' => 'string',
        'rpt_type' => 'string',
        'rpt_unit' => 'string',
        'sex' => 'string',
        'specimen_voucher' => 'string',
        'standard_name' => 'string',
        'sub_clone' => 'string',
        'sub_species' => 'string',
        'sub_strain' => 'string',
        'tissue_lib' => 'string',
        'tissue_type' => 'string',
        'transl_except' => 'string',
        'transl_table' => 'string',
        'translation' => 'string',
        'transposon' => 'string',
        'variety' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsMrna->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsPrimer; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsPrimer->table('marker_details_primer');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsPrimer->columns( 
    All => qw/ 
        marker_id
        marker_accession
        direction 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsPrimer->columns( 
    Essential => qw/ 
        marker_id
        marker_accession
        direction 
    /
);

Gramene::CDBI::Markers::MarkerDetailsPrimer->columns( 
    Ordered => qw/ 
        marker_id
        marker_accession
        direction 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'marker_accession' => 'string',
        'direction' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsPrimer->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsQtl; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsQtl->table('marker_details_qtl');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsQtl->columns( 
    All => qw/ 
        marker_id
        chromosome 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsQtl->columns( 
    Essential => qw/ 
        marker_id
        chromosome 
    /
);

Gramene::CDBI::Markers::MarkerDetailsQtl->columns( 
    Ordered => qw/ 
        marker_id
        chromosome 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'chromosome' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsQtl->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsRapd; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsRapd->table('marker_details_rapd');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsRapd->columns( 
    All => qw/ 
        marker_id
        primer1
        primer2
        band_size 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsRapd->columns( 
    Essential => qw/ 
        marker_id
        primer1
        primer2
        band_size 
    /
);

Gramene::CDBI::Markers::MarkerDetailsRapd->columns( 
    Ordered => qw/ 
        marker_id
        primer1
        primer2
        band_size 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'primer1' => 'string',
        'primer2' => 'string',
        'band_size' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsRapd->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsRflp; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsRflp->table('marker_details_rflp');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsRflp->columns( 
    All => qw/ 
        marker_id
        dna_treatment
        source_type
        source_tissue
        source_treatment
        vector
        vector_antibiotic_resistance
        fragment_size
        vector_excision_enzyme
        vector_primers
        overgo1
        overgo2 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsRflp->columns( 
    Essential => qw/ 
        marker_id
        dna_treatment
        source_type
        source_tissue
        source_treatment
        vector
        vector_antibiotic_resistance
        fragment_size
        vector_excision_enzyme
        vector_primers
        overgo1
        overgo2 
    /
);

Gramene::CDBI::Markers::MarkerDetailsRflp->columns( 
    Ordered => qw/ 
        marker_id
        dna_treatment
        source_type
        source_tissue
        source_treatment
        vector
        vector_antibiotic_resistance
        fragment_size
        vector_excision_enzyme
        vector_primers
        overgo1
        overgo2 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'dna_treatment' => 'string',
        'source_type' => 'string',
        'source_tissue' => 'string',
        'source_treatment' => 'string',
        'vector' => 'string',
        'vector_antibiotic_resistance' => 'string',
        'fragment_size' => 'number',
        'vector_excision_enzyme' => 'string',
        'vector_primers' => 'string',
        'overgo1' => 'string',
        'overgo2' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsRflp->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsSnp; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsSnp->table('marker_details_snp');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsSnp->columns( 
    All => qw/ 
        marker_id
        primer1
        primer2 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsSnp->columns( 
    Essential => qw/ 
        marker_id
        primer1
        primer2 
    /
);

Gramene::CDBI::Markers::MarkerDetailsSnp->columns( 
    Ordered => qw/ 
        marker_id
        primer1
        primer2 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'primer1' => 'string',
        'primer2' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsSnp->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsSscp; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsSscp->table('marker_details_sscp');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsSscp->columns( 
    All => qw/ 
        marker_id
        primer1
        primer2 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsSscp->columns( 
    Essential => qw/ 
        marker_id
        primer1
        primer2 
    /
);

Gramene::CDBI::Markers::MarkerDetailsSscp->columns( 
    Ordered => qw/ 
        marker_id
        primer1
        primer2 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'primer1' => 'string',
        'primer2' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsSscp->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsSsr; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsSsr->table('marker_details_ssr');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsSsr->columns( 
    All => qw/ 
        marker_id
        repeat_motif
        anneal_temperature
        product_size
        sequence_source
        remarks 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsSsr->columns( 
    Essential => qw/ 
        marker_id
        repeat_motif
        anneal_temperature
        product_size
        sequence_source
        remarks 
    /
);

Gramene::CDBI::Markers::MarkerDetailsSsr->columns( 
    Ordered => qw/ 
        marker_id
        repeat_motif
        anneal_temperature
        product_size
        sequence_source
        remarks 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'repeat_motif' => 'string',
        'anneal_temperature' => 'number',
        'product_size' => 'number',
        'sequence_source' => 'string',
        'remarks' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsSsr->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsSts; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsSts->table('marker_details_sts');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsSts->columns( 
    All => qw/ 
        marker_id
        primer1
        primer2
        reference 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsSts->columns( 
    Essential => qw/ 
        marker_id
        primer1
        primer2
        reference 
    /
);

Gramene::CDBI::Markers::MarkerDetailsSts->columns( 
    Ordered => qw/ 
        marker_id
        primer1
        primer2
        reference 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'primer1' => 'string',
        'primer2' => 'string',
        'reference' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerDetailsSts->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerImage; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerImage->table('marker_image');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerImage->columns( 
    All => qw/ 
        marker_image_id
        marker_id
        file_name
        caption
        width
        height 
    /, @extras
);

Gramene::CDBI::Markers::MarkerImage->columns( 
    Essential => qw/ 
        marker_image_id
        marker_id
        file_name
        caption
        width
        height 
    /
);

Gramene::CDBI::Markers::MarkerImage->columns( 
    Ordered => qw/ 
        marker_image_id
        marker_id
        file_name
        caption
        width
        height 
    /
);

sub to_string {
    my $self = shift;
    return $self->file_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_image_id' => 'number',
        'marker_id' => 'number',
        'file_name' => 'string',
        'caption' => 'string',
        'width' => 'number',
        'height' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerImage->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerSequence; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerSequence->table('marker_sequence');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerSequence->columns( 
    All => qw/ 
        marker_id
        seq
        seq_length
        md5_sum 
    /, @extras
);

Gramene::CDBI::Markers::MarkerSequence->columns( 
    Essential => qw/ 
        marker_id
        seq
        seq_length
        md5_sum 
    /
);

Gramene::CDBI::Markers::MarkerSequence->columns( 
    Ordered => qw/ 
        marker_id
        seq
        seq_length
        md5_sum 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'seq' => 'string',
        'seq_length' => 'number',
        'md5_sum' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerSequence->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerSynonym->table('marker_synonym');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub synonym_type {
    my $self = shift;
    return $self->synonym_type_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MarkerSynonym->columns( 
    All => qw/ 
        marker_synonym_id
        marker_id
        synonym_type_id
        marker_name 
    /, @extras
);

Gramene::CDBI::Markers::MarkerSynonym->columns( 
    Essential => qw/ 
        marker_synonym_id
        marker_id
        synonym_type_id
        marker_name 
    /
);

Gramene::CDBI::Markers::MarkerSynonym->columns( 
    Ordered => qw/ 
        marker_synonym_id
        marker_id
        synonym_type_id
        marker_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_synonym_id' => 'number',
        'marker_id' => 'number',
        'synonym_type_id' => 'number',
        'marker_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerSynonym->has_a(
    marker_id => 'Gramene::CDBI::Markers::Marker'
);

Gramene::CDBI::Markers::MarkerSynonym->has_a(
    synonym_type_id => 'Gramene::CDBI::Markers::SynonymType'
);

Gramene::CDBI::Markers::MarkerSynonym->has_many( 
    'mappings',
    'Gramene::CDBI::Markers::Mapping',
    'display_synonym_id' 
);

Gramene::CDBI::Markers::MarkerSynonym->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'display_synonym_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            mapping => [ 'marker_synonym_id', 'display_synonym_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            synonym_type => [ 'synonym_type_id', 'synonym_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Marker; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Marker->table('marker');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub germplasm {
    my $self = shift;
    return $self->germplasm_id;
}
    

sub library {
    my $self = shift;
    return $self->library_id;
}
    

sub display_synonym {
    my $self = shift;
    return $self->display_synonym_id;
}
    

sub marker_type {
    my $self = shift;
    return $self->marker_type_id;
}
    

sub source_species {
    my $self = shift;
    return $self->source_species_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Marker->columns( 
    All => qw/ 
        marker_id
        marker_type_id
        display_synonym_id
        source_species_id
        germplasm_id
        analysis_id
        library_id
        description
        date_created
        date_updated 
    /, @extras
);

Gramene::CDBI::Markers::Marker->columns( 
    Essential => qw/ 
        marker_id
        marker_type_id
        display_synonym_id
        source_species_id
        germplasm_id
        analysis_id
        library_id
        description
        date_created
        date_updated 
    /
);

Gramene::CDBI::Markers::Marker->columns( 
    Ordered => qw/ 
        marker_id
        marker_type_id
        display_synonym_id
        source_species_id
        germplasm_id
        analysis_id
        library_id
        description
        date_created
        date_updated 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'marker_type_id' => 'number',
        'display_synonym_id' => 'number',
        'source_species_id' => 'number',
        'germplasm_id' => 'number',
        'analysis_id' => 'number',
        'library_id' => 'number',
        'description' => 'string',
        'date_created' => 'string',
        'date_updated' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Marker->has_a(
    analysis_id => 'Gramene::CDBI::Markers::Analysis'
);

Gramene::CDBI::Markers::Marker->has_a(
    germplasm_id => 'Gramene::CDBI::Markers::Germplasm'
);

Gramene::CDBI::Markers::Marker->has_a(
    library_id => 'Gramene::CDBI::Markers::Library'
);

Gramene::CDBI::Markers::Marker->has_a(
    display_synonym_id => 'Gramene::CDBI::Markers::MarkerSynonym'
);

Gramene::CDBI::Markers::Marker->has_a(
    marker_type_id => 'Gramene::CDBI::Markers::MarkerType'
);

Gramene::CDBI::Markers::Marker->has_a(
    source_species_id => 'Gramene::CDBI::Markers::Species'
);

Gramene::CDBI::Markers::Marker->has_many( 
    'analytical_correspondences_from_marker',
    'Gramene::CDBI::Markers::AnalyticalCorrespondence',
    'from_marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'analytical_correspondences_to_marker',
    'Gramene::CDBI::Markers::AnalyticalCorrespondence',
    'to_marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'mappings',
    'Gramene::CDBI::Markers::Mapping',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_aflps',
    'Gramene::CDBI::Markers::MarkerDetailsAflp',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_capss',
    'Gramene::CDBI::Markers::MarkerDetailsCaps',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_clones',
    'Gramene::CDBI::Markers::MarkerDetailsClone',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_ests',
    'Gramene::CDBI::Markers::MarkerDetailsEst',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_est_clusters',
    'Gramene::CDBI::Markers::MarkerDetailsEstCluster',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_fpcs',
    'Gramene::CDBI::Markers::MarkerDetailsFpc',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_genes',
    'Gramene::CDBI::Markers::MarkerDetailsGene',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_gene_predictions',
    'Gramene::CDBI::Markers::MarkerDetailsGenePrediction',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_gene_primers',
    'Gramene::CDBI::Markers::MarkerDetailsGenePrimer',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_gsses',
    'Gramene::CDBI::Markers::MarkerDetailsGss',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_mrnas',
    'Gramene::CDBI::Markers::MarkerDetailsMrna',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_primers',
    'Gramene::CDBI::Markers::MarkerDetailsPrimer',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_qtls',
    'Gramene::CDBI::Markers::MarkerDetailsQtl',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_rapds',
    'Gramene::CDBI::Markers::MarkerDetailsRapd',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_rflps',
    'Gramene::CDBI::Markers::MarkerDetailsRflp',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_snps',
    'Gramene::CDBI::Markers::MarkerDetailsSnp',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_sscps',
    'Gramene::CDBI::Markers::MarkerDetailsSscp',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_ssrs',
    'Gramene::CDBI::Markers::MarkerDetailsSsr',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_details_stss',
    'Gramene::CDBI::Markers::MarkerDetailsSts',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_images',
    'Gramene::CDBI::Markers::MarkerImage',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_sequences',
    'Gramene::CDBI::Markers::MarkerSequence',
    'marker_id' 
);

Gramene::CDBI::Markers::Marker->has_many( 
    'marker_synonyms',
    'Gramene::CDBI::Markers::MarkerSynonym',
    'marker_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            analytical_correspondence => [ 'marker_id', 'to_marker_id' ],
            germplasm => [ 'germplasm_id', 'germplasm_id' ],
            library => [ 'library_id', 'library_id' ],
            mapping => [ 'marker_id', 'marker_id' ],
            marker_details_aflp => [ 'marker_id', 'marker_id' ],
            marker_details_caps => [ 'marker_id', 'marker_id' ],
            marker_details_clone => [ 'marker_id', 'marker_id' ],
            marker_details_est => [ 'marker_id', 'marker_id' ],
            marker_details_est_cluster => [ 'marker_id', 'marker_id' ],
            marker_details_fpc => [ 'marker_id', 'marker_id' ],
            marker_details_gene => [ 'marker_id', 'marker_id' ],
            marker_details_gene_prediction => [ 'marker_id', 'marker_id' ],
            marker_details_gene_primer => [ 'marker_id', 'marker_id' ],
            marker_details_gss => [ 'marker_id', 'marker_id' ],
            marker_details_mrna => [ 'marker_id', 'marker_id' ],
            marker_details_primer => [ 'marker_id', 'marker_id' ],
            marker_details_qtl => [ 'marker_id', 'marker_id' ],
            marker_details_rapd => [ 'marker_id', 'marker_id' ],
            marker_details_rflp => [ 'marker_id', 'marker_id' ],
            marker_details_snp => [ 'marker_id', 'marker_id' ],
            marker_details_sscp => [ 'marker_id', 'marker_id' ],
            marker_details_ssr => [ 'marker_id', 'marker_id' ],
            marker_details_sts => [ 'marker_id', 'marker_id' ],
            marker_image => [ 'marker_id', 'marker_id' ],
            marker_sequence => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'marker_id', 'marker_id' ],
            marker_type => [ 'marker_type_id', 'marker_type_id' ],
            species => [ 'source_species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Analysis; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Analysis->table('analysis');

our @extras = ();
Gramene::CDBI::Markers::Analysis->columns( 
    All => qw/ 
        analysis_id
        analysis_name
        type
        description
        last_run
        ok_to_release 
    /, @extras
);

Gramene::CDBI::Markers::Analysis->columns( 
    Essential => qw/ 
        analysis_id
        analysis_name
        type
        description
        last_run
        ok_to_release 
    /
);

Gramene::CDBI::Markers::Analysis->columns( 
    Ordered => qw/ 
        analysis_id
        analysis_name
        type
        description
        last_run
        ok_to_release 
    /
);

sub to_string {
    my $self = shift;
    return $self->analysis_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'number',
        'analysis_name' => 'string',
        'type' => 'string',
        'description' => 'string',
        'last_run' => 'string',
        'ok_to_release' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Analysis->has_many( 
    'analytical_correspondences',
    'Gramene::CDBI::Markers::AnalyticalCorrespondence',
    'analysis_id' 
);

Gramene::CDBI::Markers::Analysis->has_many( 
    'mappings',
    'Gramene::CDBI::Markers::Mapping',
    'analysis_id' 
);

Gramene::CDBI::Markers::Analysis->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'analysis_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analytical_correspondence => [ 'analysis_id', 'analysis_id' ],
            mapping => [ 'analysis_id', 'analysis_id' ],
            marker => [ 'analysis_id', 'analysis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::AnalyticalCorrespondenceType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::AnalyticalCorrespondenceType->table('analytical_correspondence_type');

our @extras = ();
Gramene::CDBI::Markers::AnalyticalCorrespondenceType->columns( 
    All => qw/ 
        analytical_correspondence_type_id
        type
        description 
    /, @extras
);

Gramene::CDBI::Markers::AnalyticalCorrespondenceType->columns( 
    Essential => qw/ 
        analytical_correspondence_type_id
        type
        description 
    /
);

Gramene::CDBI::Markers::AnalyticalCorrespondenceType->columns( 
    Ordered => qw/ 
        analytical_correspondence_type_id
        type
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analytical_correspondence_type_id' => 'number',
        'type' => 'string',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::AnalyticalCorrespondenceType->has_many( 
    'analytical_correspondences',
    'Gramene::CDBI::Markers::AnalyticalCorrespondence',
    'analytical_correspondence_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analytical_correspondence => [ 'analytical_correspondence_type_id', 'analytical_correspondence_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::GermplasmToMapSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::GermplasmToMapSet->table('germplasm_to_map_set');


sub germplasm {
    my $self = shift;
    return $self->germplasm_id;
}
    

sub map_set {
    my $self = shift;
    return $self->map_set_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::GermplasmToMapSet->columns( 
    All => qw/ 
        germplasm_to_map_set_id
        germplasm_id
        map_set_id
        relationship 
    /, @extras
);

Gramene::CDBI::Markers::GermplasmToMapSet->columns( 
    Essential => qw/ 
        germplasm_to_map_set_id
        germplasm_id
        map_set_id
        relationship 
    /
);

Gramene::CDBI::Markers::GermplasmToMapSet->columns( 
    Ordered => qw/ 
        germplasm_to_map_set_id
        germplasm_id
        map_set_id
        relationship 
    /
);

sub to_string {
    my $self = shift;
    return $self->relationship || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'germplasm_to_map_set_id' => 'number',
        'germplasm_id' => 'number',
        'map_set_id' => 'number',
        'relationship' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::GermplasmToMapSet->has_a(
    germplasm_id => 'Gramene::CDBI::Markers::Germplasm'
);

Gramene::CDBI::Markers::GermplasmToMapSet->has_a(
    map_set_id => 'Gramene::CDBI::Markers::MapSet'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm => [ 'germplasm_id', 'germplasm_id' ],
            map_set => [ 'map_set_id', 'map_set_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Library; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Library->table('library');


sub germplasm {
    my $self = shift;
    return $self->germplasm_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Library->columns( 
    All => qw/ 
        library_id
        species_id
        germplasm_id
        library_name
        center_name
        center_project
        development_stage
        tissue_type
        cell_type
        cell_line
        sex
        comments
        eo_term
        gro_term
        po_term 
    /, @extras
);

Gramene::CDBI::Markers::Library->columns( 
    Essential => qw/ 
        library_id
        species_id
        germplasm_id
        library_name
        center_name
        center_project
        development_stage
        tissue_type
        cell_type
        cell_line
        sex
        eo_term
        gro_term
        po_term 
    /
);

Gramene::CDBI::Markers::Library->columns( 
    Ordered => qw/ 
        library_id
        species_id
        germplasm_id
        library_name
        center_name
        center_project
        development_stage
        tissue_type
        cell_type
        cell_line
        sex
        comments
        eo_term
        gro_term
        po_term 
    /
);

sub to_string {
    my $self = shift;
    return $self->library_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'library_id' => 'number',
        'species_id' => 'number',
        'germplasm_id' => 'number',
        'library_name' => 'string',
        'center_name' => 'string',
        'center_project' => 'string',
        'development_stage' => 'string',
        'tissue_type' => 'string',
        'cell_type' => 'string',
        'cell_line' => 'string',
        'sex' => 'string',
        'comments' => 'string',
        'eo_term' => 'string',
        'gro_term' => 'string',
        'po_term' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Library->has_a(
    germplasm_id => 'Gramene::CDBI::Markers::Germplasm'
);

Gramene::CDBI::Markers::Library->has_a(
    species_id => 'Gramene::CDBI::Markers::Species'
);

Gramene::CDBI::Markers::Library->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'library_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm => [ 'germplasm_id', 'germplasm_id' ],
            marker => [ 'library_id', 'library_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Map; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Map->table('map');


sub map_set {
    my $self = shift;
    return $self->map_set_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Map->columns( 
    All => qw/ 
        map_id
        map_set_id
        map_name
        display_order
        location_index
        start
        end
        cmap_map_accession 
    /, @extras
);

Gramene::CDBI::Markers::Map->columns( 
    Essential => qw/ 
        map_id
        map_set_id
        map_name
        display_order
        location_index
        start
        end
        cmap_map_accession 
    /
);

Gramene::CDBI::Markers::Map->columns( 
    Ordered => qw/ 
        map_id
        map_set_id
        map_name
        display_order
        location_index
        start
        end
        cmap_map_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->map_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_id' => 'number',
        'map_set_id' => 'number',
        'map_name' => 'string',
        'display_order' => 'string',
        'location_index' => 'string',
        'start' => 'number',
        'end' => 'number',
        'cmap_map_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Map->has_a(
    map_set_id => 'Gramene::CDBI::Markers::MapSet'
);

Gramene::CDBI::Markers::Map->has_many( 
    'mappings',
    'Gramene::CDBI::Markers::Mapping',
    'map_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            map_set => [ 'map_set_id', 'map_set_id' ],
            mapping => [ 'map_id', 'map_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MapSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MapSet->table('map_set');



sub map_type {
    my $self = shift;
    return $self->map_type_id;
}
    

sub population {
    my $self = shift;
    return $self->population_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::MapSet->columns( 
    All => qw/ 
        map_set_id
        source
        map_set_name
        map_set_short_name
        map_type_id
        species_id
        germplasm_id1
        germplasm_id2
        population_id
        project
        reference
        description
        total_length
        distance_unit
        published_on
        cmap_map_set_accession
        ensembl_species_name 
    /, @extras
);

Gramene::CDBI::Markers::MapSet->columns( 
    Essential => qw/ 
        map_set_id
        source
        map_set_name
        map_set_short_name
        map_type_id
        species_id
        germplasm_id1
        germplasm_id2
        population_id
        project
        reference
        description
        total_length
        distance_unit
        published_on
        cmap_map_set_accession
        ensembl_species_name 
    /
);

Gramene::CDBI::Markers::MapSet->columns( 
    Ordered => qw/ 
        map_set_id
        source
        map_set_name
        map_set_short_name
        map_type_id
        species_id
        germplasm_id1
        germplasm_id2
        population_id
        project
        reference
        description
        total_length
        distance_unit
        published_on
        cmap_map_set_accession
        ensembl_species_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->source || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_set_id' => 'number',
        'source' => 'string',
        'map_set_name' => 'string',
        'map_set_short_name' => 'string',
        'map_type_id' => 'number',
        'species_id' => 'number',
        'germplasm_id1' => 'number',
        'germplasm_id2' => 'number',
        'population_id' => 'number',
        'project' => 'string',
        'reference' => 'string',
        'description' => 'string',
        'total_length' => 'string',
        'distance_unit' => 'string',
        'published_on' => 'string',
        'cmap_map_set_accession' => 'string',
        'ensembl_species_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MapSet->has_a(
    map_type_id => 'Gramene::CDBI::Markers::MapType'
);

Gramene::CDBI::Markers::MapSet->has_a(
    population_id => 'Gramene::CDBI::Markers::Population'
);

Gramene::CDBI::Markers::MapSet->has_a(
    species_id => 'Gramene::CDBI::Markers::Species'
);

Gramene::CDBI::Markers::MapSet->has_many( 
    'germplasm_to_map_sets',
    'Gramene::CDBI::Markers::GermplasmToMapSet',
    'map_set_id' 
);

Gramene::CDBI::Markers::MapSet->has_many( 
    'maps',
    'Gramene::CDBI::Markers::Map',
    'map_set_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm => [ 'germplasm_id2', 'germplasm_id' ],
            germplasm_to_map_set => [ 'map_set_id', 'map_set_id' ],
            map => [ 'map_set_id', 'map_set_id' ],
            map_type => [ 'map_type_id', 'map_type_id' ],
            population => [ 'population_id', 'population_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Population; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Population->table('population');


sub male_germplasm {
    my $self = shift;
    return $self->male_germplasm_id;
}
    
sub female_germplasm {
    my $self = shift;
    return $self->female_germplasm_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Population->columns( 
    All => qw/ 
        population_id
        population_name
        species_id
        male_germplasm_id
        female_germplasm_id
        generation_number
        generation_type 
    /, @extras
);

Gramene::CDBI::Markers::Population->columns( 
    Essential => qw/ 
        population_id
        population_name
        species_id
        male_germplasm_id
        female_germplasm_id
        generation_number
        generation_type 
    /
);

Gramene::CDBI::Markers::Population->columns( 
    Ordered => qw/ 
        population_id
        population_name
        species_id
        male_germplasm_id
        female_germplasm_id
        generation_number
        generation_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->population_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'population_id' => 'number',
        'population_name' => 'string',
        'species_id' => 'number',
        'male_germplasm_id' => 'number',
        'female_germplasm_id' => 'number',
        'generation_number' => 'string',
        'generation_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Population->has_a(
    male_germplasm_id => 'Gramene::CDBI::Markers::Germplasm'
);

Gramene::CDBI::Markers::Population->has_a(
    female_germplasm_id => 'Gramene::CDBI::Markers::Germplasm'
);

Gramene::CDBI::Markers::Population->has_a(
    species_id => 'Gramene::CDBI::Markers::Species'
);

Gramene::CDBI::Markers::Population->has_many( 
    'map_sets',
    'Gramene::CDBI::Markers::MapSet',
    'population_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm => [ 'female_germplasm_id', 'germplasm_id' ],
            map_set => [ 'population_id', 'population_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Germplasm; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Germplasm->table('germplasm');


sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Germplasm->columns( 
    All => qw/ 
        germplasm_id
        germplasm_name
        species_id
        description 
    /, @extras
);

Gramene::CDBI::Markers::Germplasm->columns( 
    Essential => qw/ 
        germplasm_id
        germplasm_name
        species_id
        description 
    /
);

Gramene::CDBI::Markers::Germplasm->columns( 
    Ordered => qw/ 
        germplasm_id
        germplasm_name
        species_id
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->germplasm_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'germplasm_id' => 'number',
        'germplasm_name' => 'string',
        'species_id' => 'number',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Germplasm->has_a(
    species_id => 'Gramene::CDBI::Markers::Species'
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'germplasm_to_map_sets',
    'Gramene::CDBI::Markers::GermplasmToMapSet',
    'germplasm_id' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'libraries',
    'Gramene::CDBI::Markers::Library',
    'germplasm_id' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'map_sets_germplasm1',
    'Gramene::CDBI::Markers::MapSet',
    'germplasm_id1' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'map_sets_germplasm2',
    'Gramene::CDBI::Markers::MapSet',
    'germplasm_id2' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'germplasm_id' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'populations_male_germplasm',
    'Gramene::CDBI::Markers::Population',
    'male_germplasm_id' 
);

Gramene::CDBI::Markers::Germplasm->has_many( 
    'populations_female_germplasm',
    'Gramene::CDBI::Markers::Population',
    'female_germplasm_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm_to_map_set => [ 'germplasm_id', 'germplasm_id' ],
            library => [ 'germplasm_id', 'germplasm_id' ],
            map_set => [ 'germplasm_id', 'germplasm_id2' ],
            marker => [ 'germplasm_id', 'germplasm_id' ],
            population => [ 'germplasm_id', 'female_germplasm_id' ],
            species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MapType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MapType->table('map_type');

our @extras = ();
Gramene::CDBI::Markers::MapType->columns( 
    All => qw/ 
        map_type_id
        map_type
        display_order
        description 
    /, @extras
);

Gramene::CDBI::Markers::MapType->columns( 
    Essential => qw/ 
        map_type_id
        map_type
        display_order
        description 
    /
);

Gramene::CDBI::Markers::MapType->columns( 
    Ordered => qw/ 
        map_type_id
        map_type
        display_order
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->map_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_type_id' => 'number',
        'map_type' => 'string',
        'display_order' => 'number',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MapType->has_many( 
    'map_sets',
    'Gramene::CDBI::Markers::MapSet',
    'map_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            map_set => [ 'map_type_id', 'map_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair->table('marker_details_ssr_primer_pair');

our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair->columns( 
    All => qw/ 
        marker_id
        repeat_motif
        forward_primer
        reverse_primer
        anneal_temperature
        product_size
        remarks
        sequence_source 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair->columns( 
    Essential => qw/ 
        marker_id
        repeat_motif
        forward_primer
        reverse_primer
        anneal_temperature
        product_size
        remarks
        sequence_source 
    /
);

Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair->columns( 
    Ordered => qw/ 
        marker_id
        repeat_motif
        forward_primer
        reverse_primer
        anneal_temperature
        product_size
        remarks
        sequence_source 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'repeat_motif' => 'string',
        'forward_primer' => 'string',
        'reverse_primer' => 'string',
        'anneal_temperature' => 'number',
        'product_size' => 'number',
        'remarks' => 'string',
        'sequence_source' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna->table('marker_details_unclassified_dna');

our @extras = ();
Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna->columns( 
    All => qw/ 
        marker_id
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /, @extras
);

Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna->columns( 
    Essential => qw/ 
        marker_id
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /
);

Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna->columns( 
    Ordered => qw/ 
        marker_id
        EC_number
        allele
        bound_moiety
        cell_line
        cell_type
        chromosome
        citation
        clone
        clone_lib
        codon_start
        comment
        compare
        cons_splice
        country
        dev_stage
        ecotype
        evidence
        exception
        function
        gene
        germline
        haplotype
        insertion_seq
        isolate
        isolation_source
        keyword
        lab_host
        label
        locus_tag
        macronuclear
        map
        mod_base
        mol_type
        note
        number
        origin
        phenotype
        plasmid
        pop_variant
        product
        protein_id
        pseudo
        rearranged
        ref_authors
        ref_location
        ref_pubmed
        ref_title
        ref_year
        rpt_family
        rpt_type
        rpt_unit
        sex
        specimen_voucher
        standard_name
        sub_clone
        sub_species
        sub_strain
        tissue_lib
        tissue_type
        transl_except
        transl_table
        translation
        transposon
        variety 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'EC_number' => 'string',
        'allele' => 'string',
        'bound_moiety' => 'string',
        'cell_line' => 'string',
        'cell_type' => 'string',
        'chromosome' => 'string',
        'citation' => 'string',
        'clone' => 'string',
        'clone_lib' => 'string',
        'codon_start' => 'string',
        'comment' => 'string',
        'compare' => 'string',
        'cons_splice' => 'string',
        'country' => 'string',
        'dev_stage' => 'string',
        'ecotype' => 'string',
        'evidence' => 'string',
        'exception' => 'string',
        'function' => 'string',
        'gene' => 'string',
        'germline' => 'string',
        'haplotype' => 'string',
        'insertion_seq' => 'string',
        'isolate' => 'string',
        'isolation_source' => 'string',
        'keyword' => 'string',
        'lab_host' => 'string',
        'label' => 'string',
        'locus_tag' => 'string',
        'macronuclear' => 'string',
        'map' => 'string',
        'mod_base' => 'string',
        'mol_type' => 'string',
        'note' => 'string',
        'number' => 'string',
        'origin' => 'string',
        'phenotype' => 'string',
        'plasmid' => 'string',
        'pop_variant' => 'string',
        'product' => 'string',
        'protein_id' => 'string',
        'pseudo' => 'string',
        'rearranged' => 'string',
        'ref_authors' => 'string',
        'ref_location' => 'string',
        'ref_pubmed' => 'string',
        'ref_title' => 'string',
        'ref_year' => 'number',
        'rpt_family' => 'string',
        'rpt_type' => 'string',
        'rpt_unit' => 'string',
        'sex' => 'string',
        'specimen_voucher' => 'string',
        'standard_name' => 'string',
        'sub_clone' => 'string',
        'sub_species' => 'string',
        'sub_strain' => 'string',
        'tissue_lib' => 'string',
        'tissue_type' => 'string',
        'transl_except' => 'string',
        'transl_table' => 'string',
        'translation' => 'string',
        'transposon' => 'string',
        'variety' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::MarkerType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::MarkerType->table('marker_type');

our @extras = ();
Gramene::CDBI::Markers::MarkerType->columns( 
    All => qw/ 
        marker_type_id
        marker_type
        is_sequence
        description 
    /, @extras
);

Gramene::CDBI::Markers::MarkerType->columns( 
    Essential => qw/ 
        marker_type_id
        marker_type
        is_sequence
        description 
    /
);

Gramene::CDBI::Markers::MarkerType->columns( 
    Ordered => qw/ 
        marker_type_id
        marker_type
        is_sequence
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_type_id' => 'number',
        'marker_type' => 'string',
        'is_sequence' => 'string',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::MarkerType->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'marker_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_type_id', 'marker_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Species; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Species->table('species');

our @extras = ();
Gramene::CDBI::Markers::Species->columns( 
    All => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        display_order
        description
        cmap_species_accession 
    /, @extras
);

Gramene::CDBI::Markers::Species->columns( 
    Essential => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        display_order
        description
        cmap_species_accession 
    /
);

Gramene::CDBI::Markers::Species->columns( 
    Ordered => qw/ 
        species_id
        species
        common_name
        gramene_taxonomy_id
        display_order
        description
        cmap_species_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->species || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'species_id' => 'number',
        'species' => 'string',
        'common_name' => 'string',
        'gramene_taxonomy_id' => 'string',
        'display_order' => 'number',
        'description' => 'string',
        'cmap_species_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Species->has_many( 
    'germplasms',
    'Gramene::CDBI::Markers::Germplasm',
    'species_id' 
);

Gramene::CDBI::Markers::Species->has_many( 
    'libraries',
    'Gramene::CDBI::Markers::Library',
    'species_id' 
);

Gramene::CDBI::Markers::Species->has_many( 
    'map_sets',
    'Gramene::CDBI::Markers::MapSet',
    'species_id' 
);

Gramene::CDBI::Markers::Species->has_many( 
    'markers',
    'Gramene::CDBI::Markers::Marker',
    'source_species_id' 
);

Gramene::CDBI::Markers::Species->has_many( 
    'populations',
    'Gramene::CDBI::Markers::Population',
    'species_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            germplasm => [ 'species_id', 'species_id' ],
            library => [ 'species_id', 'species_id' ],
            map_set => [ 'species_id', 'species_id' ],
            marker => [ 'species_id', 'source_species_id' ],
            population => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::SynonymType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::SynonymType->table('synonym_type');

our @extras = ();
Gramene::CDBI::Markers::SynonymType->columns( 
    All => qw/ 
        synonym_type_id
        synonym_type
        url_template
        description
        validation 
    /, @extras
);

Gramene::CDBI::Markers::SynonymType->columns( 
    Essential => qw/ 
        synonym_type_id
        synonym_type
        url_template
        description
        validation 
    /
);

Gramene::CDBI::Markers::SynonymType->columns( 
    Ordered => qw/ 
        synonym_type_id
        synonym_type
        url_template
        description
        validation 
    /
);

sub to_string {
    my $self = shift;
    return $self->synonym_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'synonym_type_id' => 'number',
        'synonym_type' => 'string',
        'url_template' => 'string',
        'description' => 'string',
        'validation' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::SynonymType->has_many( 
    'marker_synonyms',
    'Gramene::CDBI::Markers::MarkerSynonym',
    'synonym_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker_synonym => [ 'synonym_type_id', 'synonym_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::Xref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::Xref->table('xref');


sub xref_type {
    my $self = shift;
    return $self->xref_type_id;
}
    
our @extras = ();
Gramene::CDBI::Markers::Xref->columns( 
    All => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /, @extras
);

Gramene::CDBI::Markers::Xref->columns( 
    Essential => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value 
    /
);

Gramene::CDBI::Markers::Xref->columns( 
    Ordered => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'table_name' => 'string',
        'record_id' => 'number',
        'xref_type_id' => 'number',
        'xref_value' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::Xref->has_a(
    xref_type_id => 'Gramene::CDBI::Markers::XrefType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref_type => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Markers::XrefType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Markers ); };
# use base qw( Gramene::CDBI::Markers );

Gramene::CDBI::Markers::XrefType->table('xref_type');

our @extras = ();
Gramene::CDBI::Markers::XrefType->columns( 
    All => qw/ 
        xref_type_id
        xref_type
        url_template 
    /, @extras
);

Gramene::CDBI::Markers::XrefType->columns( 
    Essential => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

Gramene::CDBI::Markers::XrefType->columns( 
    Ordered => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_type_id' => 'number',
        'xref_type' => 'string',
        'url_template' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Markers::XrefType->has_many( 
    'xrefs',
    'Gramene::CDBI::Markers::Xref',
    'xref_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Markers - Gramene::CDBI-base classes for Markers

=head1 VERSION

This documentation refers to version $Revision: 1.37 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Markers;

  my @data   = Gramene::CDBI::Markers::Population->retrieve_all;
  my $object = Gramene::CDBI::Markers::Population->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Markers 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'markers'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Markers::Analysis

Corresponds to the table "analysis."  

Stringifies (C<to_name>) to "analysis_name".

Fields:

=over 4

=item * analysis_id

=item * analysis_name

=item * type

=item * description

=item * last_run

=item * ok_to_release

=back



Has Many:

=over 4

=item * analytical_correspondences => Gramene::CDBI::Markers::AnalyticalCorrespondence

=item * mappings => Gramene::CDBI::Markers::Mapping

=item * markers => Gramene::CDBI::Markers::Marker

=back


=head2 Gramene::CDBI::Markers::AnalyticalCorrespondence

Corresponds to the table "analytical_correspondence."  

Stringifies (C<to_name>) to "location".

Fields:

=over 4

=item * analytical_correspondence_id

=item * from_marker_id

=item * to_marker_id

=item * analysis_id

=item * analytical_correspondence_type_id

=item * location

=item * location_index

=item * description

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::Markers::Analysis

=item * analytical_correspondence_type => Gramene::CDBI::Markers::AnalyticalCorrespondenceType

=item * from_marker => Gramene::CDBI::Markers::Marker

=item * to_marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::AnalyticalCorrespondenceType

Corresponds to the table "analytical_correspondence_type."  

Stringifies (C<to_name>) to "type".

Fields:

=over 4

=item * analytical_correspondence_type_id

=item * type

=item * description

=back



Has Many:

=over 4

=item * analytical_correspondences => Gramene::CDBI::Markers::AnalyticalCorrespondence

=back


=head2 Gramene::CDBI::Markers::Germplasm

Corresponds to the table "germplasm."  

Stringifies (C<to_name>) to "germplasm_name".

Fields:

=over 4

=item * germplasm_id

=item * germplasm_name

=item * species_id

=item * description

=back


Has A:

=over 4

=item * species => Gramene::CDBI::Markers::Species

=back


Has Many:

=over 4

=item * germplasm_to_map_sets => Gramene::CDBI::Markers::GermplasmToMapSet

=item * libraries => Gramene::CDBI::Markers::Library

=item * map_sets_germplasm1 => Gramene::CDBI::Markers::MapSet

=item * map_sets_germplasm2 => Gramene::CDBI::Markers::MapSet

=item * markers => Gramene::CDBI::Markers::Marker

=item * populations_male_germplasm => Gramene::CDBI::Markers::Population

=item * populations_female_germplasm => Gramene::CDBI::Markers::Population

=back


=head2 Gramene::CDBI::Markers::GermplasmToMapSet

Corresponds to the table "germplasm_to_map_set."  

Stringifies (C<to_name>) to "relationship".

Fields:

=over 4

=item * germplasm_to_map_set_id

=item * germplasm_id

=item * map_set_id

=item * relationship

=back


Has A:

=over 4

=item * germplasm => Gramene::CDBI::Markers::Germplasm

=item * map_set => Gramene::CDBI::Markers::MapSet

=back



=head2 Gramene::CDBI::Markers::Library

Corresponds to the table "library."  

Stringifies (C<to_name>) to "library_name".

Fields:

=over 4

=item * library_id

=item * species_id

=item * germplasm_id

=item * library_name

=item * center_name

=item * center_project

=item * development_stage

=item * tissue_type

=item * cell_type

=item * cell_line

=item * sex

=item * comments

=item * eo_term

=item * gro_term

=item * po_term

=back


Has A:

=over 4

=item * germplasm => Gramene::CDBI::Markers::Germplasm

=item * species => Gramene::CDBI::Markers::Species

=back


Has Many:

=over 4

=item * markers => Gramene::CDBI::Markers::Marker

=back


=head2 Gramene::CDBI::Markers::Map

Corresponds to the table "map."  

Stringifies (C<to_name>) to "map_name".

Fields:

=over 4

=item * map_id

=item * map_set_id

=item * map_name

=item * display_order

=item * location_index

=item * start

=item * end

=item * cmap_map_accession

=back


Has A:

=over 4

=item * map_set => Gramene::CDBI::Markers::MapSet

=back


Has Many:

=over 4

=item * mappings => Gramene::CDBI::Markers::Mapping

=back


=head2 Gramene::CDBI::Markers::MapSet

Corresponds to the table "map_set."  

Stringifies (C<to_name>) to "source".

Fields:

=over 4

=item * map_set_id

=item * source

=item * map_set_name

=item * map_set_short_name

=item * map_type_id

=item * species_id

=item * germplasm_id1

=item * germplasm_id2

=item * population_id

=item * project

=item * reference

=item * description

=item * total_length

=item * distance_unit

=item * published_on

=item * cmap_map_set_accession

=item * ensembl_species_name

=back


Has A:

=over 4

=item * map_type => Gramene::CDBI::Markers::MapType

=item * population => Gramene::CDBI::Markers::Population

=item * species => Gramene::CDBI::Markers::Species

=back


Has Many:

=over 4

=item * germplasm_to_map_sets => Gramene::CDBI::Markers::GermplasmToMapSet

=item * maps => Gramene::CDBI::Markers::Map

=back


=head2 Gramene::CDBI::Markers::MapType

Corresponds to the table "map_type."  

Stringifies (C<to_name>) to "map_type".

Fields:

=over 4

=item * map_type_id

=item * map_type

=item * display_order

=item * description

=back



Has Many:

=over 4

=item * map_sets => Gramene::CDBI::Markers::MapSet

=back


=head2 Gramene::CDBI::Markers::Mapping

Corresponds to the table "mapping."  

Stringifies (C<to_name>) to "start".

Fields:

=over 4

=item * mapping_id

=item * marker_id

=item * map_id

=item * display_synonym_id

=item * analysis_id

=item * start

=item * end

=item * strand

=item * marker_start

=item * marker_end

=item * marker_strand

=item * cigar_line

=item * score

=item * evalue

=item * percent_identity

=item * percent_coverage

=item * alignment_length

=item * remark

=item * cmap_feature_accession

=item * date_created

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::Markers::Analysis

=item * map => Gramene::CDBI::Markers::Map

=item * marker => Gramene::CDBI::Markers::Marker

=item * display_synonym => Gramene::CDBI::Markers::MarkerSynonym

=back



=head2 Gramene::CDBI::Markers::Marker

Corresponds to the table "marker."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * marker_id

=item * marker_type_id

=item * display_synonym_id

=item * source_species_id

=item * germplasm_id

=item * analysis_id

=item * library_id

=item * description

=item * date_created

=item * date_updated

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::Markers::Analysis

=item * germplasm => Gramene::CDBI::Markers::Germplasm

=item * library => Gramene::CDBI::Markers::Library

=item * display_synonym => Gramene::CDBI::Markers::MarkerSynonym

=item * marker_type => Gramene::CDBI::Markers::MarkerType

=item * source_species => Gramene::CDBI::Markers::Species

=back


Has Many:

=over 4

=item * analytical_correspondences_from_marker => Gramene::CDBI::Markers::AnalyticalCorrespondence

=item * analytical_correspondences_to_marker => Gramene::CDBI::Markers::AnalyticalCorrespondence

=item * mappings => Gramene::CDBI::Markers::Mapping

=item * marker_details_aflps => Gramene::CDBI::Markers::MarkerDetailsAflp

=item * marker_details_capss => Gramene::CDBI::Markers::MarkerDetailsCaps

=item * marker_details_clones => Gramene::CDBI::Markers::MarkerDetailsClone

=item * marker_details_ests => Gramene::CDBI::Markers::MarkerDetailsEst

=item * marker_details_est_clusters => Gramene::CDBI::Markers::MarkerDetailsEstCluster

=item * marker_details_fpcs => Gramene::CDBI::Markers::MarkerDetailsFpc

=item * marker_details_genes => Gramene::CDBI::Markers::MarkerDetailsGene

=item * marker_details_gene_predictions => Gramene::CDBI::Markers::MarkerDetailsGenePrediction

=item * marker_details_gene_primers => Gramene::CDBI::Markers::MarkerDetailsGenePrimer

=item * marker_details_gsses => Gramene::CDBI::Markers::MarkerDetailsGss

=item * marker_details_mrnas => Gramene::CDBI::Markers::MarkerDetailsMrna

=item * marker_details_primers => Gramene::CDBI::Markers::MarkerDetailsPrimer

=item * marker_details_qtls => Gramene::CDBI::Markers::MarkerDetailsQtl

=item * marker_details_rapds => Gramene::CDBI::Markers::MarkerDetailsRapd

=item * marker_details_rflps => Gramene::CDBI::Markers::MarkerDetailsRflp

=item * marker_details_snps => Gramene::CDBI::Markers::MarkerDetailsSnp

=item * marker_details_sscps => Gramene::CDBI::Markers::MarkerDetailsSscp

=item * marker_details_ssrs => Gramene::CDBI::Markers::MarkerDetailsSsr

=item * marker_details_stss => Gramene::CDBI::Markers::MarkerDetailsSts

=item * marker_images => Gramene::CDBI::Markers::MarkerImage

=item * marker_sequences => Gramene::CDBI::Markers::MarkerSequence

=item * marker_synonyms => Gramene::CDBI::Markers::MarkerSynonym

=back


=head2 Gramene::CDBI::Markers::MarkerDetailsAflp

Corresponds to the table "marker_details_aflp."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * molecular_weight

=item * adapter1_name

=item * adapter1_restriction

=item * adapter1_sequence

=item * adapter1_complement

=item * adapter2_name

=item * adapter2_restriction

=item * adapter2_sequence

=item * adapter2_complement

=item * primer1_name

=item * primer1_common_seq

=item * primer1_overhang

=item * primer2_name

=item * primer2_common_seq

=item * primer2_overhang

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsCaps

Corresponds to the table "marker_details_caps."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * primer1

=item * primer2

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsClone

Corresponds to the table "marker_details_clone."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * clone_id

=item * forward_primer

=item * reverse_primer

=item * seq_primer

=item * clone_insert_length

=item * lab_host

=item * vector

=item * site1

=item * site2

=item * tissue_type

=item * clone

=item * date_created

=item * date_updated

=item * origin

=item * ref_authors

=item * ref_location

=item * ref_medline

=item * ref_pubmed

=item * ref_title

=item * ref_year

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsEst

Corresponds to the table "marker_details_est."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * clone

=item * comment

=item * date_created

=item * date_updated

=item * lab_host

=item * map

=item * note

=item * origin

=item * ref_authors

=item * ref_location

=item * ref_medline

=item * ref_pubmed

=item * ref_title

=item * ref_year

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsEstCluster

Corresponds to the table "marker_details_est_cluster."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * date_updated

=item * seqence_count

=item * est_count

=item * htc_count

=item * mrna_count

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsFpc

Corresponds to the table "marker_details_fpc."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * length_band

=item * length_basepair

=item * clone_count

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsGene

Corresponds to the table "marker_details_gene."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * chromosome

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsGenePrediction

Corresponds to the table "marker_details_gene_prediction."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * chromosome

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsGenePrimer

Corresponds to the table "marker_details_gene_primer."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * marker_accession

=item * primer1

=item * primer2

=item * gene

=item * gene_accession

=item * product_size

=item * anneal_temperature

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsGss

Corresponds to the table "marker_details_gss."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * read_direction

=item * insert_size

=item * template

=item * cell_line

=item * cell_type

=item * clone

=item * clone_lib

=item * comment

=item * date_created

=item * date_updated

=item * dev_stage

=item * lab_host

=item * map

=item * note

=item * origin

=item * ref_authors

=item * ref_location

=item * ref_medline

=item * ref_pubmed

=item * ref_title

=item * ref_year

=item * sex

=item * tissue_type

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsMrna

Corresponds to the table "marker_details_mrna."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * date_created

=item * date_updated

=item * EC_number

=item * allele

=item * bound_moiety

=item * cell_line

=item * cell_type

=item * chromosome

=item * citation

=item * clone

=item * clone_lib

=item * codon_start

=item * comment

=item * compare

=item * cons_splice

=item * country

=item * dev_stage

=item * ecotype

=item * evidence

=item * exception

=item * function

=item * gene

=item * germline

=item * haplotype

=item * insertion_seq

=item * isolate

=item * isolation_source

=item * keyword

=item * lab_host

=item * label

=item * locus_tag

=item * macronuclear

=item * map

=item * mod_base

=item * mol_type

=item * note

=item * number

=item * origin

=item * phenotype

=item * plasmid

=item * pop_variant

=item * product

=item * protein_id

=item * pseudo

=item * rearranged

=item * ref_authors

=item * ref_location

=item * ref_pubmed

=item * ref_title

=item * ref_year

=item * rpt_family

=item * rpt_type

=item * rpt_unit

=item * sex

=item * specimen_voucher

=item * standard_name

=item * sub_clone

=item * sub_species

=item * sub_strain

=item * tissue_lib

=item * tissue_type

=item * transl_except

=item * transl_table

=item * translation

=item * transposon

=item * variety

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsPrimer

Corresponds to the table "marker_details_primer."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * marker_accession

=item * direction

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsQtl

Corresponds to the table "marker_details_qtl."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * chromosome

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsRapd

Corresponds to the table "marker_details_rapd."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * primer1

=item * primer2

=item * band_size

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsRflp

Corresponds to the table "marker_details_rflp."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * dna_treatment

=item * source_type

=item * source_tissue

=item * source_treatment

=item * vector

=item * vector_antibiotic_resistance

=item * fragment_size

=item * vector_excision_enzyme

=item * vector_primers

=item * overgo1

=item * overgo2

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsSnp

Corresponds to the table "marker_details_snp."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * primer1

=item * primer2

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsSscp

Corresponds to the table "marker_details_sscp."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * primer1

=item * primer2

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsSsr

Corresponds to the table "marker_details_ssr."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * repeat_motif

=item * anneal_temperature

=item * product_size

=item * sequence_source

=item * remarks

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsSsrPrimerPair

Corresponds to the table "marker_details_ssr_primer_pair."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * repeat_motif

=item * forward_primer

=item * reverse_primer

=item * anneal_temperature

=item * product_size

=item * remarks

=item * sequence_source

=back




=head2 Gramene::CDBI::Markers::MarkerDetailsSts

Corresponds to the table "marker_details_sts."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * primer1

=item * primer2

=item * reference

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerDetailsUnclassifiedDna

Corresponds to the table "marker_details_unclassified_dna."  

Stringifies (C<to_name>) to "marker_id".

Fields:

=over 4

=item * marker_id

=item * EC_number

=item * allele

=item * bound_moiety

=item * cell_line

=item * cell_type

=item * chromosome

=item * citation

=item * clone

=item * clone_lib

=item * codon_start

=item * comment

=item * compare

=item * cons_splice

=item * country

=item * dev_stage

=item * ecotype

=item * evidence

=item * exception

=item * function

=item * gene

=item * germline

=item * haplotype

=item * insertion_seq

=item * isolate

=item * isolation_source

=item * keyword

=item * lab_host

=item * label

=item * locus_tag

=item * macronuclear

=item * map

=item * mod_base

=item * mol_type

=item * note

=item * number

=item * origin

=item * phenotype

=item * plasmid

=item * pop_variant

=item * product

=item * protein_id

=item * pseudo

=item * rearranged

=item * ref_authors

=item * ref_location

=item * ref_pubmed

=item * ref_title

=item * ref_year

=item * rpt_family

=item * rpt_type

=item * rpt_unit

=item * sex

=item * specimen_voucher

=item * standard_name

=item * sub_clone

=item * sub_species

=item * sub_strain

=item * tissue_lib

=item * tissue_type

=item * transl_except

=item * transl_table

=item * translation

=item * transposon

=item * variety

=back




=head2 Gramene::CDBI::Markers::MarkerImage

Corresponds to the table "marker_image."  

Stringifies (C<to_name>) to "file_name".

Fields:

=over 4

=item * marker_image_id

=item * marker_id

=item * file_name

=item * caption

=item * width

=item * height

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerSequence

Corresponds to the table "marker_sequence."  

Stringifies (C<to_name>) to "seq".

Fields:

=over 4

=item * marker_id

=item * seq

=item * seq_length

=item * md5_sum

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=back



=head2 Gramene::CDBI::Markers::MarkerSynonym

Corresponds to the table "marker_synonym."  

Stringifies (C<to_name>) to "marker_name".

Fields:

=over 4

=item * marker_synonym_id

=item * marker_id

=item * synonym_type_id

=item * marker_name

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::Markers::Marker

=item * synonym_type => Gramene::CDBI::Markers::SynonymType

=back


Has Many:

=over 4

=item * mappings => Gramene::CDBI::Markers::Mapping

=item * markers => Gramene::CDBI::Markers::Marker

=back


=head2 Gramene::CDBI::Markers::MarkerType

Corresponds to the table "marker_type."  

Stringifies (C<to_name>) to "marker_type".

Fields:

=over 4

=item * marker_type_id

=item * marker_type

=item * is_sequence

=item * description

=back



Has Many:

=over 4

=item * markers => Gramene::CDBI::Markers::Marker

=back


=head2 Gramene::CDBI::Markers::Population

Corresponds to the table "population."  

Stringifies (C<to_name>) to "population_name".

Fields:

=over 4

=item * population_id

=item * population_name

=item * species_id

=item * male_germplasm_id

=item * female_germplasm_id

=item * generation_number

=item * generation_type

=back


Has A:

=over 4

=item * male_germplasm => Gramene::CDBI::Markers::Germplasm

=item * female_germplasm => Gramene::CDBI::Markers::Germplasm

=item * species => Gramene::CDBI::Markers::Species

=back


Has Many:

=over 4

=item * map_sets => Gramene::CDBI::Markers::MapSet

=back


=head2 Gramene::CDBI::Markers::Species

Corresponds to the table "species."  

Stringifies (C<to_name>) to "species".

Fields:

=over 4

=item * species_id

=item * species

=item * common_name

=item * gramene_taxonomy_id

=item * display_order

=item * description

=item * cmap_species_accession

=back



Has Many:

=over 4

=item * germplasms => Gramene::CDBI::Markers::Germplasm

=item * libraries => Gramene::CDBI::Markers::Library

=item * map_sets => Gramene::CDBI::Markers::MapSet

=item * markers => Gramene::CDBI::Markers::Marker

=item * populations => Gramene::CDBI::Markers::Population

=back


=head2 Gramene::CDBI::Markers::SynonymType

Corresponds to the table "synonym_type."  

Stringifies (C<to_name>) to "synonym_type".

Fields:

=over 4

=item * synonym_type_id

=item * synonym_type

=item * url_template

=item * description

=item * validation

=back



Has Many:

=over 4

=item * marker_synonyms => Gramene::CDBI::Markers::MarkerSynonym

=back


=head2 Gramene::CDBI::Markers::Xref

Corresponds to the table "xref."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * xref_id

=item * table_name

=item * record_id

=item * xref_type_id

=item * xref_value

=item * comments

=back


Has A:

=over 4

=item * xref_type => Gramene::CDBI::Markers::XrefType

=back



=head2 Gramene::CDBI::Markers::XrefType

Corresponds to the table "xref_type."  

Stringifies (C<to_name>) to "xref_type".

Fields:

=over 4

=item * xref_type_id

=item * xref_type

=item * url_template

=back



Has Many:

=over 4

=item * xrefs => Gramene::CDBI::Markers::Xref

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Thu May 10 09:10:55 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
