package Gramene::CDBI::Literature;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'literature';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        author
        contributor
        data_base
        dbxref
        objectxref
        reference
        reference_abstract
        reference_comment
        reference_extra
        reference_url
        source
        source_synonym
        source_url

    /
}

# ------------------------------------------------

package Gramene::CDBI::Literature::Author; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Author->table('author');


sub contributor {
    my $self = shift;
    return $self->contributor_id;
}
    

sub reference {
    my $self = shift;
    return $self->reference_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::Author->columns( 
    All => qw/ 
        author_id
        contributor_id
        reference_id
        authorship_position 
    /, @extras
);

Gramene::CDBI::Literature::Author->columns( 
    Essential => qw/ 
        author_id
        contributor_id
        reference_id
        authorship_position 
    /
);

Gramene::CDBI::Literature::Author->columns( 
    Ordered => qw/ 
        author_id
        contributor_id
        reference_id
        authorship_position 
    /
);

sub to_string {
    my $self = shift;
    return $self->authorship_position || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'author_id' => 'number',
        'contributor_id' => 'number',
        'reference_id' => 'number',
        'authorship_position' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Author->has_a(
    contributor_id => 'Gramene::CDBI::Literature::Contributor'
);

Gramene::CDBI::Literature::Author->has_a(
    reference_id => 'Gramene::CDBI::Literature::Reference'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            contributor => [ 'contributor_id', 'contributor_id' ],
            reference => [ 'reference_id', 'reference_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::ReferenceComment; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::ReferenceComment->table('reference_comment');


sub contributor {
    my $self = shift;
    return $self->contributor_id;
}
    

sub reference {
    my $self = shift;
    return $self->reference_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::ReferenceComment->columns( 
    All => qw/ 
        reference_id
        reference_comment
        contributor_id 
    /, @extras
);

Gramene::CDBI::Literature::ReferenceComment->columns( 
    Essential => qw/ 
        reference_id
        reference_comment
        contributor_id 
    /
);

Gramene::CDBI::Literature::ReferenceComment->columns( 
    Ordered => qw/ 
        reference_id
        reference_comment
        contributor_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->reference_comment || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'reference_id' => 'number',
        'reference_comment' => 'string',
        'contributor_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::ReferenceComment->has_a(
    contributor_id => 'Gramene::CDBI::Literature::Contributor'
);

Gramene::CDBI::Literature::ReferenceComment->has_a(
    reference_id => 'Gramene::CDBI::Literature::Reference'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            contributor => [ 'contributor_id', 'contributor_id' ],
            reference => [ 'reference_id', 'reference_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::Contributor; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Contributor->table('contributor');

our @extras = ();
Gramene::CDBI::Literature::Contributor->columns( 
    All => qw/ 
        contributor_id
        contributor_name
        contributor_email
        contributor_organization
        contributor_address
        contributor_phone
        contributor_fax 
    /, @extras
);

Gramene::CDBI::Literature::Contributor->columns( 
    Essential => qw/ 
        contributor_id
        contributor_name
        contributor_email
        contributor_organization
        contributor_address
        contributor_phone
        contributor_fax 
    /
);

Gramene::CDBI::Literature::Contributor->columns( 
    Ordered => qw/ 
        contributor_id
        contributor_name
        contributor_email
        contributor_organization
        contributor_address
        contributor_phone
        contributor_fax 
    /
);

sub to_string {
    my $self = shift;
    return $self->contributor_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'contributor_id' => 'number',
        'contributor_name' => 'string',
        'contributor_email' => 'string',
        'contributor_organization' => 'string',
        'contributor_address' => 'string',
        'contributor_phone' => 'number',
        'contributor_fax' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Contributor->has_many( 
    'authors',
    'Gramene::CDBI::Literature::Author',
    'contributor_id' 
);

Gramene::CDBI::Literature::Contributor->has_many( 
    'reference_comments',
    'Gramene::CDBI::Literature::ReferenceComment',
    'contributor_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            author => [ 'contributor_id', 'contributor_id' ],
            reference_comment => [ 'contributor_id', 'contributor_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::DataBase; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::DataBase->table('data_base');

our @extras = ();
Gramene::CDBI::Literature::DataBase->columns( 
    All => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /, @extras
);

Gramene::CDBI::Literature::DataBase->columns( 
    Essential => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

Gramene::CDBI::Literature::DataBase->columns( 
    Ordered => qw/ 
        data_base_id
        name
        fullname
        datatype
        generic_url
        url_syntax 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'data_base_id' => 'number',
        'name' => 'string',
        'fullname' => 'string',
        'datatype' => 'string',
        'generic_url' => 'string',
        'url_syntax' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::Objectxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Objectxref->table('objectxref');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::Objectxref->columns( 
    All => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /, @extras
);

Gramene::CDBI::Literature::Objectxref->columns( 
    Essential => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /
);

Gramene::CDBI::Literature::Objectxref->columns( 
    Ordered => qw/ 
        objectxref_id
        table_name
        row_id
        dbxref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'objectxref_id' => 'number',
        'table_name' => 'string',
        'row_id' => 'number',
        'dbxref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Objectxref->has_a(
    dbxref_id => 'Gramene::CDBI::Literature::Dbxref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            dbxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::Dbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Dbxref->table('dbxref');

our @extras = ();
Gramene::CDBI::Literature::Dbxref->columns( 
    All => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /, @extras
);

Gramene::CDBI::Literature::Dbxref->columns( 
    Essential => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

Gramene::CDBI::Literature::Dbxref->columns( 
    Ordered => qw/ 
        dbxref_id
        xref_key
        xref_keytype
        xref_dbname
        xref_desc 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_key || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dbxref_id' => 'number',
        'xref_key' => 'string',
        'xref_keytype' => 'string',
        'xref_dbname' => 'string',
        'xref_desc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Dbxref->has_many( 
    'objectxrefs',
    'Gramene::CDBI::Literature::Objectxref',
    'dbxref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            objectxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::ReferenceAbstract; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::ReferenceAbstract->table('reference_abstract');


sub reference {
    my $self = shift;
    return $self->reference_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::ReferenceAbstract->columns( 
    All => qw/ 
        reference_id
        abstract_part_a
        abstract_part_b 
    /, @extras
);

Gramene::CDBI::Literature::ReferenceAbstract->columns( 
    Essential => qw/ 
        reference_id
        abstract_part_a
        abstract_part_b 
    /
);

Gramene::CDBI::Literature::ReferenceAbstract->columns( 
    Ordered => qw/ 
        reference_id
        abstract_part_a
        abstract_part_b 
    /
);

sub to_string {
    my $self = shift;
    return $self->abstract_part_a || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'reference_id' => 'number',
        'abstract_part_a' => 'string',
        'abstract_part_b' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::ReferenceAbstract->has_a(
    reference_id => 'Gramene::CDBI::Literature::Reference'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            reference => [ 'reference_id', 'reference_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::ReferenceExtra; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::ReferenceExtra->table('reference_extra');


sub reference {
    my $self = shift;
    return $self->reference_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::ReferenceExtra->columns( 
    All => qw/ 
        reference_id
        title
        source
        author
        year
        abstract_part_a
        abstract_part_b 
    /, @extras
);

Gramene::CDBI::Literature::ReferenceExtra->columns( 
    Essential => qw/ 
        reference_id
        title
        source
        author
        year
        abstract_part_a
        abstract_part_b 
    /
);

Gramene::CDBI::Literature::ReferenceExtra->columns( 
    Ordered => qw/ 
        reference_id
        title
        source
        author
        year
        abstract_part_a
        abstract_part_b 
    /
);

sub to_string {
    my $self = shift;
    return $self->title || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'reference_id' => 'number',
        'title' => 'string',
        'source' => 'string',
        'author' => 'string',
        'year' => 'number',
        'abstract_part_a' => 'string',
        'abstract_part_b' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::ReferenceExtra->has_a(
    reference_id => 'Gramene::CDBI::Literature::Reference'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            reference => [ 'reference_id', 'reference_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::ReferenceUrl; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::ReferenceUrl->table('reference_url');


sub reference {
    my $self = shift;
    return $self->reference_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::ReferenceUrl->columns( 
    All => qw/ 
        reference_url_id
        reference_id
        reference_url
        url_label 
    /, @extras
);

Gramene::CDBI::Literature::ReferenceUrl->columns( 
    Essential => qw/ 
        reference_url_id
        reference_id
        reference_url
        url_label 
    /
);

Gramene::CDBI::Literature::ReferenceUrl->columns( 
    Ordered => qw/ 
        reference_url_id
        reference_id
        reference_url
        url_label 
    /
);

sub to_string {
    my $self = shift;
    return $self->reference_url || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'reference_url_id' => 'number',
        'reference_id' => 'number',
        'reference_url' => 'string',
        'url_label' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::ReferenceUrl->has_a(
    reference_id => 'Gramene::CDBI::Literature::Reference'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            reference => [ 'reference_id', 'reference_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::Reference; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Reference->table('reference');


sub source {
    my $self = shift;
    return $self->source_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::Reference->columns( 
    All => qw/ 
        reference_id
        source_id
        title
        volume
        year
        start_page
        end_page
        language
        corresponding_author 
    /, @extras
);

Gramene::CDBI::Literature::Reference->columns( 
    Essential => qw/ 
        reference_id
        source_id
        title
        volume
        year
        start_page
        end_page
        language
        corresponding_author 
    /
);

Gramene::CDBI::Literature::Reference->columns( 
    Ordered => qw/ 
        reference_id
        source_id
        title
        volume
        year
        start_page
        end_page
        language
        corresponding_author 
    /
);

sub to_string {
    my $self = shift;
    return $self->title || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'reference_id' => 'number',
        'source_id' => 'number',
        'title' => 'string',
        'volume' => 'string',
        'year' => 'number',
        'start_page' => 'string',
        'end_page' => 'string',
        'language' => 'string',
        'corresponding_author' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Reference->has_a(
    source_id => 'Gramene::CDBI::Literature::Source'
);

Gramene::CDBI::Literature::Reference->has_many( 
    'authors',
    'Gramene::CDBI::Literature::Author',
    'reference_id' 
);

Gramene::CDBI::Literature::Reference->has_many( 
    'reference_abstracts',
    'Gramene::CDBI::Literature::ReferenceAbstract',
    'reference_id' 
);

Gramene::CDBI::Literature::Reference->has_many( 
    'reference_comments',
    'Gramene::CDBI::Literature::ReferenceComment',
    'reference_id' 
);

Gramene::CDBI::Literature::Reference->has_many( 
    'reference_extras',
    'Gramene::CDBI::Literature::ReferenceExtra',
    'reference_id' 
);

Gramene::CDBI::Literature::Reference->has_many( 
    'reference_urls',
    'Gramene::CDBI::Literature::ReferenceUrl',
    'reference_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            author => [ 'reference_id', 'reference_id' ],
            reference_abstract => [ 'reference_id', 'reference_id' ],
            reference_comment => [ 'reference_id', 'reference_id' ],
            reference_extra => [ 'reference_id', 'reference_id' ],
            reference_url => [ 'reference_id', 'reference_id' ],
            source => [ 'source_id', 'source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::SourceSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::SourceSynonym->table('source_synonym');


sub source {
    my $self = shift;
    return $self->source_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::SourceSynonym->columns( 
    All => qw/ 
        source_synonym_id
        source_id
        source_synonym 
    /, @extras
);

Gramene::CDBI::Literature::SourceSynonym->columns( 
    Essential => qw/ 
        source_synonym_id
        source_id
        source_synonym 
    /
);

Gramene::CDBI::Literature::SourceSynonym->columns( 
    Ordered => qw/ 
        source_synonym_id
        source_id
        source_synonym 
    /
);

sub to_string {
    my $self = shift;
    return $self->source_synonym || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'source_synonym_id' => 'number',
        'source_id' => 'number',
        'source_synonym' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::SourceSynonym->has_a(
    source_id => 'Gramene::CDBI::Literature::Source'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            source => [ 'source_id', 'source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::SourceUrl; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::SourceUrl->table('source_url');


sub source {
    my $self = shift;
    return $self->source_id;
}
    
our @extras = ();
Gramene::CDBI::Literature::SourceUrl->columns( 
    All => qw/ 
        source_url_id
        source_id
        source_url
        url_label 
    /, @extras
);

Gramene::CDBI::Literature::SourceUrl->columns( 
    Essential => qw/ 
        source_url_id
        source_id
        source_url
        url_label 
    /
);

Gramene::CDBI::Literature::SourceUrl->columns( 
    Ordered => qw/ 
        source_url_id
        source_id
        source_url
        url_label 
    /
);

sub to_string {
    my $self = shift;
    return $self->source_url || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'source_url_id' => 'number',
        'source_id' => 'number',
        'source_url' => 'string',
        'url_label' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::SourceUrl->has_a(
    source_id => 'Gramene::CDBI::Literature::Source'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            source => [ 'source_id', 'source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Literature::Source; {

BEGIN { our @ISA = qw( Gramene::CDBI::Literature ); };
# use base qw( Gramene::CDBI::Literature );

Gramene::CDBI::Literature::Source->table('source');

our @extras = ();
Gramene::CDBI::Literature::Source->columns( 
    All => qw/ 
        source_id
        source_name
        ISSN 
    /, @extras
);

Gramene::CDBI::Literature::Source->columns( 
    Essential => qw/ 
        source_id
        source_name
        ISSN 
    /
);

Gramene::CDBI::Literature::Source->columns( 
    Ordered => qw/ 
        source_id
        source_name
        ISSN 
    /
);

sub to_string {
    my $self = shift;
    return $self->source_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'source_id' => 'number',
        'source_name' => 'string',
        'ISSN' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Literature::Source->has_many( 
    'references',
    'Gramene::CDBI::Literature::Reference',
    'source_id' 
);

Gramene::CDBI::Literature::Source->has_many( 
    'source_synonyms',
    'Gramene::CDBI::Literature::SourceSynonym',
    'source_id' 
);

Gramene::CDBI::Literature::Source->has_many( 
    'source_urls',
    'Gramene::CDBI::Literature::SourceUrl',
    'source_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            reference => [ 'source_id', 'source_id' ],
            source_synonym => [ 'source_id', 'source_id' ],
            source_url => [ 'source_id', 'source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Literature - Gramene::CDBI-base classes for Literature

=head1 VERSION

This documentation refers to version $Revision: 1.8 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Literature;

  my @data   = Gramene::CDBI::Literature::Source->retrieve_all;
  my $object = Gramene::CDBI::Literature::Source->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Literature 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'literature'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Literature::Author

Corresponds to the table "author."  

Stringifies (C<to_name>) to "authorship_position".

Fields:

=over 4

=item * author_id

=item * contributor_id

=item * reference_id

=item * authorship_position

=back


Has A:

=over 4

=item * contributor => Gramene::CDBI::Literature::Contributor

=item * reference => Gramene::CDBI::Literature::Reference

=back



=head2 Gramene::CDBI::Literature::Contributor

Corresponds to the table "contributor."  

Stringifies (C<to_name>) to "contributor_name".

Fields:

=over 4

=item * contributor_id

=item * contributor_name

=item * contributor_email

=item * contributor_organization

=item * contributor_address

=item * contributor_phone

=item * contributor_fax

=back



Has Many:

=over 4

=item * authors => Gramene::CDBI::Literature::Author

=item * reference_comments => Gramene::CDBI::Literature::ReferenceComment

=back


=head2 Gramene::CDBI::Literature::DataBase

Corresponds to the table "data_base."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * data_base_id

=item * name

=item * fullname

=item * datatype

=item * generic_url

=item * url_syntax

=back




=head2 Gramene::CDBI::Literature::Dbxref

Corresponds to the table "dbxref."  

Stringifies (C<to_name>) to "xref_key".

Fields:

=over 4

=item * dbxref_id

=item * xref_key

=item * xref_keytype

=item * xref_dbname

=item * xref_desc

=back



Has Many:

=over 4

=item * objectxrefs => Gramene::CDBI::Literature::Objectxref

=back


=head2 Gramene::CDBI::Literature::Objectxref

Corresponds to the table "objectxref."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * objectxref_id

=item * table_name

=item * row_id

=item * dbxref_id

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Literature::Dbxref

=back



=head2 Gramene::CDBI::Literature::Reference

Corresponds to the table "reference."  

Stringifies (C<to_name>) to "title".

Fields:

=over 4

=item * reference_id

=item * source_id

=item * title

=item * volume

=item * year

=item * start_page

=item * end_page

=item * language

=item * corresponding_author

=back


Has A:

=over 4

=item * source => Gramene::CDBI::Literature::Source

=back


Has Many:

=over 4

=item * authors => Gramene::CDBI::Literature::Author

=item * reference_abstracts => Gramene::CDBI::Literature::ReferenceAbstract

=item * reference_comments => Gramene::CDBI::Literature::ReferenceComment

=item * reference_extras => Gramene::CDBI::Literature::ReferenceExtra

=item * reference_urls => Gramene::CDBI::Literature::ReferenceUrl

=back


=head2 Gramene::CDBI::Literature::ReferenceAbstract

Corresponds to the table "reference_abstract."  

Stringifies (C<to_name>) to "abstract_part_a".

Fields:

=over 4

=item * reference_id

=item * abstract_part_a

=item * abstract_part_b

=back


Has A:

=over 4

=item * reference => Gramene::CDBI::Literature::Reference

=back



=head2 Gramene::CDBI::Literature::ReferenceComment

Corresponds to the table "reference_comment."  

Stringifies (C<to_name>) to "reference_comment".

Fields:

=over 4

=item * reference_id

=item * reference_comment

=item * contributor_id

=back


Has A:

=over 4

=item * contributor => Gramene::CDBI::Literature::Contributor

=item * reference => Gramene::CDBI::Literature::Reference

=back



=head2 Gramene::CDBI::Literature::ReferenceExtra

Corresponds to the table "reference_extra."  

Stringifies (C<to_name>) to "title".

Fields:

=over 4

=item * reference_id

=item * title

=item * source

=item * author

=item * year

=item * abstract_part_a

=item * abstract_part_b

=back


Has A:

=over 4

=item * reference => Gramene::CDBI::Literature::Reference

=back



=head2 Gramene::CDBI::Literature::ReferenceUrl

Corresponds to the table "reference_url."  

Stringifies (C<to_name>) to "reference_url".

Fields:

=over 4

=item * reference_url_id

=item * reference_id

=item * reference_url

=item * url_label

=back


Has A:

=over 4

=item * reference => Gramene::CDBI::Literature::Reference

=back



=head2 Gramene::CDBI::Literature::Source

Corresponds to the table "source."  

Stringifies (C<to_name>) to "source_name".

Fields:

=over 4

=item * source_id

=item * source_name

=item * ISSN

=back



Has Many:

=over 4

=item * references => Gramene::CDBI::Literature::Reference

=item * source_synonyms => Gramene::CDBI::Literature::SourceSynonym

=item * source_urls => Gramene::CDBI::Literature::SourceUrl

=back


=head2 Gramene::CDBI::Literature::SourceSynonym

Corresponds to the table "source_synonym."  

Stringifies (C<to_name>) to "source_synonym".

Fields:

=over 4

=item * source_synonym_id

=item * source_id

=item * source_synonym

=back


Has A:

=over 4

=item * source => Gramene::CDBI::Literature::Source

=back



=head2 Gramene::CDBI::Literature::SourceUrl

Corresponds to the table "source_url."  

Stringifies (C<to_name>) to "source_url".

Fields:

=over 4

=item * source_url_id

=item * source_id

=item * source_url

=item * url_label

=back


Has A:

=over 4

=item * source => Gramene::CDBI::Literature::Source

=back



=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:22:11 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
