package Gramene::CDBI::GrameneSearch;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'gramene_search';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        doc_search
        module_search

    /
}

# ------------------------------------------------

package Gramene::CDBI::GrameneSearch::DocSearch; {

BEGIN { our @ISA = qw( Gramene::CDBI::GrameneSearch ); };
# use base qw( Gramene::CDBI::GrameneSearch );

Gramene::CDBI::GrameneSearch::DocSearch->table('doc_search');

our @extras = ();
Gramene::CDBI::GrameneSearch::DocSearch->columns( 
    All => qw/ 
        path
        title
        contents 
    /, @extras
);

Gramene::CDBI::GrameneSearch::DocSearch->columns( 
    Essential => qw/ 
        path
        title
        contents 
    /
);

Gramene::CDBI::GrameneSearch::DocSearch->columns( 
    Ordered => qw/ 
        path
        title
        contents 
    /
);

sub to_string {
    my $self = shift;
    return $self->path || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'path' => 'string',
        'title' => 'string',
        'contents' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::GrameneSearch::ModuleSearch; {

BEGIN { our @ISA = qw( Gramene::CDBI::GrameneSearch ); };
# use base qw( Gramene::CDBI::GrameneSearch );

Gramene::CDBI::GrameneSearch::ModuleSearch->table('module_search');

our @extras = ();
Gramene::CDBI::GrameneSearch::ModuleSearch->columns( 
    All => qw/ 
        module_name
        table_name
        record_id
        record_text 
    /, @extras
);

Gramene::CDBI::GrameneSearch::ModuleSearch->columns( 
    Essential => qw/ 
        module_name
        table_name
        record_id
        record_text 
    /
);

Gramene::CDBI::GrameneSearch::ModuleSearch->columns( 
    Ordered => qw/ 
        module_name
        table_name
        record_id
        record_text 
    /
);

sub to_string {
    my $self = shift;
    return $self->module_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'module_name' => 'string',
        'table_name' => 'string',
        'record_id' => 'number',
        'record_text' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::GrameneSearch - Gramene::CDBI-base classes for GrameneSearch

=head1 VERSION

This documentation refers to version $Revision: 1.1 $.

=head1 SYNOPSIS

  use Gramene::CDBI::GrameneSearch;

  my @data   = Gramene::CDBI::GrameneSearch::DocSearch->retrieve_all;
  my $object = Gramene::CDBI::GrameneSearch::DocSearch->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the GrameneSearch 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'gramene_search'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::GrameneSearch::DocSearch

Corresponds to the table "doc_search."  

Stringifies (C<to_name>) to "path".

Fields:

=over 4

=item * path

=item * title

=item * contents

=back




=head2 Gramene::CDBI::GrameneSearch::ModuleSearch

Corresponds to the table "module_search."  

Stringifies (C<to_name>) to "module_name".

Fields:

=over 4

=item * module_name

=item * table_name

=item * record_id

=item * record_text

=back




=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 14:54:30 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
